/*
 * ex21.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <f2.h>
#if !defined HAVE_LIBx1f4i0
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <sf.h>
#include <types.h>

#define __DETAIL_CALL__			0

#define __COPY_DATA__			1
/*
 * keep it on to test the expressing copying
 */

#define MAKE_SINGLE(a, b)		a

#define e21a_fix \
    sizeof(e21a_set) / sizeof(struct x1f4_linetext_type)

#define I_BILL(t)			(*((X1f4_E4_C_BILL *) (t)))
#define I_MODE(t)			(*((X1f4_E4_C_MODE *) (t)))
#define I_REAL(t)			(*((X1f4_E4_C_REAL *) (t)))
#define I_TEXT(t)			(*((X1f4_E4_C_TEXT *) (t)))

#define l_BILL(e, output) \
    {									      \
	X1f4_E4_C_BILL *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

#define l_REAL(e, output) \
    {									      \
	X1f4_E4_C_REAL *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

#define l_TEXT(e, output) \
    {									      \
	X1f4_E4_C_TEXT *l;						      \
									      \
	l = (output);							      \
	*l = (X1f4_E4_C_TEXT) (e);					      \
    }

#define l_USER(e, output) \
    {									      \
	X1f4_E4_C_USER *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
    void **state;
} context_type;

typedef struct line_type {
    struct line_type *line_data;
} line_type;

static int beta_list(void *, void *, void **);
static int case_bits(void *, void *, void **);
static int case_fail(void *, void *, void **);
static int case_mask(void *, void *, void **);
static int case_pass(void *, void *, void **);
static int case_swap(void *, void *, void **);
static int case_text(void *, void *, void **);
static int case_type(void *, void *, void **);
static int deck_list(void *, void *, void **);
#if __DETAIL_CALL__
static int detail_call(const struct x1f4_function_type *);
#endif				/* __DETAIL_CALL__ */
static int fix_text(const void *, const struct x1f4_function_type *, void **);
static int high_bits(void *, void *, void **);
static int high_fail(void *, void *, void **);
static int high_mask(void *, void *, void **);
static int high_pass(void *, void *, void **);
static int high_swap(void *, void *, void **);
static int high_text(void *, void *, void **);
static int high_type(void *, void *, void **);
static int link_expression(void *, void *, void *);
static int merge_sets(struct x1f4_function_type **);
static int pick_bits(void *, void *, void **);
static int pick_fail(void *, void *, void **);
static int pick_mask(void *, void *, void **);
static int pick_pass(void *, void *, void **);
static int pick_swap(void *, void *, void **);
static int pick_text(void *, void *, void **);
static int pick_type(void *, void *, void **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);
static int size_list(void *, void *, void **);
static int test_cast(void *, const char *, unsigned);
static int type_ex21(void *, void *);

static void print_functions(void);
static void usage(void);

static const int c_____m_m[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE,
    X1f4_E4_MODE,
    X1f4_E4_POST_XSET,
    0
/* *INDENT-ON* */
}, c_____s_s[] = {
/* *INDENT-OFF* */
    X1f4_E4_SLIP,
    X1f4_E4_SLIP,
    X1f4_E4_POST_XSET,
    X1f4_E4_POST_XSET
/* *INDENT-ON* */
}, c_____s_t[] = {
/* *INDENT-OFF* */
    X1f4_E4_SLIP,
    X1f4_E4_TEXT,
    X1f4_E4_POST_XSET,
    0
/* *INDENT-ON* */
};
static struct x1f4_linetext_type e21e_set[] = {
/* *INDENT-OFF* */
    {	{	"/$/",			high_bits,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								},
    {	{	"/%/",			deck_list,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								},
    {	{	"/^/",			high_pass,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								},
    {	{	"/&/",			high_mask,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								},
    {	{	"/?/",			high_type,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								},
    {	{	"/:/",			high_swap,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								},
    {	{	"/*/",			high_fail,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								},
    {	{	"/@/",			high_text,
		0,			c_____m_m,	1,
		0,					3	},
	NULL								}
/* *INDENT-ON* */
};
static const struct x1f4_2ndscope_type c10xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 0,		NULL				}
/* *INDENT-ON* */
}, c11xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 1,		NULL				}
/* *INDENT-ON* */
}, c12xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 2,		NULL				}
/* *INDENT-ON* */
}, c13xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 3,		NULL				}
/* *INDENT-ON* */
}, c14xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 4,		NULL				}
/* *INDENT-ON* */
}, c15xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 5,		NULL				}
/* *INDENT-ON* */
}, c16xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 6,		NULL				}
/* *INDENT-ON* */
}, c17xx_2nd[] = {
/* *INDENT-OFF* */
    {	X1f4_E4_CALL_LINK,
	e21e_set + 7,		NULL				}
/* *INDENT-ON* */
};
static struct x1f4_linetext_type e21a_set[] = {
/* *INDENT-OFF* */
    {	{	"bits",			case_bits,
		X1f4_E4_MODE,		c_____m_m,	1,
		X1f4_E4_SIDE_LIST | X1f4_E4_TEXT_LINK,	4	},
	NULL								},
    {	{	"count",		beta_list,
		X1f4_E4_MODE,		NULL,		0,
		X1f4_E4_SIDE_LIST | X1f4_E4_TEXT_LINK,	5	},
	NULL								},
    {	{	"dispense",		case_pass,
		X1f4_E4_MODE,		NULL,		0,
		X1f4_E4_LINK_PASS | X1f4_E4_SIDE_LIST | X1f4_E4_TEXT_LINK,
							8	},
	NULL								},
    {	{	"mask",			case_mask,
		X1f4_E4_MODE,		c_____m_m,	2,
		X1f4_E4_KEEP_CALL | X1f4_E4_POST_TYPE | X1f4_E4_SIDE_LIST
		| X1f4_E4_TEXT_LINK,
							4	},
	NULL								},
    {	{	"type",			case_type,
		X1f4_E4_TEXT,		c_____s_s,	1,
		X1f4_E4_SLIP_LIST | X1f4_E4_TEXT_LINK,	4	},
	NULL								},
    {	{	"swap",			case_swap,
		X1f4_E4_MODE,		c_____s_s,	2,
		X1f4_E4_POST_TYPE | X1f4_E4_SLIP_LIST | X1f4_E4_TEXT_LINK,
							4	},
	NULL								},
    {	{	"stats",		case_fail,
		X1f4_E4_MODE,		c_____s_s,	1,
		X1f4_E4_POST_TYPE | X1f4_E4_SIDE_LIST | X1f4_E4_SLIP_LIST
		| X1f4_E4_TEXT_LINK,
							5	},
	NULL								},
    {	{	"noise",		case_text,
		X1f4_E4_MODE,		c_____s_t,	2,
		X1f4_E4_POST_TYPE | X1f4_E4_SLIP_LIST | X1f4_E4_TEXT_LINK,
							5	},
	NULL								},
    {	{	"z_bits",		high_bits,
		X1f4_E4_MODE,		NULL,		0,
		0,					6	},
	NULL								},
    {	{	"z_count",		deck_list,
		X1f4_E4_MODE,		NULL,		0,
		0,					7	},
	NULL								},
    {	{	"z_dispense",		high_pass,
		X1f4_E4_MODE,		NULL,		0,
		0,					10	},
	NULL								},
    {	{	"z_mask",		high_mask,
		X1f4_E4_MODE,		NULL,		0,
		0,					6	},
	NULL								},
    {	{	"z_type",		high_type,
		X1f4_E4_TEXT,		NULL,		0,
		0,					6	},
	NULL								},
    {	{	"z_swap",		high_swap,
		X1f4_E4_MODE,		NULL,		0,
		0,					6	},
	NULL								},
    {	{	"z_stats",		high_fail,
		X1f4_E4_MODE,		NULL,		0,
		0,					7	},
	NULL								},
    {	{	"z_noise",		high_text,
		X1f4_E4_MODE,		NULL,		0,
		0,					7	},
	NULL								},
    {	{	NULL,			NULL,
		0,			NULL,		0,
		0,					1	},
	NULL								}
/* *INDENT-ON* */
};
static struct x1f4_operator_type e21d_set[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("$", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c10xx_2nd			},
    {	MAKE_SINGLE("%", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c11xx_2nd			},
    {	MAKE_SINGLE("^", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c12xx_2nd			},
    {	MAKE_SINGLE("&", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c13xx_2nd			},
    {	MAKE_SINGLE("?", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c14xx_2nd			},
    {	MAKE_SINGLE(":", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c15xx_2nd			},
    {	MAKE_SINGLE("*", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c16xx_2nd			},
    {	MAKE_SINGLE("@", " "),  NULL,		0210,
	X1f4_E4_TEXT,		c_____m_m,
	X1f4_E4_E2ND_LINK,			1,
	NULL,			c17xx_2nd			}
/* *INDENT-ON* */
}, *ex21_e4_1_arithmetics[] = {
/* *INDENT-OFF* */
    e21d_set + 0,
    e21d_set + 1,
    e21d_set + 2,
    e21d_set + 3,
    e21d_set + 4,
    e21d_set + 5,
    e21d_set + 6,
    e21d_set + 7
/* *INDENT-ON* */
};
static struct x1f4_variable_type e21b_set[] = {
/* *INDENT-OFF* */
    {	"i_bits",		0,
	0,					6		},
    {	"i_count",		0,
	0,					7		},
    {	"i_dispense",		0,
	0,					10		},
    {	"i_mask",		0,
	0,					6		},
    {	"i_type",		0,
	0,					6		},
    {	"i_swap",		0,
	0,					6		},
    {	"i_stats",		0,
	0,					7		},
    {	"i_noise",		0,
	0,					7		},
    {	NULL,			0,
	0,					1		}
/* *INDENT-ON* */
};
static void *e4fine = NULL;
static const void *e21c_set[] = {
/* *INDENT-OFF* */
    e21a_set + 0,
    e21a_set + 1,
    e21a_set + 2,
    e21a_set + 3,
    e21a_set + 4,
    e21a_set + 5,
    e21a_set + 6,
    e21a_set + 7
/* *INDENT-ON* */
};

static int
beta_list(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = size_list;

    *text = function_data;

    return 0;
}


static int
case_bits(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = pick_bits;

    *text = function_data;

    return 0;
}


static int
case_fail(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = pick_fail;

    *text = function_data;

    return 0;
}


static int
case_mask(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = pick_mask;

    *text = function_data;

    return 0;
}


static int
case_pass(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = pick_pass;

    *text = function_data;

    return 0;
}


static int
case_swap(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = pick_swap;

    *text = function_data;

    return 0;
}


static int
case_text(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = pick_text;

    *text = function_data;

    return 0;
}


static int
case_type(void *context, void *data, void **text)
{
    struct x1f4_function_type *function_data;

    function_data = data;

    function_data->function = pick_type;

    *text = function_data;

    return 0;
}


static int
deck_list(void *text, void *output, void **input)
{
    l_USER(e21a_set + 1, output);

    return 0;
}


#if __DETAIL_CALL__
static int
detail_call(const struct x1f4_function_type *function_data)
{
    x1f4_print_hfunction(stderr, function_data, "called ");

    return 0;
}
#endif				/* __DETAIL_CALL__ */


static int
fix_text(const void *context, const struct x1f4_function_type *function_data,
	 void **link)
{
    return 0;
}


static int
high_bits(void *text, void *output, void **input)
{
    l_USER(e21a_set + 0, output);

    return 0;
}


static int
high_fail(void *text, void *output, void **input)
{
    l_USER(e21a_set + 6, output);

    return 0;
}


static int
high_mask(void *text, void *output, void **input)
{
    l_USER(e21a_set + 3, output);

    return 0;
}


static int
high_pass(void *text, void *output, void **input)
{
    l_USER(e21a_set + 2, output);

    return 0;
}


static int
high_swap(void *text, void *output, void **input)
{
    l_USER(e21a_set + 5, output);

    return 0;
}


static int
high_text(void *text, void *output, void **input)
{
    l_USER(e21a_set + 7, output);

    return 0;
}


static int
high_type(void *text, void *output, void **input)
{
    l_USER(e21a_set + 4, output);

    return 0;
}


static int
link_expression(void *expression, void *context, void *output)
{
    int status;
    void *slip;

    status = x1f4_head_expression(expression, &slip);
    if (status) {
    } else {
	int excess, slipping;

	while (1) {
	    struct x1f4_transfer_type transfer;

	    status = x1f4_slip_expression
		(expression, slip, 0, &slipping, output, &transfer);
	    if (status) {
		break;
	    } else {
		if (slipping) {
		    if (transfer.function_data->flags & X1f4_E4_TEXT_LINK) {
			status = transfer.function_data->function
			    (transfer.context, transfer.output,
			     transfer.input);
		    } else {
			status = transfer.function_data->function
			    (NULL, transfer.output, transfer.input);
		    }

		    if (status) {
			break;
		    }
		} else {
		    break;
		}
	    }
	}

	excess = x1f4_tail_expression(expression, &slip);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    function_data = (struct x1f4_function_type *)
	malloc((count + e21a_fix) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	const struct x1f4_linetext_type *linetext_data;
	unsigned i;

	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	linetext_data = e21a_set;
	i = e21a_fix;
	for (; i; i--) {
	    *function_data++ = linetext_data->function;
	    linetext_data++;
	}
    }

    return status;
}


static int
pick_bits(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE call = 0, mode;
    const int *args;
    struct x1f4_function_type *function_data;
    unsigned count;

    function_data = context;

    mode = I_MODE(input[0]);

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    args = function_data->args;

    count = function_data->count;
    for (--count; count; count--) {
	int type;

	input++;

	type = *++args;
	if (type == X1f4_E4_MODE) {
	    X1f4_E4_C_MODE pick;

	    pick = I_MODE(input[0]);
	    if (pick < 0) {
	    } else {
		if (sizeof(X1f4_E4_C_MODE) << 3 < pick) {
		} else {
		    call |= mode & (X1f4_E4_C_MODE) 1 << pick;
		}
	    }
	} else {
	    if (type == X1f4_E4_BILL) {
		X1f4_E4_C_BILL pick;

		pick = I_BILL(input[0]);
		if (sizeof(X1f4_E4_C_MODE) << 3 < pick) {
		} else {
		    call |= mode & (X1f4_E4_C_MODE) 1 << pick;
		}
	    }
	}
    }

    l_MODE(call, output);

    return 0;
}


static int
pick_fail(void *context, void *output, void **input)
{
    const int *args;
    int call;
    struct x1f4_function_type *function_data;

    function_data = context;

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    args = function_data->args;

    if (*args == X1f4_E4_MODE) {
	call = 0;
    } else {
	if (*args == X1f4_E4_REAL) {
	    call = 1;
	} else {
	    call = -1;
	}
    }

    if (call ^ -1) {
	X1f4_E4_C_MODE mode = 0;
	X1f4_E4_C_REAL real = 1;
	unsigned count;
	void **shift;

	shift = input;

	count = function_data->count - 1;
	for (; count; count--) {
	    int type;

	    shift++;

	    type = *++args;
	    if (type == X1f4_E4_MODE) {
		X1f4_E4_C_MODE pick;

		pick = I_MODE(shift[0]);
		if (call) {
		    real *= pick;
		} else {
		    mode += pick;
		}
	    } else {
		if (type == X1f4_E4_BILL) {
		    X1f4_E4_C_BILL pick;

		    pick = I_BILL(shift[0]);
		    if (call) {
			real *= pick;
		    } else {
			mode += pick;
		    }
		} else {
		    if (type == X1f4_E4_REAL) {
			X1f4_E4_C_REAL pick;

			pick = I_REAL(shift[0]);
			if (call) {
			    real *= pick;
			} else {
			    mode += pick;
			}
		    }
		}
	    }
	}

	if (call) {
	    l_REAL(real, input[0]);
	} else {
	    l_MODE(mode, input[0]);
	}
    }

    l_MODE(call, output);

    return 0;
}


static int
pick_mask(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE call = 0, mode;
    const int *args;
    struct x1f4_function_type *function_data;
    unsigned count;
    void **shift;

    function_data = context;

    mode = I_MODE(input[1]);

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    args = function_data->args;

    shift = input + 1;

    count = function_data->count - 2;
    for (; count; count--) {
	int type;

	shift++;

	type = *++args;
	if (type == X1f4_E4_MODE) {
	    X1f4_E4_C_MODE pick;

	    pick = I_MODE(shift[0]);
	    if (pick < 0) {
	    } else {
		if (sizeof(X1f4_E4_C_MODE) << 3 < pick) {
		} else {
		    call |= mode & (X1f4_E4_C_MODE) 1 << pick;
		}
	    }
	} else {
	    if (type == X1f4_E4_BILL) {
		X1f4_E4_C_BILL pick;

		pick = I_BILL(shift[0]);
		if (sizeof(X1f4_E4_C_MODE) << 3 < pick) {
		} else {
		    call |= mode & (X1f4_E4_C_MODE) 1 << pick;
		}
	    }
	}
    }

    l_MODE(call, output);
    l_MODE(call, input[0]);

    return 0;
}


static int
pick_pass(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE pass = 0;
    struct x1f4_function_type *function_data;

    function_data = context;

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    if (function_data->flags & X1f4_E4_POST_TYPE) {
	const int *args;
	unsigned count, i;
	void **shift;

	args = function_data->args;

	shift = input;

	count = function_data->count;
	i = count;
	for (; i; i--) {
	    if (args[count] & X1f4_E4_POST_XSET) {
		int type;

		pass++;

		type = *args;
		if (type == X1f4_E4_MODE) {
		    l_MODE(I_MODE(shift[0]) >> 1, shift[0]);
		} else {
		    if (type == X1f4_E4_BILL) {
			l_BILL(I_BILL(shift[0]) >> 1, shift[0]);
		    }
		}
	    }

	    args++;

	    shift++;
	}
    }

    l_MODE(pass, output);

    return 0;
}


static int
pick_swap(void *context, void *output, void **input)
{
    int swap = 0, type;
    struct x1f4_function_type *function_data;

    function_data = context;

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    type = *function_data->args;

    if (type != *(function_data->args + 1)) {
	swap = 1;
    } else {
	if (type == X1f4_E4_BILL) {
	    X1f4_E4_C_BILL bill;

	    bill = I_BILL(input[0]);
	    l_BILL(I_BILL(input[1]), input[0]);
	    l_BILL(bill, input[1]);
	} else {
	    if (type == X1f4_E4_MODE) {
		X1f4_E4_C_MODE mode;

		mode = I_MODE(input[0]);
		l_MODE(I_MODE(input[1]), input[0]);
		l_MODE(mode, input[1]);
	    } else {
		if (type == X1f4_E4_REAL) {
		    X1f4_E4_C_REAL real;

		    real = I_REAL(input[0]);
		    l_REAL(I_REAL(input[1]), input[0]);
		    l_REAL(real, input[1]);
		} else {
		    if (type == X1f4_E4_TEXT) {
			X1f4_E4_C_TEXT text;

			text = I_TEXT(input[0]);
			l_TEXT(I_TEXT(input[1]), input[0]);
			l_TEXT(text, input[1]);
		    } else {
			swap = 1;
		    }
		}
	    }
	}
    }

    l_MODE(swap, output);

    return 0;
}


static int
pick_text(void *context, void *output, void **input)
{
    struct x1f4_function_type *function_data;

    function_data = context;

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    if (*function_data->args == X1f4_E4_MODE) {
	l_MODE(strlen(I_TEXT(input[1])), input[0]);
    } else {
	if (*function_data->args == X1f4_E4_TEXT) {
	    l_TEXT(I_TEXT(input[1]), input[0]);
	} else {
	}
    }

    l_MODE(0, output);

    return 0;
}


static int
pick_type(void *context, void *output, void **input)
{
    const char *text;
    struct x1f4_function_type *function_data;

    function_data = context;

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    switch (*function_data->args) {
    case X1f4_E4_BILL:
	text = "cardinal";
	break;
    case X1f4_E4_MODE:
	text = "integer";
	break;
    case X1f4_E4_REAL:
	text = "real";
	break;
    case X1f4_E4_TEXT:
	text = "text";
	break;
    default:
	text = "other";
    }

    l_TEXT(text, output);

    return 0;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }
    if (status && e4fine) {
	variable_data = e21b_set;
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = e21c_set + (variable_data - e21b_set);
	    *variable = variable_data;
	} else {
	}
    }

    return status;
}


static int
size_list(void *context, void *output, void **input)
{
    struct x1f4_function_type *function_data;

    function_data = context;

#if __DETAIL_CALL__
    detail_call(function_data);
#endif				/* __DETAIL_CALL__ */

    l_MODE(function_data->count, output);

    return 0;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


static int
type_ex21(void *self, void *e4fine)
{
    int status = 0;
    struct x1f4_linetext_type *linetext_data, *linetext_slip;
    struct x1f4_operator_type *operator_data;
    struct x1f4_variable_type *variable_data;
    unsigned count, i;

    count = (e21a_fix - 1) >> 1;

    variable_data = e21b_set;
    operator_data = e21d_set;
    linetext_slip = e21e_set;
    linetext_data = e21a_set;
    i = count;
    for (; i; i--) {
	int type;

	status = x1f4_type_e4fine(e4fine, &type, linetext_data);
	if (status) {
	    break;
	} else {
	    linetext_data[count].function.type = type;
	    linetext_slip->function.type = type;
	    operator_data->type = type;
	    variable_data->type = type;
	    linetext_data++;
	    linetext_slip++;
	    operator_data++;
	    variable_data++;
	}
    }

    return status;
}


static void
print_functions(void)
{
    struct x1f4_linetext_type *linetext_data;

    linetext_data = e21a_set;
    while (linetext_data->function.flags
	   & (X1f4_E4_SIDE_LIST | X1f4_E4_SLIP_LIST)) {
	x1f4_print_function(stdout, &linetext_data->function);
	linetext_data++;
    }
}


static void
usage(void)
{
    puts("Usage: ex21 [OPTIONS] EXPRESSION [TYPE NAME VALUE]\n\
Demonstrate variadic functions, evaluate EXPRESSION.\n\
\n\
Options:\n\
  -M, --stat-storage		stat expression memory storage requirements\n\
  -P, --print			print expression\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
  -s, --sequence		sequence expression evaluation\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --list			list available functions and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_memory = 0, do_optimize = 0, do_print = 0, do_sequence = 0,
	do_storage = 0, list_functions = 0, status = 1;
    unsigned precision = 3, print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "list",	    0x00,   NULL,   0x01    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "sequence",	    0x00,   NULL,   0x73    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "MPmop:s", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 021:
		print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'M':
		do_storage = 1;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 's':
		do_sequence = 1;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (list_functions | do_print) {
	    precision_specs = NULL;
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (list_functions == 1) {
	x1f4_print_functions(stdout, x1f4_e4_defaults);
	print_functions();
    } else if ((argc - optind) % 3 != 1) {
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
#if defined HAVE_LIBx1f4l0
	    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
	    struct x1f4_eerecord_type eerecord;
	    struct x1f4_variable_type *variable_data;
	    unsigned flags = X1f4_E4_DDACCESS;
	    void **state = NULL, *x1f4_expression;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    expression = argv[optind];
	    optind++;

	    status = libx1f4i0_init_list
		(argv[0], argc, argv, optind, (void *) &variable_data, &state);
	    if (status) {
		break;
	    }

#if defined HAVE_LIBx1f4l0
	    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		struct x1f4_e4fine_type pick_fine;

		if (1) {
		    libx1f4i0_line_valist
			(valist, do_memory, do_storage,
			 &pick_fine.resource_set.free,
			 &pick_fine.resource_set.link,
			 &pick_fine.resource_set.mode,
			 &pick_fine.resource_set.data, &dlist);
		}

		x1f4_init_e4fine(&e4fine, X1f4_E4FINE_RESOURCE, &pick_fine);
	    } else {
#endif				/* HAVE_LIBx1f4l0 */
		x1f4_init_e4fine(&e4fine, 0, NULL);
#if defined HAVE_LIBx1f4l0
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    if (e4fine) {
		if (type_ex21(argv[0], e4fine)) {
		    x1f4_fini_e4fine(&e4fine);
		    if (1) {
			e4fine = NULL;
		    }
		}
	    }

	    {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
		    struct context_type context;
		    struct x1f4_attributes_type attributes;
		    const struct x1f4_operator_type *const *operator1s;

		    context.state = state;
		    context.variable_data = variable_data;

		    attributes.bcollect_set.eerecord_data = &eerecord;
		    attributes.function_set.fix = fix_text;
		    attributes.function_set.get = libx1f4i0_select_function;
		    attributes.function_set.context = function_data;
		    x1f4_llink_operator1s(&operator1s);
		    libx1f4i0_join_sets
			(&attributes.operator1s, operator1s,
			 (void *) ex21_e4_1_arithmetics, 8);
		    x1f4_llink_operator2s(&attributes.operator2s);
		    attributes.terminator = 0;
		    attributes.transfer_set.fine = e4fine;
		    attributes.variable_set.get = select_variable;
		    attributes.variable_set.context = &context;

		    flags |= X1f4_E4_BCOLLECT | X1f4_E4_ESTORAGE;

		    if (e4fine) {
			flags |= X1f4_E4_TRANSFER;
		    }

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			flags |= X1f4_E4_RESOURCE;
			libx1f4i0_line_valist
			    (valist, do_memory, do_storage,
			     &attributes.resource_set.free,
			     &attributes.resource_set.link,
			     &attributes.resource_set.mode,
			     &attributes.resource_set.context, &dlist);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    {
			status = x1f4_init_expression
			    (&x1f4_expression, expression, flags, &attributes);
#if __COPY_DATA__
			if (status) {
			} else {
			    void *expression;

			    status = x1f4_copy_expression
				(&expression, x1f4_expression);
			    if (status) {
			    } else {
				x1f4_fini_expression(&x1f4_expression);
				x1f4_expression = expression;
			    }
			}
#endif				/* __COPY_DATA__ */
		    }

		    free(function_data);
		}
	    }
	    if (status) {
		if (status == X1f4_E4_ALLOC_ERROR) {
		    perror(argv[0]);
		} else {
		    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
			    expression);
		    fprintf(stderr, "%s: ", argv[0]);
		    x1f4_stat_expression(stderr, test_cast, &eerecord, NULL);
		    fprintf(stderr, "\n");
		}
	    } else {
		if (do_optimize) {
		    x1f4_line_expression(x1f4_expression);
		}

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    if (do_storage) {
			unsigned size;

			x1f4_size_xalist(valist, &size);
			fprintf(stderr, " %7u\n", size);
		    }
		}
#endif				/* HAVE_LIBx1f4l0 */

		if (do_print) {
		    x1f4_print_expression
			(stdout, x1f4_expression, print_flags);
		    putchar('\n');
		} else {
		    struct lead_type output;

		    if (do_sequence) {
			status = link_expression
			    (x1f4_expression, NULL, &output);
		    } else {
			status = x1f4_link_expression
			    (x1f4_expression, &output);
		    }
		    if (status) {
			fprintf(stderr, "%s: cannot evaluate `%s'\n", argv[0],
				expression);
		    } else {
			libx1f4i0_type_data
			    (x1f4_type_expression(x1f4_expression), precision,
			     &output);
		    }
		}

		x1f4_fini_expression(&x1f4_expression);
	    }

	    if (e4fine) {
		x1f4_fini_e4fine(&e4fine);
	    }

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		libx1f4i0_fini_valist
		    (argv[0], valist, do_memory, do_storage, &status);
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    libx1f4i0_fini_list(argv[0], (void *) &variable_data);
	} while (0);
    }

    return status;
}
