/*
 * ex20.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <ckeytree.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <f2.h>
#if !defined HAVE_LIBx1f4i0
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#if !defined HAVE_LIBx1f4i0
# include <lcardinal-wx.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <line.h>
#if !defined HAVE_LIBx1f4i0
# include <lintegral-x.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <sf.h>
#include <types.h>

#define I_BILL(i)			(*((X1f4_E4_C_BILL *) (i)))
#define I_MODE(i)			(*((X1f4_E4_C_MODE *) (i)))
#define I_REAL(r)			(*((X1f4_E4_C_REAL *) (r)))
#define I_USER(t)			(*((X1f4_E4_C_USER *) (t)))
#define I_TEXT(t)			(*((X1f4_E4_C_TEXT *) (t)))

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
    void **state;
} context_type;

typedef union state_type {
    X1f4_E4_C_BILL bill;
    X1f4_E4_C_MODE mode;
    X1f4_E4_C_REAL real;
    X1f4_E4_C_USER user;
    X1f4_E4_C_TEXT text;
} state_type;

extern const struct x1f4_function_type _libx1f4i0_e4_line[];
extern const struct x1f4_operator_type *_libx1f4i0_e4_line_1[],
    *_libx1f4i0_e4_line_2[];

void *global_context;

static int lookup_type(const void *, int, const char **, unsigned *);
static int merge_sets(struct x1f4_function_type **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);
static int test_cast(void *, const char *, unsigned);

static void fprint_line(FILE *, unsigned, struct line_type *);
static void usage(void);

static const struct x1f4_ckeytree_type type_0[] = {
/* *INDENT-OFF* */
    {	"cardinal",	{	X1f4_E4_BILL	},	NULL	},
    {	"integer",	{	X1f4_E4_MODE	},	NULL	},
    {	"line",		{	EX20_E4_LINE	},	NULL	},
    {	"pair",		{	EX20_E4_PAIR	},	NULL	},
    {	"quad",		{	EX20_E4_QUAD	},	NULL	},
    {	"real",		{	X1f4_E4_REAL	},	NULL	},
    {	"text",		{	X1f4_E4_TEXT	},	NULL	},
    {	NULL,		{	0		},	NULL	}
/* *INDENT-ON* */
}, type_root[] = {
/* *INDENT-OFF* */
    {	"",		{	0		},	type_0	},
    {	NULL,		{	0		},	NULL	}
/* *INDENT-ON* */
};

static int
lookup_type(const void *none, int type, const char **name, unsigned *size)
{
    int status;

    if (type == EX20_E4_LINE) {
	*name = "line";
	*size = 4;

	status = 0;
    } else {
	if (type == EX20_E4_PAIR) {
	    *name = "pair";
	    *size = 4;

	    status = 0;
	} else {
#if 0
	    if (type == EX20_E4_QUAD) {
#endif				/* 0 */
		*name = "quad";
		*size = 4;

		status = 0;
#if 0
	    } else {
		status = -1;
	    }
#endif				/* 0 */
	}
    }

    return status;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    function_data = (struct x1f4_function_type *)
	malloc((count + 10) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, _libx1f4i0_e4_line,
	       sizeof(struct x1f4_function_type) * 10);
    }

    return status;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


static void
fprint_line(FILE *file, unsigned count, struct line_type *line_data)
{
    long *data;

    data = line_data->data;
    fprintf(file, "%ld", *data);
    --count;
    for (; count; count--) {
	data++;
	fprintf(file, " %ld", *data);
    }
}


static void
usage(void)
{
    puts("Usage: ex20 [OPTIONS] EXPRESSION [TYPE NAME VALUE...]\n\
Evaluate EXPRESSION.\n\
\n\
Options:\n\
  -M, --stat-storage		stat expression memory storage requirements\n\
  -P, --print			print expression\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_memory = 0, do_optimize = 0, do_print = 0, do_storage = 0,
	list_functions = 0, status = 1;
    unsigned precision = 3, print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "MPmop:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 021:
		print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'M':
		do_storage = 1;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (list_functions) {
	    precision_specs = NULL;
	}

	if (do_print) {
	    precision_specs = NULL;
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (argc - optind < 1) {
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
#if defined HAVE_LIBx1f4l0
	    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
	    struct miss_type miss;
	    struct x1f4_eerecord_type eerecord;
	    struct x1f4_variable_type *variable_data;
	    unsigned count, flags = X1f4_E4_DDACCESS;
	    void **state = NULL, *x1f4_expression;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    miss.line_data = NULL;

	    expression = argv[optind];
	    optind++;

	    global_context = &miss;

	    count = (argc - optind + 2) / 3;
	    if (!count) {
		variable_data = NULL;
	    } else {
		variable_data = (struct x1f4_variable_type *)
		    malloc((sizeof(struct x1f4_variable_type) + sizeof(void *)
			    + sizeof(union state_type)) * count
			   + sizeof(struct x1f4_variable_type));
		if (!variable_data) {
		    break;
		} else {
		    struct x1f4_variable_type *slide1;
		    union state_type *slide3;
		    unsigned i = 0;
		    void **slide2;

		    slide1 = variable_data;
		    slide2 = (void **) (slide1 + count + 1);
		    slide3 = (union state_type *) (slide2 + count);

		    state = slide2;

		    while (optind < argc) {
			int status;
			unsigned type;

			if (argc - optind < 3) {
			    fprintf(stderr, "%s: cannot parse variable %u de"
				    "finition: not enough arguments\n",
				    argv[0], i);
			    break;
			}

			status =
			    x1f4_find_ctkey(argv[optind], type_root, &type);
			if (status) {
			    fprintf(stderr, "%s: cannot parse type specifica"
				    "tion for variable %u: `%s'\n", argv[0], i,
				    argv[optind]);
			    break;
			}

			optind++;

			status = libx1f4i0_miss_list
			    (argv[0], argv[optind], i, variable_data);
			if (status) {
			    break;
			}

			slide1->name = argv[optind];
			slide1->length = strlen(argv[optind]);

			optind++;

			if (X1f4_E4_LAST < type) {
			    long *data;
			    struct line_type *line;
			    unsigned j = 0, scale;

			    if (type == EX20_E4_LINE) {
				scale = 8;
			    } else if (type == EX20_E4_PAIR) {
				scale = 2;
			    } else {
				scale = 4;
			    }

			    if (argc - optind < scale) {
				fprintf(stderr, "%s: cannot parse variable %"
					"u definition: not enough arguments\n",
					argv[0], i);
				break;
			    }

			    line = (struct line_type *)
				malloc(sizeof(struct line_type));
			    if (!line) {
				perror(argv[0]);
				break;
			    }

			    line->line_data = miss.line_data;
			    miss.line_data = line;

			    data = line->data;

			    for (; j < scale; j++) {
				long cell;

				if (x1f4_parse_xlintegral
				    (&cell, argv[optind], NULL, 0)) {
				    fprintf(stderr, "%s: cannot parse value "
					    "specification for variable %u, "
					    "cell %u: `%s'\n", argv[0], i, j,
					    argv[optind]);
				    break;
				}

				optind++;

				*data++ = cell;
			    }

			    if (j < scale) {
				break;
			    }

			    optind--;

			    slide3->user = line;
			} else if (type == X1f4_E4_BILL) {
			    X1f4_E4_C_BILL bill;

			    if (x1f4_parse_wxlcardinal
				(&bill, argv[optind], NULL, 0)) {
				fprintf(stderr, "%s: cannot parse value spec"
					"ification for variable %u: `%s'\n",
					argv[0], i, argv[optind]);
				break;
			    }

			    slide3->bill = bill;
			} else if (type == X1f4_E4_MODE) {
			    X1f4_E4_C_MODE mode;

			    if (x1f4_parse_xlintegral
				(&mode, argv[optind], NULL, 0)) {
				fprintf(stderr, "%s: cannot parse value spec"
					"ification for variable %u: `%s'\n",
					argv[0], i, argv[optind]);
				break;
			    }

			    slide3->mode = mode;
			} else if (type == X1f4_E4_REAL) {
			    X1f4_E4_C_REAL real;
			    char *error;

			    real = strtod(argv[optind], &error);
			    if (*error || error == argv[optind]) {
				fprintf(stderr, "%s: cannot parse value spec"
					"ification for variable %u: `%s'\n",
					argv[0], i, argv[optind]);
				break;
			    }

			    slide3->real = real;
			} else {
			    slide3->text = argv[optind];
			}

			optind++;

			slide1->type = type;
			slide1->flags = 0;

			*slide2 = slide3;

			slide1++;
			slide2++;
			slide3++;

			i++;
		    }
		    if (optind < argc) {
			while (miss.line_data) {
			    struct line_type *line;

			    line = miss.line_data;
			    miss.line_data = line->line_data;
			    free(line);
			}

			free(variable_data);

			status = 1;

			break;
		    } else {
			slide1->name = NULL;
		    }
		}

#if 0
		{
		    struct x1f4_variable_type *slide1;
		    union state_type *slide3;
		    void **slide2;

		    slide1 = variable_data;
		    slide2 = (void **) (slide1 + count + 1);
		    slide3 = (union state_type *) (slide2 + count);

		    slide1 = variable_data;
		    while (slide1->name) {
			switch (slide1->type) {
			case EX20_E4_LINE:
			    fputs("line", stderr);
			    break;
			case EX20_E4_PAIR:
			    fputs("pair", stderr);
			    break;
			case EX20_E4_QUAD:
			    fputs("quad", stderr);
			    break;
			case X1f4_E4_BILL:
			    fputs("cardinal", stderr);
			    break;
			case X1f4_E4_MODE:
			    fputs("integer", stderr);
			    break;
			case X1f4_E4_REAL:
			    fputs("real", stderr);
			    break;
			default:
			    fputs("text", stderr);
			}
			fputs(" ", stderr);
			fputs(slide1->name, stderr);
			fputs(" ", stderr);
			switch (slide1->type) {
			case EX20_E4_LINE:
			    fprint_line(stderr, 8, I_USER(slide3));
			    break;
			case EX20_E4_PAIR:
			    fprint_line(stderr, 2, I_USER(slide3));
			    break;
			case EX20_E4_QUAD:
			    fprint_line(stderr, 4, I_USER(slide3));
			    break;
			case X1f4_E4_BILL:
			    fprintf(stderr, "%lu", I_BILL(slide3));
			    break;
			case X1f4_E4_MODE:
			    fprintf(stderr, "%ld", I_MODE(slide3));
			    break;
			case X1f4_E4_REAL:
			    x1f4_fprint_wfloat
				(stderr, 0, 0, precision, I_REAL(slide3));
			    break;
			default:
			    fputs(I_TEXT(slide3), stderr);
			}
			fputs("\n", stderr);

			slide1++;
			slide3++;
		    }
		}
#endif				/* 0 */
	    }

#if defined HAVE_LIBx1f4l0
	    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

	    {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
		    struct context_type context;
		    struct x1f4_attributes_type attributes;
		    const struct x1f4_operator_type *const *operator1s,
			*const *operator2s;

		    context.state = state;
		    context.variable_data = variable_data;

		    attributes.bcollect_set.eerecord_data = &eerecord;
		    attributes.function_set.get = libx1f4i0_select_function;
		    attributes.function_set.context = function_data;
		    x1f4_llink_operator1s(&operator1s);
		    libx1f4i0_join_sets
			(&attributes.operator1s, operator1s,
			 _libx1f4i0_e4_line_1, 6);
		    x1f4_llink_operator2s(&operator2s);
		    libx1f4i0_join_sets
			(&attributes.operator2s, operator2s,
			 _libx1f4i0_e4_line_2, 21);
		    attributes.terminator = 0;
		    attributes.variable_set.get = select_variable;
		    attributes.variable_set.context = &context;

		    flags |= X1f4_E4_BCOLLECT | X1f4_E4_ESTORAGE;

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			flags |= X1f4_E4_RESOURCE;
			libx1f4i0_line_valist
			    (valist, do_memory, do_storage,
			     &attributes.resource_set.free,
			     &attributes.resource_set.link,
			     &attributes.resource_set.mode,
			     &attributes.resource_set.context, &dlist);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    {
			status = x1f4_init_expression
			    (&x1f4_expression, expression, flags, &attributes);
		    }

		    if (attributes.operator2s != operator2s) {
			free((void *) attributes.operator2s);
		    }

		    if (attributes.operator1s != operator1s) {
			free((void *) attributes.operator1s);
		    }

		    free(function_data);
		}
	    }
	    if (status) {
		if (status == X1f4_E4_ALLOC_ERROR) {
		    perror(argv[0]);
		} else {
		    struct x1f4_eelookup_type eelookup;

		    eelookup.type_l.fix = lookup_type;
		    eelookup.type_l.context = NULL;
		    eelookup.type_q.e4fine = NULL;

		    fprintf(stderr, "%s: ", argv[0]);
		    x1f4_stat_expression
			(stderr, test_cast, &eerecord, &eelookup);
		    fprintf(stderr, "\n");
		}
	    } else {
		if (do_optimize) {
		    x1f4_line_expression(x1f4_expression);
		}

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    if (do_storage) {
			unsigned size;

			x1f4_size_xalist(valist, &size);
			fprintf(stderr, " %7u\n", size);
		    }
		}
#endif				/* HAVE_LIBx1f4l0 */

		if (do_print) {
		    x1f4_print_expression
			(stdout, x1f4_expression, print_flags);
		    putchar('\n');
		} else {
		    union state_type output;

		    status = x1f4_link_expression(x1f4_expression, &output);
		    if (status) {
			fprintf(stderr, "%s: cannot evaluate `%s'\n", argv[0],
				expression);
		    } else {
			unsigned type;

			type = x1f4_type_expression(x1f4_expression);
			if (type == EX20_E4_LINE) {
			    fprint_line(stdout, 8, I_USER(&output));
			    putchar('\n');
			} else if (type == EX20_E4_PAIR) {
			    fprint_line(stdout, 2, I_USER(&output));
			    putchar('\n');
			} else if (type == EX20_E4_QUAD) {
			    fprint_line(stdout, 4, I_USER(&output));
			    putchar('\n');
			} else {
			    libx1f4i0_type_data(type, precision, &output);
			}
		    }
		}

		x1f4_fini_expression(&x1f4_expression);
	    }

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		libx1f4i0_fini_valist
		    (argv[0], valist, do_memory, do_storage, &status);
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    while (miss.line_data) {
		struct line_type *line;

		line = miss.line_data;
		miss.line_data = line->line_data;
		free(line);
	    }

	    if (variable_data) {
		free(variable_data);
	    }
	} while (0);
    }

    return status;
}
