/*
 * ev18.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <cardinal.f-f.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <f2.h>
#if !defined HAVE_LIBx1f4i0
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <sf.h>
#include <types.h>

#define MAKE_DOUBLE(a, b)		a b
#define MAKE_SINGLE(a, b)		a

#define I_MODE(i)			(*((X1f4_E4_C_MODE *) (i)))
#define I_REAL(r)			(*((X1f4_E4_C_REAL *) (r)))
#define I_TEXT(r)			(*((X1f4_E4_C_TEXT *) (r)))

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }
#define l_REAL(e, output) \
    {									      \
	X1f4_E4_C_REAL *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }
#define l_TEXT(e, output) \
    {									      \
	X1f4_E4_C_TEXT *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

#define HOOK_AXXX(context, output, input, f_1, f_2, f_3) \
    {									      \
	X1f4_E4_C_MODE mode;						      \
	unsigned text;							      \
									      \
	x1f4_hook_expression((context), &text);				      \
	x1f4_fprint_fcardinal(stdout, 0, 3, 2, text & 7);		      \
	fputc(' ', stdout);						      \
									      \
	do {								      \
	    if (text & (1 << (2 - (f_1)))) {				      \
	    } else {							      \
		x1f4_clip_expression((context), (context), 1 << (2 - (f_1))); \
	    }								      \
									      \
	    mode = I_MODE((input[(f_1)]));				      \
	    if (mode) {							      \
		break;							      \
	    }								      \
									      \
	    if (text & (1 << (2 - (f_2)))) {				      \
	    } else {							      \
		x1f4_clip_expression((context), (context), 1 << (2 - (f_2))); \
	    }								      \
									      \
	    mode = I_MODE((input[(f_2)]));				      \
	    if (mode) {							      \
		break;							      \
	    }								      \
									      \
	    if (text & (1 << (2 - (f_3)))) {				      \
	    } else {							      \
		x1f4_clip_expression((context), (context), 1 << (2 - (f_3))); \
	    }								      \
									      \
	    mode = I_MODE((input[(f_3)]));				      \
	} while (0);							      \
									      \
	l_MODE(mode, (output));						      \
    }

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
    void **state;
} context_type;

static int c23xx_m__(void *, void **);
static int c23xx_r__(void *, void **);
static int c23xx_t__(void *, void **);
static int hook_a123(void *, void *, void **);
static int hook_a132(void *, void *, void **);
static int hook_a213(void *, void *, void **);
static int hook_a231(void *, void *, void **);
static int hook_a312(void *, void *, void **);
static int hook_a321(void *, void *, void **);
static int keep_mode(void *, void *, void **);
static int keep_real(void *, void *, void **);
static int keep_text(void *, void *, void **);
static int lose_mode(void *, void *, void **);
static int lose_real(void *, void *, void **);
static int lose_text(void *, void *, void **);
static int merge_sets(struct x1f4_function_type **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);

static void usage(void);

static const int c_____m__[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE
/* *INDENT-ON* */
}, c_____m_3[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE,
    X1f4_E4_MODE,
    X1f4_E4_MODE
/* *INDENT-ON* */
}, c_____r__[] = {
/* *INDENT-OFF* */
    X1f4_E4_REAL
/* *INDENT-ON* */
}, c_____t__[] = {
/* *INDENT-OFF* */
    X1f4_E4_TEXT
/* *INDENT-ON* */
};
static const struct x1f4_function_type e18_set[] = {
/* *INDENT-OFF* */
    {	"MODE",			keep_mode,
	X1f4_E4_MODE,		c_____m__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	"Mode",			lose_mode,
	X1f4_E4_MODE,		c_____m__,	1,
	0,					4		},
    {	"REAL",			keep_real,
	X1f4_E4_REAL,		c_____r__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	"Real",			lose_real,
	X1f4_E4_REAL,		c_____r__,	1,
	0,					4		},
    {	"TEXT",			keep_text,
	X1f4_E4_TEXT,		c_____t__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	"Text",			lose_text,
	X1f4_E4_TEXT,		c_____t__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	"and_1_2_3",		hook_a123,
	X1f4_E4_MODE,		c_____m_3,	3,
	0,					9		},
    {	"and_1_3_2",		hook_a132,
	X1f4_E4_MODE,		c_____m_3,	3,
	0,					9		},
    {	"and_2_1_3",		hook_a213,
	X1f4_E4_MODE,		c_____m_3,	3,
	0,					9		},
    {	"and_2_3_1",		hook_a231,
	X1f4_E4_MODE,		c_____m_3,	3,
	0,					9		},
    {	"and_3_1_2",		hook_a312,
	X1f4_E4_MODE,		c_____m_3,	3,
	0,					9		},
    {	"and_3_2_1",		hook_a321,
	X1f4_E4_MODE,		c_____m_3,	3,
	0,					9		},
    {	NULL,			NULL,
	0,			NULL,		0,
	0,					1		}
/* *INDENT-ON* */
};
static const struct x1f4_operator_type e23xx_m__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("#", " "),  c23xx_m__,	0210,
	X1f4_E4_MODE,		c_____m__,
	X1f4_E4_KEEP_CALL,	1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e23xx_r__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("#", "#"),  c23xx_r__,	0210,
	X1f4_E4_REAL,		c_____r__,
	X1f4_E4_KEEP_CALL,	1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e23xx_t__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("#", " "),  c23xx_t__,	0210,
	X1f4_E4_TEXT,		c_____t__,
	X1f4_E4_KEEP_CALL,	1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, *ev18_e4_1_arithmetics[] = {
/* *INDENT-OFF* */
    e23xx_m__,
    e23xx_r__,
    e23xx_t__
/* *INDENT-ON* */
};

static int
c23xx_m__(void *output, void **input)
{
    l_MODE(I_MODE(input[0]), output);

    return 0;
}


static int
c23xx_r__(void *output, void **input)
{
    l_REAL(I_REAL(input[0]), output);

    return 0;
}


static int
c23xx_t__(void *output, void **input)
{
    l_TEXT(I_TEXT(input[0]), output);

    return 0;
}


static int
hook_a123(void *context, void *output, void **input)
{
    HOOK_AXXX(context, output, input, 0, 1, 2);

    return 0;
}


static int
hook_a132(void *context, void *output, void **input)
{
    HOOK_AXXX(context, output, input, 0, 2, 1);

    return 0;
}


static int
hook_a213(void *context, void *output, void **input)
{
    HOOK_AXXX(context, output, input, 1, 0, 2);

    return 0;
}


static int
hook_a231(void *context, void *output, void **input)
{
    HOOK_AXXX(context, output, input, 1, 2, 0);

    return 0;
}


static int
hook_a312(void *context, void *output, void **input)
{
    HOOK_AXXX(context, output, input, 2, 0, 1);

    return 0;
}


static int
hook_a321(void *context, void *output, void **input)
{
    HOOK_AXXX(context, output, input, 2, 1, 0);

    return 0;
}


static int
keep_mode(void *context, void *output, void **input)
{
    l_MODE(I_MODE(input[0]), output);

    return 0;
}


static int
keep_real(void *context, void *output, void **input)
{
    l_REAL(I_REAL(input[0]), output);

    return 0;
}


static int
keep_text(void *context, void *output, void **input)
{
    l_TEXT(I_TEXT(input[0]), output);

    return 0;
}


static int
lose_mode(void *context, void *output, void **input)
{
    l_MODE(I_MODE(input[0]), output);

    return 0;
}


static int
lose_real(void *context, void *output, void **input)
{
    l_REAL(I_REAL(input[0]), output);

    return 0;
}


static int
lose_text(void *context, void *output, void **input)
{
    l_TEXT(I_TEXT(input[0]), output);

    return 0;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    function_data = (struct x1f4_function_type *)
	malloc((count + 13) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, e18_set, sizeof(struct x1f4_function_type) * 13);
    }

    return status;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
}


static void
usage(void)
{
    puts("Usage: ev18 [OPTIONS] EXPRESSION [TYPE NAME VALUE]\n\
Test late function arguments evaluation, evaluate EXPRESSION.\n\
\n\
Options:\n\
  -P, --print			print expression\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --list			list available functions and exit\n\
      --list-1			list available unary operators and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_optimize = 0, do_print = 0, list_functions = 0, status = 0;
    unsigned precision = 3, print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "list",	    0x00,   NULL,   0x01    },
		{   "list-1",	    0x00,   NULL,   0x02    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Pop:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 002:
		list_functions = 2;
		break;
	    case 021:
		print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (do_print) {
	    precision_specs = NULL;
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (list_functions == 1) {
	struct x1f4_function_type *function_data;

	if (merge_sets(&function_data)) {
	    status = 1;
	    perror(argv[0]);
	} else {
	    x1f4_print_functions(stdout, function_data);

	    free(function_data);
	}
    } else if (list_functions == 2) {
	const struct x1f4_operator_type *const *operAtor1s, *const *operator1s;

	x1f4_llink_operator1s(&operator1s);
	libx1f4i0_join_sets
	    (&operAtor1s, operator1s, ev18_e4_1_arithmetics, 3);
	x1f4_print_operators(stdout, operAtor1s, 1);
	if (operAtor1s != operator1s) {
	    free((void *) operAtor1s);
	}
    } else if ((argc - optind) % 3 != 1) {
	status = 1;
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
	    struct x1f4_variable_type *variable_data;
	    unsigned flags = X1f4_E4_DDACCESS;
	    void **state = NULL, *x1f4_expression;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    status = 1;

	    expression = argv[optind];
	    optind++;

	    status = libx1f4i0_init_list
		(argv[0], argc, argv, optind, (void *) &variable_data, &state);
	    if (status) {
		break;
	    }

#if defined HAVE_LIBx1f4l0
	    {
		x1f4_init_valist(&valist, 1024);
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
		    struct context_type context;
		    struct x1f4_attributes_type attributes;
		    const struct x1f4_operator_type *const *operator1s;

		    context.state = state;
		    context.variable_data = variable_data;

		    attributes.function_set.get = libx1f4i0_select_function;
		    attributes.function_set.context = function_data;
		    x1f4_llink_operator1s(&operator1s);
		    libx1f4i0_join_sets
			(&attributes.operator1s, operator1s,
			 ev18_e4_1_arithmetics, 3);
		    x1f4_llink_operator2s(&attributes.operator2s);
		    attributes.terminator = 0;
		    attributes.variable_set.get = select_variable;
		    attributes.variable_set.context = &context;

		    flags |= X1f4_E4_ESTORAGE;

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			flags |= X1f4_E4_RESOURCE;
			if (1) {
			    attributes.resource_set.context = valist;
			    attributes.resource_set.free = x1f4_free_valist;
			    attributes.resource_set.link = x1f4_link_valist;
			    attributes.resource_set.mode = x1f4_mode_valist;
			}
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    {
			status = x1f4_init_expression
			    (&x1f4_expression, expression, flags, &attributes);
		    }

		    if (attributes.operator1s != operator1s) {
			free((void *) attributes.operator1s);
		    }

		    free(function_data);
		}
	    }
	    if (status) {
		if (status == X1f4_E4_ALLOC_ERROR) {
		    perror(argv[0]);
		} else {
		    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
			    expression);
		}
	    } else {
		if (do_optimize) {
		    x1f4_line_expression(x1f4_expression);
		}

		if (do_print) {
		    x1f4_print_expression
			(stdout, x1f4_expression, print_flags);
		    putchar('\n');
		} else {
		    struct lead_type output;

		    status = x1f4_link_expression(x1f4_expression, &output);
		    if (status) {
			fprintf(stderr, "%s: cannot evaluate `%s'\n", argv[0],
				expression);
		    } else {
			libx1f4i0_type_data
			    (x1f4_type_expression(x1f4_expression), precision,
			     &output);
		    }
		}

		x1f4_fini_expression(&x1f4_expression);
	    }

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		unsigned linger;

		if (1) {
		    x1f4_size_valist(valist, &linger);
		}
		if (linger) {
		    fprintf(stderr, "%s: memory is leaking\n", argv[0]);
		    status = 1;
		}

		if (1) {
		    x1f4_fini_valist(&valist);
		}
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    libx1f4i0_fini_list(argv[0], (void *) &variable_data);
	} while (0);
    }

    return status;
}
