/*
 * ev15.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <f2.h>
#if !defined HAVE_LIBx1f4i0
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <sf.h>
#include <types.h>

#define MAKE_DOUBLE(a, b)		a b
#define MAKE_SINGLE(a, b)		a

#define I_MODE(i)			(*((X1f4_E4_C_MODE *) (i)))
#define I_REAL(r)			(*((X1f4_E4_C_REAL *) (r)))
#define I_TEXT(r)			(*((X1f4_E4_C_TEXT *) (r)))

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }
#define l_REAL(e, output) \
    {									      \
	X1f4_E4_C_REAL *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }
#define l_TEXT(e, output) \
    {									      \
	X1f4_E4_C_TEXT *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

typedef X1f4_E4_C_TEXT e4_c_text;

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
    unsigned count;
    void **state;
} context_type;

static int c10xx_m__(void *, void **);
static int c10xx_t__(void *, void **);
static int c11xx_t__(void *, void **);
static int c12xx_b__(void *, void **);
static int c12xx_m__(void *, void **);
static int c12xx_r__(void *, void **);
static int c12xx_t__(void *, void **);
static int c13xx_t_t(void *, void **);
static int c14xx_t_t(void *, void **);
static int c15xx_t_t(void *, void **);
static int c16xx_m_t(void *, void **);
static int c16xx_t_m(void *, void **);
static int c17xx_m_t(void *, void **);
static int c17xx_t_m(void *, void **);
static int c18xx_m_t(void *, void **);
static int c18xx_t_m(void *, void **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);

static void usage(void);

static const int c_____b__[] = {
/* *INDENT-OFF* */
    X1f4_E4_BILL
/* *INDENT-ON* */
}, c_____m__[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE
/* *INDENT-ON* */
}, c_____m_t[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE,
    X1f4_E4_TEXT
/* *INDENT-ON* */
}, c_____r__[] = {
/* *INDENT-OFF* */
    X1f4_E4_REAL
/* *INDENT-ON* */
}, c_____t__[] = {
/* *INDENT-OFF* */
    X1f4_E4_TEXT
/* *INDENT-ON* */
}, c_____t_m[] = {
/* *INDENT-OFF* */
    X1f4_E4_TEXT,
    X1f4_E4_MODE
/* *INDENT-ON* */
}, c_____t_t[] = {
/* *INDENT-OFF* */
    X1f4_E4_TEXT,
    X1f4_E4_TEXT
/* *INDENT-ON* */
};
static const struct x1f4_operator_type e10xx_m__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("$", " "),  c10xx_m__,	0210,
	X1f4_E4_TEXT,		c_____m__,
	X1f4_E4_KEEP_CALL,	1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e10xx_t__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("$", " "),  c10xx_t__,	0210,
	X1f4_E4_TEXT,		c_____t__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e11xx_t__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("#", " "),  c11xx_t__,	0210,
	X1f4_E4_MODE,		c_____t__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e12xx_b__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("?", " "),  c12xx_b__,	0210,
	X1f4_E4_TEXT,		c_____b__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e12xx_m__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("?", " "),  c12xx_m__,	0210,
	X1f4_E4_TEXT,		c_____m__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e12xx_r__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("?", " "),  c12xx_r__,	0210,
	X1f4_E4_TEXT,		c_____r__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e12xx_t__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("?", " "),  c12xx_t__,	0210,
	X1f4_E4_TEXT,		c_____t__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e13xx_t_t[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("^", " "),  c13xx_t_t,	0410,
	X1f4_E4_MODE,		c_____t_t,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e14xx_t_t[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("/", " "),  c14xx_t_t,	0420,
	X1f4_E4_TEXT,		c_____t_t,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e15xx_t_t[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("*", " "),  c15xx_t_t,	0430,
	X1f4_E4_TEXT,		c_____t_t,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e16xx_m_t[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("+", " "),  c16xx_m_t,	0440,
	X1f4_E4_TEXT,		c_____m_t,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e16xx_t_m[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("+", " "),  c16xx_t_m,	0440,
	X1f4_E4_TEXT,		c_____t_m,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e17xx_m_t[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("-", " "),  c17xx_m_t,	0450,
	X1f4_E4_TEXT,		c_____m_t,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e17xx_t_m[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("-", " "),  c17xx_t_m,	0450,
	X1f4_E4_TEXT,		c_____t_m,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e18xx_m_t[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("!", " "),  c18xx_m_t,	0460,
	X1f4_E4_MODE,		c_____m_t,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e18xx_t_m[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("!", " "),  c18xx_t_m,	0460,
	X1f4_E4_MODE,		c_____t_m,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, *ev15_e4_1_arithmetics[] = {
/* *INDENT-OFF* */
    e10xx_m__,
    e10xx_t__,
    e11xx_t__,
    e12xx_b__,
    e12xx_m__,
    e12xx_r__,
    e12xx_t__
/* *INDENT-ON* */
}, *ev15_e4_2_arithmetics[] = {
/* *INDENT-OFF* */
    e13xx_t_t,
    e14xx_t_t,
    e15xx_t_t,
    e16xx_m_t,
    e16xx_t_m,
    e17xx_m_t,
    e17xx_t_m,
    e18xx_m_t,
    e18xx_t_m
/* *INDENT-ON* */
};
static void *static_text;

static int
c10xx_m__(void *output, void **input)
{
    X1f4_E4_C_TEXT text;
    struct context_type *context_data;
    unsigned count;

    context_data = static_text;

    count = context_data->count;
    if (!count) {
	text = "";
    } else {
	X1f4_E4_C_MODE index;

	index = I_MODE(input[0]);
	if (!(index < count)) {
	    text = "";
	} else {
	    text = (X1f4_E4_C_TEXT) context_data->variable_data[index].name;
	}
    }

    l_TEXT(text, output);

    return 0;
}


static int
c10xx_t__(void *output, void **input)
{
    char *environment;

    environment = getenv(I_TEXT(input[0]));
    if (!environment) {
	environment = "";
    }

    l_TEXT(environment, output);

    return 0;
}


static int
c11xx_t__(void *output, void **input)
{
    l_MODE(strlen(I_TEXT(input[0])), output);

    return 0;
}


static int
c12xx_b__(void *output, void **input)
{
    l_TEXT("cardinal", output);

    return 0;
}


static int
c12xx_m__(void *output, void **input)
{
    l_TEXT("integer", output);

    return 0;
}


static int
c12xx_r__(void *output, void **input)
{
    l_TEXT("real", output);

    return 0;
}


static int
c12xx_t__(void *output, void **input)
{
    l_TEXT("text", output);

    return 0;
}


static int
c13xx_t_t(void *output, void **input)
{
    l_MODE(strcmp(I_TEXT(input[0]), I_TEXT(input[1])), output);

    return 0;
}


static int
c14xx_t_t(void *output, void **input)
{
    e4_c_text major, minor;
    unsigned fl1st, fl2nd;

    major = I_TEXT(input[0]);
    fl1st = strlen(major);
    minor = I_TEXT(input[1]);
    fl2nd = strlen(minor);
    if (fl1st < fl2nd) {
	l_TEXT(major, output);
    } else {
	if (fl1st ^ fl2nd) {
	    l_TEXT(minor, output);
	} else {
	    l_TEXT("", output);
	}
    }

    return 0;
}


static int
c15xx_t_t(void *output, void **input)
{
    e4_c_text major, minor;
    unsigned fl1st, fl2nd;

    major = I_TEXT(input[0]);
    fl1st = strlen(major);
    minor = I_TEXT(input[1]);
    fl2nd = strlen(minor);
    if (fl1st < fl2nd) {
	l_TEXT(minor, output);
    } else {
	if (fl1st ^ fl2nd) {
	    l_TEXT(major, output);
	} else {
	    l_TEXT("", output);
	}
    }

    return 0;
}


static int
c16xx_m_t(void *output, void **input)
{
    e4_c_text minor;
    unsigned fl1st, fl2nd;

    fl1st = I_MODE(input[0]);
    minor = I_TEXT(input[1]);
    fl2nd = strlen(minor);
    if (fl1st < fl2nd) {
	l_TEXT("head", output);
    } else {
	if (fl1st ^ fl2nd) {
	    l_TEXT(minor, output);
	} else {
	    l_TEXT("", output);
	}
    }

    return 0;
}


static int
c16xx_t_m(void *output, void **input)
{
    e4_c_text major;
    unsigned fl1st, fl2nd;

    major = I_TEXT(input[0]);
    fl1st = strlen(major);
    fl2nd = I_MODE(input[1]);
    if (fl1st < fl2nd) {
	l_TEXT(major, output);
    } else {
	if (fl1st ^ fl2nd) {
	    l_TEXT("tail", output);
	} else {
	    l_TEXT("", output);
	}
    }

    return 0;
}


static int
c17xx_m_t(void *output, void **input)
{
    e4_c_text minor;
    unsigned fl1st, fl2nd;

    fl1st = I_MODE(input[0]);
    minor = I_TEXT(input[1]);
    fl2nd = strlen(minor);
    if (fl1st < fl2nd) {
	l_TEXT(minor, output);
    } else {
	if (fl1st ^ fl2nd) {
	    l_TEXT("head", output);
	} else {
	    l_TEXT("", output);
	}
    }

    return 0;
}


static int
c17xx_t_m(void *output, void **input)
{
    e4_c_text major;
    unsigned fl1st, fl2nd;

    major = I_TEXT(input[0]);
    fl1st = strlen(major);
    fl2nd = I_MODE(input[1]);
    if (fl1st < fl2nd) {
	l_TEXT("tail", output);
    } else {
	if (fl1st ^ fl2nd) {
	    l_TEXT(major, output);
	} else {
	    l_TEXT("", output);
	}
    }

    return 0;
}


static int
c18xx_m_t(void *output, void **input)
{
    l_MODE(I_MODE(input[0]) + strlen(I_TEXT(input[1])), output);

    return 0;
}


static int
c18xx_t_m(void *output, void **input)
{
    l_MODE(strlen(I_TEXT(input[0])) + I_MODE(input[1]), output);

    return 0;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
}


static void
usage(void)
{
    puts("Usage: ev15 [OPTIONS] EXPRESSION [TYPE NAME VALUE]\n\
Test operators evaluation, evaluate EXPRESSION.\n\
\n\
Options:\n\
  -P, --print			print expression\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --list-1			list available unary operators and exit\n\
      --list-2			list available binary operators and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_optimize = 0, do_print = 0, list_functions = 0, status = 0;
    unsigned precision = 3, print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "list-1",	    0x00,   NULL,   0x02    },
		{   "list-2",	    0x00,   NULL,   0x03    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Pop:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 002:
		list_functions = 2;
		break;
	    case 003:
		list_functions = 3;
		break;
	    case 021:
		print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (do_print) {
	    precision_specs = NULL;
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (list_functions == 2) {
	const struct x1f4_operator_type *const *operAtor1s, *const *operator1s;

	x1f4_llink_operator1s(&operator1s);
	libx1f4i0_join_sets(&operAtor1s, operator1s, ev15_e4_1_arithmetics, 7);
	x1f4_print_operators(stdout, operAtor1s, 1);
	if (operAtor1s != operator1s) {
	    free((void *) operAtor1s);
	}
    } else if (list_functions == 3) {
	const struct x1f4_operator_type *const *operAtor2s, *const *operator2s;

	x1f4_llink_operator2s(&operator2s);
	libx1f4i0_join_sets(&operAtor2s, operator2s, ev15_e4_2_arithmetics, 9);
	x1f4_print_operators(stdout, operAtor2s, 2);
	if (operAtor2s != operator2s) {
	    free((void *) operAtor2s);
	}
    } else if ((argc - optind) % 3 != 1) {
	status = 1;
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
	    struct context_type context;
	    struct x1f4_variable_type *variable_data;
	    unsigned count, flags = X1f4_E4_DDACCESS;
	    void **state = NULL, *x1f4_expression;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    status = 1;

	    expression = argv[optind];
	    optind++;

	    static_text = &context;

	    status = libx1f4i0_lead_list
		(argv[0], argc, argv, optind, (void *) &variable_data, &state,
		 &count);
	    if (status) {
		break;
	    }

#if defined HAVE_LIBx1f4l0
	    {
		x1f4_init_valist(&valist, 1024);
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    {
		struct x1f4_attributes_type attributes;
		const struct x1f4_operator_type *const *operator1s,
		    *const *operator2s;

		context.count = count;
		context.state = state;
		context.variable_data = variable_data;

		attributes.function_set.get = libx1f4i0_select_function;
		attributes.function_set.context = x1f4_e4_defaults;
		x1f4_llink_operator1s(&operator1s);
		libx1f4i0_join_sets(&attributes.operator1s, operator1s,
				    ev15_e4_1_arithmetics, 7);
		x1f4_llink_operator2s(&operator2s);
		libx1f4i0_join_sets(&attributes.operator2s, operator2s,
				    ev15_e4_2_arithmetics, 9);
		attributes.terminator = 0;
		attributes.variable_set.get = select_variable;
		attributes.variable_set.context = &context;

		flags |= X1f4_E4_ESTORAGE;

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    flags |= X1f4_E4_RESOURCE;
		    if (1) {
			attributes.resource_set.context = valist;
			attributes.resource_set.free = x1f4_free_valist;
			attributes.resource_set.link = x1f4_link_valist;
			attributes.resource_set.mode = x1f4_mode_valist;
		    }
		}
#endif				/* HAVE_LIBx1f4l0 */

		{
		    status = x1f4_init_expression
			(&x1f4_expression, expression, flags, &attributes);
		}

		if (attributes.operator1s != operator1s) {
		    free((void *) attributes.operator1s);
		}
		if (attributes.operator2s != operator2s) {
		    free((void *) attributes.operator2s);
		}
	    }
	    if (status) {
		if (status == X1f4_E4_ALLOC_ERROR) {
		    perror(argv[0]);
		} else {
		    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
			    expression);
		}
	    } else {
		if (do_optimize) {
		    x1f4_line_expression(x1f4_expression);
		}

		if (do_print) {
		    x1f4_print_expression
			(stdout, x1f4_expression, print_flags);
		    putchar('\n');
		} else {
		    struct lead_type output;

		    status = x1f4_link_expression(x1f4_expression, &output);
		    if (status) {
			fprintf(stderr, "%s: cannot evaluate `%s'\n", argv[0],
				expression);
		    } else {
			libx1f4i0_type_data
			    (x1f4_type_expression(x1f4_expression), precision,
			     &output);
		    }
		}

		x1f4_fini_expression(&x1f4_expression);
	    }

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		unsigned linger;

		if (1) {
		    x1f4_size_valist(valist, &linger);
		}
		if (linger) {
		    fprintf(stderr, "%s: memory is leaking\n", argv[0]);
		    status = 1;
		}

		if (1) {
		    x1f4_fini_valist(&valist);
		}
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    libx1f4i0_fini_list(argv[0], (void *) &variable_data);
	} while (0);
    }

    return status;
}
