/*
 * ev13.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <ckeytree.h>
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <sf.h>
#include <types.h>

typedef struct context_type {
    struct x1f4_stateset_type *stateset_data;
    struct x1f4_variable_type *variable_data;
    unsigned class;
    void *shift, **state;
} context_type;

typedef struct textset_type {
    struct x1f4_variable_type *variable_data, *variable_miss;
    void *shift;
} textset_type;

static int fix_state(const void *, const struct x1f4_variable_type *, void **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);

static void usage(void);

static const struct x1f4_ckeytree_type type_0[] = {
/* *INDENT-OFF* */
    {	"cardinal",	{	X1f4_E4_BILL	},	NULL	},
    {	"integer",	{	X1f4_E4_MODE	},	NULL	},
    {	"real",		{	X1f4_E4_REAL	},	NULL	},
    {	"text",		{	X1f4_E4_TEXT	},	NULL	},
    {	NULL,		{	0		},	NULL	}
/* *INDENT-ON* */
}, type_root[] = {
/* *INDENT-OFF* */
    {	"",		{	0		},	type_0	},
    {	NULL,		{	0		},	NULL	}
/* *INDENT-ON* */
};

static int
fix_state(const void *context, const struct x1f4_variable_type *variable_data,
	  void **state)
{
    int status;
    const struct textset_type *textset_data;

    textset_data = context;
    if (variable_data < textset_data->variable_data) {
	status = 1;
    } else {
	if (variable_data < textset_data->variable_miss) {
	    *state = (struct lead_type *) textset_data->shift
		+ (variable_data - textset_data->variable_data);

	    status = 0;
	} else {
	    status = 1;
	}
    }

    return status;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    unsigned index;

	    status = 0;

	    *variable = variable_data;

	    index = variable_data - context_data->variable_data;
	    if (index < context_data->class) {
		*state = context_data->stateset_data;
	    } else {
		*state = context_data->state[index];
	    }
	}
    }

    return status;
}


static void
usage(void)
{
    puts("Usage: ev13 [OPTIONS] EXPRESSION COUNT [TYPE NAME VALUE]...\n\
Test reference variables evaluation, evaluate EXPRESSION for COUNT reference\n\
variables and some extra non reference variables.\n\
\n\
Options:\n\
  -M, --stat-storage		stat expression memory storage requirements\n\
  -c, --cycle CYCLE		cycle over EXPRESSION for CYCLE times\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_memory = 0, do_optimize = 0, do_storage = 0, list_functions = 0,
	status = 1;
    unsigned cycle = 1, precision = 3;

    unsetenv("POSIXLY_CORRECT");

    {
	char *cycle_specs = NULL, *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "cycle",	    0x01,   NULL,   0x63    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Mc:mop:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 'M':
		do_storage = 1;
		break;
	    case 'c':
		cycle_specs = optarg;
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (list_functions) {
	    precision_specs = NULL;
	}

	if (cycle_specs) {
	    status = x1f4_parse_wxcardinal(&cycle, cycle_specs, NULL, 0);
	    if (status) {
		fprintf(stderr, "%s: cannot parse cycle specification: `%s'\n",
			argv[0], cycle_specs);

		return 1;
	    }
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if ((argc - optind) % 3 != 2) {
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
#if defined HAVE_LIBx1f4l0
	    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
	    struct x1f4_stateset_type stateset;
	    struct x1f4_variable_type *variable_data;
	    unsigned class, count, flags = X1f4_E4_DDACCESS;
	    void *shift = NULL, **state = NULL, *x1f4_expression;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    expression = argv[optind];
	    optind++;

	    status = x1f4_parse_wxcardinal(&class, argv[optind], NULL, 0);
	    if (status) {
		fprintf(stderr, "%s: cannot parse count specification: `%s'"
			"\n", argv[0], argv[optind]);
		break;
	    }

	    optind++;

	    count = (argc - optind) / 3;

	    if (count < class) {
		status = 1;
		fprintf(stderr, "%s: count specification (%u) is exceeding c"
			"ount maximum (%u)\n", argv[0], class, count);
		break;
	    }

	    if (!count) {
		variable_data = NULL;
	    } else {
		variable_data = (struct x1f4_variable_type *)
		    malloc((sizeof(struct x1f4_variable_type) + sizeof(void *)
			    + sizeof(struct lead_type)) * (count - class)
			   + (sizeof(struct x1f4_variable_type)
			      + sizeof(struct lead_type) * cycle) * class
			   + sizeof(struct x1f4_variable_type));
		if (!variable_data) {
		    break;
		} else {
		    struct x1f4_variable_type *slide1;
		    struct lead_type *slide3;
		    unsigned i, j;
		    void **slide2;

		    slide1 = variable_data;
		    slide2 = (void **) (slide1 + count + 1);
		    slide3 = (struct lead_type *) (slide2 + count - class);

		    slide2 -= class;

		    j = class;

		    shift = slide3;

		    state = slide2;

		    for (i = count; i; i--) {
			int status;
			unsigned type;

			status =
			    x1f4_find_ctkey(argv[optind], type_root, &type);
			if (status) {
			    fprintf(stderr, "%s: cannot parse type specifica"
				    "tion for variable %u: `%s'\n", argv[0],
				    count - i, argv[optind]);
			    break;
			}

			optind++;

			status = libx1f4i0_miss_list
			    (argv[0], argv[optind], count - i, variable_data);
			if (status) {
			    break;
			}

			slide1->name = argv[optind];
			slide1->length = strlen(argv[optind]);

			optind++;

			status = libx1f4i0_lead_data
			    (argv[0], argv[optind], type, count - i,
			     &slide3->data);
			if (status) {
			    break;
			}

			optind++;

			slide1->type = type;
			slide1->flags = 0;

			if (j) {
			    j--;
			    if (j) {
			    } else {
				slide3 += class * (cycle - 1);
			    }

			    slide1->flags = X1f4_E4_REFERENCE;
			} else {
			    slide1->flags = 0;

			    *slide2 = slide3;
			}

			slide1++;
			slide2++;
			slide3++;
		    }
		    if (i) {
			free(variable_data);
			status = 1;
			break;
		    } else {
			slide1->name = NULL;
		    }
		}
	    }

#if defined HAVE_LIBx1f4l0
	    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

	    {
		struct context_type context;
		struct x1f4_attributes_type attributes;

		context.class = class;
		context.state = state;
		context.stateset_data = &stateset;
		context.variable_data = variable_data;

		attributes.function_set.get = libx1f4i0_select_function;
		attributes.function_set.context = x1f4_e4_defaults;
		x1f4_llink_operator1s(&attributes.operator1s);
		x1f4_llink_operator2s(&attributes.operator2s);
		attributes.terminator = 0;
		attributes.variable_set.get = select_variable;
		attributes.variable_set.context = &context;

		flags |= X1f4_E4_ESTORAGE;

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    flags |= X1f4_E4_RESOURCE;
		    libx1f4i0_line_valist
			(valist, do_memory, do_storage,
			 &attributes.resource_set.free,
			 &attributes.resource_set.link,
			 &attributes.resource_set.mode,
			 &attributes.resource_set.context, &dlist);
		}
#endif				/* HAVE_LIBx1f4l0 */

		{
		    status = x1f4_init_expression
			(&x1f4_expression, expression, flags, &attributes);
		}
	    }
	    if (status) {
		if (status == X1f4_E4_ALLOC_ERROR) {
		    perror(argv[0]);
		} else {
		    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
			    expression);
		}
	    } else {
		struct textset_type textset;

		if (do_optimize) {
		    x1f4_line_expression(x1f4_expression);
		}

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    if (do_storage) {
			unsigned size;

			x1f4_size_xalist(valist, &size);
			fprintf(stderr, " %7u\n", size);
		    }
		}
#endif				/* HAVE_LIBx1f4l0 */

		stateset.context = &textset;
		stateset.get = fix_state;

		textset.shift = shift;
		textset.variable_data = variable_data;
		textset.variable_miss = variable_data + class;

		while (cycle) {
		    status = x1f4_lead_expression(x1f4_expression);
		    if (status) {
			fprintf(stderr, "%s: cannot evaluate `%s'\n", argv[0],
				expression);
			break;
		    } else {
			struct lead_type *state_data;
			unsigned j;
			struct x1f4_variable_type *variable_slip;

			variable_slip = variable_data;

			state_data = textset.shift;

			putchar(' ');

			j = class;
			for (; j; j--) {
			    putchar(' ');

			    libx1f4i0_post_data
				(variable_slip->type, precision, state_data);

			    state_data++;

			    variable_slip++;
			}

			cycle--;
			if (cycle) {
			    struct lead_type *state_data;

			    state_data = textset.shift;
			    state_data += class;

			    memcpy(state_data, textset.shift,
				   sizeof(struct lead_type) * class);

			    textset.shift = state_data;
			}
		    }
		}

		putchar('\n');

		x1f4_fini_expression(&x1f4_expression);
	    }

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		libx1f4i0_fini_valist
		    (argv[0], valist, do_memory, do_storage, &status);
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    if (variable_data) {
		free(variable_data);
	    }
	} while (0);
    }

    return status;
}
