/*
 * xshell.c
 * Copyright (C) 2006-2012, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <errno.h>
#include <fcntl.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <xshell-names.h>

#define ARGV_MASK			(1u << 0x00)
#define BACK_MASK			(1u << 0x01)
#define C3RD_MASK			(1u << 0x02)
#define CALL_MASK			(1u << 0x03)
#define HEAD_MASK			(1u << 0x04)
#define LEAD_MASK			(1u << 0x05)
#define LIFT_MASK			(1u << 0x06)
#define LOOK_MASK			(1u << 0x07)
#define MASK_MASK			(1u << 0x08)
#define PATH_MASK			(1u << 0x09)
#define TAIL_MASK			(1u << 0x0a)
#define TYPE_MASK			(1u << 0x0b)

#define KILL_C3RD			0
#define MISS_C3RD			1
#define NULL_C3RD			2

#define C3RD				MISS_C3RD
#define NONE				0

#define FLAT				-1

#define Record(record)			((struct record_type *) (record))

typedef struct record_type {
    struct {
	int lead, lift;
    } adapter;
    struct {
	int (*call) (void *);
	void *back;
    } cleanup;
    struct {
	int fd_0, fd_1, wait;
    } context;
    struct {
	const char *const *argv, *path;
    } command;
    struct {
	int (*head) (void *), (*tail) (void *),
	    (*type) (void *, const char *, unsigned);
	void *look;
    } printer;
    struct {
	unsigned c3rd, mask;
    } trailer;
} record_type;

typedef struct xshell_type {
    struct {
	int lead, lift;
    } adapter;
    struct {
	int (*call) (void *);
	void *back;
    } cleanup;
    struct {
	const char *const *argv, *path;
    } command;
    struct {
	int (*head) (void *), (*tail) (void *),
	    (*type) (void *, const char *, unsigned);
	void *look;
    } printer;
    struct {
	unsigned c3rd, mask;
    } trailer;
} xshell_type;

static int deck_xshell(void *);
static int ever_stream(int *);
static int exec_xshell(void *);

static int
deck_xshell(void *record)
{
    int status = -1, pipes_0[2], pipes_1[2];
    struct record_type *record_data;

    record_data = record;

    pipes_0[0] = -1;
    pipes_0[1] = -1;
    pipes_1[0] = -1;
    pipes_1[1] = -1;

    fflush(stdout);

    fsync(STDOUT_FILENO);

    do {
	int lead, lift;
	pid_t wait;

	lead = record_data->adapter.lead;
	lift = record_data->adapter.lift;

	if (lift ^ FLAT) {
	} else {
	    if (ever_stream(pipes_1)) {
		break;
	    }
	}
	if (lead ^ FLAT) {
	} else {
	    if (ever_stream(pipes_0)) {
		break;
	    }
	}

	if (!(wait = fork())) {
	    do {
		const char *const *argv, *null_argv[2];

		if (lead ^ FLAT) {
		    if (dup2(lead, STDIN_FILENO) == -1) {
			break;
		    } else {
			if (close(lead)) {
			    break;
			}
		    }
		} else {
		    if (dup2(pipes_0[0], STDIN_FILENO) == -1) {
			break;
		    } else {
			if (close(pipes_0[0])) {
			    break;
			} else {
			    if (close(pipes_0[1])) {
				break;
			    }
			}
		    }
		}
		if (lift ^ FLAT) {
		    if (dup2(lift, STDOUT_FILENO) == -1) {
			break;
		    } else {
			if (close(lift)) {
			    break;
			}
		    }
		} else {
		    if (dup2(pipes_1[1], STDOUT_FILENO) == -1) {
			break;
		    } else {
			if (close(pipes_1[0])) {
			    break;
			} else {
			    if (close(pipes_1[1])) {
				break;
			    }
			}
		    }
		}

		{
		    unsigned c3rd;

		    c3rd = record_data->trailer.c3rd;
		    if (c3rd != MISS_C3RD) {
			if (c3rd == KILL_C3RD) {
			    close(STDERR_FILENO);
			} else {
			    int fd;

			    fd = open("/dev/null", O_WRONLY);
			    if (fd != -1) {
				if (dup2(fd, STDERR_FILENO) == -1) {
				    break;
				} else {
				    close(fd);
				}
			    }
			}
		    }
		}

		if (record_data->cleanup.call) {
		    if (record_data->cleanup.call(record_data->cleanup.back)) {
			break;
		    }
		}

		argv = record_data->command.argv;
		if (argv) {
		} else {
		    argv = null_argv;
		    null_argv[0] = record_data->command.path;
		    null_argv[1] = NULL;
		}

		execvp(record_data->command.path, (void *) argv);

		exec_xshell(record_data);
	    } while (0);

	    _exit(127);
	} else {
	    if (wait < (pid_t) 0) {
		break;
	    }
	}

	if (lead ^ FLAT) {
	    record_data->context.fd_1 = lead;
	} else {
	    if (close(pipes_0[0])) {
	    } else {
		pipes_0[0] = -1;
	    }

	    record_data->context.fd_1 = pipes_0[1];
	}
	if (lift ^ FLAT) {
	    record_data->context.fd_0 = lift;
	} else {
	    if (close(pipes_1[1])) {
	    } else {
		pipes_1[1] = -1;
	    }

	    record_data->context.fd_0 = pipes_1[0];
	}

	record_data->context.wait = wait;

	status = 0;
    } while (0);

    if (status) {
	if (pipes_0[0] != -1) {
	    close(pipes_0[0]);
	}
	if (pipes_0[1] != -1) {
	    close(pipes_0[1]);
	}
	if (pipes_1[0] != -1) {
	    close(pipes_1[0]);
	}
	if (pipes_1[1] != -1) {
	    close(pipes_1[1]);
	}
    }

    return status;
}


static int
ever_stream(int *fds)
{
    int status;

    status = pipe(fds);
    if (status) {
	fds[0] = -1;
	fds[1] = -1;
    } else {
	unsigned i = 0;

	for (; i < 2; i++) {
	    if (fcntl(fds[i], F_SETFD, 1) == -1) {
		status = 1;

		close(fds[0]);
		close(fds[1]);

		break;
	    }
	}
    }

    return status;
}


static int
exec_xshell(void *record)
{
    do {
	int (*post) (void *), (*type) (void *, const char *, unsigned);
	void *look;

	look = Record(record)->printer.look;

	post = Record(record)->printer.head;
	if (post) {
	    if (post(look)) {
		break;
	    }
	}

	type = Record(record)->printer.type;
	if (type) {
	    if (type(look, "cannot execute `", 16)) {
		break;
	    }
	    if (type(look, Record(record)->command.path,
		     strlen(Record(record)->command.path))) {
		break;
	    }
	    if (type(look, "': ", 3)) {
		break;
	    }
	    if (type(look, strerror(errno), strlen(strerror(errno)))) {
		break;
	    }
	}

	post = Record(record)->printer.tail;
	if (post) {
	    if (post(look)) {
		break;
	    }
	}
    } while (0);

    return -1;
}


int
x1f4_deck_xshell(void *record)
{
    int status;
    struct record_type *record_data;

    record_data = record;

    if (!record_data->command.path) {
	status = 0;
    } else {
	status = deck_xshell(record);
    }

    return status;
}


int
x1f4_fini_xshell(void **address)
{
    struct record_type *record_data;

    record_data = *address;

    free(record_data);

    return 0;
}


int
x1f4_hook_xshell(void *record, pid_t *wait)
{
    struct record_type *record_data;

    record_data = record;

    *wait = record_data->context.wait;

    return 0;
}


int
x1f4_init_xshell(void **address)
{
    int status;
    struct record_type *record_data;

    record_data = (struct record_type *) malloc(sizeof(struct record_type));
    if (!record_data) {
	status = -1;
    } else {
	status = 0;

	record_data->adapter.lead = FLAT;
	record_data->adapter.lift = FLAT;
	record_data->command.argv = NULL;
	record_data->command.path = NULL;
	record_data->cleanup.back = NULL;
	record_data->cleanup.call = NULL;
	record_data->printer.head = NULL;
	record_data->printer.look = NULL;
	record_data->printer.tail = NULL;
	record_data->printer.type = NULL;
	record_data->trailer.c3rd = C3RD;
	record_data->trailer.mask = NONE;

	*address = record_data;
    }

    return status;
}


int
x1f4_mode_xshell(void *record, struct xshell_type *xshell_data, unsigned mask)
{
    struct record_type *record_data;

    record_data = record;

    if (mask & ARGV_MASK) {
	record_data->command.argv = xshell_data->command.argv;
    }
    if (mask & BACK_MASK) {
	record_data->cleanup.back = xshell_data->cleanup.back;
    }
    if (mask & C3RD_MASK) {
	record_data->trailer.c3rd = xshell_data->trailer.c3rd;
    }
    if (mask & CALL_MASK) {
	record_data->cleanup.call = xshell_data->cleanup.call;
    }
    if (mask & HEAD_MASK) {
	record_data->printer.head = xshell_data->printer.head;
    }
    if (mask & LEAD_MASK) {
	record_data->adapter.lead = xshell_data->adapter.lead;
    }
    if (mask & LIFT_MASK) {
	record_data->adapter.lift = xshell_data->adapter.lift;
    }
    if (mask & LOOK_MASK) {
	record_data->printer.look = xshell_data->printer.look;
    }
    if (mask & MASK_MASK) {
	record_data->trailer.mask = xshell_data->trailer.mask;
    }
    if (mask & PATH_MASK) {
	record_data->command.path = xshell_data->command.path;
    }
    if (mask & TAIL_MASK) {
	record_data->printer.tail = xshell_data->printer.tail;
    }
    if (mask & TYPE_MASK) {
	record_data->printer.type = xshell_data->printer.type;
    }

    return 0;
}


int
x1f4_pipe_xshell(void *record, int *fd_0, int *fd_1)
{
    struct record_type *record_data;

    record_data = record;

    *fd_0 = record_data->context.fd_0;
    *fd_1 = record_data->context.fd_1;

    return 0;
}


int
x1f4_wait_xshell(void *record)
{
    struct record_type *record_data;

    record_data = record;

    if (record_data->adapter.lift ^ FLAT) {
    } else {
	close(record_data->context.fd_0);
    }
    if (record_data->adapter.lead ^ FLAT) {
    } else {
	close(record_data->context.fd_1);
    }

    return 0;
}
