/*
 * lxfile-a.e.c
 * Copyright (C) 2008, 2009, 2010, 2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#include <errno.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#include <string.h>

#include <exerrors.h>
#include <file.h>
#if !defined HAVE_LIBx1f4i0
# include <lintegral.v-d.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <lxfile-inter.h>
#include <lxfile-types.h>

#define false(e)			0

#define lxfile(lxfile) \
    ((struct lxfile_type *) (lxfile))

int
_libx1f4i0_lxfile_stat_else(void *lxfile, const char *path)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "will not open `", 15);
	    if (status) {
		break;
	    } else {
		status = push(data, path, strlen(path));
		if (status) {
		    break;
		} else {
		    status = push
			(data, "' for both read exclusive and write exclusiv"
			 "e were specified", 60);
		    if (status) {
			break;
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_file(void *lxfile, int class)
{
    int status;

    if (class == FILE_LINK_CLASS) {
	status = _libx1f4i0_lxfile_stat_link(lxfile);
    } else {
	if (class == FILE_LOSE_CLASS) {
	    status = _libx1f4i0_lxfile_stat_lose(lxfile);
	} else {
	    if (class == FILE_MODE_CLASS) {
		status = _libx1f4i0_lxfile_stat_mode(lxfile);
	    } else {
		if (false(class == FILE_OPEN_CLASS)) {
		} else {
		    if (false(class == FILE_OVER_CLASS)) {
		    } else {
			if (class == FILE_PUSH_CLASS) {
			    status = _libx1f4i0_lxfile_stat_push(lxfile);
			} else {
			    if (class == FILE_READ_CLASS) {
				status = _libx1f4i0_lxfile_stat_read(lxfile);
			    } else {
				if (class == FILE_SEEK_CLASS) {
				    status =
					_libx1f4i0_lxfile_stat_seek(lxfile);
				} else {
				    if (class == FILE_ZERO_CLASS) {
					status = _libx1f4i0_lxfile_stat_zero
					    (lxfile);
				    } else {
					status = X1f4_EX_CRITICAL;
				    }
				}
			    }
			}
		    }
		}
	    }
	}
    }

    return status;
}


int
_libx1f4i0_lxfile_stat_fine(void *lxfile, const char *path)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot set the close-on-exec flag on `", 38);
	    if (status) {
		break;
	    } else {
		status = push(data, path, strlen(path));
		if (status) {
		    break;
		} else {
		    status = push(data, "': ", 3);
		    if (status) {
			break;
		    } else {
			char *error;

			error = strerror(errno);
			status = push(data, error, strlen(error));
			if (status) {
			    break;
			}
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_free(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot free memory", 18);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CRITICAL;
}


int
_libx1f4i0_lxfile_stat_line(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot write: file open read-only", 33);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_link(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot allocate memory", 22);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CRITICAL;
}


int
_libx1f4i0_lxfile_stat_lose(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot close file: ", 19);
	    if (status) {
		break;
	    } else {
		char *error;

		error = strerror(errno);
		status = push(data, error, strlen(error));
		if (status) {
		    break;
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CRITICAL;
}


int
_libx1f4i0_lxfile_stat_love(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot write: file not open", 27);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_miss(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot seek: file not open", 26);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_mode(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot reallocate memory", 24);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CRITICAL;
}


int
_libx1f4i0_lxfile_stat_node(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot read: file open write-only", 33);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_none(void *lxfile, const char *path)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "will not open `", 15);
	    if (status) {
		break;
	    } else {
		status = push(data, path, strlen(path));
		if (status) {
		    break;
		} else {
		    status = push
			(data, "' for neither read nor write access was requ"
			 "ested", 49);
		    if (status) {
			break;
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_null(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot read: file not open", 26);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_open(void *lxfile, const char *path)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot open `", 13);
	    if (status) {
		break;
	    } else {
		status = push(data, path, strlen(path));
		if (status) {
		    break;
		} else {
		    status = push(data, "': ", 3);
		    if (status) {
			break;
		    } else {
			char *error;

			error = strerror(errno);
			status = push(data, error, strlen(error));
			if (status) {
			    break;
			}
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_peep(void *lxfile, long peep)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "out of range cache buffer access (", 34);
	    if (status) {
		break;
	    } else {
		status = x1f4_vprint_dlintegral(data, push, peep);
		if (status) {
		    break;
		} else {
		    status = push(data, ")", 1);
		    if (status) {
			break;
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_pipe(void *lxfile, const char *path)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "will not open `", 15);
	    if (status) {
		break;
	    } else {
		status = push(data, path, strlen(path));
		if (status) {
		    break;
		} else {
		    status = push
			(data, "' for read/write access is not yet supported",
			 44);
		    if (status) {
			break;
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_push(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot write file: ", 19);
	    if (status) {
		break;
	    } else {
		char *error;

		error = strerror(errno);
		status = push(data, error, strlen(error));
		if (status) {
		    break;
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_rate(void *lxfile, long read)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "negative read request (", 23);
	    if (status) {
		break;
	    } else {
		status = x1f4_vprint_dlintegral(data, push, read);
		if (status) {
		    break;
		} else {
		    status = push(data, ")", 1);
		    if (status) {
			break;
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_read(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot read file: ", 18);
	    if (status) {
		break;
	    } else {
		char *error;

		error = strerror(errno);
		status = push(data, error, strlen(error));
		if (status) {
		    break;
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_seek(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot seek file: ", 18);
	    if (status) {
		break;
	    } else {
		char *error;

		error = strerror(errno);
		status = push(data, error, strlen(error));
		if (status) {
		    break;
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_skip(void *lxfile, long skip)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "negative skip request (", 23);
	    if (status) {
		break;
	    } else {
		status = x1f4_vprint_dlintegral(data, push, skip);
		if (status) {
		    break;
		} else {
		    status = push(data, ")", 1);
		    if (status) {
			break;
		    }
		}
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}


int
_libx1f4i0_lxfile_stat_zero(void *lxfile)
{
    do {
	int (*line) (void *), (*post) (void *),
	    (*push) (void *, const char *, unsigned), status;
	void *data;

	data = lxfile(lxfile)->link_e.data;

	line = lxfile(lxfile)->link_e.line;
	if (line) {
	    status = line(data);
	    if (status) {
		break;
	    }
	}

	push = lxfile(lxfile)->link_e.push;
	if (push) {
	    status = push(data, "cannot write file", 17);
	    if (status) {
		break;
	    }
	}

	post = lxfile(lxfile)->link_e.post;
	if (post) {
	    status = post(data);
	    if (status) {
		break;
	    }
	}
    } while (0);

    return X1f4_EX_CAN_CONTINUE;
}
