/*
 * e4.7.1.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#include <e4-config.h>

#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#include <stddef.h>
#include <stdlib.h>
#include <string.h>

#include <ctype.0.h>
#include <e4-byte.h>
#include <e4-defs.h>
#include <e4-inter.h>
#include <e4-parse.h>
#include <e4-types.h>

#define SCALAR_BITS \
    ((1 << BILL) | (1 << MODE) | (1 << REAL))

#define EXPRESSION(expression) \
    ((e4_expression_type *) expression)

#define E_FREE(mcontext, mdata) \
    (EXPRESSION(mcontext)->m.free(EXPRESSION(mcontext)->m.data, (mdata)))
#define E_LINK(mcontext, mdata, size) \
    (EXPRESSION(mcontext)->m.link					      \
	(EXPRESSION(mcontext)->m.data, (void *) (mdata), (size)))
#define E_MODE(mcontext, mdata, size) \
    (EXPRESSION(mcontext)->m.mode					      \
	(EXPRESSION(mcontext)->m.data, (void *) (mdata), (size)))

#define M_FREE(mcontext, mdata) \
    (mcontext)->m.free((mcontext)->m.data, (mdata))
#define M_LINK(mcontext, mdata, size) \
    (mcontext)->m.link((mcontext)->m.data, (void *) (mdata), (size))
#define M_MODE(mcontext, mdata, size) \
    (mcontext)->m.mode((mcontext)->m.data, (void *) (mdata), (size))

#if __ALLOW_FUNCTION_POINTERS__
# define case_odb2(lock_data, lock_type, call, parser_data) \
    ((lock_data)->odbx_data == _x1f4_e4_pick				      \
     ? 0 : fix_odb2((lock_data), (lock_type), (call), (parser_data)))
#else
# define case_odb2(lock_data, lock_type, call, parser_data) \
    fix_odb2((lock_data), (lock_type), (call), (parser_data))
#endif				/* __ALLOW_FUNCTION_POINTERS__ */

typedef struct half_type {
    int *none;
    struct e4_hack_type *hack_data;
} half_type;

typedef struct lock_type {
    int type;
    struct e4_atom_type *atom_data;
    struct lock_type *lock_data;
    const struct e4_odbx_type *odbx_data;
} lock_type;

extern const struct e4_odbx_type _x1f4_e4_implicit[], _x1f4_e4_pick[];

static int call_hack(struct e4_hack_type *, struct parser_type *);
static int case_atom(struct e4_atom_type *, int *, struct parser_type *);
static int deck_atom(struct e4_atom_type *, struct parser_type *);
static int fix_hack(struct e4_hack_type *);
static int fix_odb2(struct lock_type *, int *, int *, struct parser_type *);
static int fix_side(struct e4_atom_type *);
static int fix_type(struct e4_hack_type *, int, int, struct parser_type *,
		    const struct e4_odbx_type *);
static int head_list(struct e4_hack_type **, const struct e4_line_type *,
		     struct parser_type *);
static int land_list(struct e4_hack_type **, const struct e4_line_type *,
		     unsigned, unsigned, struct parser_type *);
static int line_hack(struct e4_hack_type *, int (**) (x1f4_e4_LOAD_ARGS_0),
		     const int *, unsigned, unsigned);
static int link_hack(struct e4_hack_type *, int *,
		     int (**) (x1f4_e4_LOAD_ARGS_0), void *);
static int mind_ever(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int mind_this(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int near_bill(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int near_ever(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int near_last(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int near_mode(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int near_real(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int near_text(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int near_this(struct e4_atom_type *, struct half_type *,
		     struct parser_type *);
static int set_type(struct e4_odb1_type *, int, int);
static int type_call(struct e4_hack_type *, int *, int, int,
		     int (**) (x1f4_e4_LOAD_ARGS_0), struct parser_type *);
static int type_hack(struct e4_hack_type *, int *, int, int,
		     int (**) (x1f4_e4_LOAD_ARGS_0), struct parser_type *);
static int type_last(struct e4_hack_type *, int *, const int *, unsigned,
		     const struct e4_last_type *,
		     int (**) (x1f4_e4_LOAD_ARGS_0), struct e4_atom_type *,
		     struct parser_type *);
static int type_miss(struct e4_hack_type *, int *, const int *, unsigned,
		     const struct e4_last_type *, struct e4_atom_type *,
		     struct parser_type *);
static int type_post(struct e4_hack_type *, unsigned,
		     const struct e4_last_type *, struct e4_atom_type *,
		     struct parser_type *);

static void free_odb1(struct e4_odb1_type *, void *);
static void free_odbx(struct e4_hack_type *, void *);
#if __ALLOW_RIGHT_FIRST__
static void swap_call(struct e4_atom_type *, unsigned);
#endif				/* __ALLOW_RIGHT_FIRST__ */

static int (*const mind_atom[]) (struct e4_atom_type *, struct half_type *,
				 struct parser_type *) = {
/* *INDENT-OFF* */
    near_bill,
    near_mode,
    near_real,
    near_text,
    mind_ever,
    near_last,
    near_last,
    near_last,
    mind_this
/* *INDENT-ON* */
}, (*const near_atom[]) (struct e4_atom_type *, struct half_type *,
			 struct parser_type *) = {
/* *INDENT-OFF* */
    near_bill,
    near_mode,
    near_real,
    near_text,
    near_ever,
    near_last,
    near_last,
    near_last,
    near_this
/* *INDENT-ON* */
};

static int
call_hack(struct e4_hack_type *hack_data, struct parser_type *parser_data)
{
    int status = 0;
    struct e4_atom_type *atom_data;
    unsigned count;

    atom_data = hack_data->atoms;

    count = hack_data->count;

    do {
	struct e4_odb2_type *odb2_data;

	odb2_data = atom_data->odb2._2nd;
	if (odb2_data) {
	    struct e4_odb2_type odb2, *odb2_link = NULL;
	    unsigned land = 0;

	    do {
		int reset = 0;
		const struct e4_odbx_type *odbx_data;

		odbx_data = odb2_data->odbx_data;

#if __ALLOW_RIGHT_FIRST__
		if (odbx_data->flags & LEFT_LAST) {
		    struct e4_atom_type *lead_atom, *miss_atom;
		    unsigned left, post;

		    miss_atom = atom_data;
		    while (land) {
			struct e4_odb2_type *odb2_text;

			miss_atom--;
			land--;
			odb2_text = miss_atom->odb2._2nd;
			while (odb2_text) {
			    odb2_text = odb2_text->odb2_data;
			    land++;
			}
		    }
		    lead_atom = miss_atom;
		    land = 1;
		    do {
			struct e4_odb2_type *odb2_text;

			lead_atom--;
			land--;
			odb2_text = lead_atom->odb2._2nd;
			while (odb2_text) {
			    odb2_text = odb2_text->odb2_data;
			    land++;
			}
		    } while (land);

		    left = miss_atom - lead_atom;
		    post = atom_data - miss_atom + 1;

		    swap_call(lead_atom, left + post);

		    miss_atom = lead_atom + post;

		    swap_call(lead_atom, post);
		    swap_call(miss_atom, left);

		    miss_atom--;

		    if (odb2_link) {
			odb2_link->odb2_data = NULL;
		    } else {
			miss_atom->odb2._2nd = NULL;
		    }

		    odb2_link = atom_data->odb2._2nd;
		    if (odb2_link) {
			struct e4_odb2_type *odb2_call;

			while (1) {
			    land++;
			    odb2_call = odb2_link->odb2_data;
			    if (odb2_call) {
				land++;
				odb2_link = odb2_call->odb2_data;
				if (odb2_link) {
				} else {
				    odb2_call->odb2_data = odb2_data;
				    odb2_link = odb2_call;
				    if (1) {
					break;
				    }
				}
			    } else {
				odb2_link->odb2_data = odb2_data;
				if (1) {
				    break;
				}
			    }
			}
		    } else {
			atom_data->odb2._2nd = odb2_data;
		    }
		}
#endif				/* __ALLOW_RIGHT_FIRST__ */

		if (odbx_data->flags & E2ND_LINK) {
		    const struct e4_2nde_type *e2nd_data;

		    e2nd_data = odbx_data->extension2;
		    if (e2nd_data->bits & CALL_LINK) {
			struct e4_atom_type *lead_atom, *miss_atom;
			struct e4_hack_type *hack_text;
			const struct e4_line_type *line_data;

			miss_atom = atom_data;
			while (land) {
			    struct e4_odb2_type *odb2_text;

			    miss_atom--;
			    land--;
			    odb2_text = miss_atom->odb2._2nd;
			    while (odb2_text) {
				odb2_text = odb2_text->odb2_data;
				land++;
			    }
			}
			lead_atom = miss_atom;
			land = 1;
			do {
			    struct e4_odb2_type *odb2_text;

			    lead_atom--;
			    land--;
			    odb2_text = lead_atom->odb2._2nd;
			    while (odb2_text) {
				odb2_text = odb2_text->odb2_data;
				land++;
			    }
			} while (land);

			line_data = e2nd_data->line;

			status = land_list
			    (&hack_text, line_data, miss_atom - lead_atom,
			     atom_data - miss_atom + 1, parser_data);
			if (status) {
			    break;
			} else {
			    const int *args;
			    int (**post) (x1f4_e4_LOAD_ARGS_0);
			    struct e4_odb2_type *odb2_lock;
			    unsigned class, flags;

			    odb2_lock = odb2_data->odb2_data;

			    reset = 1;

			    if (odb2_link) {
				odb2_link->odb2_data = NULL;
			    } else {
				atom_data->odb2._2nd = NULL;
			    }

			    args = line_data->last.args + 2;

			    flags = parser_data->expression_data->flags;

			    class = line_data->last.flags;

			    post = (int (**) (x1f4_e4_LOAD_ARGS_0))
				(hack_text + 2);

			    memcpy
				(hack_text->atoms, lead_atom,
				 (char *) miss_atom - (char *) lead_atom);

			    line_hack(hack_text, post, args + 0, class, flags);

			    lead_atom->odb2._2nd = odb2_lock;
			    lead_atom->type = LAST;
			    lead_atom->data.last.base = hack_text;
			    lead_atom->odb1 = NULL;

			    if (class & TEXT_LINK) {
				_x1f4_e4_link_laSt(lead_atom);
			    } else {
				_x1f4_e4_link_last(lead_atom);
			    }

			    hack_text++;

			    post++;

			    memcpy
				(hack_text->atoms, miss_atom,
				 (char *) (atom_data + 1)
				 - (char *) miss_atom);

			    line_hack(hack_text, post, args + 1, class, flags);

			    if (count ^ 1) {
				memmove
				    (lead_atom + 1, atom_data + 1,
				     (count - 1)
				     * sizeof(struct e4_atom_type));
			    }

			    hack_data->count -= atom_data - lead_atom;

			    atom_data = lead_atom;

			    M_FREE(parser_data, odb2_data);

			    odb2.odb2_data = odb2_lock;

			    odb2_data = &odb2;

			    land--;
			}
		    }
		}

		land++;

		if (reset) {
		    odb2_link = NULL;
		} else {
		    odb2_link = odb2_data;
		}

		odb2_data = odb2_data->odb2_data;
	    } while (odb2_data);
	    if (status) {
		break;
	    }
	}

	count--;

	atom_data++;
    } while (count);

    if (status) {
    } else {
	if (parser_data->flags & PACK_ESTORAGE) {
	    M_MODE(parser_data, &hack_data->atoms,
		   hack_data->count * sizeof(struct e4_atom_type));
	}
    }

    return status;
}


static int
case_atom(struct e4_atom_type *atom_data, int *atom_type,
	  struct parser_type *parser_data)
{
    int status = 0;
    const struct e4_odbx_type *const *list;

    list = parser_data->attributes_data->odb1_data;
    if (!list) {
	status = _x1f4_e4_080704_error_g(parser_data->back_data);
    } else {
	int type;
	struct e4_odb1_type *base_odb1 = NULL, *last_odb1 = NULL, *odb1_data;

	type = *atom_type;

	odb1_data = atom_data->odb1;
	while (odb1_data) {
	    const char *e;
	    struct text_type *text_data;
	    unsigned length;

	    text_data = (struct text_type *) (odb1_data + 1);

	    e = text_data->data;
	    length = text_data->size;
	    while (length) {
		const struct e4_odbx_type *const *odbx, *odbx_data,
		    *odbx_text = NULL;
		unsigned toxic = 0;

		odbx = list;

		odbx_data = *odbx;
		while (odbx_data) {
		    unsigned shift;

		    shift = odbx_data->length;
		    if (toxic < shift) {
			if (shift < length + 1) {
			    if (*odbx_data->args == type) {
				const char *name;

				name = odbx_data->name;
				if (memcmp
				    ((void *) (e - shift), name, shift)) {
				} else {
				    odbx_text = odbx_data;
				    toxic = shift;
				}
			    }
			}
		    }

		    odbx++;
		    odbx_data = *odbx;
		}
		if (!odbx_text) {
		    status = _x1f4_e4_080704_error_i(parser_data->back_data);
		    break;
		} else {
		    struct e4_odb1_type *ocb1_data;

		    e -= toxic;
		    length -= toxic;

		    status = M_LINK(parser_data, &ocb1_data,
				    sizeof(struct e4_odb1_type));
		    if (status) {
			status = ALLOC_ERROR;
			break;
		    } else {
			if (base_odb1) {
			    last_odb1->odb1_data = ocb1_data;
			} else {
			    base_odb1 = ocb1_data;
			}

			last_odb1 = ocb1_data;

			ocb1_data->odb1_data = NULL;
			ocb1_data->odbx_data = odbx_text;

			type = odbx_text->type;
		    }
		}
	    }
	    if (status) {
		break;
	    }

	    odb1_data = odb1_data->odb1_data;
	}

	if (status) {
	    free_odb1(base_odb1, parser_data->expression_data);
	} else {
	    *atom_type = type;

	    free_odb1(atom_data->odb1, parser_data->expression_data);

	    atom_data->odb1 = base_odb1;
	}
    }

    return status;
}


static int
deck_atom(struct e4_atom_type *lead_atom, struct parser_type *parser_data)
{
    struct e4_odb1_type *odb1_data, *odb1_text = NULL;
    unsigned status = 0;

    odb1_data = lead_atom->odb1;
    while (odb1_data) {
	struct e4_odb1_type *odb1_lock;
	const struct e4_odbx_type *odbx_data;

	odb1_lock = odb1_data->odb1_data;

	odbx_data = odb1_data->odbx_data;
	if (odbx_data->flags & E2ND_LINK) {
	    const struct e4_2nde_type *e2nd_data;

	    e2nd_data = odbx_data->extension2;
	    if (e2nd_data->bits & CALL_LINK) {
		struct e4_hack_type *hack_text = NULL;
		const struct e4_line_type *line_data;

		line_data = e2nd_data->line;

		status = head_list(&hack_text, line_data, parser_data);
		if (status) {
		    break;
		} else {
		    struct e4_atom_type *atom_data;
		    unsigned class, flags;

		    flags = parser_data->expression_data->flags;

		    class = line_data->last.flags;

		    atom_data = hack_text->atoms;

		    *atom_data = *lead_atom;

		    atom_data->odb2._2nd = NULL;

		    if (odb1_text) {
			odb1_text->odb1_data = NULL;
		    } else {
			atom_data->odb1 = NULL;
		    }

		    line_hack
			(hack_text,
			 (int (**) (x1f4_e4_LOAD_ARGS_0)) (hack_text + 1),
			 NULL, class, flags);

		    lead_atom->type = LAST;
		    lead_atom->data.last.base = hack_text;
		    lead_atom->odb1 = odb1_lock;

		    if (class & TEXT_LINK) {
			_x1f4_e4_link_laSt(lead_atom);
		    } else {
			_x1f4_e4_link_last(lead_atom);
		    }

		    M_FREE(parser_data, odb1_data);
		}
	    } else {
		odb1_text = odb1_data;
	    }
	} else {
	    odb1_text = odb1_data;
	}

	odb1_data = odb1_lock;
    }

    return status;
}


static int
fix_hack(struct e4_hack_type *hack_data)
{
    struct e4_atom_type *atom_data;

    while (1) {
	atom_data = hack_data->atoms;
	if (atom_data->type == THIS) {
	    hack_data = atom_data->data.hack.data;
	} else {
	    fix_side(atom_data);
	    break;
	}
    }

    return 0;
}


static int
fix_odb2(struct lock_type *lock_data, int *lock_type, int *call,
	 struct parser_type *parser_data)
{
    int major, minor, status = 0;

    major = lock_data->type;
    minor = *lock_type;

    do {
	int miss_type;
	const int *args;
	const struct e4_odbx_type *odbx_data;
	unsigned flags;

	odbx_data = lock_data->odbx_data;
	args = odbx_data->args;
	if (major != args[0] || minor != args[1]) {
	    if (0) {
	    } else {
		const char *name;
		const struct e4_odbx_type *const *list;
		unsigned length;

		length = odbx_data->length;
		name = odbx_data->name;
		list = parser_data->attributes_data->odb2_data;
		odbx_data = *list;
		while (odbx_data) {
		    const char *text;

		    text = odbx_data->name;
		    if (odbx_data->length == length) {
			args = odbx_data->args;
			if (args[0] == major) {
			    if (args[1] == minor) {
				if (memcmp(odbx_data->name, name, length)) {
				} else {
				    miss_type = odbx_data->type;
				    break;
				}
			    }
			}
			if ((odbx_data->flags & (CALL_LOCK | LEFT_XSET))
			    == (CALL_LOCK | LEFT_XSET) && BASE_LOCK < major) {
			    if (major == minor) {
				if (memcmp(odbx_data->name, name, length)) {
				} else {
				    miss_type = major;
				    break;
				}
			    } else {
			    }
			}
		    }

		    list++;
		    odbx_data = *list;
		}

		if (odbx_data) {
		    lock_data->odbx_data = odbx_data;
		} else {
		    status = _x1f4_e4_080704_error_6
			(parser_data->back_data, name, length, major, minor);
		    if (1) {
			break;
		    }
		}
	    }
	} else {
	    miss_type = odbx_data->type;
	}

	flags = odbx_data->flags;

	if (flags & E2ND_LINK) {
	    if (((struct e4_2nde_type *)
		 odbx_data->extension2)->bits & CALL_LINK) {
		*call = 1;
	    }
	}

#if __ALLOW_RIGHT_FIRST__
	if (flags & LEFT_LAST) {
	    *call = 1;
	}
#endif				/* __ALLOW_RIGHT_FIRST__ */

	do {
	    if (!(flags & LEFT_XSET)) {
		struct e4_atom_type *atom_data;

		atom_data = lock_data->atom_data;
		if (atom_data) {
		    int type;

		    type = atom_data->type;
		    if (type == EVER) {
			if (atom_data->data.ever.ever_data->flags
			    & READ_ONLY) {
			} else {
			    fix_side(atom_data);
			}
		    } else {
			if (atom_data->type != THIS) {
			} else {
			    struct e4_hack_type *hack_data;

			    hack_data = atom_data->data.hack.data;
			    if (hack_data->flags & READ_ONLY) {
			    } else {
				fix_hack(hack_data);
			    }
			}
		    }
		}
	    } else {
		struct e4_atom_type *atom_data;

		atom_data = lock_data->atom_data;
		if (!atom_data) {
		    status = _x1f4_e4_080704_error_b(parser_data->back_data);
		    break;
		} else {
		    int type;

		    type = atom_data->type;
		    if (type == EVER) {
			if (atom_data->data.ever.ever_data->flags
			    & READ_ONLY) {
			    status = _x1f4_e4_080704_error_7
				(parser_data->back_data, atom_data,
				 parser_data->expression);
			    break;
			}
		    } else {
			if (atom_data->type != THIS) {
			    status = _x1f4_e4_080704_error_b
				(parser_data->back_data);
			    break;
			} else {
			    struct e4_hack_type *hack_data;

			    hack_data = atom_data->data.hack.data;
			    if (hack_data->flags & READ_ONLY) {
				status = _x1f4_e4_080704_error_c
				    (parser_data->back_data, hack_data,
				     parser_data->expression);
				break;
			    }
			}
		    }
		}
	    }
	} while (0);
	if (status) {
	    break;
	}

	*lock_type = miss_type;
    } while (0);

    return status;
}


static int
fix_side(struct e4_atom_type *atom_data)
{
    const struct e4_ever_type *ever_data;
    int type;

    ever_data = atom_data->data.ever.ever_data;

    type = ever_data->type;
    if (ever_data->flags & REFERENCE) {
	if (type == BILL) {
	    atom_data->load = _x1f4_e4_load_Bill;
	} else if (type == MODE) {
	    atom_data->load = _x1f4_e4_load_Mode;
	} else if (type == REAL) {
	    atom_data->load = _x1f4_e4_load_Real;
	} else if (type == TEXT) {
	    atom_data->load = _x1f4_e4_load_Text;
	} else {
	    atom_data->load = _x1f4_e4_load_User;
	}
    } else {
	if (type == BILL) {
	    atom_data->load = _x1f4_e4_load_BILL;
	} else if (type == MODE) {
	    atom_data->load = _x1f4_e4_load_MODE;
	} else if (type == REAL) {
	    atom_data->load = _x1f4_e4_load_REAL;
	} else if (type == TEXT) {
	    atom_data->load = _x1f4_e4_load_TEXT;
	} else {
	    atom_data->load = _x1f4_e4_load_USER;
	}
    }

    return 0;
}


static int
fix_type(struct e4_hack_type *hack_data, int type, int name,
	 struct parser_type *parser_data,
	 const struct e4_odbx_type *odbx_data)
{
    int status;
    struct e4_odb1_type *odb1_data;

    status = M_LINK(parser_data, &odb1_data, sizeof(struct e4_odb1_type));
    if (status) {
	status = ALLOC_ERROR;
    } else {
	odb1_data->odb1_data = NULL;
	if (hack_data->count == 1) {
	    struct e4_atom_type *atom_data;
	    struct e4_odb1_type *atom_odb1;

	    atom_data = hack_data->atoms;
	    atom_odb1 = atom_data->odb1;
	    if (atom_odb1) {
		while (atom_odb1->odb1_data) {
		    atom_odb1 = atom_odb1->odb1_data;
		}
		atom_odb1->odb1_data = odb1_data;
	    } else {
		atom_data->odb1 = odb1_data;
	    }

	    if (odbx_data) {
		odb1_data->odbx_data = odbx_data;
	    } else {
		set_type(odb1_data, name, type);
	    }
	} else {
	    struct e4_hack_type *hack_this;

	    status =
		M_LINK(parser_data, &hack_this, sizeof(struct e4_hack_type));
	    if (status) {
		M_FREE(parser_data, odb1_data);
		status = ALLOC_ERROR;
	    } else {
		struct e4_atom_type *atom_data;

		status = M_LINK(parser_data, &atom_data,
				sizeof(struct e4_atom_type));
		if (status) {
		    M_FREE(parser_data, hack_this);
		    M_FREE(parser_data, odb1_data);
		    status = ALLOC_ERROR;
		} else {
		    atom_data->data.hack.data = hack_this;
		    atom_data->load = _x1f4_e4_load_this;
		    atom_data->odb1 = odb1_data;
		    atom_data->odb2._2nd = NULL;
		    atom_data->type = THIS;
		    *hack_this = *hack_data;
		    hack_data->count = 1;
		    hack_data->atoms = atom_data;
		    hack_data->flags = READ_ONLY;
		    _x1f4_e4_fast_hack
			(hack_data, parser_data->expression_data->flags,
			 &atom_data->data.hack.load);

		    if (odbx_data) {
			odb1_data->odbx_data = odbx_data;
		    } else {
			set_type(odb1_data, name, type);
		    }
		}
	    }
	}
    }

    return status;
}


static int
head_list(struct e4_hack_type **hack, const struct e4_line_type *line_data,
	  struct parser_type *parser_data)
{
#define BACK_HACK(count, excess) \
    (offset + (sizeof(struct e4_hack_type)				      \
	       + sizeof(int (*) (x1f4_e4_LOAD_ARGS_0))))

    int status;
    struct e4_hack_type *hack_data;
    unsigned offset;

    offset = sizeof(struct e4_base_type);

    status = M_LINK(parser_data, &hack_data, BACK_HACK(count, excess));
    if (status) {
	status = ALLOC_ERROR;
    } else {
	void *fast;

	status = M_LINK(parser_data, &fast, sizeof(struct e4_atom_type));
	if (status) {
	    status = ALLOC_ERROR;

	    M_FREE(parser_data, hack_data);
	} else {
	    hack_data = (struct e4_hack_type *) ((char *) hack_data + offset);

	    hack_base(hack_data)->last_data = &line_data->last;

	    if (line_data->last.flags & TEXT_LINK) {
		hack_base(hack_data)->text = (void *) line_data->text;
	    }

	    *hack = hack_data;

	    hack_data->count = 1;
	    hack_data->atoms = fast;
	    /*
	     * likely unused, will set to a possibly incorrect value
	     */
	    hack_data->flags = READ_ONLY;
	}
    }

    return status;

#undef BACK_HACK
}


static int
land_list(struct e4_hack_type **hack, const struct e4_line_type *line_data,
	  unsigned lead, unsigned miss, struct parser_type *parser_data)
{
#define BACK_HACK(count, excess) \
    (offset + ((sizeof(struct e4_hack_type)				      \
		+ sizeof(int (*) (x1f4_e4_LOAD_ARGS_0))) << 1))

    int status;
    struct e4_hack_type *hack_data;
    unsigned offset;

    offset = sizeof(struct e4_base_type);

    status = M_LINK(parser_data, &hack_data, BACK_HACK(count, excess));
    if (status) {
	status = ALLOC_ERROR;
    } else {
	void *fast;

	status =
	    M_LINK(parser_data, &fast, lead * sizeof(struct e4_atom_type));
	if (status) {
	    status = ALLOC_ERROR;

	    M_FREE(parser_data, hack_data);
	} else {
	    void *ever;

	    status =
		M_LINK(parser_data, &ever, miss * sizeof(struct e4_atom_type));
	    if (status) {
		status = ALLOC_ERROR;

		M_FREE(parser_data, fast);
		M_FREE(parser_data, hack_data);
	    } else {
		hack_data = (struct e4_hack_type *)
		    ((char *) hack_data + offset);

		hack_base(hack_data)->last_data = &line_data->last;

		if (line_data->last.flags & TEXT_LINK) {
		    hack_base(hack_data)->text = (void *) line_data->text;
		}

		*hack = hack_data;

		hack_data->count = lead;
		hack_data->atoms = fast;
		/*
		 * likely unused, will set to a possibly incorrect value
		 */
		hack_data->flags = READ_ONLY;

		hack_data++;

		hack_data->count = miss;
		hack_data->atoms = ever;
		/*
		 * likely unused, will set to a possibly incorrect value
		 */
		hack_data->flags = READ_ONLY & 0;
	    }
	}
    }

    return status;

#undef BACK_HACK
}


static int
line_hack(struct e4_hack_type *hack_data, int (**post) (x1f4_e4_LOAD_ARGS_0),
	  const int *bits, unsigned flags, unsigned e_flags)
{
    unsigned class;

    _x1f4_e4_fast_hack(hack_data, e_flags, post);

    if (flags & POST_TYPE) {
	class = *bits;
    } else {
	class = 0;
    }

    _x1f4_e4_fast_miss(hack_data, class, post);

    return 0;
}


static int
link_hack(struct e4_hack_type *hack_data, int *type,
	  int (**load) (x1f4_e4_LOAD_ARGS_0), void *parser)
{
    int call = 0, hack_type = 0, (*const *link_atom)
	(struct e4_atom_type *, struct half_type *, struct parser_type *),
	none, status = 0;
    struct e4_atom_type *atom_data;
    struct half_type half;
    struct lock_type *lock_data = NULL;
    struct parser_type *parser_data;
    unsigned count, i;

    parser_data = parser;

    count = hack_data->count;

    link_atom = count == 1 ? mind_atom : near_atom;

    half.hack_data = hack_data;
    half.none = &none;

    atom_data = hack_data->atoms;
    for (i = count; i; i--) {
	none = atom_data->type;

	status = link_atom[none](atom_data, &half, parser_data);
	if (status) {
	    break;
	}

	if (1) {
	    const struct e4_odbx_type *odbx_data;

	    odbx_data = atom_data->odb2._1st;
	    atom_data->odb2._2nd = NULL;

	    if (atom_data->odb1) {
		status = case_atom(atom_data, &none, parser_data);
		if (status) {
		    atom_data++;
		    break;
		} else {
		    status = deck_atom(atom_data, parser_data);
		    if (status) {
			atom_data++;
			break;
		    } else {
			hack_data->flags |= READ_ONLY;
		    }
		}
	    }

	    if (count == 1) {
		hack_type = none;
	    } else if (i == 1) {
		struct e4_odb2_type *odb2_data = NULL;

		do {
		    struct lock_type *free_lock;
		    void *data;

		    status = M_LINK
			(parser_data, &data, sizeof(struct e4_odb2_type));
		    if (status) {
			status = ALLOC_ERROR;
			break;
		    }
		    if (odb2_data) {
			odb2_data->odb2_data = data;
			odb2_data = data;
		    } else {
			atom_data->odb2._2nd = data;
			odb2_data = data;
		    }

		    odb2_data->odb2_data = NULL;

		    status = case_odb2(lock_data, &none, &call, parser_data);
		    if (status) {
			break;
		    }

		    odb2_data->odbx_data = lock_data->odbx_data;

		    free_lock = lock_data;
		    lock_data = lock_data->lock_data;
		    M_FREE(parser_data, free_lock);
		} while (lock_data);
		if (status) {
		    atom_data++;
		    break;
		}

		hack_type = none;
	    } else {
		if (!lock_data) {
		    status = M_LINK
			(parser_data, &lock_data, sizeof(struct lock_type));
		    if (status) {
			status = ALLOC_ERROR;
			atom_data++;
			break;
		    }

		    lock_data->atom_data = atom_data;
		    lock_data->lock_data = NULL;
		    lock_data->odbx_data = odbx_data;
		    lock_data->type = none;
		} else {
		    int priority;
		    const struct e4_odbx_type *odbx_text;

		    odbx_text = lock_data->odbx_data;
		    priority = odbx_data->priority;
		    if (priority < odbx_text->priority
			|| (priority == odbx_text->priority
			    && (odbx_data->flags & odbx_text->flags
				& BACK_LINK))) {
			struct lock_type *next_lock;

			status = M_LINK
			    (parser_data, &next_lock,
			     sizeof(struct lock_type));
			if (status) {
			    status = ALLOC_ERROR;
			    atom_data++;
			    break;
			}

			next_lock->lock_data = lock_data;
			lock_data = next_lock;

			lock_data->atom_data = atom_data;
			lock_data->odbx_data = odbx_data;
			lock_data->type = none;
		    } else {
			struct e4_odb2_type *odb2_data = NULL;
			struct lock_type *free_lock = NULL;

			do {
			    void *data;

			    status = M_LINK
				(parser_data, &data,
				 sizeof(struct e4_odb2_type));
			    if (status) {
				status = ALLOC_ERROR;
				break;
			    }
			    if (odb2_data) {
				odb2_data->odb2_data = data;
				odb2_data = data;
			    } else {
				atom_data->odb2._2nd = data;
				odb2_data = data;
			    }

			    odb2_data->odb2_data = NULL;

			    status = case_odb2
				(lock_data, &none, &call, parser_data);
			    if (status) {
				break;
			    }

			    odb2_data->odbx_data = lock_data->odbx_data;

			    if (free_lock) {
				M_FREE(parser_data, free_lock);
			    }
			    free_lock = lock_data;
			    lock_data = lock_data->lock_data;

			    if (lock_data) {
			    } else {
				break;
			    }
			    if (priority < lock_data->odbx_data->priority) {
				break;
			    }
			} while (1);
			if (status) {
			    atom_data++;
			    break;
			}

			free_lock->lock_data = lock_data;
			lock_data = free_lock;

			lock_data->atom_data = NULL;
			lock_data->odbx_data = odbx_data;
			lock_data->type = none;
		    }
		}
	    }
	}

	atom_data++;
    }

    while (lock_data) {
	struct lock_type *free_lock;

	free_lock = lock_data;
	lock_data = lock_data->lock_data;
	M_FREE(parser_data, free_lock);
    }

    if (call) {
	if (status) {
	} else {
	    status = call_hack(hack_data, parser_data);
	}
    }

    if (status) {
	hack_data->count =
	    atom_data - (struct e4_atom_type *) hack_data->atoms;
	free_odbx(hack_data, parser_data->expression_data);
	hack_data->count = count;
    } else {
	*type = hack_type;

	_x1f4_e4_fast_hack
	    (hack_data, parser_data->expression_data->flags, load);
    }

    return status;
}


static int
mind_ever(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    const struct e4_ever_type *ever_data;
    unsigned flags;

    ever_data = atom_data->data.ever.ever_data;
    flags = ever_data->flags;
    *half_data->none = ever_data->type;
    if (flags & READ_ONLY) {
	fix_side(atom_data);
    } else {
	if (1) {
	    half_data->hack_data->flags &= ~READ_ONLY;
	}
	if (flags & REFERENCE) {
	    atom_data->load = _x1f4_e4_load_clip;
	} else {
	    atom_data->load = _x1f4_e4_load_ever;
	}
    }

    return 0;
}


static int
mind_this(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    int status;

    status = link_hack
	(atom_data->data.hack.data, half_data->none,
	 &atom_data->data.hack.load, parser_data);
    if (status) {
    } else {
	if (1) {
	    atom_data->load = _x1f4_e4_load_this;
	    if (1) {
		if (((struct e4_hack_type *) atom_data->data.hack.data)->flags
		    & READ_ONLY) {
		} else {
		    half_data->hack_data->flags &= ~READ_ONLY;
		}
	    }
	}
    }

    return status;
}


static int
near_bill(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    atom_data->load = _x1f4_e4_load_bill;

    return 0;
}


static int
near_ever(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    const struct e4_ever_type *ever_data;
    unsigned flags;

    ever_data = atom_data->data.ever.ever_data;
    flags = ever_data->flags;
    *half_data->none = ever_data->type;
    if (flags & READ_ONLY) {
	fix_side(atom_data);
    } else {
	if (flags & REFERENCE) {
	    atom_data->load = _x1f4_e4_load_clip;
	} else {
	    atom_data->load = _x1f4_e4_load_ever;
	}
    }

    return 0;
}


static int
near_last(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    int (**post) (x1f4_e4_LOAD_ARGS_0), status = 0, type;
    const int *args;
    struct e4_hack_type *hack_data;
    const struct e4_last_type *last_data;
    unsigned count, flags, j, miss = 0;

    type = atom_data->type;

    hack_data = atom_data->data.last.base;
    last_data = hack_base(hack_data)->last_data;

    flags = last_data->flags;

    if (0) {
    } else {
	args = last_data->args;
    }

    count = last_data->count;

    j = count;

    post = (int (**) (x1f4_e4_LOAD_ARGS_0)) (hack_data + j);

    while (j) {
	int type;
	unsigned class;

	status = link_hack(hack_data, &type, post, parser_data);
	if (status) {
	    break;
	}

	--j;

	if (!(flags & POST_TYPE)) {
	    class = 0;

	    if (type != *args) {
		status = type_last
		    (hack_data, &type, args, j, last_data, post, atom_data,
		     parser_data);
		if (status) {
		    hack_data++;
		    break;
		}
	    }
	} else {
	    class = args[count];

	    if (!(class & HERE_XSET)) {
		if (type != *args) {
		    status = type_last
			(hack_data, &type, args, j, last_data, post, atom_data,
			 parser_data);
		    if (status) {
			hack_data++;
			break;
		    } else {
			class = args[count];
		    }
		}
	    } else {
		if (hack_data->flags & READ_ONLY) {
		    status = type_post
			(hack_data, j, last_data, atom_data, parser_data);
		    if (status) {
			hack_data++;
			break;
		    }
		} else {
		    if (type != *args) {
			status = type_miss
			    (hack_data, &type, args, j, last_data, atom_data,
			     parser_data);
			if (status) {
			    hack_data++;
			    break;
			}
		    } else {
			if (0) {
			} else {
			    miss = 1;
			}
		    }
		}
	    }
	}

	_x1f4_e4_fast_miss(hack_data, class, post);

	post++;

	hack_data++;

	args++;
    }
    if (status) {
	j = last_data->count - j;
	for (; j; j--) {
	    --hack_data;
	    free_odbx(hack_data, parser_data->expression_data);
	}
    } else {
	*half_data->none = last_data->type;

	if (type == PICK) {
	    atom_data->load = _x1f4_e4_load_pick;
	} else {
	    if (type == LOCK) {
		atom_data->load = _x1f4_e4_load_lock;
	    } else {
		if (1) {
		    if (1) {
			if (last_data->flags & TEXT_LINK) {
			    _x1f4_e4_link_laSt(atom_data);
			} else {
			    _x1f4_e4_link_last(atom_data);
			}
		    }
		}
	    }
	}
    }

    return status;
}


static int
near_mode(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    atom_data->load = _x1f4_e4_load_mode;

    return 0;
}


static int
near_real(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    atom_data->load = _x1f4_e4_load_real;

    return 0;
}


static int
near_text(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    atom_data->load = _x1f4_e4_load_text;

    return 0;
}


static int
near_this(struct e4_atom_type *atom_data, struct half_type *half_data,
	  struct parser_type *parser_data)
{
    int status;

    status = link_hack
	(atom_data->data.hack.data, half_data->none,
	 &atom_data->data.hack.load, parser_data);
    if (status) {
    } else {
	if (1) {
	    atom_data->load = _x1f4_e4_load_this;
	}
    }

    return status;
}


static int
set_type(struct e4_odb1_type *odb1_data, int name, int type)
{
    if (name == BILL) {
	if (type == MODE) {
	    odb1_data->odbx_data = _x1f4_e4_implicit + 0;
	} else {
	    odb1_data->odbx_data = _x1f4_e4_implicit + 1;
	}
    } else {
	if (name == MODE) {
	    if (type == BILL) {
		odb1_data->odbx_data = _x1f4_e4_implicit + 2;
	    } else {
		odb1_data->odbx_data = _x1f4_e4_implicit + 3;
	    }
	} else {
	    if (type == BILL) {
		odb1_data->odbx_data = _x1f4_e4_implicit + 4;
	    } else {
		odb1_data->odbx_data = _x1f4_e4_implicit + 5;
	    }
	}
    }

    return 0;
}


static int
type_call(struct e4_hack_type *hack_data, int *type, int miss, int name,
	  int (**load) (x1f4_e4_LOAD_ARGS_0), struct parser_type *parser_data)
{
    int status;

    /*
     * the _load_ method needs be set if hack is modified.
     * _fix_type_ does modify it in certain cases, yet for such cases
     * (single atom, unary operator) the subsequent call to _x1f4_e4_fast_miss_
     * sets the _load_ method.
     */

    do {
	if ((miss | name) & ~15) {
	} else {
	    if ((1 << miss) & SCALAR_BITS && (1 << name) & SCALAR_BITS) {
		status = fix_type(hack_data, miss, name, parser_data, NULL);
		if (status) {
		} else {
		    *type = name;
		}

		break;
	    }
	}

	if (parser_data->flags & IMPLICIT_LINK) {
	    int (*lock) (const void *, int, int, const struct e4_odbx_type **);
	    const struct e4_attributes_type *attributes_data;

	    attributes_data = parser_data->attributes_data;

	    lock = attributes_data->trap_0.get;
	    if (lock) {
		const struct e4_odbx_type *odbx_data;

		if (lock
		    (attributes_data->trap_0.context, name, miss,
		     &odbx_data)) {
		} else {
		    status = fix_type
			(hack_data, miss, name, parser_data, odbx_data);
		    if (status) {
		    } else {
			*type = name;
		    }

		    break;
		}
	    }
	}

	if (parser_data->flags & COMPOSER_LINK) {
	    int (*lock) (const void *, int, int, const struct e4_line_type **);
	    const struct e4_attributes_type *attributes_data;

	    attributes_data = parser_data->attributes_data;

	    lock = attributes_data->turn_0.get;
	    if (lock) {
		const struct e4_line_type *line_data;

		if (lock
		    (attributes_data->turn_0.context, name, miss,
		     &line_data)) {
		} else {
		    struct e4_atom_type *atom_data;

		    /*
		     * TODO
		     *
		     * make a function out of this block.  as for the
		     * IMPLICIT_LINK fix.
		     */
		    status = M_LINK(parser_data, &atom_data,
				    sizeof(struct e4_atom_type));
		    if (status) {
			status = ALLOC_ERROR;
		    } else {
			struct e4_hack_type *hack_this;
			const struct e4_last_type *last_data;
			unsigned offset;

			last_data = &line_data->last;

#define BACK_HACK(count, excess) \
    (offset +           (sizeof(struct e4_hack_type)			      \
			 + sizeof(int (*) (x1f4_e4_LOAD_ARGS_0))))


			offset = sizeof(struct e4_base_type);

			status = M_LINK(parser_data, &hack_this,
					BACK_HACK(count, excess));
			if (status) {
			    status = ALLOC_ERROR;
			    M_FREE(parser_data, atom_data);
			} else {
			    int (**post) (x1f4_e4_LOAD_ARGS_0);

			    hack_this = (struct e4_hack_type *)
				((char *) hack_this + offset);

			    if (last_data->flags & TEXT_LINK) {
				hack_base(hack_this)->text =
				    (void *) line_data->text;
			    }

			    atom_data->odb1 = NULL;
			    atom_data->odb2._1st = NULL;
			    atom_data->type = LAST;
			    atom_data->data.last.base = hack_this;
			    hack_base(hack_this)->last_data = last_data;

			    _x1f4_e4_fast_miss(hack_data, 0, load);

			    post = (void *) (hack_this + 1);
			    *post = *load;

			    *hack_this = *hack_data;

			    hack_data->count = 1;
			    hack_data->atoms = atom_data;
			    hack_data->flags = READ_ONLY;

			    _x1f4_e4_fast_hack
				(hack_data,
				 parser_data->expression_data->flags, load);

			    if (1) {
				if (last_data->flags & TEXT_LINK) {
				    _x1f4_e4_link_laSt(atom_data);
				} else {
				    _x1f4_e4_link_last(atom_data);
				}
			    }

			    *type = name;
			}

#undef BACK_HACK
		    }

		    break;
		}
	    }
	}

	status = PARSE_ERROR;
    } while (0);

    return status;
}


static int
type_hack(struct e4_hack_type *hack_data, int *type, int miss, int name,
	  int (**load) (x1f4_e4_LOAD_ARGS_0), struct parser_type *parser_data)
{
    int status;

    status = type_call(hack_data, type, miss, name, load, parser_data);
    if (status) {
	if (status == PARSE_ERROR) {
	    status = _x1f4_e4_130607_error_0
		(parser_data->back_data, name, miss);
	}
    }

    return status;
}


static int
type_last(struct e4_hack_type *hack_data, int *type, const int *args,
	  unsigned j, const struct e4_last_type *last_data,
	  int (**load) (x1f4_e4_LOAD_ARGS_0), struct e4_atom_type *atom_data,
	  struct parser_type *parser_data)
{
    int miss, name, status;

    miss = *type;
    name = *args;

    if (name == SIDE) {
	if (miss == VOID) {
	    status = _x1f4_e4_130704_error_0(parser_data->back_data);
	} else {
	    int *down;

	    down = (int *) args;
	    *down = miss;

	    status = 0;

	    if (hack_data->flags & READ_ONLY) {
	    } else {
		unsigned flags;

		flags = last_data->flags;
		if (flags & LINK_PASS) {
		    if (flags & FLAT_LINE) {
			const struct e4_last_type *last_text;

			last_text = (void *)
			    ((struct e4_flat_type *) last_data - 2)->flat_data;
			if (last_text->count < last_data->count - j) {
			    down[last_data->count] |= HERE_XSET;
			}
		    } else {
			down[last_data->count] |= HERE_XSET;
		    }
		}
	    }
	}
    } else {
	status = type_call(hack_data, type, miss, name, load, parser_data);
	if (status) {
	    if (status == PARSE_ERROR) {
		status = _x1f4_e4_080704_error_k
		    (parser_data->back_data, atom_data, last_data->count - j,
		     name, miss, parser_data->expression);
	    }
	}
    }

    return status;
}


static int
type_miss(struct e4_hack_type *hack_data, int *type, const int *args,
	  unsigned j, const struct e4_last_type *last_data,
	  struct e4_atom_type *atom_data, struct parser_type *parser_data)
{
    int status;

    if (*args == SIDE) {
	int *down;

	down = (int *) args;
	*down = *type;

	status = 0;
    } else {
	status = _x1f4_e4_081014_error_1
	    (parser_data->back_data, hack_data, atom_data,
	     last_data->count - j, *args, *type, parser_data->expression);
    }

    return status;
}


static int
type_post(struct e4_hack_type *hack_data, unsigned j,
	  const struct e4_last_type *last_data, struct e4_atom_type *atom_data,
	  struct parser_type *parser_data)
{
    return _x1f4_e4_081014_error_2
	(parser_data->back_data, hack_data, atom_data, last_data->count - j,
	 parser_data->expression);
}


static void
free_odb1(struct e4_odb1_type *odb1_data, void *expression)
{
    while (odb1_data) {
	struct e4_odb1_type *free_odb1;

	free_odb1 = odb1_data;
	odb1_data = odb1_data->odb1_data;
	E_FREE(expression, free_odb1);
    }
}


static void
free_odbx(struct e4_hack_type *hack_data, void *expression)
{
    struct e4_atom_type *atom_data;
    unsigned i;

    atom_data = hack_data->atoms;
    i = hack_data->count;
    for (; i; i--) {
	_x1f4_e4_free_odbx(atom_data, expression);
	atom_data++;
    }
}


#if __ALLOW_RIGHT_FIRST__
static void
swap_call(struct e4_atom_type *atom_data, unsigned count)
{
    struct e4_atom_type *atom_text;

    atom_text = atom_data + count;
    count >>= 1;
    for (; count; count--) {
	struct e4_atom_type atom;

	atom = *--atom_text;
	*atom_text = *atom_data;
	*atom_data++ = atom;
    }
}
#endif				/* __ALLOW_RIGHT_FIRST__ */


int
_x1f4_e4_link_hack(struct e4_hack_type *hack_data, int *type,
		   int (**load) (x1f4_e4_LOAD_ARGS_0), void *parser)
{
    int status;

    status = link_hack(hack_data, type, load, parser);
    if (status) {
    } else {
	struct parser_type *parser_data;

	parser_data = parser;
	if (parser_data->flags & SEQUENCE_LINK) {
	    int miss, name;

	    miss = *type;
	    name = parser_data->attributes_data->link_z.type;
	    if (name ^ miss) {
		status = type_hack
		    (hack_data, type, miss, name, load, parser_data);
	    }
	}
    }

    return status;
}
