/*
 * e4-e.0.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if !defined HAVE_LIBx1f4i0
# include <cstring.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <e4-defs.h>
#include <e4-inter.h>
#include <e4-types.h>
#if !defined HAVE_LIBx1f4i0
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */

extern const struct e4_odbx_type _x1f4_e4_implicit[], _x1f4_e4_pick[];

static void print_Atom(FILE *, struct e4_atom_type *, unsigned, unsigned *);
static void print_atom(FILE *, struct e4_atom_type *, unsigned);
static void print_bill(FILE *, struct e4_atom_type *, unsigned);
static void print_ever(FILE *, struct e4_atom_type *, unsigned);
static void print_hack(FILE *, struct e4_hack_type *, unsigned);
static void print_last(FILE *, struct e4_atom_type *, unsigned);
static void print_lock(FILE *, struct e4_atom_type *, unsigned);
static void print_mode(FILE *, struct e4_atom_type *, unsigned);
static void print_odb1(FILE *, struct e4_odb1_type *, unsigned);
static void print_odb2(FILE *, const struct e4_odbx_type *, unsigned);
static void print_pick(FILE *, struct e4_atom_type *, unsigned);
static void print_real(FILE *, struct e4_atom_type *, unsigned);
static void print_text(FILE *, struct e4_atom_type *, unsigned);
static void print_this(FILE *, struct e4_atom_type *, unsigned);

static void (*print_type[]) (FILE *, struct e4_atom_type *, unsigned) = {
/* *INDENT-OFF* */
    print_bill,
    print_mode,
    print_real,
    print_text,
    print_ever,
    print_last,
    print_lock,
    print_pick,
    print_this
/* *INDENT-ON* */
};

static void
print_Atom(FILE *file, struct e4_atom_type *atom_data, unsigned flags,
	   unsigned *state)
{
    unsigned c;

    c = *state;
    for (; c; c--) {
	fputc('(', file);
    }
    print_atom(file, atom_data, flags);
    state++;
    c = *state;
    for (; c; c--) {
	fputc(')', file);
    }
}


static void
print_atom(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    struct e4_odb1_type *odb1_data;

    odb1_data = atom_data->odb1;
    if (odb1_data) {
	print_odb1(file, odb1_data, flags);
    }

    print_type[atom_data->type](file, atom_data, flags);
}


static void
print_bill(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    fprintf(file, "%lu", *(C_BILL *) &atom_data->data.lead.data);
    if (flags & DETAIL_CONSTANTS) {
	fputc('z', file);
    }
}


static void
print_ever(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    fputs(atom_data->data.ever.ever_data->name, file);
}


static void
print_hack(FILE *file, struct e4_hack_type *hack_data, unsigned flags)
{
    unsigned class;

    class = hack_data->count;
    if (class == 1) {
	print_atom(file, hack_data->atoms, flags);
    } else {
	struct e4_odb2_type **odb2_data;
	unsigned count, miss;

	miss = class
	    * (sizeof(struct e4_atom_type *) + (sizeof(unsigned) << 2));
	count = class - 1;
	miss += count * sizeof(struct e4_odb2_type *);
	odb2_data = (struct e4_odb2_type **) malloc(miss);
	if (odb2_data) {
	    struct e4_atom_type **atom_slip;
	    unsigned *atoms;

	    atom_slip = (void *) (odb2_data + count);
	    atoms = (unsigned *) (atom_slip + count + 1);

	    memset(odb2_data, 0, count * sizeof(struct e4_odb2_type *));
	    memset(atoms, 0, class * sizeof(unsigned) << 2);

	    {
		struct e4_atom_type *atom_data;
		unsigned i = 0;

		atom_data = hack_data->atoms;
		_x1f4_e4_rule_list
		    (atom_data, count, atoms, odb2_data, atom_slip);
		atoms += class << 1;
		print_Atom(file, *atom_slip, flags, atoms);
		atom_slip++;
		atoms += 2;
		for (; i < count; i++) {
		    const struct e4_odbx_type *odbx_data;

		    odbx_data = (*odb2_data)->odbx_data;
		    if (odbx_data != _x1f4_e4_pick) {
			print_odb2(file, odbx_data, flags);
		    }
		    print_Atom(file, *atom_slip, flags, atoms);
		    atom_slip++;
		    atoms += 2;
		    odb2_data++;
		}

		odb2_data -= count;
	    }

	    free(odb2_data);
	}
    }
}


static void
print_last(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    struct e4_hack_type *hack_data;
    const struct e4_last_type *last_data;
    unsigned count;

    hack_data = atom_data->data.last.base;
    last_data = hack_base(hack_data)->last_data;
    fputs(last_data->name, file);
    fputc('(', file);
    count = last_data->count;
    if (count) {
	print_hack(file, hack_data, flags);
	count--;
	while (count) {
	    hack_data++;
	    count--;
	    fputs(", ", file);
	    print_hack(file, hack_data, flags);
	}
    }

    fputc(')', file);
}


static void
print_lock(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    struct e4_fine_type *fine_data;
    struct e4_hack_type *hack_data;
    const struct e4_last_type *last_data;
    unsigned count;

    fine_data = lock_fine(atom_data);

    hack_data = atom_data->data.last.base;
    last_data = hack_base(hack_data)->last_data;
    fputs(fine_data->ever_data->name, file);
    fputc('(', file);
    count = last_data->count;
    if (count) {
	print_hack(file, hack_data, flags);
	count--;
	while (count) {
	    hack_data++;
	    count--;
	    fputs(", ", file);
	    print_hack(file, hack_data, flags);
	}
    }

    fputc(')', file);
}


static void
print_mode(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    fprintf(file, "%ld", *(C_MODE *) &atom_data->data.lead.data);
    if (flags & DETAIL_CONSTANTS) {
	fputc('i', file);
    }
}


static void
print_odb1(FILE *file, struct e4_odb1_type *odb1_data, unsigned flags)
{
    const struct e4_odbx_type *odbx_data;

    odbx_data = odb1_data->odbx_data;
    odb1_data = odb1_data->odb1_data;
    if (odb1_data) {
	print_odb1(file, odb1_data, flags);
    }
    do {
	if (flags & DETAIL_IMPLICITS) {
	} else {
	    const struct e4_odbx_type *implicit;
	    unsigned i = 6;

	    implicit = _x1f4_e4_implicit;
	    for (; i; i--) {
		if (odbx_data == implicit) {
		    break;
		} else {
		    implicit++;
		}
	    }
	    if (i) {
		break;
	    }
	}

	fputs(odbx_data->name, file);
	if (flags & DETAIL_OPERATORS) {
	    int type_f;
	    const int *args;

	    args = odbx_data->args;
	    type_f = *args;
	    if (type_f == BILL) {
		fputc(',', file);
	    } else if (type_f == MODE) {
		fputc('.', file);
	    } else if (type_f == REAL) {
		fputc('*', file);
	    } else if (type_f == TEXT) {
		fputc('o', file);
	    } else {
		fputc('x', file);
	    }
	}
    } while (0);
}


static void
print_odb2(FILE *file, const struct e4_odbx_type *odbx_data, unsigned flags)
{
    fputc(' ', file);
    if (flags & DETAIL_OPERATORS) {
	int type_f;
	const int *args;

	args = odbx_data->args;
	type_f = *args;
	if (type_f == BILL) {
	    fputc(',', file);
	} else if (type_f == MODE) {
	    fputc('.', file);
	} else if (type_f == REAL) {
	    fputc('*', file);
	} else if (type_f == TEXT) {
	    fputc('o', file);
	} else {
	    fputc('x', file);
	}
	fputs(odbx_data->name, file);
	args++;
	type_f = *args;
	if (type_f == BILL) {
	    fputc(',', file);
	} else if (type_f == MODE) {
	    fputc('.', file);
	} else if (type_f == REAL) {
	    fputc('*', file);
	} else if (type_f == TEXT) {
	    fputc('o', file);
	} else {
	    fputc('x', file);
	}
    } else {
	fputs(odbx_data->name, file);
    }

    fputc(' ', file);
}


static void
print_pick(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    struct e4_hack_type *hack_data;
    const struct e4_last_type *last_data;
    unsigned count;

    hack_data = atom_data->data.last.base;
    last_data = hack_base(hack_data)->last_data;
    fputc('(', file);
    count = last_data->count;
    if (count) {
	print_hack(file, hack_data, flags);
	count--;
	while (count) {
	    hack_data++;
	    count--;
	    fputs(", ", file);
	    print_hack(file, hack_data, flags);
	}
    }

    fputc(')', file);
}


static void
print_real(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    x1f4_fprint_wfloat(file, 0, 0, 6, *(C_REAL *) &atom_data->data.lead.data);
    if (flags & DETAIL_CONSTANTS) {
	fputc('r', file);
    }
}


static void
print_text(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    fputc('"', file);
    x1f4_print_cstring(file, *(C_TEXT *) &atom_data->data.lead.data);
    fputc('"', file);
}


static void
print_this(FILE *file, struct e4_atom_type *atom_data, unsigned flags)
{
    fputc('(', file);
    print_hack(file, atom_data->data.hack.data, flags);
    fputc(')', file);
}


void
x1f4_print_expression(FILE *file, struct e4_expression_type *expression_data,
		      unsigned flags)
{
    print_hack(file, &expression_data->hack_data, flags);
}
