/*
 * tf21.c
 * Copyright (C) 2002-2014, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/time.h>

#include <aime.h>
#include <f2.h>
#include <inter.h>
#include <sf.h>
#include <tfxx.h>
#include <types.h>

#define TF21_ST_RING			(X1f4_E4_LAST + 1)

#define MAKE_SINGLE(a, b)		a

#define I_MISS(m)			(*((TF21_ST_C_RING *) (m)))
#define I_TEXT(t)			(*((X1f4_E4_C_TEXT *) (t)))

#define TF21_ST_C_RING			struct miss_type *

#define l_MISS(e, output) \
    {									      \
	TF21_ST_C_RING *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

#define l_TEXT(e, output) \
    {									      \
	X1f4_E4_C_TEXT *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

typedef struct link_type {
    struct link_type *link_data;
    void *data;
} link_type;

typedef struct miss_type {
    char *data;
    unsigned size;
} miss_type;

typedef struct text_type {
    struct link_type *link_data;
    void *c1;
} text_type;

extern const struct x1f4_function_type _libx1f4i0_e4_io[];

static int c21xx_l_l(void *, void **);
static int flat_core(void *, void *);
static int flat_none(void *, void **);
#if defined HAVE_LIBx1f4l0
static int free_core(void *);
#endif				/* HAVE_LIBx1f4l0 */
static int line_none(void *, void **);
static int link_core(void **, unsigned);
static int link_miss(void *, void *, void **);
static int link_program(const char *, void *, struct text_type *, void *,
			const char *);
static int lookup_type(const void *, int, const char **, unsigned *);
static int make_miss(void *, void *, void **);
static int make_null(void *, void *, void **);
static int make_text(void *, void *, void **);
static int merge_sets(struct x1f4_function_type **);
static int miss_list(struct text_type *);
static int seek_line(const char *, unsigned, unsigned *);
#if defined HAVE_LIBx1f4l0
static int slip_core(void **, unsigned);
#endif				/* HAVE_LIBx1f4l0 */
static int test_cast(void *, const char *, unsigned);

static void usage(void);

static const int c_____l__[] = {
/* *INDENT-OFF* */
    TF21_ST_RING
/* *INDENT-ON* */
}, c_____l_l[] = {
/* *INDENT-OFF* */
    TF21_ST_RING,
    TF21_ST_RING
/* *INDENT-ON* */
}, c_____t__[] = {
/* *INDENT-OFF* */
    X1f4_E4_TEXT
/* *INDENT-ON* */
};
static struct x1f4_datatype_type datatypes[] = {
/* *INDENT-OFF* */
    {	"string",	flat_none,	NULL,
	line_none,	NULL,
	flat_none,	NULL,		TF21_ST_RING,
	6,		NULL					},
    {	NULL,		NULL,		NULL,
	NULL,		NULL,
	NULL,		NULL,		0,
	4,		NULL					}
/* *INDENT-ON* */
};
static const struct x1f4_function_type functions[] = {
/* *INDENT-OFF* */
    {	"string_text",		make_miss,
	TF21_ST_RING,		c_____t__,	1,
	0,					11		},
    {	"text_string",		make_null,
	X1f4_E4_TEXT,		c_____l__,	1,
	0,					11		},
    {	NULL,			NULL,
	0,			NULL,		0,
	0,					1		}
/* *INDENT-ON* */
};
static const struct x1f4_operator_type e21xx_l_l[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("=", " "),  c21xx_l_l,	0400,
	TF21_ST_RING,		c_____l_l,
	X1f4_E4_LEFT_XSET,	1,
	NULL,			NULL				}
/* *INDENT-ON* */
};
const struct x1f4_operator_type *operators[] = {
/* *INDENT-OFF* */
    e21xx_l_l
/* *INDENT-ON* */
};
static void *static_text;

static int
c21xx_l_l(void *output, void **input)
{
    int status;
    struct miss_type *miss_data;

    miss_data = I_MISS(input[0]);
    if (!miss_data) {
    } else {
#if defined HAVE_LIBx1f4l0
	free_core(miss_data);
#else
	free(miss_data);
#endif				/* HAVE_LIBx1f4l0 */

	l_MISS(NULL, output);
	l_MISS(NULL, input[0]);
    }

    miss_data = I_MISS(input[1]);
    if (!miss_data) {
	status = 0;
    } else {
	void *cause[1];

	cause[0] = &miss_data->data;

	status = link_miss(static_text, output, cause);
	if (!status) {
	    l_MISS(I_MISS(output), input[0]);
	}
    }

    return status;
}


static int
flat_core(void *context, void *data)
{
#if defined HAVE_LIBx1f4l0
    free_core(((struct link_type *) data)->data);
    free_core(data);
#else
    free(((struct link_type *) data)->data);
    free(data);
#endif				/* HAVE_LIBx1f4l0 */

    return 0;
}


static int
flat_none(void *context, void **address)
{
    void *data;

    data = *address;
    if (data) {
	*address = NULL;

#if defined HAVE_LIBx1f4l0
	free_core(data);
#else
	free(data);
#endif				/* HAVE_LIBx1f4l0 */
    }

    return 0;
}


#if defined HAVE_LIBx1f4l0
static int
free_core(void *data)
{
    struct x1f4_c1_type *c1_data;
    void *list;

    c1_data = ((struct text_type *) static_text)->c1;

    list = c1_data->resource_set.context;

    if (!list) {
	free(data);
    } else {
	c1_data->resource_set.free(list, data);
    }

    return 0;
}
#endif				/* HAVE_LIBx1f4l0 */


static int
line_none(void *context, void **address)
{
    *address = NULL;

    return 0;
}


static int
link_core(void **post, unsigned size)
{
    int status;
    struct link_type *link_data;

#if defined HAVE_LIBx1f4l0
    slip_core((void *) &link_data, sizeof(struct link_type));
#else
    link_data = (struct link_type *) malloc(sizeof(struct link_type));
#endif				/* HAVE_LIBx1f4l0 */
    if (!link_data) {
	status = -1;
    } else {
#if defined HAVE_LIBx1f4l0
	slip_core(post, size);
#else
	*post = malloc(size);
#endif				/* HAVE_LIBx1f4l0 */
	if (!*post) {
	    status = -1;

#if defined HAVE_LIBx1f4l0
	    free_core(link_data);
#else
	    free(link_data);
#endif				/* HAVE_LIBx1f4l0 */
	} else {
	    struct text_type *text_data;

	    text_data = static_text;

	    link_data->link_data = text_data->link_data;
	    link_data->data = *post;

	    text_data->link_data = link_data;

	    status = 0;
	}
    }

    return status;
}


static int
link_miss(void *context, void *output, void **input)
{
    X1f4_E4_C_TEXT text;
    int status;
    struct miss_type *miss_data;
    unsigned size;

    text = I_TEXT(input[0]);

    size = strlen(text) + 1;

#if defined HAVE_LIBx1f4l0
    slip_core((void *) &miss_data, sizeof(struct miss_type) + size);
#else
    miss_data = (struct miss_type *) malloc(sizeof(struct miss_type) + size);
#endif				/* HAVE_LIBx1f4l0 */
    if (!miss_data) {
	status = -1;
    } else {
	void *data;

	status = 0;

	data = miss_data + 1;

	miss_data->data = data;
	miss_data->size = size;

	memcpy(data, text, size);

	l_MISS(miss_data, output);
    }

    return status;
}


static int
link_program(const char *self, void *proGram, struct text_type *text_data,
	     void *data, const char *program)
{
    int status = 0;
#if __BETA_PROGRAM__
    void *near;
#endif				/* __BETA_PROGRAM__ */

#if __RAIL_PROGRAM__
    x1f4_rail_shuffle(proGram);
#endif				/* __RAIL_PROGRAM__ */

#if __BETA_PROGRAM__
    status = x1f4_near_shuffle(&near, proGram);
    if (status) {
    } else {
	int excess;

# define proGram			near
#endif				/* __BETA_PROGRAM__ */

#if __PROGRAM_SLIP__
	x1f4_head_program(proGram);
	while (!x1f4_tail_program(proGram)) {
	    status = x1f4_slip_program(proGram);
	    if (text_data->link_data) {
		miss_list(text_data);
	    }
	    if (status) {
		break;
	    }
	}
#else
	status = x1f4_link_program(proGram);
	if (1) {
	    /*
	     * this sucks - memory is being freed only when program completes
	     */
	    if (text_data->link_data) {
		miss_list(text_data);
	    }
	}
#endif				/* __PROGRAM_SLIP__ */

	if (status) {
#if __LOOK_PROGRAM__
	    unsigned copy, miss;
#endif				/* __LOOK_PROGRAM__ */

#if __LOOK_PROGRAM__
	    x1f4_look_program(proGram, &copy);
	    seek_line(data, copy, &miss);
	    fprintf(stderr, "%s: %s: %u: cannot execute program\n", self,
		    program, miss);
#else
	    fprintf(stderr, "%s: cannot execute `%s'\n", self, program);
#endif				/* __LOOK_PROGRAM__ */
	}

#if __BETA_PROGRAM__
# undef proGram

	excess = x1f4_side_shuffle(&near, proGram, text_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }
#endif				/* __BETA_PROGRAM__ */

    return status;
}


static int
lookup_type(const void *none, int type, const char **name, unsigned *size)
{
    int status;

    if (type == TF21_ST_RING) {
	*name = "string";
	*size = 6;

	status = 0;
    } else {
	status = -1;
    }

    return status;
}


static int
make_miss(void *context, void *output, void **input)
{
    X1f4_E4_C_TEXT text;
    int status;
    struct miss_type *miss_data;
    unsigned size;

    text = I_TEXT(input[0]);

    size = strlen(text) + 1;

    link_core((void *) &miss_data, sizeof(struct miss_type) + size);
    if (!miss_data) {
	status = -1;
    } else {
	void *data;

	status = 0;

	data = miss_data + 1;

	miss_data->data = data;
	miss_data->size = size;

	memcpy(data, text, size);

	l_MISS(miss_data, output);
    }

    return status;
}


static int
make_null(void *context, void *output, void **input)
{
    int status;
    struct miss_type *miss_data;

    miss_data = I_MISS(input[0]);
    if (!miss_data) {
	status = 0;
	l_TEXT((char *) x1f4_c1_empty_string, output);
    } else {
	status = make_text(context, output, input);
    }

    return status;
}


static int
make_text(void *context, void *output, void **input)
{
    char *data;
    int status;
    struct miss_type *miss_data;
    unsigned size;

    miss_data = I_MISS(input[0]);

    size = miss_data->size;

    link_core((void *) &data, size);
    if (!miss_data) {
	status = -1;
    } else {
	status = 0;

	memcpy(data, miss_data->data, size);

	l_TEXT(data, output);
    }

    return status;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count, trans;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    x1f4_count_functions(_libx1f4i0_e4_io, &trans);
    function_data = (struct x1f4_function_type *) malloc
	((count + trans + 2 + 1) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, _libx1f4i0_e4_io,
	       trans * sizeof(struct x1f4_function_type));
	function_data += trans;
	memcpy(function_data, functions,
	       sizeof(struct x1f4_function_type) << 1);
    }

    return status;
}


static int
miss_list(struct text_type *text_data)
{
    struct link_type *link_data;

    link_data = text_data->link_data;

    text_data->link_data = NULL;

    while (link_data) {
	struct link_type *free_list;

	free_list = link_data->link_data;
	flat_core(text_data, link_data);
	link_data = free_list;
    }

    return 0;
}


static int
seek_line(const char *data, unsigned copy, unsigned *miss)
{
    unsigned line = 0;

    while (copy) {
	if (*data == 10) {
	    line++;
	}

	copy--;
	data++;
    }

    *miss = line;

    return 0;
}


#if defined HAVE_LIBx1f4l0
static int
slip_core(void **post, unsigned size)
{
    struct x1f4_c1_type *c1_data;
    void *list;

    c1_data = ((struct text_type *) static_text)->c1;

    list = c1_data->resource_set.context;

    if (!list) {
	*post = malloc(size);
    } else {
	if (c1_data->resource_set.link(list, post, size)) {
	    *post = NULL;
	}
    }

    return 0;
}
#endif				/* HAVE_LIBx1f4l0 */


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


static void
usage(void)
{
    puts("Usage: tf21 [OPTIONS] PATH [TYPE NAME VALUE]\n\
Test application defined data types interpretation,\n\
execute program read from PATH.\n\
\n\
Options:\n\
  -M, --stat-storage		stat program memory storage requirements\n\
  -P, --print			print program\n\
  -Q, --basic			print program\n\
  -c				execute the PROGRAM program instead program\n\
				read from the PROGRAM file\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
  -t, --time			time execution\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_immediate = 0, do_memory = 0, do_optimize = 0, do_print = 0,
	do_storage = 0, do_time = 0, status = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	int fast = ~0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "basic",	    0x00,   NULL,   0x51    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "time",	    0x00,   NULL,   0x74    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "MPQcmot", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 'M':
		do_storage = 1;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'Q':
		do_print = 2;
		break;
	    case 'c':
		do_immediate = 1;
		break;
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 't':
		do_time = 1;
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}
    }

    if ((argc - optind) % 3 != 1) {
	fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' for "
		"more information.\n", argv[0], argv[0]);

	return 1;
    } else {
	const char *program;
	unsigned count;
	void *context = NULL, *state_data;

	program = argv[optind];
	optind++;

	status = libx1f4i0_init_fine
	    (optind, argc, argv, &count, &state_data, &context);
	if (status) {
	} else {
	    unsigned size;
	    void *data;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    if (do_immediate) {
		data = (void *) program;
		size = strlen(program);
	    } else {
		status = libx1f4i0_read_file(&data, &size, program, 1);
	    }
	    if (status) {
		libx1f4i0_stat_failure(argv[0], status, program);
	    } else {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
#if defined HAVE_LIBx1f4l0
		    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
		    struct text_type text;
		    struct x1f4_c1_type c1;
		    struct x1f4_c1record_type c1record;
		    const struct x1f4_operator_type *const *operator1s,
			*const *operator2s;
		    unsigned flags = X1f4_C1_BCOLLECT | X1f4_C1_DDACCESS;
		    void *proGram;

		    {
			char *mine;

			mine = data;
			mine[size] = 0;
		    }

		    if (do_optimize) {
			flags |= X1f4_C1_OPTIMIZE;
		    }

		    static_text = &text;

		    datatypes[0].context = &text;

#if defined HAVE_LIBx1f4l0
		    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

		    c1.datatype_set.miss = datatypes;
		    c1.function_set.get = libx1f4i0_select_function;
		    c1.function_set.context = function_data;
		    x1f4_llink_operator1s(&operator1s);
		    libx1f4i0_join_sets(&c1.operator1s, operator1s, NULL, 0);
		    x1f4_llink_operator2s(&operator2s);
		    libx1f4i0_join_sets
			(&c1.operator2s, operator2s, operators, 1);
		    c1.variable_set.context = context;

		    set_xxrecord(c1, &c1record);

		    flags |= X1f4_C1_TYPELIST;

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			flags |= X1f4_C1_RESOURCE;
			libx1f4i0_line_valist
			    (valist, do_memory, do_storage,
			     &c1.resource_set.free, &c1.resource_set.link,
			     &c1.resource_set.mode, &c1.resource_set.context,
			     &dlist);
		    } else {
			if (1) {
			    c1.resource_set.context = valist;
			}
		    }
#else
		    if (2) {
			if (1) {
			    c1.resource_set.context = NULL;
			}
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    status = x1f4_init_program(&proGram, data, flags, &c1);
		    if (status) {
			if (status == X1f4_C1_ALLOC_ERROR) {
			    perror(argv[0]);
			} else {
			    struct x1f4_eelookup_type eelookup;

			    eelookup.type_l.fix = lookup_type;
			    eelookup.type_l.context = NULL;
			    eelookup.type_q.e4fine = NULL;

			    c1record.pick = 1;

			    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
				    program);
			    fprintf(stderr, "%s: ", argv[0]);
			    x1f4_stat_program
				(stderr, test_cast, data, &c1record,
				 &eelookup);
			    fprintf(stderr, "\n");
			}
		    } else {
			if (do_print) {
			    if (do_print == 1) {
#if __PROGRAM_DUMP__
				x1f4_vprint_program
				    (stdout, test_cast, proGram, 0);
#endif				/* __PROGRAM_DUMP__ */
			    } else {
#if __PROGRAM_DUMP__
				x1f4_vprint_bprogram
				    (stdout, test_cast, proGram, 0, 4);
#endif				/* __PROGRAM_DUMP__ */
			    }
			} else {
			    struct timeval start;

			    if (do_time) {
				gettimeofday(&start, NULL);
			    }

			    text.c1 = &c1;
			    text.link_data = NULL;

			    status = link_program
				(argv[0], proGram, &text, data, program);
			    if (status) {
			    } else {
				if (do_time) {
				    double count;
				    struct timeval finish;

				    gettimeofday(&finish, NULL);
				    count = finish.tv_sec;
				    count -= start.tv_sec;
				    count *= 1000000;
				    count += finish.tv_usec;
				    count -= start.tv_usec;
				    count /= 1000000;
				    fprintf(stderr, "execution time: %.6f se"
					    "conds\n", count);
				}
			    }
			}

#if defined HAVE_LIBx1f4l0
			if (valist) {
			    if (do_storage) {
				unsigned size;

				x1f4_size_xalist(valist, &size);
				fprintf(stderr, " %7u\n", size);
			    }
			}
#endif				/* HAVE_LIBx1f4l0 */

			x1f4_fini_program(&proGram);
		    }

		    if (c1.operator2s != operator2s) {
			free((void *) c1.operator2s);
		    }

		    if (c1.operator1s != operator1s) {
			free((void *) c1.operator1s);
		    }

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			libx1f4i0_fini_valist
			    (argv[0], valist, do_memory, do_storage, &status);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    free(function_data);
		}

		if (do_immediate) {
		} else {
		    free(data);
		}
	    }
	}

	if (context) {
	    x1f4_air_state(context);
	}

	if (state_data) {
	    free(state_data);
	}
    }

    return status;
}
