/*
 * tf20.c
 * Copyright (C) 2002-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#include <f2.h>
#include <inter.h>
#include <line.h>
#include <sf.h>
#include <types.h>

extern const struct x1f4_function_type _libx1f4i0_e4_io[],
    _libx1f4i0_e4_line[], _libx1f4i0_e4_line_io[];
extern const struct x1f4_operator_type *_libx1f4i0_e4_line_1[],
    *_libx1f4i0_e4_line_2[];

void *global_context;

static int link_none(void *, void **);
static int lookup_type(const void *, int, const char **, unsigned *);
static int merge_sets(struct x1f4_function_type **);
static int test_cast(void *, const char *, unsigned);

static void usage(void);

static struct x1f4_datatype_type datatypes[] = {
/* *INDENT-OFF* */
    {	"line",		NULL,		NULL,
	NULL,		link_none,
	NULL,		NULL,		EX20_E4_LINE,
	4,		NULL					},
    {	"pair",		NULL,		NULL,
	NULL,		link_none,
	NULL,		NULL,		EX20_E4_PAIR,
	4,		NULL					},
    {	"quad",		NULL,		NULL,
	NULL,		link_none,
	NULL,		NULL,		EX20_E4_QUAD,
	4,		NULL					},
    {	NULL,		NULL,		NULL,
	NULL,		NULL,
	NULL,		NULL,		0,
	4,		NULL					}
/* *INDENT-ON* */
};

static int
link_none(void *subtext, void **address)
{
    int status;
    struct line_type *line_data;

    line_data = (struct line_type *) malloc(sizeof(struct line_type));
    if (!line_data) {
	status = -1;
    } else {
	struct miss_type *miss_data;

	status = 0;

	miss_data = subtext;

	line_data->line_data = miss_data->line_data;
	miss_data->line_data = line_data;

	*address = line_data;
    }

    return status;
}


static int
lookup_type(const void *none, int type, const char **name, unsigned *size)
{
    int status;

    if (type == EX20_E4_LINE) {
	*name = "line";
	*size = 4;

	status = 0;
    } else {
	if (type == EX20_E4_PAIR) {
	    *name = "pair";
	    *size = 4;

	    status = 0;
	} else {
#if 0
	    if (type == EX20_E4_QUAD) {
#endif				/* 0 */
		*name = "quad";
		*size = 4;

		status = 0;
#if 0
	    } else {
		status = -1;
	    }
#endif				/* 0 */
	}
    }

    return status;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count, scale, shift, trans;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    x1f4_count_functions(_libx1f4i0_e4_io, &trans);
    x1f4_count_functions(_libx1f4i0_e4_line, &scale);
    x1f4_count_functions(_libx1f4i0_e4_line_io, &shift);
    function_data = (struct x1f4_function_type *)
	malloc((count + trans + scale + shift + 1)
	       * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, _libx1f4i0_e4_io,
	       trans * sizeof(struct x1f4_function_type));
	function_data += trans;
	memcpy(function_data, _libx1f4i0_e4_line,
	       scale * sizeof(struct x1f4_function_type));
	function_data += scale;
	memcpy(function_data, _libx1f4i0_e4_line_io,
	       (shift + 1) * sizeof(struct x1f4_function_type));
    }

    return status;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


static void
usage(void)
{
    puts("Usage: tf20 [OPTIONS] FILE\n\
Exit successfully?\n\
\n\
Options:\n\
  -M, --stat-storage		stat program memory storage requirements\n\
  -P, --print			print program\n\
  -Q, --basic			print program\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_memory = 0, do_optimize = 0, do_print = 0, do_storage = 0,
	status = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	int fast = ~0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "basic",	    0x00,   NULL,   0x51    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "MPQmo", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 'M':
		do_storage = 1;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'Q':
		do_print = 2;
		break;
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}
    }

    if (argc - optind != 1) {
	fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' for "
		"more information.\n", argv[0], argv[0]);

	return 1;
    } else {
	{
	    unsigned size;
	    void *data;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    status = libx1f4i0_read_file(&data, &size, argv[optind], 1);
	    if (status) {
		libx1f4i0_stat_failure(argv[0], status, argv[optind]);
	    } else {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
#if defined HAVE_LIBx1f4l0
		    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
		    struct miss_type miss;
		    struct x1f4_c1_type c1;
		    struct x1f4_c1record_type c1record;
		    const struct x1f4_operator_type *const *operator1s,
			*const *operator2s;
		    unsigned flags = X1f4_C1_BCOLLECT;
		    void *program;

		    {
			char *mine;

			mine = data;
			mine[size] = 0;
		    }

		    if (do_optimize) {
			flags |= X1f4_C1_OPTIMIZE;
		    }

#if defined HAVE_LIBx1f4l0
		    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

		    datatypes[0].context = &miss;
		    datatypes[1].context = &miss;
		    datatypes[2].context = &miss;

		    miss.line_data = NULL;

		    global_context = &miss;

		    c1.bcollect_set.c1record_data = &c1record;
		    c1.datatype_set.miss = datatypes;
		    c1.function_set.get = libx1f4i0_select_function;
		    c1.function_set.context = function_data;
		    x1f4_llink_operator1s(&operator1s);
		    libx1f4i0_join_sets
			(&c1.operator1s, operator1s, _libx1f4i0_e4_line_1, 6);
		    x1f4_llink_operator2s(&operator2s);
		    libx1f4i0_join_sets
			(&c1.operator2s, operator2s, _libx1f4i0_e4_line_2, 21);
		    c1.variable_set.context = NULL;

		    flags |= X1f4_C1_TYPELIST;

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			flags |= X1f4_C1_RESOURCE;
			libx1f4i0_line_valist
			    (valist, do_memory, do_storage,
			     &c1.resource_set.free, &c1.resource_set.link,
			     &c1.resource_set.mode, &c1.resource_set.context,
			     &dlist);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    status = x1f4_init_program(&program, data, flags, &c1);
		    if (status) {
			if (status == X1f4_C1_ALLOC_ERROR) {
			    perror(argv[0]);
			} else {
			    struct x1f4_eelookup_type eelookup;

			    eelookup.type_l.fix = lookup_type;
			    eelookup.type_l.context = NULL;
			    eelookup.type_q.e4fine = NULL;

			    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
				    argv[optind]);
			    fprintf(stderr, "%s: ", argv[0]);
			    x1f4_stat_program
				(stderr, test_cast, data, &c1record,
				 &eelookup);
			    fprintf(stderr, "\n");
			}
		    } else {
			if (do_print) {
			    if (do_print == 1) {
				x1f4_vprint_program
				    (stdout, test_cast, program, 0);
			    } else {
				x1f4_vprint_bprogram
				    (stdout, test_cast, program, 0, 4);
			    }
			} else {
			    status = x1f4_link_program(program);
			    if (status) {
				fprintf(stderr, "%s: cannot execute `%s'\n",
					argv[0], argv[optind]);
			    }
			}

#if defined HAVE_LIBx1f4l0
			if (valist) {
			    if (do_storage) {
				unsigned size;

				x1f4_size_xalist(valist, &size);
				fprintf(stderr, " %7u\n", size);
			    }
			}
#endif				/* HAVE_LIBx1f4l0 */

			x1f4_fini_program(&program);
		    }

		    while (miss.line_data) {
			struct line_type *line;

			line = miss.line_data;
			miss.line_data = line->line_data;
			free(line);
		    }

		    if (c1.operator2s != operator2s) {
			free((void *) c1.operator2s);
		    }

		    if (c1.operator1s != operator1s) {
			free((void *) c1.operator1s);
		    }

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			libx1f4i0_fini_valist
			    (argv[0], valist, do_memory, do_storage, &status);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    free(function_data);
		}

		free(data);
	    }
	}
    }

    return status;
}
