/*
 * tf18.c
 * Copyright (C) 2002-2014, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <integral.v-d.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <types.h>

#define __ROLL_VALIST__			1

#if defined HAVE_LIBx1f4l0
#else
# undef __ROLL_VALIST__
# define __ROLL_VALIST__		0
#endif				/* HAVE_LIBx1f4l0 */

#define DATA_BITS \
    X1f4_LXDATA_EVERLINK

#define DATE_BITS \
    X1f4_LXDATE_EVERLINK

#define DECQ_BITS \
    X1f4_LXDECQ_EVERLINK | X1f4_LXDECQ_RESETNEW

#define FILE_BITS \
    X1f4_LXFILE_EVERLINK | X1f4_LXFILE_SETCLOSE

#define LINE_BITS \
    X1f4_LXLINE_EVERLINK | X1f4_LXLINE_RESETNEW

#define LIST_BITS \
    X1f4_LXLIST_EVERLINK | X1f4_LXLIST_RESETNEW

#define POLL_BITS \
   X1f4_LXPOLL_EVERLINK

#define PORT_BITS \
    X1f4_LXPORT_EVERLINK | X1f4_LXPORT_SETCLOSE

#define TRAP_BITS \
    X1f4_LXTRAP_REPORTER

#define A113_BITS \
    X1f4_A1_BCOLLECT | X1f4_A1_CASTTYPE | X1f4_A1_COMPOSER | X1f4_A1_DDACCESS \
    | X1f4_A1_LEADCAST | X1f4_A1_OPTIMIZE | X1f4_A1_SCOMMENT		      \
    | X1f4_A1_SIDELIST | X1f4_A1_TEXTFLAT | X1f4_A1_TRANSFER		      \
    | X1f4_A1_TYPELIST

#define LIST_TYPES			4

#define I_MODE(i)			(*((X1f4_E4_C_MODE *) (i)))
#define I_TEXT(i)			(*((X1f4_E4_C_TEXT *) (i)))

typedef struct context_type {
    char **argv;
    const char *fix, *program, *self;
    int argc, class, do_memory, do_storage, error, exit, immediate,
	(*line) (struct context_type *, unsigned *), pick_dash, sequenced,
	slip;
#if __ROLL_VALIST__
    struct list_type dlist;
#endif				/* __ROLL_VALIST__ */
    struct x1f4_indexset_type *indexset_data;
    struct x1f4_track_type *eport, plain, track;
    unsigned bytes, call, size;
    void *logic, *record;
#if __ROLL_VALIST__
    void *valist;
#endif				/* __ROLL_VALIST__ */
    const void *data;
} context_type;

static int copy_error(void *, const char *, unsigned);
static int copy_output(void *, const char *, unsigned);
static int dumb_mode(void *, void *, void **);
static int ever_list(void *, void *, struct x1f4_variable_type *);
static int fini_record(struct context_type *);
static int fini_valist(struct context_type *);
static int fini_xset(struct context_type *);
static int flat_text(void *);
static int flush_error(struct context_type *, const char *);
static int init_record(struct context_type *);
static int init_valist(struct context_type *);
static int init_xset(struct context_type *);
static int line_flat(void *);
static int line_shuffle(struct context_type *, unsigned *);
static int list_text(void *, const void *);
static int look_type(const void *, int, const char **, unsigned *);
static int mode_argc(void *, void *, void **);
static int pick_aime(void *, void *, void **);
static int pick_shuffle(struct context_type *, unsigned *);
static int post_flat(void *);
static int push_flat(void *, const char *, unsigned);
static int rule_fset(struct context_type *, const struct x1f4_function_type *);
static int run_logic(struct context_type *, int, char **);
static int seek_line(struct context_type *, unsigned, unsigned *);
static int select_context(const void *, const struct x1f4_function_type *,
			  void **);
static int select_logic(const char *, unsigned, const void *,
			const struct x1f4_function_type **);
static int set_composer(struct context_type *, void **,
			int (**) (void *, int, int,
				  const struct x1f4_linetext_type **));
static int set_frame(struct context_type *, const char *, void *, void *);
static int set_logic(struct context_type *, const char *, const void *, int *);
static int set_track(struct context_type *, const char *, void *, void *);
static int stat_argv(void *, int);
static int test_cast(void *, const char *, unsigned);
static int text_argv(void *, void *, void **);
static int type_fsdeck(struct context_type *);
static int wait_some(struct context_type *);

static void SIGCHLD_handler(int);
static void list_0010(struct context_type *);
static void list_0220(struct context_type *);
static void usage(void);

static int *static_class;
static const int c_____m__[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE
/* *INDENT-ON* */
}, c_____t_t[] = {
/* *INDENT-OFF* */
    X1f4_E4_TEXT,
    X1f4_E4_TEXT
/* *INDENT-ON* */
};
static sig_atomic_t SIGCHLD_received;
static const struct x1f4_function_type tf18_set[] = {
/* *INDENT-OFF* */
    {	"aime",			pick_aime,
	X1f4_E4_MODE,		c_____t_t,	2,
	X1f4_E4_KEEP_CALL,			4		},
    {	"argc",			mode_argc,
	X1f4_E4_MODE,		NULL,		0,
	X1f4_E4_KEEP_CALL,			4		},
    {	"argv",			text_argv,
	X1f4_E4_TEXT,		c_____m__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	"exit",			dumb_mode,
	X1f4_E4_VOID,		c_____m__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	NULL,			NULL,
	0,			NULL,		0,
	0,					1		}
/* *INDENT-ON* */
};
static struct x1f4_lxnear_type lxnear;
static struct x1f4_lxtrap_type lxtrap;
const static struct x1f4_lxtile_type tiles[] = {
/* *INDENT-OFF* */
    {	x1f4_note_lxcast,	0,		NULL		},
    {	x1f4_note_lxdata,	DATA_BITS,	NULL		},
    {	x1f4_note_lxdate,	DATE_BITS,	NULL		},
    {	x1f4_note_lxdecq,	DECQ_BITS,	NULL		},
    {	x1f4_note_lxfile,	FILE_BITS,	NULL		},
    {	x1f4_note_lxline,	LINE_BITS,	NULL		},
    {	x1f4_note_lxlist,	LIST_BITS,	NULL		},
    {	x1f4_note_lxnear,	0,		&lxnear		},
    {	x1f4_note_lxpoll,	POLL_BITS,	NULL		},
    {	x1f4_note_lxport,	PORT_BITS,	NULL		},
    {	x1f4_note_lxtext,	0,		NULL		},
    {	x1f4_note_lxtrap,	TRAP_BITS,	&lxtrap		}
/* *INDENT-ON* */
};
static const struct x1f4_textpipe_type near_pipes[] = {
/* *INDENT-OFF* */
    {	"o",		copy_output,		1,	NULL	},
    {	"v",		copy_error,		1,	NULL	}
/* *INDENT-ON* */
};
static void *static_context;

static int
copy_error(void *none, const char *data, unsigned size)
{
    fwrite(data, size, 1, stderr);

    return 0;
}


static int
copy_output(void *context, const char *data, unsigned size)
{
    struct context_type *context_data;

    context_data = context;
    if (context_data->slip) {
	context_data->slip = 0;
	fwrite(context_data->fix, context_data->bytes, 1, stdout);
    }

    fwrite(data, size, 1, stdout);

    return 0;
}


static int
dumb_mode(void *context, void *output, void **input)
{
    ((struct context_type *) static_context)->exit = I_MODE(input[0]) << 1 | 1;

    return 1;
}


static int
ever_list(void *none, void *data, struct x1f4_variable_type *variable_data)
{
    puts(variable_data->name);

    return 0;
}


static int
fini_record(struct context_type *context_data)
{
    const char *name;
    int status;
    struct x1f4_datatype_type *datatype_data;

    datatype_data = context_data->indexset_data->datatype_set.datatype_data;

    name = datatype_data->name;
    while (name) {
	if (datatype_data->size == 6) {
	    if (memcmp(name, "record", 6)) {
	    } else {
		break;
	    }
	}

	datatype_data++;
	name = datatype_data->name;
    }

    if (name) {
	status = datatype_data->flat
	    (datatype_data->context, &context_data->record);
    } else {
	status = 1;
    }

    return status;
}


static int
fini_valist(struct context_type *context_data)
{
#if !__ROLL_VALIST__
# define status				0
#endif				/* !__ROLL_VALIST__ */

#if __ROLL_VALIST__
    int status = 0;
    void *valist;
#endif				/* __ROLL_VALIST__ */

#if __ROLL_VALIST__
    valist = context_data->valist;
    if (valist) {
	libx1f4i0_fini_valist
	    (context_data->self, valist, context_data->do_memory,
	     context_data->do_storage, &status);
	if (status) {
	} else {
	    if (context_data->do_memory) {
		fprintf(stderr, " %7u max allocation\n",
			context_data->dlist.ever);
	    }
	}
    }
#endif				/* __ROLL_VALIST__ */

    return status;

#if !__ROLL_VALIST__
# undef status
#endif				/* !__ROLL_VALIST__ */
}


static int
fini_xset(struct context_type *context_data)
{
    int excess, status;

    status = fini_record(context_data);

    excess = x1f4_flat_indexset(context_data->indexset_data);
    if (excess) {
	if (status) {
	} else {
	    status = excess;
	}
    }

    return status;
}


static int
flat_text(void *context)
{
    int status;
    struct context_type *context_data;
    struct x1f4_indexset_type *indexset_data;

    context_data = context;

    indexset_data = context_data->indexset_data;

    status = indexset_data->autodeck_set.deck
	(&indexset_data->autodeck_set.text);

    if (SIGCHLD_received) {
	int excess;

	excess = wait_some(context_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}


static int
flush_error(struct context_type *context_data, const char *program)
{
    if (context_data->error) {
    } else {
	fflush(stdout);
	fprintf(stderr, "%s:", context_data->self);
	if (program) {
	    fprintf(stderr, " %s:", program);
	}
	if (1) {
	    int (*line) (struct context_type *, unsigned *);

	    line = context_data->line;
	    if (line) {
		unsigned copy, miss;

		line(context_data, &copy);
		seek_line(context_data, copy, &miss);
		fprintf(stderr, " %u:", miss + 1);
	    }
	}

	context_data->error = 1;

	fprintf(stderr, " cannot execute program\n");
    }

    return 17;
}


static int
init_record(struct context_type *context_data)
{
    const char *name;
    int status;
    struct x1f4_datatype_type *datatype_data;

    datatype_data = context_data->indexset_data->datatype_set.datatype_data;

    name = datatype_data->name;
    while (name) {
	if (datatype_data->size == 6) {
	    if (memcmp(name, "record", 6)) {
	    } else {
		break;
	    }
	}

	datatype_data++;
	name = datatype_data->name;
    }

    if (name) {
	int type;
	void **record, *text;

	text = context_data->indexset_data->variable_set.text;

	record = &context_data->record;

	type = datatype_data->type;

	status = datatype_data->line(datatype_data->context, record);
	if (status) {
	} else {
	    status = x1f4_push_variable
		(&text, "context", 7, type, X1f4_E4_READ_ONLY, record);
	    if (status) {
		datatype_data->flat(datatype_data->context, record);
	    }
	}

	context_data->indexset_data->variable_set.text = text;
    } else {
	status = 1;
    }

    return status;
}


static int
init_valist(struct context_type *context_data)
{
#if __ROLL_VALIST__
    libx1f4i0_init_valist
	(&context_data->valist, context_data->do_memory,
	 context_data->do_storage);
#endif				/* __ROLL_VALIST__ */

#if __ROLL_VALIST__
    context_data->dlist.ever = 0;
    context_data->dlist.size = 0;
#endif				/* __ROLL_VALIST__ */

#if __ROLL_VALIST__
    context_data->dlist.valist = context_data->valist;
#endif				/* __ROLL_VALIST__ */

    return 0;
}


static int
init_xset(struct context_type *context_data)
{
    int status;
    struct x1f4_textpipe_type here_pipes[2];
    struct x1f4_textport_type textport;
    unsigned textport_bits = 0;

    x1f4_rule_eproxy(&context_data->track);

    context_data->track.data = &context_data->eport;

    textport_bits = X1f4_TEXTPORT_CASTTYPE | X1f4_TEXTPORT_TEXTFLAT;

    textport.lasttype_set.type = X1f4_E4_LAST;

    textport.tilelong_set.lxtile_data = tiles;
    textport.tilelong_set.miss =
	sizeof(tiles) / sizeof(struct x1f4_lxtile_type);

    here_pipes[0] = near_pipes[0];
    here_pipes[0].text = context_data;
    here_pipes[1] = near_pipes[1];
    here_pipes[1].text = context_data;

    lxnear.nearpipe_set.textpipe_data = here_pipes;
    lxnear.nearpipe_set.miss = 2;

    lxtrap.reporter_set.track = &context_data->eport;

    textport.textflat_set = context_data->track;

    textport.autodeck_set.class = &context_data->class;

#if __ROLL_VALIST__
    if (context_data->valist) {
	textport_bits |= X1f4_TEXTPORT_RESOURCE;
	libx1f4i0_line_valist
	    (context_data->valist, context_data->do_memory,
	     context_data->do_storage, &textport.resource_set.free,
	     &textport.resource_set.link, &textport.resource_set.mode,
	     &textport.resource_set.data, &context_data->dlist);
    }
#endif				/* __ROLL_VALIST__ */

    status = x1f4_fast_indexset
	(context_data->indexset_data, textport_bits, &textport);
    if (status) {
    } else {
	status = type_fsdeck(context_data);
	if (status) {
	} else {
	    status = init_record(context_data);
	    if (status) {
	    } else {
		context_data->plain.data = context_data;
		context_data->plain.line = line_flat;
		context_data->plain.post = post_flat;
		context_data->plain.push = push_flat;

		context_data->eport = &context_data->plain;
	    }
	}

	if (status) {
	    x1f4_flat_indexset(context_data->indexset_data);
	}
    }

    return status;
}


static int
line_flat(void *context)
{
    struct context_type *context_data;

    context_data = context;

    fflush(stdout);
    fputs(context_data->self, stderr);
    fputs(":", stderr);
    if (context_data->program) {
	fputs(" ", stderr);
	fputs(context_data->program, stderr);
	fputs(":", stderr);
    }
    if (1) {
	int (*line) (struct context_type *, unsigned *);

	line = context_data->line;
	if (line) {
	    unsigned copy, miss;

	    line(context, &copy);
	    seek_line(context, copy, &miss);
	    fprintf(stderr, " %u:", miss + 1);
	}
    }

    fputs(" ", stderr);

    return 0;
}


static int
line_shuffle(struct context_type *context_data, unsigned *copy)
{
    return x1f4_look_shuffle(context_data->logic, copy);
}


static int
list_text(void *context, const void *text)
{
    x1f4_print_htfunction(stdout, text, "", context, look_type);

    return 0;
}


static int
look_type(const void *context, int type, const char **name, unsigned *size)
{
    int delete = 1;
    struct x1f4_indexset_type *indexset_data;
    const struct x1f4_nodetype_type *nodetype_data;
    unsigned count;

    indexset_data = ((struct context_type *) context)->indexset_data;

    nodetype_data = indexset_data->nodetype_set.nodetype_data;
    nodetype_data += LIST_TYPES;

    count = indexset_data->nodetype_set.miss - LIST_TYPES;
    for (; count; count--) {
	if (nodetype_data->code == type) {
	    *name = nodetype_data->name;
	    *size = nodetype_data->size;

	    delete = 0;

	    break;
	} else {
	    nodetype_data++;
	}
    }
    if (delete) {
	if (type == X1f4_E4_OBJECT) {
	    *name = "object";
	    *size = 6;

	    delete = 0;
	} else {
	    if (type == X1f4_E4_VOID) {
		*name = "void";
		*size = 4;

		delete = 0;
	    }
	}
    }

    return delete;
}


static int
mode_argc(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE *l;

    l = (void *) (output);
    *l = ((struct context_type *) static_context)->argc;

    return 0;
}


static int
pick_aime(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE *l;
    const char *program;
    int class, error, exit, status;
    struct context_type *context_data;
    unsigned size;
    const void *data;
    void *logic;

    context_data = static_context;

    class = context_data->class;
    error = context_data->error;

    exit = context_data->exit;

    logic = context_data->logic;

    data = context_data->data;
    size = context_data->size;

    program = context_data->program;

    context_data->data = I_TEXT(input[1]);
    context_data->size = strlen(context_data->data);

    context_data->program = I_TEXT(input[0]);

    l = (void *) (output);

    if (0) {
    } else {
	int exit = 0;

	status = set_logic
	    (context_data, I_TEXT(input[0]), I_TEXT(input[1]), &exit);

	*l = exit;
	if (status) {
	    status = X1f4_EX_CAN_CONTINUE;
	}
    }

    context_data->class = class;
    context_data->error = error;

    context_data->exit = exit;

    context_data->logic = logic;

    context_data->size = size;
    context_data->data = data;

    context_data->program = program;

    return status;
}


static int
pick_shuffle(struct context_type *context_data, unsigned *copy)
{
    return x1f4_seek_shuffle(context_data->logic, copy);
}


static int
post_flat(void *context)
{
    ((struct context_type *) context)->error = 1;

    fputs("\n", stderr);

    return 0;
}


static int
push_flat(void *context, const char *data, unsigned size)
{
    fwrite(data, size, 1, stderr);

    return 0;
}


static int
rule_fset(struct context_type *context_data,
	  const struct x1f4_function_type *function_data)
{
    const char *name;
    int status;

    name = function_data->name;
    if (name) {
	void *fsdeck;

	fsdeck = context_data->indexset_data->function_set.text;

	do {
	    status = x1f4_post_mxpath
		(fsdeck, name, function_data->length, function_data);
	    if (status) {
		break;
	    } else {
		function_data++;
	    }

	    name = function_data->name;
	} while (name);
    } else {
	status = 0;
    }

    return status;
}


static int
run_logic(struct context_type *context_data, int argc, char **argv)
{
    int excess, exit = 0, status;
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

#if __ROLL_VALIST__
    context_data->valist = NULL;
#endif				/* __ROLL_VALIST__ */

    init_valist(context_data);

    status = init_xset(context_data);
    if (status) {
    } else {
	const char *program;

	program = argv[optind];
	optind++;

	SIGCHLD_received = 0;

	signal(SIGCHLD, SIGCHLD_handler);

	if (context_data->immediate) {
	    if (0) {
	    } else {
		context_data->data = program;
		context_data->size = strlen(program);

		context_data->program = NULL;

		status = set_logic(context_data, NULL, program, &exit);
	    }
	} else {
	    unsigned size;
	    void *data;

	    do {
		if (context_data->pick_dash) {
		    if (program[0] == '-') {
			if (program[1]) {
			} else {
			    status = libx1f4i0_read_dash
				(&data, &size, 1);
			    if (1) {
				break;
			    }
			}
		    }
		}

		status = libx1f4i0_read_file(&data, &size, program, 1);
	    } while (0);
	    if (status) {
		libx1f4i0_stat_failure(argv[0], status, program);
	    } else {
		if (1) {
		    char *mine;

		    mine = data;
		    mine[size] = 0;
		}

		context_data->data = data;
	        context_data->size = strlen(data);

		context_data->program = program;

		status = set_logic(context_data, program, data, &exit);

		free(data);
	    }
	}

	signal(SIGCHLD, SIG_DFL);

	excess = fini_xset(context_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    excess = fini_valist(context_data);
    if (excess) {
	if (status) {
	} else {
	    status = excess;
	}
    }

    if (status) {
    } else {
	status = exit;
    }

    return status;
}


static int
seek_line(struct context_type *context_data, unsigned copy, unsigned *miss)
{
    const char *data;
    unsigned line = 0;

    /*
     * there is no way of knowing whether this is in the right piece of source.
     * minimize damage, in case it isn't don't search beyond its end.
     */
    if (copy < context_data->size) {
    } else {
	copy = 0;
    }

    data = context_data->data;
    while (copy) {
	if (*data == 10) {
	    line++;
	}

	copy--;
	data++;
    }

    *miss = line;

    return 0;
}


static int
select_context(const void *context, const struct x1f4_function_type *miss,
	       void **text)
{
    *text = (void *)
	((const struct x1f4_linetext_type *)
	 ((const char *) miss
	  - offsetof(struct x1f4_linetext_type, function)))->context;

    return 0;
}


static int
select_logic(const char *f, unsigned length, const void *context,
	     const struct x1f4_function_type **function)
{
    int status;
    struct context_type *context_data;
    const void *mind;

    context_data = (void *) context;

    status = x1f4_find_mxpath
	(context_data->indexset_data->function_set.text, f, length, &mind);
    if (status) {
	status = 0;

	*function = mind;
    } else {
	status = 1;
    }

    return status;
}


static int
set_composer(struct context_type *context_data, void **text,
	     int (**lock) (void *, int, int,
			   const struct x1f4_linetext_type **))
{
    struct x1f4_lxslip_type *lxslip_data;

    lxslip_data = context_data->indexset_data->sliplong_set.lxslip_data;
    while (lxslip_data->note != x1f4_note_lxcast) {
	lxslip_data++;
    }

    *lock = x1f4_lock_lxcast;

    *text = lxslip_data->slip;

    return 0;
}


static int
set_frame(struct context_type *context_data, const char *program,
	  void *proGram, void *degree)
{
    int status;

    context_data->line = pick_shuffle;

    context_data->logic = proGram;

    status = x1f4_long_shuffle
	(proGram, &context_data->class, context_data, flat_text, degree);
    if (status) {
	if (context_data->exit) {
	    status = 0;
	    *(X1f4_E4_C_MODE *) degree = context_data->exit >> 1;
	} else {
	    flush_error(context_data, program);
	}
    }

    return status;
}


static int
set_logic(struct context_type *context_data, const char *program,
	  const void *data, int *exit)
{
    int status;
    struct x1f4_a1_type a1;
    struct x1f4_a1record_type a1record;
    unsigned flags = A113_BITS;
    void *proGram;

    a1.autolead_set.context = context_data->indexset_data->autodeck_set.less;
    a1.autolead_set.link = x1f4_link_e4lf;
    a1.autolead_set.pick = x1f4_pick_e4lf;
    a1.autolink_set = context_data->indexset_data->autolink_set;
    a1.bcollect_set.a1record_data = &a1record;
    set_composer(context_data, &a1.composer_set.context, &a1.composer_set.get);
    a1.datatype_set.miss =
	context_data->indexset_data->datatype_set.datatype_data;
    a1.function_set.fix = select_context;
    a1.function_set.get = select_logic;
    a1.function_set.context = context_data;
    x1f4_llink_operator1s(&a1.operator1s);
    a1.operator2s = context_data->indexset_data->operator_set.operator2s;
    x1f4_pset_lxcast
	((void *) a1.composer_set.context, (void *) &a1.sidetype_set.miss);
    a1.textflat_set = context_data->track;
    a1.transfer_set.fine = context_data->indexset_data->transfer_set.fine;
    a1.variable_set.context = context_data->indexset_data->variable_set.text;

#if __ROLL_VALIST__
    if (context_data->valist) {
	flags |= X1f4_A1_RESOURCE;
	libx1f4i0_line_valist
	    (context_data->valist, context_data->do_memory,
	     context_data->do_storage, &a1.resource_set.free,
	     &a1.resource_set.link, &a1.resource_set.mode,
	     &a1.resource_set.context, &context_data->dlist);
    }
#endif				/* __ROLL_VALIST__ */

    context_data->line = NULL;

    status = x1f4_init_shuffle(&proGram, data, flags, &a1);
    if (status) {
	if (status == X1f4_A1_ALLOC_ERROR) {
	    perror(context_data->self);
	} else {
	    struct x1f4_eelookup_type eelookup;

	    fprintf(stderr, "%s: cannot parse ", context_data->self);
	    if (program) {
		fprintf(stderr, "`%s'", program);
	    } else {
		fprintf(stderr, "program");
	    }
	    fprintf(stderr, "\n");
	    fprintf(stderr, "%s: ", context_data->self);
	    eelookup.type_l.context = context_data;
	    eelookup.type_l.fix = look_type;
	    eelookup.type_q.e4fine =
		context_data->indexset_data->transfer_set.fine;
	    a1record.pick = 1;
	    x1f4_stat_shuffle(stderr, test_cast, data, &a1record, &eelookup);
	    fprintf(stderr, "\n");
	}
    } else {
	status = x1f4_pipe_e4ll
	    (context_data->indexset_data->autodeck_set.less, proGram);
	if (status) {
	    perror(context_data->self);
	} else {
	    X1f4_E4_C_MODE degree;
	    struct x1f4_track_type *iport;

	    context_data->class = 0;
	    context_data->error = 0;

	    context_data->exit = 0;

	    static_class = &context_data->class;

	    iport = context_data->eport;

	    context_data->eport = &context_data->plain;

	    if (context_data->sequenced) {
		status = set_track(context_data, program, proGram, &degree);
	    } else {
		status = set_frame(context_data, program, proGram, &degree);
	    }

	    context_data->eport = iport;

	    if (status) {
	    } else {
		if (*exit) {
		} else {
		    *exit = degree;
		}
	    }
	}

#if __ROLL_VALIST__
	if (context_data->valist) {
	    if (context_data->do_storage) {
		unsigned size;

		x1f4_size_xalist(context_data->valist, &size);
		fprintf(stderr, " %7u\n", size);
	    }
	}
#endif				/* __ROLL_VALIST__ */

	x1f4_fini_shuffle(&proGram);
    }

    return status;
}


static int
set_track(struct context_type *context_data, const char *program,
	  void *proGram, void *degree)
{
    int status;
    void *subtext;

    x1f4_rail_shuffle(proGram);

    status = x1f4_near_shuffle(&subtext, proGram);
    if (status) {
    } else {
	int excess;
	struct x1f4_indexset_type *indexset_data;
	unsigned bytes;

	indexset_data = context_data->indexset_data;

	context_data->logic = subtext;

	context_data->line = line_shuffle;

	if (context_data->fix) {
	    bytes = strlen(context_data->fix);
	} else {
	    bytes = 0;
	}

	context_data->bytes = bytes;

	x1f4_head_shuffle(subtext);
	while (!x1f4_tail_shuffle(subtext)) {
	    status = x1f4_slip_shuffle(subtext);

	    context_data->slip = bytes;

	    if (context_data->class) {
		int excess;

		context_data->class = 0;

		excess = indexset_data->autodeck_set.deck
		    (&indexset_data->autodeck_set.text);
		if (excess) {
		    if (status) {
		    } else {
			status = excess;
		    }
		}

		if (SIGCHLD_received) {
		    excess = wait_some(context_data);
		    if (excess) {
			if (status) {
			} else {
			    status = excess;
			}
		    }
		}
	    }

	    if (status) {
		break;
	    }
	}

	if (status) {
	    if (context_data->exit) {
		status = 0;
	    } else {
		flush_error(context_data, program);
	    }
	} else {
	    x1f4_post_shuffle(subtext, degree);
	}

	excess = x1f4_side_shuffle(&subtext, proGram, context_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}

	if (status) {
	} else {
	    if (context_data->exit) {
		*(X1f4_E4_C_MODE *) degree = context_data->exit >> 1;
	    }
	}
    }

    return status;
}


static int
stat_argv(void *context, int index)
{
    line_flat(context);
    push_flat(context, "out of range argv index: ", 25);
    x1f4_vprint_dintegral(context, push_flat, index);
    post_flat(context);

    return -1;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return size ^ fwrite(name, 1, size, cast);
}


static int
text_argv(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE index;
    int status = 0;
    struct context_type *context_data;

    context_data = static_context;

    index = I_MODE(input[0]);
    if (index < 0 || context_data->argc < index + 1) {
	status = stat_argv(static_context, index);
    } else {
	X1f4_E4_C_TEXT *l;

	l = (void *) (output);
	if (index) {
	    *l = context_data->argv[index];
	} else {
	    char *program;

	    program = (char *) context_data->program;

	    *l = program ? program : (char *) x1f4_c1_empty_string;
	}
    }

    return status;
}


static int
type_fsdeck(struct context_type *context_data)
{
    rule_fset(context_data, tf18_set);

    return 0;
}


static int
wait_some(struct context_type *context_data)
{
    int status;
    struct x1f4_lxslip_type *lxslip_data;
    void *lxport;

    SIGCHLD_received = 0;

    lxslip_data = context_data->indexset_data->sliplong_set.lxslip_data;
    while (lxslip_data->note != x1f4_note_lxport) {
	lxslip_data++;
    }

    lxport = lxslip_data->slip;

    while (1) {
	pid_t wait;

	wait = waitpid(-1, NULL, WNOHANG);
	if (wait == -1) {
	    status = 0;
	    if (1) {
		break;
	    }
	} else {
	    if (wait) {
		x1f4_wait_lxport(lxport, wait);
	    } else {
		status = 0;
		if (1) {
		    break;
		}
	    }
	}
    }

    return status;
}


static void
SIGCHLD_handler(int signal)
{
    SIGCHLD_received = 1;
    *static_class = 1;
}


static void
list_0010(struct context_type *context_data)
{
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    context_data->argv = (void *) context_data;

    if (init_xset(context_data)) {
    } else {
	x1f4_lime_mxdeck(indexset.function_set.text, context_data, list_text);

	fini_xset(context_data);
    }
}


static void
list_0220(struct context_type *context_data)
{
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    context_data->argv = (void *) context_data;

    if (init_xset(context_data)) {
    } else {
	x1f4_list_state(indexset.variable_set.text, NULL, ever_list);

	fini_xset(context_data);
    }
}


static void
usage(void)
{
    puts("Usage: tf18 [OPTIONS] PROGRAM\n\
Execute PROGRAM.\n\
\n\
Options:\n\
  -M, --stat-storage		stat program memory storage requirements\n\
  -c				execute the PROGRAM program instead program\n\
				read from the PROGRAM file\n\
  -f, --framed			run function calls in a hierarchical fashion\n\
  -m, --stat-memory		stat memory operations\n\
  -s, --sequenced		flatten call hierarchy\n\
  -t, --text TEXT		output TEXT before each first output in\n\
				sequences not interrupted by control transfer\
\n\
      --dash			recognize dash for standard input in PROGRAM\n\
      --data			list defined constants and exit\n\
      --list			list available functions and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int list_functions = 0, status = 0;
    struct context_type context;

    context.immediate = 0;

    context.do_memory = 0;
    context.do_storage = 0;

    context.pick_dash = 0;

    context.sequenced = 0;

    context.fix = NULL;
    context.bytes = 0;
    context.slip = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	int fast = ~0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "dash",	    0x00,   NULL,   0x04    },
		{   "data",	    0x00,   NULL,   0x02    },
		{   "framed",	    0x00,   NULL,   0x66    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "list",	    0x00,   NULL,   0x01    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   "sequenced",    0x00,   NULL,   0x73    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "text",	    0x01,   NULL,   0x74    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Mcfmst:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 002:
		list_functions = 2;
		break;
	    case 004:
		context.pick_dash = 1;
		break;
	    case 'M':
		context.do_storage = 1;
		break;
	    case 'c':
		context.immediate = 1;
		break;
	    case 'f':
		context.sequenced = 0;
		break;
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case 'm':
		context.do_memory = 1;
		break;
	    case 's':
		context.sequenced = 1;
		break;
	    case 't':
		context.fix = optarg;
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}
    }

    context.argc = argc - optind;
    context.argv = argv + optind;

    if (list_functions) {
	if (list_functions == 1) {
	    list_0010(&context);
	} else {
	    if (list_functions == 2) {
		list_0220(&context);
	    }
	}
    } else {
	if (context.argv ? !(argc - optind) : (argc - optind) % 3 != 1) {
	    fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' "
		    "for more information.\n", argv[0], argv[0]);

	    return 1;
	} else {
	    context.self = argv[0];

	    static_context = &context;

	    status = run_logic(&context, argc, argv);
	}
    }

    return status;
}
