/*
 * tf15.c
 * Copyright (C) 2002-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/time.h>

#include <aime.h>
#include <float1.h>
#include <inter.h>
#include <sf.h>
#include <types.h>

#define __SLIP_SHUFFLE__		1

#define I_MODE(t)			(*((X1f4_E4_C_MODE *) (t)))
#define I_USER(t)			(*((X1f4_E4_C_USER *) (t)))

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

extern const struct x1f4_function_type _libx1f4i0_e4_io[];

extern const void *const x1f4_a1_walk_link;

typedef struct atom_type {
    X1f4_E4_C_MODE half, line, news;
    int mean, slip;
    struct atom_type *atom_data;
    struct timeval s;
    struct x1f4_dispatch_type dispatch;
    void *args[2], *context;
} atom_type;

static int case_null(void *, void *, void **);
static int line_miss(void *, int, void *, struct x1f4_dispatch_type **);
#if __SLIP_SHUFFLE__
static int link_program(const char *, void *, void *, const char *);
#endif				/* __SLIP_SHUFFLE__ */
static int mean_call(void *, void *, void **);
static int mean_miss(void *, void *, void **);
static int merge_sets(struct x1f4_function_type **);
static int sail_miss(void *, int, void *, struct x1f4_dispatch_type **);
static int test_cast(void *, const char *, unsigned);
static int time_call(void *, void *, void **);
static int time_miss(void *, void *, void **);

static void usage(void);

static int c___m_e_m[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE,
    X1f4_E4_CALL,
    X1f4_E4_MODE
/* *INDENT-ON* */
}, c_____f__[] = {
/* *INDENT-OFF* */
    X1f4_E4_CALL
/* *INDENT-ON* */
}, c_____m__[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE
/* *INDENT-ON* */
};
static const struct x1f4_function_type functions[] = {
/* *INDENT-OFF* */
    {	"average",		mean_miss,
	X1f4_E4_MODE,		c___m_e_m,	3,
	X1f4_E4_WALK_LINK,			7		},
    {	"time",			time_miss,
	X1f4_E4_VOID,		c_____f__,	1,
	X1f4_E4_WALK_LINK,			4		},
    {	NULL,			NULL,
	0,			NULL,		0,
	0,					1		}
/* *INDENT-ON* */
};
static const struct x1f4_linetext_type timedcall[] = {
/* *INDENT-OFF* */
    {	{	"",
		NULL,
		X1f4_E4_VOID,
		NULL,
		0,
		0,
		0		},
	NULL							},
    {	{	"",
		NULL,
		X1f4_E4_MODE,
		c_____m__,
		1,
		0,
		0		},
	NULL							}
/* *INDENT-ON* */
};
static void *static_text;

static int
case_null(void *context, void *output, void **input)
{
    fputs("0s\n", stdout);

    l_MODE(0, output);

    return 0;
}


static int
line_miss(void *atom, int success, void *output,
	  struct x1f4_dispatch_type **dispatch)
{
    if (success) {
    } else {
	struct atom_type *atom_data, **atoms;

	atom_data = atom;

	atom_data->news += atom_data->half;

	atom_data->slip--;
	if (atom_data->slip) {
	    *dispatch = &atom_data->dispatch;
	} else {
	    *dispatch = NULL;
	}

	if (atom_data->slip) {
	} else {
	    if (atom_data->mean) {
		struct timeval f, s;

		s = atom_data->s;

		gettimeofday(&f, NULL);
		x1f4_fprint_wfloat
		    (stdout, 0, 0, 3,
		     ((double) (f.tv_sec - s.tv_sec)
		      + (double) (f.tv_usec - s.tv_usec) / 1000000)
		     / atom_data->mean);

		l_MODE(atom_data->news / atom_data->mean, output);
	    } else {
		fputc('0', stdout);

		l_MODE(0, output);
	    }

	    fputs("s\n", stdout);

	    atoms = atom_data->context;
	    *atoms = atom_data->atom_data;

	    free(atom_data);
	}
    }

    return success;
}


#if __SLIP_SHUFFLE__
static int
link_program(const char *self, void *proGram, void *context,
	     const char *program)
{
    int status;
    void *near;

    x1f4_rail_shuffle(proGram);

    status = x1f4_near_shuffle(&near, proGram);
    if (status) {
    } else {
	int excess;

# define proGram			near

	x1f4_head_shuffle(proGram);
	while (!x1f4_tail_shuffle(proGram)) {
	    status = x1f4_slip_shuffle(proGram);
	    if (status) {
		break;
	    }
	}

	if (status) {
	    fprintf(stderr, "%s: cannot execute `%s'\n", self, program);
	}

# undef proGram

	excess = x1f4_side_shuffle(&near, proGram, context);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}
#endif				/* __SLIP_SHUFFLE__ */


static int
mean_call(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE mean, news = 0;
    int (*function) (void *, void *, void **), status = 0;
    struct timeval f, s;
    struct x1f4_linetext_type *linetext_data;

    linetext_data = I_USER(input[1]);

    mean = I_MODE(input[0]);
    if (mean < 0) {
	mean = 0;
    } else {
	if (mean < 1048576) {
	} else {
	    mean = 1048576;
	}
    }

    gettimeofday(&s, NULL);

    function = linetext_data->function.function;

    if (linetext_data->function.flags & X1f4_E4_TEXT_LINK) {
	int i;
	void *text;

	text = (void *) linetext_data->context;

	for (i = mean; i; i--) {
	    X1f4_E4_C_MODE mode;

	    status = function(text, &mode, input + 2);
	    if (status) {
		break;
	    } else {
		news += mode;
	    }
	}
    } else {
	int i;

	for (i = mean; i; i--) {
	    X1f4_E4_C_MODE mode;

	    status = function(NULL, &mode, input + 2);
	    if (status) {
		break;
	    } else {
		news += mode;
	    }
	}
    }

    if (status) {
    } else {
	if (mean) {
	    gettimeofday(&f, NULL);
	    x1f4_fprint_wfloat
		(stdout, 0, 0, 3,
		 ((double) (f.tv_sec - s.tv_sec)
		  + (double) (f.tv_usec - s.tv_usec) / 1000000) / mean);

	    l_MODE(news / mean, output);
	} else {
	    fputc('0', stdout);

	    l_MODE(0, output);
	}

	fputs("s\n", stdout);
    }

    return status;
}


static int
mean_miss(void *context, void *output, void **input)
{
    int status;

    if (output == x1f4_a1_walk_link) {
	X1f4_E4_C_MODE mean;
	void **port;

	port = input[1];

	mean = I_MODE(port[0]);
	if (mean < 0) {
	    mean = 0;
	} else {
	    if (mean < 1048576) {
	    } else {
		mean = 1048576;
	    }
	}

	if (mean) {
	    struct atom_type *atom_data;

	    atom_data = malloc(sizeof(struct atom_type));
	    if (atom_data) {
		struct atom_type **atoms;
		struct x1f4_linetext_type *linetext_data;

		status = 0;

		atoms = static_text;
		atom_data->atom_data = *atoms;
		*atoms = atom_data;

		linetext_data = I_USER(port[1]);

		input[2] = &atom_data->dispatch;

		atom_data->dispatch.transfer.context =
		    (void *) linetext_data->context;
		atom_data->dispatch.transfer.function_data =
		    &linetext_data->function;
		atom_data->dispatch.transfer.input = atom_data->args + 1;
		atom_data->dispatch.transfer.output = &atom_data->half;

		atom_data->dispatch.back = atom_data;
		atom_data->dispatch.call = line_miss;

		atom_data->context = static_text;

		atom_data->args[0] = port[-1];
		atom_data->args[1] = &atom_data->line;

		atom_data->line = I_MODE(port[2]);
		atom_data->mean = mean;
		atom_data->news = 0;
		atom_data->slip = mean;

		gettimeofday(&atom_data->s, NULL);
	    } else {
		status = -1;
	    }
	} else {
	    status = case_null(context, input[0], input[1]);
	    if (status) {
	    } else {
		input[2] = NULL;
	    }
	}
    } else {
	status = mean_call(context, output, input);
    }

    return status;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count, trans;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    x1f4_count_functions(_libx1f4i0_e4_io, &trans);
    function_data = (struct x1f4_function_type *)
	malloc((count + trans + 2) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, _libx1f4i0_e4_io,
	       trans * sizeof(struct x1f4_function_type));
	function_data += trans;
	memcpy(function_data, functions,
	       sizeof(struct x1f4_function_type) << 1);
    }

    return status;
}


static int
sail_miss(void *atom, int success, void *output,
	  struct x1f4_dispatch_type **dispatch)
{
    if (success) {
    } else {
	struct atom_type *atom_data, **atoms;
	struct timeval f, s;

	atom_data = atom;

	*dispatch = NULL;

	s = atom_data->s;

	if (1) {
	    gettimeofday(&f, NULL);
	    x1f4_fprint_wfloat
		(stdout, 0, 0, 3,
		 (double) (f.tv_sec - s.tv_sec)
		 + (double) (f.tv_usec - s.tv_usec) / 1000000);
	    fputs("s\n", stdout);
	}

	atoms = atom_data->context;
	*atoms = atom_data->atom_data;

	free(atom_data);
    }

    return success;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


static int
time_call(void *context, void *output, void **input)
{
    int status;
    struct timeval f, s;
    struct x1f4_linetext_type *linetext_data;

    linetext_data = I_USER(input[0]);

    gettimeofday(&s, NULL);

    if (linetext_data->function.flags & X1f4_E4_TEXT_LINK) {
	void *text;

	text = (void *) linetext_data->context;

	status = linetext_data->function.function(text, NULL, NULL);
    } else {
	status = linetext_data->function.function(NULL, NULL, NULL);
    }

    if (1) {
	gettimeofday(&f, NULL);
	x1f4_fprint_wfloat
	    (stdout, 0, 0, 3,
	     (double) (f.tv_sec - s.tv_sec)
	     + (double) (f.tv_usec - s.tv_usec) / 1000000);
	fputs("s\n", stdout);
    }

    return status;
}


static int
time_miss(void *context, void *output, void **input)
{
    int status;

    if (output == x1f4_a1_walk_link) {
	struct atom_type *atom_data;

	atom_data = malloc(sizeof(struct atom_type));
	if (atom_data) {
	    struct atom_type **atoms;
	    struct x1f4_linetext_type *linetext_data;
	    void **port;

	    status = 0;

	    atoms = static_text;
	    atom_data->atom_data = *atoms;
	    *atoms = atom_data;

	    port = input[1];

	    linetext_data = I_USER(port[0]);

	    input[2] = &atom_data->dispatch;

	    atom_data->dispatch.transfer.context =
		(void *) linetext_data->context;
	    atom_data->dispatch.transfer.function_data =
		&linetext_data->function;
	    atom_data->dispatch.transfer.input = atom_data->args + 1;
	    atom_data->dispatch.transfer.output = NULL;

	    atom_data->dispatch.back = atom_data;
	    atom_data->dispatch.call = sail_miss;

	    atom_data->context = static_text;

	    atom_data->args[0] = port[-1];

	    gettimeofday(&atom_data->s, NULL);
	} else {
	    status = -1;
	}
    } else {
	status = time_call(context, output, input);
    }

    return status;
}


static void
usage(void)
{
    puts("Usage: tf15 [OPTIONS] PATH [TYPE NAME VALUE]\n\
Execute program read from PATH.\n\
\n\
Options:\n\
  -M, --stat-storage		stat program memory storage requirements\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_memory = 0, do_optimize = 0, do_storage = 0, status = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	int fast = ~0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Mmo", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 'M':
		do_storage = 1;
		break;
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case 'm':
		do_memory = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}
    }

    if ((argc - optind) % 3 != 1) {
	fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' for "
		"more information.\n", argv[0], argv[0]);

	return 1;
    } else {
	unsigned size;
	void *context = NULL, *data, *state_data;

	do {
	    const char *program;
	    unsigned count;

	    program = argv[optind];
	    optind++;

	    status = libx1f4i0_init_fine
		(optind, argc, argv, &count, &state_data, &context);
	    if (status) {
		break;
	    }

	    status = libx1f4i0_read_file(&data, &size, program, 1);
	    if (status) {
		libx1f4i0_stat_failure(argv[0], status, program);
	    } else {
		struct x1f4_function_type *function_data;
		void *e4fine;

		if (x1f4_init_e4fine(&e4fine, 0, NULL)) {
		    e4fine = NULL;
		}

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
#if defined HAVE_LIBx1f4l0
		    struct list_type dlist;
#endif				/* HAVE_LIBx1f4l0 */
		    struct x1f4_a1_type a1;
		    struct x1f4_a1record_type a1record;
		    unsigned flags = X1f4_A1_BCOLLECT | X1f4_A1_DDACCESS;
		    void *proGram;
#if defined HAVE_LIBx1f4l0
		    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

		    {
			char *mine;

			mine = data;
			mine[size] = 0;
		    }

		    if (do_optimize) {
			flags |= X1f4_A1_OPTIMIZE;
		    }

#if defined HAVE_LIBx1f4l0
		    libx1f4i0_init_valist(&valist, do_memory, do_storage);
#endif				/* HAVE_LIBx1f4l0 */

		    a1.function_set.get = libx1f4i0_select_function;
		    a1.function_set.context = function_data;
		    x1f4_llink_operator1s(&a1.operator1s);
		    x1f4_llink_operator2s(&a1.operator2s);
		    a1.variable_set.context = context;

		    a1.bcollect_set.a1record_data = &a1record;

		    if (e4fine) {
			flags |= X1f4_A1_TRANSFER;
			a1.transfer_set.fine = e4fine;
			x1f4_type_e4fine(e4fine, c_____f__, timedcall);
			x1f4_type_e4fine(e4fine, c___m_e_m + 1, timedcall + 1);
		    }

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			flags |= X1f4_A1_RESOURCE;
			libx1f4i0_line_valist
			    (valist, do_memory, do_storage,
			     &a1.resource_set.free, &a1.resource_set.link,
			     &a1.resource_set.mode, &a1.resource_set.context,
			     &dlist);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    status = x1f4_init_shuffle(&proGram, data, flags, &a1);
		    if (status) {
			if (status == X1f4_A1_ALLOC_ERROR) {
			    perror(argv[0]);
			} else {
			    struct x1f4_eelookup_type eelookup;

			    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
				    program);
			    fprintf(stderr, "%s: ", argv[0]);
			    a1record.pick = 1;
			    eelookup.type_l.fix = NULL;
			    x1f4_stat_shuffle
				(stderr, test_cast, data, &a1record,
				 &eelookup);
			    fprintf(stderr, "\n");
			}
		    } else {
			struct atom_type *atom_data = NULL;

			static_text = &atom_data;

#if __SLIP_SHUFFLE__
			status = link_program
			    (argv[0], proGram, &atom_data, program);
#else
			status = x1f4_link_shuffle(proGram, &atom_data);
#endif				/* __SLIP_SHUFFLE__ */

			while (atom_data) {
			    struct atom_type *atom_slip;

			    atom_slip = atom_data->atom_data;
			    free(atom_data);
			    atom_data = atom_slip;
			}

#if defined HAVE_LIBx1f4l0
			if (valist) {
			    if (do_storage) {
				unsigned size;

				x1f4_size_xalist(valist, &size);
				fprintf(stderr, " %7u\n", size);
			    }
			}
#endif				/* HAVE_LIBx1f4l0 */

			x1f4_fini_shuffle(&proGram);
		    }

#if defined HAVE_LIBx1f4l0
		    if (valist) {
			libx1f4i0_fini_valist
			    (argv[0], valist, do_memory, do_storage, &status);
		    }
#endif				/* HAVE_LIBx1f4l0 */

		    free(function_data);
		}

		if (e4fine) {
		    x1f4_fini_e4fine(&e4fine);
		}

		free(data);
	    }
	} while (0);

	if (context) {
	    x1f4_air_state(context);
	}

	if (state_data) {
	    free(state_data);
	}
    }

    return status;
}
