/*
 * tc10.c
 * Copyright (C) 2002-2010, 2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#include <inter.h>
#include <sf.h>
#include <tfxx.h>

extern const struct x1f4_function_type _libx1f4i0_e4_io[];

static int merge_sets(struct x1f4_function_type **);
static int test_cast(void *, const char *, unsigned);

static void usage(void);

static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count, trans;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    x1f4_count_functions(_libx1f4i0_e4_io, &trans);
    function_data = (struct x1f4_function_type *)
	malloc((count + trans + 1) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, _libx1f4i0_e4_io,
	       (trans + 1) * sizeof(struct x1f4_function_type));
    }

    return status;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return 1 ^ fwrite(name, size, 1, cast);
}


static void
usage(void)
{
    puts("Usage: tc10 [OPTIONS] PATH\n\
Execute program read from PATH.\n\
\n\
Options:\n\
  -o, --optimize		enable optimizations\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_optimize = 0, status = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	int fast = ~0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "help",	    0x00,   NULL,   0x68    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "o", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}
    }

    if (argc - optind != 1) {
	fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' for "
		"more information.\n", argv[0], argv[0]);

	return 1;
    } else {
	{
	    unsigned size;
	    void *data;

	    status = libx1f4i0_read_file(&data, &size, argv[optind], 1);
	    if (status) {
		libx1f4i0_stat_failure(argv[0], status, argv[optind]);
	    } else {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
		    struct x1f4_c1_type c1;
		    struct x1f4_c1record_type c1record;
		    unsigned flags = X1f4_C1_BCOLLECT | X1f4_C1_DDACCESS;
		    void *program;

		    {
			char *mine;

			mine = data;
			mine[size] = 0;
		    }

		    if (do_optimize) {
			flags |= X1f4_C1_OPTIMIZE;
		    }

		    c1.function_set.get = libx1f4i0_select_function;
		    c1.function_set.context = function_data;
		    x1f4_llink_operator1s(&c1.operator1s);
		    x1f4_llink_operator2s(&c1.operator2s);
		    c1.variable_set.context = NULL;

		    set_xxrecord(c1, &c1record);

		    status = x1f4_init_program(&program, data, flags, &c1);
		    if (status) {
			if (status == X1f4_C1_ALLOC_ERROR) {
			    perror(argv[0]);
			} else {
			    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
				    argv[optind]);
			    fprintf(stderr, "%s: ", argv[0]);
			    x1f4_stat_program
				(stderr, test_cast, data, &c1record, NULL);
			    fprintf(stderr, "\n");
			}
		    } else {
			if (0) {
			} else {
			    status = x1f4_link_program(program);
			    if (status) {
				fprintf(stderr, "%s: cannot execute `%s'\n",
					argv[0], argv[optind]);
			    }
			}

			x1f4_fini_program(&program);
		    }

		    free(function_data);
		}

		free(data);
	    }
	}
    }

    return status;
}
