/*
 * logique48.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <sf.h>

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
    void **state;
} context_type;

typedef union state_type {
    X1f4_E4_C_BILL bill;
    X1f4_E4_C_MODE mode;
    X1f4_E4_C_REAL real;
    X1f4_E4_C_TEXT text;
} state_type;

static int run_lanes(void **, unsigned, unsigned, unsigned);
static int run_logic(void *, unsigned, unsigned);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);

static void print_header(struct x1f4_variable_type *, unsigned, unsigned);
static void usage(void);

static int
run_lanes(void **slide9, unsigned count, unsigned size, unsigned precision)
{
    int status = 0;

    for (; count; count--) {
	status = run_logic(*slide9, size, precision);
	if (status) {
	    status = count;
	    break;
	} else {
	    slide9++;
	}
    }

    if (!status) {
	putchar('\n');
    }

    return status;
}


static int
run_logic(void *x1f4_expression, unsigned size, unsigned precision)
{
    int status;
    union state_type output;

    status = x1f4_link_expression(x1f4_expression, &output);
    if (status) {
    } else {
	putchar(' ');
	libx1f4i0_copy_data
	    (x1f4_type_expression(x1f4_expression), size, precision, &output);
    }

    return status;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
}


static void
print_header(struct x1f4_variable_type *slide1, unsigned count, unsigned size)
{
    unsigned l;

    l = count;
    for (; l; l--) {
	printf(" %*s", (int) size, slide1->name);
	slide1++;
    }

    {
	putchar('\n');
    }
}


static void
usage(void)
{
    puts("Usage: logique48 [OPTIONS] COUNT [BIT] [LOGIC]\n\
Evaluate LOGIC(s) for COUNT BIT(s) binaries.\n\
\n\
Options:\n\
  -P, --print			print expression\n\
  -m, --min-width		set field minimum width\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
  -s, --skip-header		do not print header\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_optimize = 0, do_print = 0, skip_header = 0, status = 1;
    unsigned min_width = 0, precision = 3, print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *min_width_specs = NULL, *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "min-width",    0x01,   NULL,   0x6d    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   "skip-header",  0x00,   NULL,   0x73    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Pm:op:s", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 021:
		print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'm':
		min_width_specs = optarg;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 's':
		skip_header = 1;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (do_print) {
	    precision_specs = NULL;
	}

	if (min_width_specs) {
	    if (x1f4_parse_wxcardinal(&min_width, min_width_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse minimum field width specif"
			"ication: `%s'\n", argv[0], min_width_specs);

		return 1;
	    }
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (!(argc - optind)) {
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    struct x1f4_variable_type *variable_data;
	    union state_type *slide7 = NULL;
	    unsigned count, flags = X1f4_E4_DDACCESS, logic, size;
	    void **state = NULL;

	    status = x1f4_parse_wxcardinal(&count, argv[optind], NULL, 0);
	    if (status) {
		fprintf(stderr, "%s: cannot parse count specification: `%s'"
			"\n", argv[0], argv[optind]);
		break;
	    }

	    optind++;

	    logic = argc - optind;

	    if (logic < count) {
		fprintf(stderr, "%s: count specification (%u) is exceeding a"
			"rguments count (%u)\n", argv[0], count, logic);
		status = 1;
		break;
	    }

	    size = min_width;

	    if (!count) {
		variable_data = NULL;
	    } else {
		variable_data = (struct x1f4_variable_type *)
		    malloc((sizeof(struct x1f4_variable_type) + sizeof(void *)
			    + sizeof(union state_type)) * count
			   + sizeof(struct x1f4_variable_type));
		if (!variable_data) {
		    break;
		} else {
		    struct x1f4_variable_type *slide1;
		    union state_type *slide3;
		    unsigned i;
		    void **slide2;

		    slide1 = variable_data;
		    slide2 = (void **) (slide1 + count + 1);
		    slide3 = (union state_type *) (slide2 + count);

		    state = slide2;

		    for (i = count; i; i--) {
			unsigned length;

			status = libx1f4i0_miss_list
			    (argv[0], argv[optind], count - i, variable_data);
			if (status) {
			    break;
			}

			length = strlen(argv[optind]);

			slide1->name = argv[optind];
			slide1->length = length;

			if (size < length) {
			    size = length;
			}

			optind++;

			slide3->mode = 0;
			slide1->type = X1f4_E4_MODE;
			slide1->flags = 0;

			*slide2 = slide3;

			slide1++;
			slide2++;
			slide3++;
		    }
		    if (i) {
			free(variable_data);
			break;
		    } else {
			slide7 = slide3;

			slide1->name = NULL;
		    }
		}
	    }

	    logic -= count;
	    if (!logic) {
		unsigned i = 0, l;

		if (!skip_header) {
		    print_header(variable_data, count, size);
		}

		l = 1 << count;
		for (; i < l; i++) {
		    union state_type *slide3;
		    unsigned b, e = 0;

		    slide3 = slide7;

		    b = i;

		    for (; e < count; e++) {
			--slide3;
			slide3->mode = b & 1;
			b >>= 1;
		    }
		    for (e = count; e; e--) {
			printf(" %*d", (int) size, (int) slide3->mode);
			slide3++;
		    }

		    {
			putchar('\n');
		    }
		}
	    } else {
		void **deletion_data;

		deletion_data = (void **) malloc(sizeof(void *) * logic);
		if (!deletion_data) {
		    perror(argv[0]);
		} else {
		    unsigned i;
		    void **slide9;

		    slide9 = deletion_data;
		    i = logic;
		    for (; i; i--) {
			const char *expression;
			struct context_type context;
			struct x1f4_attributes_type attributes;

			expression = argv[optind];

			context.state = state;
			context.variable_data = variable_data;

			attributes.function_set.get =
			    libx1f4i0_select_function;
			attributes.function_set.context = x1f4_e4_defaults;
			x1f4_llink_operator1s(&attributes.operator1s);
			x1f4_llink_operator2s(&attributes.operator2s);
			attributes.terminator = 0;
			attributes.variable_set.get = select_variable;
			attributes.variable_set.context = &context;

			status = x1f4_init_expression
			    (slide9, expression, flags, &attributes);
			if (status) {
			    logic -= i;

			    if (status == X1f4_E4_ALLOC_ERROR) {
				perror(argv[0]);
			    } else {
				fprintf(stderr, "%s: cannot parse `%s'\n",
					argv[0], expression);
			    }

			    break;
			} else {
			    if (do_optimize) {
				x1f4_line_expression(*slide9);
			    }

			    optind++;
			    slide9++;
			}
		    }

		    if (!status) {
			if (do_print) {
			    slide9 = deletion_data;
			    i = logic;
			    for (; i; i--) {
				x1f4_print_expression
				    (stdout, *slide9, print_flags);
				putchar('\n');
				if (1) {
				    slide9++;
				}
			    }
			} else {
			    int missed = ~0;

			    if (!count) {
				missed = run_lanes
				    (deletion_data, logic, 0, precision);
				if (missed) {
				    status = 1;
				}
			    } else {
				unsigned i = 0, l;

				if (!skip_header) {
				    print_header(variable_data, count, size);
				}

				l = 1 << count;
				for (; i < l; i++) {
				    union state_type *slide3;
				    unsigned b, e = 0;

				    slide3 = slide7;

				    b = i;

				    for (; e < count; e++) {
					--slide3;
					slide3->mode = b & 1;
					b >>= 1;
				    }
				    for (e = count; e; e--) {
					printf(" %*d", (int) size,
					       (int) slide3->mode);
					slide3++;
				    }

				    missed = run_lanes
					(deletion_data, logic, size,
					 precision);
				    if (missed) {
					putchar('\n');
					fflush(stdout);
					status = 1;
					break;
				    }
				}
			    }
			    if (status) {
				fprintf(stderr, "%s: cannot evaluate `%s'\n",
					argv[0], argv[optind - missed]);
			    }
			}
		    }

		    slide9 = deletion_data;
		    i = logic;
		    for (; i; i--) {
			x1f4_fini_expression(slide9);
			if (1) {
			    slide9++;
			}
		    }

		    free(deletion_data);
		}
	    }

	    if (variable_data) {
		free(variable_data);
	    }
	} while (0);
    }

    return status;
}
