/*
 * ex12.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <sf.h>
#include <types.h>

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }
#define l_REAL(e, output) \
    {									      \
	X1f4_E4_C_REAL *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }
#define l_TEXT(e, output) \
    {									      \
	X1f4_E4_C_TEXT *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
    void **state;
} context_type;

static int lose_mode(void *, void *, void **);
static int lose_real(void *, void *, void **);
static int lose_text(void *, void *, void **);
static int merge_sets(struct x1f4_function_type **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);

static void usage(void);

static const struct x1f4_function_type e12_set[] = {
/* *INDENT-OFF* */
    {	"love",			lose_text,
	X1f4_E4_TEXT,		NULL,		0,
	0,					4		},
    {	"null",			lose_mode,
	X1f4_E4_MODE,		NULL,		0,
	0,					4		},
    {	"zero",			lose_real,
	X1f4_E4_REAL,		NULL,		0,
	0,					4		},
    {	NULL,			NULL,
	0,			NULL,		0,
	0,					1		}
/* *INDENT-ON* */
};

static int
lose_mode(void *context, void *output, void **input)
{
    l_MODE(0, output);

    return 0;
}


static int
lose_real(void *context, void *output, void **input)
{
    l_REAL(0, output);

    return 0;
}


static int
lose_text(void *context, void *output, void **input)
{
    l_TEXT("love", output);

    return 0;
}


static int
merge_sets(struct x1f4_function_type **set)
{
    int status = 0;
    struct x1f4_function_type *function_data;
    unsigned count;

    x1f4_count_functions(x1f4_e4_defaults, &count);
    function_data = (struct x1f4_function_type *)
	malloc((count + 4) * sizeof(struct x1f4_function_type));
    if (!function_data) {
	status = 1;
    } else {
	*set = function_data;
	memcpy(function_data, x1f4_e4_defaults,
	       count * sizeof(struct x1f4_function_type));
	function_data += count;
	memcpy(function_data, e12_set, sizeof(struct x1f4_function_type) << 2);
    }

    return status;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
}


static void
usage(void)
{
    puts("Usage: ex12 [OPTIONS] EXPRESSION [TYPE NAME VALUE]\n\
Test functions evaluation, evaluate EXPRESSION.\n\
\n\
Options:\n\
  -P, --print			print expression\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
      --list			list available functions and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_print = 0, list_functions = 0, status = 1;
    unsigned precision = 3;

    unsetenv("POSIXLY_CORRECT");

    {
	char *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "list",	    0x00,   NULL,   0x01    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Pp:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (do_print) {
	    precision_specs = NULL;
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (list_functions == 1) {
	struct x1f4_function_type *function_data;

	if (merge_sets(&function_data)) {
	    status = 1;
	    perror(argv[0]);
	} else {
	    x1f4_print_functions(stdout, function_data);

	    free(function_data);
	}
    } else if ((argc - optind) % 3 != 1) {
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
	    struct x1f4_variable_type *variable_data;
	    void **state = NULL, *x1f4_expression;

	    expression = argv[optind];
	    optind++;

	    status = libx1f4i0_init_list
		(argv[0], argc, argv, optind, (void *) &variable_data, &state);
	    if (status) {
		break;
	    }

	    {
		struct x1f4_function_type *function_data;

		if (merge_sets(&function_data)) {
		    status = 1;
		    perror(argv[0]);
		} else {
		    struct context_type context;
		    struct x1f4_attributes_type attributes;

		    context.state = state;
		    context.variable_data = variable_data;

		    attributes.function_set.get = libx1f4i0_select_function;
		    attributes.function_set.context = function_data;
		    x1f4_llink_operator1s(&attributes.operator1s);
		    x1f4_llink_operator2s(&attributes.operator2s);
		    attributes.terminator = 0;
		    attributes.variable_set.get = select_variable;
		    attributes.variable_set.context = &context;

		    status = x1f4_init_expression
			(&x1f4_expression, expression, X1f4_E4_DDACCESS,
			 &attributes);
		    if (status) {
			if (status == X1f4_E4_ALLOC_ERROR) {
			    perror(argv[0]);
			} else {
			    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
				    expression);
			}
		    } else {
			if (do_print) {
			    x1f4_print_expression(stdout, x1f4_expression, 0);
			    putchar('\n');
			} else {
			    struct lead_type output;

			    status = x1f4_link_expression
				(x1f4_expression, &output);
			    if (status) {
				fprintf(stderr, "%s: cannot evaluate `%s'\n",
					argv[0], expression);
			    } else {
				libx1f4i0_type_data
				    (x1f4_type_expression(x1f4_expression),
				     precision, &output);
			    }
			}

			x1f4_fini_expression(&x1f4_expression);
		    }

		    free(function_data);
		}
	    }

	    libx1f4i0_fini_list(argv[0], (void *) &variable_data);
	} while (0);
    }

    return status;
}
