/*
 * ev16.c
 * Copyright (C) 2006-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4i0
# include <libx1f4i0.h>
#endif				/* HAVE_LIBx1f4i0 */
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <math.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <f2.h>
#if !defined HAVE_LIBx1f4i0
# include <float1.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <sf.h>
#include <types.h>

#define MAKE_DOUBLE(a, b)		a b
#define MAKE_SINGLE(a, b)		a

#define I_MODE(i)			(*((X1f4_E4_C_MODE *) (i)))
#define I_REAL(r)			(*((X1f4_E4_C_REAL *) (r)))
#define I_TEXT(t)			(*((X1f4_E4_C_TEXT *) (t)))

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }
#define l_REAL(e, output) \
    {									      \
	X1f4_E4_C_REAL *l;						      \
									      \
	l = (output);							      \
	*l = (e);							      \
    }

typedef struct context_type {
    struct x1f4_variable_type *variable_data;
    void **state;
} context_type;

static int c23xx_m__(void *, void **);
static int c23xx_r__(void *, void **);
static int c32xx_m__(void *, void **);
static int select_variable(const char *, unsigned, const void *,
			   const struct x1f4_variable_type **, void **);

static void usage(void);

static const int c_____m__[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE
/* *INDENT-ON* */
}, c_____r__[] = {
/* *INDENT-OFF* */
    X1f4_E4_REAL
/* *INDENT-ON* */
};
static const struct x1f4_operator_type e23xx_m__[] = {
/* *INDENT-OFF* */
    {	MAKE_DOUBLE("#", "#"),  c23xx_m__,	0210,
	X1f4_E4_MODE,		c_____m__,
	0,			2,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e23xx_r__[] = {
/* *INDENT-OFF* */
    {	MAKE_DOUBLE("#", "#"),  c23xx_r__,	0210,
	X1f4_E4_REAL,		c_____r__,
	0,			2,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e32xx_m__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("#", " "),  c32xx_m__,	0210,
	X1f4_E4_MODE,		c_____m__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e33xx_m__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("@", " "),  c23xx_m__,	0210,
	X1f4_E4_MODE,		c_____m__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e33xx_r__[] = {
/* *INDENT-OFF* */
    {	MAKE_SINGLE("@", " "),  c23xx_r__,	0210,
	X1f4_E4_REAL,		c_____r__,
	0,			1,
	NULL,			NULL				}
/* *INDENT-ON* */
}, e42xx_m__[] = {
/* *INDENT-OFF* */
    {	MAKE_DOUBLE("@", "@"),  c32xx_m__,	0210,
	X1f4_E4_MODE,		c_____m__,
	0,			2,
	NULL,			NULL				}
/* *INDENT-ON* */
}, *ev16_e4_1_arithmetics[] = {
/* *INDENT-OFF* */
    e23xx_m__,
    e23xx_r__,
    e32xx_m__,
    e33xx_m__,
    e33xx_r__,
    e42xx_m__
/* *INDENT-ON* */
};

static int
c23xx_m__(void *output, void **input)
{
    X1f4_E4_C_MODE m;
    int status = 0;

    m = I_MODE(input[0]);
    if (!m) {
	status = 1;
    } else {
	X1f4_E4_C_MODE e;
	int c = 0;

	if (m < 0) {
	    m = -m;
	}
	e = m;
	while (e) {
	    c++;
	    e >>= 1;
	}
	if (!(m & (((X1f4_E4_C_MODE) 1 << (c - 1)) - 1))) {
	    c--;
	}

	l_MODE(c, output);
    }

    return status;
}


static int
c23xx_r__(void *output, void **input)
{
    double r, s, t;
    int status = 0;

    r = I_REAL(input[0]);
    if (!r) {
	status = 1;
    } else {
	if (r < 0) {
	    r = -r;
	}
	r = log2(r);
	s = modf(r, &t);
	if (s) {
	    t++;
	}

	l_REAL(t, output);
    }

    return status;
}


static int
c32xx_m__(void *output, void **input)
{
    X1f4_E4_C_MODE m;
    unsigned s;
    unsigned char *a, *e;

    m = I_MODE(input[0]);
    s = sizeof(m);
    a = (unsigned char *) &m;
    e = a + s;
    s >>= 1;
    while (s) {
	unsigned c, o;

	e--;

	c = *a;
	c = c << 4 | c >> 4;
	c = (c & 0x33) << 2 | (c & 0xcc) >> 2;
	c = (c & 0x55) << 1 | (c & 0xaa) >> 1;
	o = *e;
	o = o << 4 | o >> 4;
	o = (o & 0x33) << 2 | (o & 0xcc) >> 2;
	o = (o & 0x55) << 1 | (o & 0xaa) >> 1;

	*a = o;
	*e = c;

	a++;

	s--;
    }

    l_MODE(m, output);

    return 0;
}


static int
select_variable(const char *f, unsigned length, const void *context,
		const struct x1f4_variable_type **variable, void **state)
{
    int status = X1f4_E4_PARSE_ERROR;
    const struct context_type *context_data;
    const struct x1f4_variable_type *variable_data;

    context_data = context;
    variable_data = context_data->variable_data;
    if (variable_data) {
	while (variable_data->name) {
	    if (length == variable_data->length
		&& !memcmp((void *) f, variable_data->name, length)) {
		break;
	    }
	    variable_data++;
	}
	if (variable_data->name) {
	    status = 0;
	    *state = context_data->state
		[variable_data - context_data->variable_data];
	    *variable = variable_data;
	}
    }

    return status;
}


static void
usage(void)
{
    puts("Usage: ev16 [OPTIONS] EXPRESSION [TYPE NAME VALUE]\n\
Test operators evaluation, evaluate EXPRESSION.\n\
\n\
Options:\n\
  -P, --print			print expression\n\
  -o, --optimize		enable optimizations\n\
  -p, --precision DIGITS	set precision for printing reals to DIGITS\n\
      --detail-constants	detail constants when printing expression\n\
      --detail-operators	detail operators when printing expression\n\
      --list-1			list available unary operators and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int do_optimize = 0, do_print = 0, list_functions = 0, status = 0;
    unsigned precision = 3, print_flags = 0;

    unsetenv("POSIXLY_CORRECT");

    {
	char *precision_specs = NULL;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "detail-constants",
				    0x00,   NULL,   0x12    },
		{   "detail-operators",
				    0x00,   NULL,   0x11    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "list-1",	    0x00,   NULL,   0x02    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   "precision",    0x01,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long(argc, argv, "Pop:", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 002:
		list_functions = 2;
		break;
	    case 021:
		print_flags |= X1f4_E4_DETAIL_OPERATORS;
		break;
	    case 022:
		print_flags |= X1f4_E4_DETAIL_CONSTANTS;
		break;
	    case 'P':
		do_print = 1;
		break;
	    case 'o':
		do_optimize = 1;
		break;
	    case 'p':
		precision_specs = optarg;
		break;
	    case 'h':
		usage();

		return 0;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);

		return 0;
	    case '?':
		return 1;
	    }
	}

	if (do_print) {
	    precision_specs = NULL;
	}

	if (precision_specs) {
	    if (x1f4_parse_wxcardinal(&precision, precision_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse precision specification: `"
			"%s'\n", argv[0], precision_specs);

		return 1;
	    }
	}
    }

    if (list_functions == 2) {
	const struct x1f4_operator_type *const *operAtor1s, *const *operator1s;

	x1f4_llink_operator1s(&operator1s);
	libx1f4i0_join_sets(&operAtor1s, operator1s, ev16_e4_1_arithmetics, 6);
	x1f4_print_operators(stdout, operAtor1s, 1);
	if (operAtor1s != operator1s) {
	    free((void *) operAtor1s);
	}
    } else if ((argc - optind) % 3 != 1) {
	status = 1;
	fprintf(stderr, "%s: wrong number of arguments\nType `%s --help' for"
		" more information.\n", argv[0], argv[0]);
    } else {
	do {
	    const char *expression;
	    struct x1f4_variable_type *variable_data;
	    unsigned flags = X1f4_E4_DDACCESS;
	    void **state = NULL, *x1f4_expression;
#if defined HAVE_LIBx1f4l0
	    void *valist = NULL;
#endif				/* HAVE_LIBx1f4l0 */

	    status = 1;

	    expression = argv[optind];
	    optind++;

	    status = libx1f4i0_init_list
		(argv[0], argc, argv, optind, (void *) &variable_data, &state);
	    if (status) {
		break;
	    }

#if defined HAVE_LIBx1f4l0
	    {
		x1f4_init_valist(&valist, 1024);
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    {
		struct context_type context;
		struct x1f4_attributes_type attributes;
		const struct x1f4_operator_type *const *operator1s;

		context.state = state;
		context.variable_data = variable_data;

		attributes.function_set.get = libx1f4i0_select_function;
		attributes.function_set.context = x1f4_e4_defaults;
		x1f4_llink_operator1s(&operator1s);
		libx1f4i0_join_sets
		    (&attributes.operator1s, operator1s, ev16_e4_1_arithmetics,
		     6);
		x1f4_llink_operator2s(&attributes.operator2s);
		attributes.terminator = 0;
		attributes.variable_set.get = select_variable;
		attributes.variable_set.context = &context;

		flags |= X1f4_E4_ESTORAGE;

#if defined HAVE_LIBx1f4l0
		if (valist) {
		    flags |= X1f4_E4_RESOURCE;
		    if (1) {
			attributes.resource_set.context = valist;
			attributes.resource_set.free = x1f4_free_valist;
			attributes.resource_set.link = x1f4_link_valist;
			attributes.resource_set.mode = x1f4_mode_valist;
		    }
		}
#endif				/* HAVE_LIBx1f4l0 */

		{
		    status = x1f4_init_expression
			(&x1f4_expression, expression, flags, &attributes);
		}

		if (attributes.operator1s != operator1s) {
		    free((void *) attributes.operator1s);
		}
	    }
	    if (status) {
		if (status == X1f4_E4_ALLOC_ERROR) {
		    perror(argv[0]);
		} else {
		    fprintf(stderr, "%s: cannot parse `%s'\n", argv[0],
			    expression);
		}
	    } else {
		if (do_optimize) {
		    x1f4_line_expression(x1f4_expression);
		}

		if (do_print) {
		    x1f4_print_expression
			(stdout, x1f4_expression, print_flags);
		    putchar('\n');
		} else {
		    struct lead_type output;

		    status = x1f4_link_expression(x1f4_expression, &output);
		    if (status) {
			fprintf(stderr, "%s: cannot evaluate `%s'\n", argv[0],
				expression);
		    } else {
			libx1f4i0_type_data
			    (x1f4_type_expression(x1f4_expression), precision,
			     &output);
		    }
		}

		x1f4_fini_expression(&x1f4_expression);
	    }

#if defined HAVE_LIBx1f4l0
	    if (valist) {
		unsigned linger;

		if (1) {
		    x1f4_size_valist(valist, &linger);
		}
		if (linger) {
		    fprintf(stderr, "%s: memory is leaking\n", argv[0]);
		    status = 1;
		}

		if (1) {
		    x1f4_fini_valist(&valist);
		}
	    }
#endif				/* HAVE_LIBx1f4l0 */

	    libx1f4i0_fini_list(argv[0], (void *) &variable_data);
	} while (0);
    }

    return status;
}
