/*
 * aime.c
 * Copyright (C) 2002-2014, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#if defined HAVE_FEATURES_H
# include <features.h>
#endif				/* HAVE_FEATURES_H */

#include <getopt.h>
#if defined HAVE_LIBx1f4l0
# include <libx1f4l0.h>
#endif				/* HAVE_LIBx1f4l0 */
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>

#include <aime.h>
#if !defined HAVE_LIBx1f4i0
# include <cardinal-wx.h>
# include <integral.v-d.h>
#endif				/* !HAVE_LIBx1f4i0 */
#include <inter.h>
#include <types.h>

#define __ROLL_VALIST__			1

#if defined HAVE_LIBx1f4l0
#else
# undef __ROLL_VALIST__
# define __ROLL_VALIST__		0
#endif				/* HAVE_LIBx1f4l0 */

#define DATA_BITS \
    X1f4_LXDATA_EVERLINK

#define DATE_BITS \
    X1f4_LXDATE_EVERLINK

#define DECQ_BITS \
    X1f4_LXDECQ_EVERLINK | X1f4_LXDECQ_RESETNEW

#define FILE_BITS \
    X1f4_LXFILE_EVERLINK | X1f4_LXFILE_SETCLOSE

#define LINE_BITS \
    X1f4_LXLINE_EVERLINK | X1f4_LXLINE_RESETNEW

#define LIST_BITS \
    X1f4_LXLIST_EVERLINK | X1f4_LXLIST_RESETNEW

#define NEAR_BITS \
    X1f4_LXNEAR_LONGPIPE

#define POLL_BITS \
   X1f4_LXPOLL_EVERLINK

#define PORT_BITS \
    X1f4_LXPORT_EVERLINK | X1f4_LXPORT_SETCLOSE

#define TRAP_BITS \
    X1f4_LXTRAP_REPORTER

#define I_MODE(i)			(*((X1f4_E4_C_MODE *) (i)))

#define l_MODE(e, output) \
    {									      \
	X1f4_E4_C_MODE *l;						      \
									      \
	l = (void *) (output);						      \
	*l = (e);							      \
    }

#define l_TEXT(e, output) \
    {									      \
	X1f4_E4_C_TEXT *l;						      \
									      \
	l = (void *) (output);						      \
	*l = (e);							      \
    }

typedef struct context_type {
    char **argv;
    const char *preprocessor, *program, *self;
    int argc, class, disallow_comments, do_memory, do_optimize, do_storage,
	do_trace, error, exit, immediate,
	(*line) (struct context_type *, unsigned *), pick_dash, preprocess,
	sequenced;
#if __ROLL_VALIST__
    struct list_type dlist;
#endif				/* __ROLL_VALIST__ */
    const struct x1f4_function_type *function_slip;
    struct x1f4_indexset_type *indexset_data;
    struct x1f4_track_type *eport, plain, track;
    unsigned bail_net, bail_out, call, initial;
    void *bailer, *logic;
#if __ROLL_VALIST__
    void *valist;
#endif				/* __ROLL_VALIST__ */
    const void *data, *text;
} context_type;

typedef struct window_type {
    int (*fini) (struct context_type *), (*init) (struct context_type *);
} window_type;

extern const struct x1f4_function_type _libx1f4i0_lead_0[];

static int blow_vxbeta(struct context_type *);
static int case_window(struct context_type *);
static int copy_error(void *, const char *, unsigned);
static int copy_output(void *, const char *, unsigned);
static int dumb_mode(void *, void *, void **);
static int ever_list(void *, void *, struct x1f4_variable_type *);
static int ever_last(void *, const char *, unsigned);
static int ever_name(void *, const char *, unsigned);
static int ever_star(void *, const char *, unsigned);
static int fast_call(void *, const struct x1f4_function_type *);
static int fini_tf13(struct context_type *);
static int fini_valist(struct context_type *);
static int fit_logic(struct context_type *, const void *);
static int fix_logic(struct context_type *, const void *);
static int flat_lxbail(struct context_type *);
static int flat_text(void *);
static int flush_error(struct context_type *);
static int init_tf13(struct context_type *);
static int init_valist(struct context_type *);
static int lay_logic(struct context_type *, const void *);
static int line_flat(void *);
static int line_program(struct context_type *, unsigned *);
static int line_shuffle(struct context_type *, unsigned *);
static int link_lxbail(struct context_type *, unsigned *,
		       struct x1f4_textport_type *);
static int link_vxbeta(struct context_type *);
static int list_text(void *, const void *);
static int lose_call(void *, const struct x1f4_function_type *);
static int mode_argc(void *, void *, void **);
static int pick_shuffle(struct context_type *, unsigned *);
static int pipe_shuffle(struct context_type *, void *);
static int post_flat(void *);
static int push_flat(void *, const char *, unsigned);
static int rule_fset(struct context_type *, const struct x1f4_function_type *);
static int rule_line(struct context_type *);
static int rule_source(void *, unsigned, unsigned);
static int run_frame(struct context_type *, void *);
static int run_logic(struct context_type *, const void *);
static int run_track(struct context_type *, void *);
static int seek_line(struct context_type *, unsigned, const char **,
		     unsigned *, unsigned *, unsigned);
static int set_composer(struct context_type *, void **,
			int (**) (void *, int, int,
				  const struct x1f4_linetext_type **));
static int set_frame(struct context_type *, void *, void *);
static int set_logic(struct context_type *, const void *);
static int set_track(struct context_type *, void *, void *);
static int stat_argv(void *, int);
static int test_cast(void *, const char *, unsigned);
static int tempt_logic(struct context_type *, int, char **);
static int text_argv(void *, void *, void **);
static int type_fsdeck(struct context_type *);
static int wait_some(struct context_type *);

static void SIGCHLD_handler(int);
static void list_0010(struct context_type *);
static void list_0220(struct context_type *);
static void list_0230(struct context_type *);
static void usage(void);

static int *static_class;
static const int c_____m__[] = {
/* *INDENT-OFF* */
    X1f4_E4_MODE
/* *INDENT-ON* */
};
static sig_atomic_t SIGCHLD_received;
static const struct window_type riddle[] = {
/* *INDENT-OFF* */
    {	fini_valist,		init_valist			},
    {	blow_vxbeta,		link_vxbeta			},
    {	case_window,		type_fsdeck			}
/* *INDENT-ON* */
};
static struct x1f4_datapipe_type long_pipes[] = {
/* *INDENT-OFF* */
    {	"b",			x1f4_pipe_lxdata,
	0,			1,		NULL		},
    {	"f",			x1f4_pipe_lxfile,
	0,			1,		NULL		}
/* *INDENT-ON* */
};
static const struct x1f4_function_type command_set[] = {
/* *INDENT-OFF* */
    {	"argc",			mode_argc,
	X1f4_E4_MODE,		NULL,		0,
	X1f4_E4_KEEP_CALL,			4		},
    {	"argv",			text_argv,
	X1f4_E4_TEXT,		c_____m__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	NULL,			NULL,
	0,			NULL,		0,
	0,					1		}
/* *INDENT-ON* */
}, t13_set[] = {
/* *INDENT-OFF* */
    {	"exit",			dumb_mode,
	X1f4_E4_VOID,		c_____m__,	1,
	X1f4_E4_KEEP_CALL,			4		},
    {	NULL,			NULL,
	0,			NULL,		0,
	0,					1		}
/* *INDENT-ON* */
};
static struct x1f4_lxnear_type lxnear;
static struct x1f4_lxtrap_type lxtrap;
const static struct x1f4_lxtile_type tiles[] = {
/* *INDENT-OFF* */
    {	x1f4_note_lxcast,	0,		NULL		},
    {	x1f4_note_lxdata,	DATA_BITS,	NULL		},
    {	x1f4_note_lxdate,	DATE_BITS,	NULL		},
    {	x1f4_note_lxdecq,	DECQ_BITS,	NULL		},
    {	x1f4_note_lxfile,	FILE_BITS,	NULL		},
    {	x1f4_note_lxline,	LINE_BITS,	NULL		},
    {	x1f4_note_lxlist,	LIST_BITS,	NULL		},
    {	x1f4_note_lxmath,	0,		NULL		},
    {	x1f4_note_lxnear,	NEAR_BITS,	&lxnear		},
    {	x1f4_note_lxpoll,	POLL_BITS,	NULL		},
    {	x1f4_note_lxport,	PORT_BITS,	NULL		},
    {	x1f4_note_lxtext,	0,		NULL		},
    {	x1f4_note_lxtrap,	TRAP_BITS,	&lxtrap		}
/* *INDENT-ON* */
};
static const struct x1f4_textpipe_type near_pipes[] = {
/* *INDENT-OFF* */
    {	"o",		copy_output,		1,	NULL	},
    {	"v",		copy_error,		1,	NULL	}
/* *INDENT-ON* */
};
static void *static_context;

static int
blow_vxbeta(struct context_type *context_data)
{
    int excess, status;

    if (context_data->bailer) {
	status = flat_lxbail(context_data);
    } else {
	status = 0;
    }

    excess = x1f4_flat_indexset(context_data->indexset_data);
    if (excess) {
    } else {
	if (status) {
	} else {
	    status = excess;
	}
    }

    return status;
}


static int
case_window(struct context_type *context_data)
{
    return 0;
}


static int
copy_error(void *none, const char *data, unsigned size)
{
    fwrite(data, size, 1, stderr);

    return 0;
}


static int
copy_output(void *none, const char *data, unsigned size)
{
    fwrite(data, size, 1, stdout);

    return 0;
}


static int
dumb_mode(void *context, void *output, void **input)
{
    ((struct context_type *) static_context)->exit = I_MODE(input[0]) << 1 | 1;

    return 1;
}


static int
ever_list(void *indexset, void *data, struct x1f4_variable_type *variable_data)
{
    int type;
    struct x1f4_eelookup_type *eelookup_data;

    eelookup_data =
	&((struct x1f4_indexset_type *) indexset)->eelookup_set.eelookup;

    type = variable_data->type;
    if (X1f4_E4_CALL < type) {
	const struct x1f4_linetext_type *linetext_data;

	if (x1f4_pick_e4fine
	    (((struct x1f4_indexset_type *) indexset)->transfer_set.fine, type,
	     &linetext_data)) {
	    x1f4_vprint_xtfunction
		(NULL, copy_output, variable_data, ever_star,
		 &linetext_data->function, eelookup_data);
	    putchar('\n');
	} else {
	    puts(variable_data->name);
	}
    } else {
	x1f4_dump_type(NULL, ever_name, type, eelookup_data);
	putchar(' ');
	puts(variable_data->name);
    }

    return 0;
}


static int
ever_last(void *function, const char *data, unsigned size)
{
    struct x1f4_function_type *function_data;

    function_data = function;
    copy_error(NULL, function_data->name, function_data->length);

    return 0;
}


static int
ever_name(void *none, const char *data, unsigned size)
{
    if (size) {
	if (*data == '`') {
	    data++;
	    size--;
	} else {
	    if (data[size - 1] == '\'') {
		size--;
	    }
	}

	copy_output(NULL, data, size);
    }

    return 0;
}


static int
ever_star(void *variable, const char *data, unsigned size)
{
    struct x1f4_variable_type *variable_data;

    variable_data = variable;
    copy_output(NULL, "(*", 2);
    copy_output(NULL, variable_data->name, variable_data->length);
    copy_output(NULL, ")", 1);

    return 0;
}


static int
fast_call(void *context, const struct x1f4_function_type *function_data)
{
    struct context_type *context_data;
    unsigned call;

    context_data = context;

    call = context_data->call;

    call++;

    context_data->call = call;

    for (; call; call--) {
	fputs("    ", stderr);
    }

    fputs("<---- ", stderr);
    x1f4_vprint_xtfunction
	(stderr, test_cast, (void *) function_data, ever_last, function_data,
	 &((struct context_type *)
	   context)->indexset_data->eelookup_set.eelookup);
    fputs(" ----\n", stderr);

    return 0;
}


static int
fini_tf13(struct context_type *context_data)
{
    int excess, status = 0;
    static const struct window_type *window_data;
    unsigned access;

    access = sizeof(riddle) / sizeof(struct window_type);

    window_data = riddle + access;

    for (; access; access--) {
	window_data--;
	excess = window_data->fini(context_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}


static int
fini_valist(struct context_type *context_data)
{
#if !__ROLL_VALIST__
# define status				0
#endif				/* !__ROLL_VALIST__ */

#if __ROLL_VALIST__
    int status = 0;
    void *valist;
#endif				/* __ROLL_VALIST__ */

#if __ROLL_VALIST__
    valist = context_data->valist;
    if (valist) {
	libx1f4i0_fini_valist
	    (context_data->self, valist, context_data->do_memory,
	     context_data->do_storage, &status);
	if (status) {
	} else {
	    if (context_data->do_memory) {
		fprintf(stderr, " %7u max allocation\n",
			context_data->dlist.ever);
	    }
	}
    }
#endif				/* __ROLL_VALIST__ */

    return status;

#if !__ROLL_VALIST__
# undef status
#endif				/* !__ROLL_VALIST__ */
}


static int
fit_logic(struct context_type *context_data, const void *data)
{
    int status;
    unsigned bits = 0;

    if (context_data->disallow_comments) {
    } else {
	bits |= X1f4_A1_SCOMMENT; 
    }

    static_context = context_data;

    if (x1f4_peer_shuffle(data, bits)) {
	status = set_logic(context_data, data);
    } else {
	status = run_logic(context_data, data);
    }

    return status;
}


static int
fix_logic(struct context_type *context_data, const void *data)
{
    int status;

    if (context_data->preprocess) {
	status = lay_logic(context_data, data);
    } else {
	status = fit_logic(context_data, data);
    }

    return status;
}


static int
flat_lxbail(struct context_type *context_data)
{
    return x1f4_fini_lxbail(&context_data->bailer);
}


static int
flat_text(void *context)
{
    int status;
    struct context_type *context_data;
    struct x1f4_indexset_type *indexset_data;

    context_data = context;

    indexset_data = context_data->indexset_data;

    status = indexset_data->autodeck_set.deck
	(&indexset_data->autodeck_set.text);

    if (SIGCHLD_received) {
	if (wait_some(context_data)) {
	    status = X1f4_EX_CANNOT_CONTINUE;
	}
    }

    return status;
}


static int
flush_error(struct context_type *context_data)
{
    if (context_data->error) {
    } else {
	fflush(stdout);
	fprintf(stderr, "%s:", context_data->self);
	rule_line(context_data);
	fprintf(stderr, " cannot execute program\n");
    }

    return 17;
}


static int
init_tf13(struct context_type *context_data)
{
    int status;
    const struct window_type *window_data;
    unsigned access;

#if __ROLL_VALIST__
    context_data->valist = NULL;
#endif				/* __ROLL_VALIST__ */

    context_data->plain.data = context_data;
    context_data->plain.line = line_flat;
    context_data->plain.post = post_flat;
    context_data->plain.push = push_flat;

    x1f4_rule_eproxy(&context_data->track);

    context_data->track.data = &context_data->eport;

    context_data->eport = &context_data->plain;

    access = sizeof(riddle) / sizeof(struct window_type);

    window_data = riddle;

    for (; access; access--) {
	status = window_data->init(context_data);
	if (status) {
	    break;
	} else {
	    window_data++;
	}
    }

    if (status) {
	access = sizeof(riddle) / sizeof(struct window_type) - access;

	for (; access; access--) {
	    window_data--;
	    window_data->fini(context_data);
	}
    }

    return status;
}


static int
init_valist(struct context_type *context_data)
{
#if __ROLL_VALIST__
    libx1f4i0_init_valist
	(&context_data->valist, context_data->do_memory,
	 context_data->do_storage);
#endif				/* __ROLL_VALIST__ */

#if __ROLL_VALIST__
    context_data->dlist.ever = 0;
    context_data->dlist.size = 0;
#endif				/* __ROLL_VALIST__ */

#if __ROLL_VALIST__
    context_data->dlist.valist = context_data->valist;
#endif				/* __ROLL_VALIST__ */

    return 0;
}


static int
lay_logic(struct context_type *context_data, const void *data)
{
    int status;
    void *text;

    status = libx1f4i0_cell_text
	(context_data->self, context_data->preprocessor, data, &text);
    if (status) {
    } else {
	context_data->text = text;

	status = fit_logic(context_data, text);

	free(text);
    }

    return status;
}


static int
line_flat(void *context)
{
    int delete;

    if (((struct context_type *) context)->line) {
	fflush(stdout);
	fputs(((struct context_type *) context)->self, stderr);
	fputs(":", stderr);
	rule_line(context);
	fputs(" ", stderr);
	delete = 0;
    } else {
	delete = -1;
    }

    return delete;
}


static int
line_program(struct context_type *context_data, unsigned *copy)
{
    return x1f4_look_program(context_data->logic, copy);
}


static int
line_shuffle(struct context_type *context_data, unsigned *copy)
{
    return x1f4_look_shuffle(context_data->logic, copy);
}


static int
link_lxbail(struct context_type *context_data, unsigned *bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;
    struct x1f4_lxbail_type lxbail;
    void *text;

    lxbail.recovery_set.down = context_data->bail_net;
    lxbail.recovery_set.tear = context_data->bail_out;

    lxbail.textflat_set = context_data->track;

    status = x1f4_init_lxbail(&text, X1f4_LXBAIL_TEXTFLAT, &lxbail);
    if (status) {
    } else {
	context_data->bailer = text;

	*bits |= X1f4_TEXTPORT_MISSBAIL;

	textport_data->missbail_set.text = text;
	textport_data->missbail_set.call = x1f4_call_lxbail;
	textport_data->missbail_set.fine = x1f4_fine_lxbail;
	textport_data->missbail_set.miss = x1f4_miss_lxbail;
    }

    return status;
}


static int
link_vxbeta(struct context_type *context_data)
{
    struct x1f4_textport_type textport;
    unsigned textport_bits;

    textport_bits = X1f4_TEXTPORT_CASTTYPE | X1f4_TEXTPORT_SUB2LIST
	| X1f4_TEXTPORT_SUBFLIST | X1f4_TEXTPORT_TEXTFLAT;

    textport.lasttype_set.type = X1f4_E4_LAST;

    textport.tilelong_set.lxtile_data = tiles;
    textport.tilelong_set.miss =
	sizeof(tiles) / sizeof(struct x1f4_lxtile_type);

#if __ROLL_VALIST__
    if (context_data->valist) {
	textport_bits |= X1f4_TEXTPORT_CODELINK;
	libx1f4i0_line_valist
	    (context_data->valist, context_data->do_memory,
	     context_data->do_storage, &textport.codelink_set.free,
	     &textport.codelink_set.link, &textport.codelink_set.mode,
	     &textport.codelink_set.data, &context_data->dlist);
    }
#endif				/* __ROLL_VALIST__ */

#if __ROLL_VALIST__
    if (context_data->valist) {
	textport_bits |= X1f4_TEXTPORT_RESOURCE;
	libx1f4i0_line_valist
	    (context_data->valist, context_data->do_memory,
	     context_data->do_storage, &textport.resource_set.free,
	     &textport.resource_set.link, &textport.resource_set.mode,
	     &textport.resource_set.data, &context_data->dlist);
    }
#endif				/* __ROLL_VALIST__ */

    lxnear.longpipe_set.datapipe_data = long_pipes;
    lxnear.longpipe_set.miss = 2;

    lxnear.nearpipe_set.textpipe_data = near_pipes;
    lxnear.nearpipe_set.miss = 2;

    lxtrap.reporter_set.track = &context_data->eport;

    textport.textflat_set = context_data->track;

    textport.autodeck_set.class = &context_data->class;

    if (context_data->bail_out) {
	link_lxbail(context_data, &textport_bits, &textport);
    }

    return x1f4_fast_indexset
	(context_data->indexset_data, textport_bits, &textport);
}


static int
list_text(void *context, const void *text)
{
    x1f4_print_htfunction
	(stdout, text, "", ((struct context_type *) context)->indexset_data,
	 x1f4_look_indexset);

    return 0;
}


static int
lose_call(void *context, const struct x1f4_function_type *function_data)
{
    struct context_type *context_data;
    unsigned call;

    context_data = context;

    call = context_data->call;

    call--;

    context_data->call = call;

    for (++call; call; call--) {
	fputs("    ", stderr);
    }

    fputs(" ---- ", stderr);
    x1f4_vprint_xtfunction
	(stderr, test_cast, (void *) function_data, ever_last, function_data,
	 &((struct context_type *)
	   context)->indexset_data->eelookup_set.eelookup);
    fputs(" ---->\n", stderr);

    return 0;
}


static int
mode_argc(void *context, void *output, void **input)
{
    l_MODE(((struct context_type *) static_context)->argc, output);

    return 0;
}


static int
pick_shuffle(struct context_type *context_data, unsigned *copy)
{
    return x1f4_seek_shuffle(context_data->logic, copy);
}


static int
pipe_shuffle(struct context_type *context_data, void *shuffle)
{
    x1f4_a1screen_type screen;

    screen.data = context_data;

    screen.back = lose_call;
    screen.fast = fast_call;
    screen.lose = lose_call;

    context_data->call = -1;

    return x1f4_pipe_shuffle(shuffle, &screen);
}


static int
post_flat(void *context)
{
    ((struct context_type *) context)->error = 1;

    fputs("\n", stderr);

    return 0;
}


static int
push_flat(void *context, const char *data, unsigned size)
{
    fwrite(data, size, 1, stderr);

    return 0;
}


static int
rule_fset(struct context_type *context_data,
	  const struct x1f4_function_type *function_data)
{
    const char *name;
    int status;

    name = function_data->name;
    if (name) {
	void *fsdeck;

	fsdeck = context_data->indexset_data->function_set.text;

	do {
	    status = x1f4_post_mxpath
		(fsdeck, name, function_data->length, function_data);
	    if (status) {
		break;
	    } else {
		function_data++;
	    }

	    name = function_data->name;
	} while (name);
    } else {
	status = 0;
    }

    return status;
}


static int
rule_line(struct context_type *context_data)
{
    int (*line) (struct context_type *, unsigned *);

    line = context_data->line;
    if (line) {
	unsigned copy;

	line(context_data, &copy);
	fputs(" ", stderr);
	rule_source(context_data, context_data->initial, copy);
	fputs(":", stderr);
    } else {
	if (context_data->program) {
	    fputs(" ", stderr);
	    fputs(context_data->program, stderr);
	    fputs(":", stderr);
	}
    }

    return 17;
}


static int
rule_source(void *context, unsigned pick, unsigned offset)
{
    const char *name;
    struct context_type *context_data;
    unsigned miss, size;

    context_data = context;

    seek_line(context_data, offset, &name, &size, &miss, pick);

    if (size) {
	fwrite(name, size, 1, stderr);
	fputs(": ", stderr);
    }

    fprintf(stderr, "%u", miss);

    return 0;
}


static int
run_frame(struct context_type *context_data, void *proGram)
{
    int status;

    context_data->class = 0;
    context_data->error = 0;

    context_data->exit = 0;

    context_data->logic = proGram;

    context_data->line = line_program;

    status = x1f4_note_program
	(proGram, &context_data->class, context_data, flat_text);
    if (status) {
	if (context_data->exit) {
	    status = context_data->exit >> 1;
	} else {
	    flush_error(context_data);
	}
    }

    return status;
}


static int
run_logic(struct context_type *context_data, const void *data)
{
    int status;
    struct x1f4_c1_type c1;
    struct x1f4_c1record_type c1record;
    unsigned flags;
    void *proGram;

    x1f4_xset_program(&c1, &flags, context_data->indexset_data);

    flags |= X1f4_C1_BCOLLECT | X1f4_C1_COMPOSER | X1f4_C1_SIDELIST;

    if (context_data->disallow_comments) {
	flags &= ~X1f4_C1_SCOMMENT;
    }

    if (context_data->do_optimize) {
	flags |= X1f4_C1_OPTIMIZE;
    }

    c1.bcollect_set.c1record_data = &c1record;
    set_composer(context_data, &c1.composer_set.context, &c1.composer_set.get);
    x1f4_pset_lxcast
	((void *) c1.composer_set.context, (void *) &c1.sidetype_set.miss);

#if __ROLL_VALIST__
    if (context_data->valist) {
	flags |= X1f4_C1_RESOURCE;
	libx1f4i0_line_valist
	    (context_data->valist, context_data->do_memory,
	     context_data->do_storage, &c1.resource_set.free,
	     &c1.resource_set.link, &c1.resource_set.mode,
	     &c1.resource_set.context, &context_data->dlist);
    }
#endif				/* __ROLL_VALIST__ */

    context_data->line = NULL;

    status = x1f4_init_program(&proGram, data, flags, &c1);
    if (status) {
	if (status == X1f4_C1_ALLOC_ERROR) {
	    perror(context_data->self);
	} else {
	    fprintf(stderr, "%s: cannot parse ", context_data->self);
	    if (context_data->program) {
		fprintf(stderr, "`%s'", context_data->program);
	    } else {
		fprintf(stderr, "program");
	    }
	    fprintf(stderr, "\n");
	    fprintf(stderr, "%s: ", context_data->self);
	    c1record.hint.data = context_data;
	    c1record.hint.rule = rule_source;
	    c1record.pick = context_data->initial;
	    x1f4_lame_program
		(stderr, test_cast, data, X1f4_C1_HINT_MASK, &c1record,
		 &context_data->indexset_data->eelookup_set.eelookup);
	    fprintf(stderr, "\n");
	}
    } else {
	if (0) {
	} else {
	    if (context_data->sequenced) {
		status = run_track(context_data, proGram);
	    } else {
		status = run_frame(context_data, proGram);
	    }
	}

	if (context_data->bailer) {
	    x1f4_pass_lxbail(context_data->bailer);
	}

#if __ROLL_VALIST__
	if (context_data->valist) {
	    if (context_data->do_storage) {
		unsigned size;

		x1f4_size_xalist(context_data->valist, &size);
		fprintf(stderr, " %7u\n", size);
	    }
	}
#endif				/* __ROLL_VALIST__ */

	x1f4_fini_program(&proGram);
    }

    return status;
}


static int
run_track(struct context_type *context_data, void *proGram)
{
    int status = 0;
    struct x1f4_indexset_type *indexset_data;

    indexset_data = context_data->indexset_data;

    context_data->class = 0;
    context_data->error = 0;

    context_data->exit = 0;

    context_data->logic = proGram;

    context_data->line = line_program;

    x1f4_head_program(proGram);
    while (!x1f4_tail_program(proGram)) {
	status = x1f4_slip_program(proGram);

	if (context_data->class) {
	    int excess;

	    context_data->class = 0;

	    excess = indexset_data->autodeck_set.deck
		(&indexset_data->autodeck_set.text);
	    if (excess) {
		if (status) {
		} else {
		    status = excess;
		}
	    }

	    if (SIGCHLD_received) {
		excess = wait_some(context_data);
		if (excess) {
		    if (status) {
		    } else {
			status = excess;
		    }
		}
	    }
	}

	if (status) {
	    break;
	}
    }

    if (status) {
	if (context_data->exit) {
	    status = context_data->exit >> 1;
	} else {
	    flush_error(context_data);
	}
    }

    return status;
}


static int
seek_line(struct context_type *context_data, unsigned copy, const char **name,
	  unsigned *size, unsigned *miss, unsigned pick)
{
    unsigned line = 0;

    if (context_data->preprocess) {
	const char *fail = NULL;
	unsigned slip = 0;

	libx1f4i0_line_text(context_data->text, copy, &fail, &slip, &line);

	if (slip) {
	    *name = fail;
	    *size = slip;

	    line++;
	} else {
	    line += pick;

	    fail = context_data->program;
	    if (fail) {
		*name = fail;
		*size = strlen(fail);
	    } else {
		*name = NULL;
		*size = 0;
	    }
	}
    } else {
	const char *data, *fail;

	fail = context_data->program;
	if (fail) {
	    *name = fail;
	    *size = strlen(fail);
	} else {
	    *name = NULL;
	    *size = 0;
	}

	data = context_data->data;
	while (copy) {
	    if (*data == 10) {
		line++;
	    }

	    copy--;
	    data++;
	}

	line += pick;
    }

    *miss = line;

    return 0;
}


static int
set_composer(struct context_type *context_data, void **text,
	     int (**lock) (void *, int, int,
			   const struct x1f4_linetext_type **))
{
    struct x1f4_lxslip_type *lxslip_data, *lxslip_text;

    lxslip_text = context_data->indexset_data->sliplong_set.lxslip_data;

    lxslip_data = lxslip_text;
    while (lxslip_data->note != x1f4_note_lxcast) {
	lxslip_data++;
    }

    *lock = x1f4_lock_lxcast;

    *text = lxslip_data->slip;

    while (lxslip_text->note != x1f4_note_lxtrap) {
	lxslip_text++;
    }

    x1f4_qfix_lxtrap(lxslip_text->slip, lxslip_data->slip, x1f4_lock_lxcast);

    return 0;
}


static int
set_frame(struct context_type *context_data, void *proGram, void *degree)
{
    int status;

    context_data->line = pick_shuffle;

    context_data->logic = proGram;

    status = x1f4_long_shuffle
	(proGram, &context_data->class, context_data, flat_text, degree);
    if (status) {
	if (context_data->exit) {
	    status = 0;
	    *(X1f4_E4_C_MODE *) degree = context_data->exit >> 1;
	} else {
	    flush_error(context_data);
	}
    }

    return status;
}


static int
set_logic(struct context_type *context_data, const void *data)
{
    int status;
    struct x1f4_a1_type a1;
    struct x1f4_a1record_type a1record;
    unsigned flags;
    void *proGram;

    x1f4_xset_shuffle(&a1, &flags, context_data->indexset_data);

    flags |= X1f4_A1_BCOLLECT | X1f4_A1_COMPOSER | X1f4_A1_SIDELIST
	| X1f4_A1_TEXTFLAT;

    if (context_data->disallow_comments) {
	flags &= ~X1f4_A1_SCOMMENT;
    }

    if (context_data->do_optimize) {
	flags |= X1f4_A1_OPTIMIZE;
    }

    a1.bcollect_set.a1record_data = &a1record;
    set_composer(context_data, &a1.composer_set.context, &a1.composer_set.get);
    x1f4_pset_lxcast
	((void *) a1.composer_set.context, (void *) &a1.sidetype_set.miss);
    a1.textflat_set = context_data->track;

#if __ROLL_VALIST__
    if (context_data->valist) {
	flags |= X1f4_A1_RESOURCE;
	libx1f4i0_line_valist
	    (context_data->valist, context_data->do_memory,
	     context_data->do_storage, &a1.resource_set.free,
	     &a1.resource_set.link, &a1.resource_set.mode,
	     &a1.resource_set.context, &context_data->dlist);
    }
#endif				/* __ROLL_VALIST__ */

    context_data->line = NULL;

    status = x1f4_init_shuffle(&proGram, data, flags, &a1);
    if (status) {
	if (status == X1f4_A1_ALLOC_ERROR) {
	    perror(context_data->self);
	} else {
	    fprintf(stderr, "%s: cannot parse ", context_data->self);
	    if (context_data->program) {
		fprintf(stderr, "`%s'", context_data->program);
	    } else {
		fprintf(stderr, "program");
	    }
	    fprintf(stderr, "\n");
	    fprintf(stderr, "%s: ", context_data->self);
	    a1record.hint.data = context_data;
	    a1record.hint.rule = rule_source;
	    a1record.pick = context_data->initial;
	    x1f4_lame_shuffle
		(stderr, test_cast, data, X1f4_A1_HINT_MASK, &a1record,
		 &context_data->indexset_data->eelookup_set.eelookup);
	    fprintf(stderr, "\n");
	}
    } else {
	if (0) {
	} else {
	    status = x1f4_pipe_e4ll
		(context_data->indexset_data->autodeck_set.less, proGram);
	    if (status) {
		perror(context_data->self);
	    } else {
		X1f4_E4_C_MODE degree;

		if (context_data->do_trace) {
		    pipe_shuffle(context_data, proGram);
		}

		context_data->class = 0;
		context_data->error = 0;

		context_data->exit = 0;

		if (context_data->sequenced) {
		    status = set_track(context_data, proGram, &degree);
		} else {
		    status = set_frame(context_data, proGram, &degree);
		}

		if (status) {
		} else {
		    status = degree;
		}
	    }
	}

	if (context_data->bailer) {
	    x1f4_pass_lxbail(context_data->bailer);
	}

#if __ROLL_VALIST__
	if (context_data->valist) {
	    if (context_data->do_storage) {
		unsigned size;

		x1f4_size_xalist(context_data->valist, &size);
		fprintf(stderr, " %7u\n", size);
	    }
	}
#endif				/* __ROLL_VALIST__ */

	x1f4_fini_shuffle(&proGram);
    }

    return status;
}


static int
set_track(struct context_type *context_data, void *proGram, void *degree)
{
    int status;
    void *subtext;

    x1f4_rail_shuffle(proGram);

    status = x1f4_near_shuffle(&subtext, proGram);
    if (status) {
    } else {
	int excess;
	struct x1f4_indexset_type *indexset_data;

	indexset_data = context_data->indexset_data;

	context_data->logic = subtext;

	context_data->line = line_shuffle;

	x1f4_head_shuffle(subtext);
	while (!x1f4_tail_shuffle(subtext)) {
	    status = x1f4_slip_shuffle(subtext);

	    if (context_data->class) {
		int excess;

		context_data->class = 0;

		excess = indexset_data->autodeck_set.deck
		    (&indexset_data->autodeck_set.text);
		if (excess) {
		    if (status) {
		    } else {
			status = excess;
		    }
		}

		if (SIGCHLD_received) {
		    excess = wait_some(context_data);
		    if (excess) {
			if (status) {
			} else {
			    status = excess;
			}
		    }
		}
	    }

	    if (status) {
		break;
	    }
	}

	if (status) {
	    if (context_data->exit) {
		status = 0;
	    } else {
		flush_error(context_data);
	    }
	} else {
	    x1f4_post_shuffle(subtext, degree);
	}

	excess = x1f4_side_shuffle(&subtext, proGram, context_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}

	if (status) {
	} else {
	    if (context_data->exit) {
		*(X1f4_E4_C_MODE *) degree = context_data->exit >> 1;
	    }
	}
    }

    return status;
}


static int
stat_argv(void *context, int index)
{
    line_flat(context);
    push_flat(context, "out of range argv index: ", 25);
    x1f4_vprint_dintegral(context, push_flat, index);
    post_flat(context);

    return -1;
}


static int
test_cast(void *cast, const char *name, unsigned size)
{
    return size ^ fwrite(name, 1, size, cast);
}


static int
tempt_logic(struct context_type *context_data, int argc, char **argv)
{
    int status;
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    status = init_tf13(context_data);
    if (status) {
    } else {
	const char *program;
	unsigned count;
	void *state_data;

	program = argv[optind];
	optind++;

	if (context_data->argv) {
	    state_data = NULL;
	} else {
	    status = libx1f4i0_init_fine
		(optind, argc, argv, &count, &state_data,
		 &indexset.variable_set.text);
	}
	if (status) {
	} else {
	    SIGCHLD_received = 0;

	    static_class = &context_data->class;

	    signal(SIGCHLD, SIGCHLD_handler);

	    signal(SIGPIPE, SIG_IGN);

	    if (0) {
	    } else {
		if (context_data->immediate) {
		    if (0) {
		    } else {
			context_data->data = program;

			context_data->program = NULL;

			status = fix_logic(context_data, program);
		    }
		} else {
		    unsigned size;
		    void *data;

		    do {
			if (context_data->pick_dash) {
			    if (program[0] == '-') {
				if (program[1]) {
				} else {
				    status = libx1f4i0_read_dash
					(&data, &size, 1);
				    if (1) {
					break;
				    }
				}
			    }
			}

			status = libx1f4i0_read_file(&data, &size, program, 1);
		    } while (0);
		    if (status) {
			libx1f4i0_stat_failure(argv[0], status, program);
		    } else {
			if (1) {
			    char *mine;

			    mine = data;
			    mine[size] = 0;
			}

			context_data->data = data;

			context_data->program = program;

			status = fix_logic(context_data, data);

			free(data);
		    }
		}
	    }

	    signal(SIGPIPE, SIG_DFL);

	    signal(SIGCHLD, SIG_DFL);
	}

	if (state_data) {
	    free(state_data);
	}

	if (1) {
	    int excess;

	    excess = fini_tf13(context_data);
	    if (excess) {
		if (status) {
		} else {
		    status = excess;
		}
	    }
	}
    }

    return status;
}


static int
text_argv(void *context, void *output, void **input)
{
    X1f4_E4_C_MODE index;
    int argc, status = 0;

    argc = ((struct context_type *) static_context)->argc;

    index = I_MODE(input[0]);
    if (index < 0 ? index < -argc : !(index < argc)) {
	status = stat_argv(static_context, index);
    } else {
	if (index < 0) {
	    index += argc;
	}
	if (index) {
	    l_TEXT(((struct context_type *) static_context)->argv[index],
		   output);
	} else {
	    char *program;

	    program = (char *)
		((struct context_type *) static_context)->program;
	    if (program) {
		l_TEXT(program, output);
	    } else {
		l_TEXT((char *) x1f4_c1_empty_string, output);
	    }
	}
    }

    return status;
}


static int
type_fsdeck(struct context_type *context_data)
{
    if (0) {
    } else {
	rule_fset(context_data, _libx1f4i0_lead_0);
	rule_fset(context_data, t13_set);
	if (context_data->argv) {
	    rule_fset(context_data, command_set);
	}
    }

    return 0;
}


static int
wait_some(struct context_type *context_data)
{
    int status;
    struct x1f4_lxslip_type *lxslip_data;
    void *lxport;

    SIGCHLD_received = 0;

    lxslip_data = context_data->indexset_data->sliplong_set.lxslip_data;
    while (lxslip_data->note != x1f4_note_lxport) {
	lxslip_data++;
    }

    lxport = lxslip_data->slip;

    while (1) {
	pid_t wait;

	wait = waitpid(-1, NULL, WNOHANG);
	if (wait == -1) {
	    status = 0;
	    if (1) {
		break;
	    }
	} else {
	    if (wait) {
		x1f4_wait_lxport(lxport, wait);
	    } else {
		status = 0;
		if (1) {
		    break;
		}
	    }
	}
    }

    return status;
}


static void
SIGCHLD_handler(int signal)
{
    SIGCHLD_received = 1;
    *static_class = 1;
}


static void
list_0010(struct context_type *context_data)
{
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    context_data->argv = (void *) context_data;

    if (init_tf13(context_data)) {
    } else {
	x1f4_lime_mxdeck(indexset.function_set.text, context_data, list_text);

	fini_tf13(context_data);
    }
}


static void
list_0220(struct context_type *context_data)
{
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    context_data->argv = (void *) context_data;

    if (init_tf13(context_data)) {
    } else {
	x1f4_list_state(indexset.variable_set.text, &indexset, ever_list);

	fini_tf13(context_data);
    }
}


static void
list_0230(struct context_type *context_data)
{
    struct x1f4_indexset_type indexset;

    context_data->indexset_data = &indexset;

    context_data->argv = (void *) context_data;

    if (init_tf13(context_data)) {
    } else {
	x1f4_print_tboperators
	    (stdout, context_data->indexset_data->operator_set.operator2s, 0,
	     context_data->indexset_data, x1f4_look_indexset);

	fini_tf13(context_data);
    }
}


static void
usage(void)
{
    puts("Usage: aime [OPTIONS] PROGRAM [TYPE NAME VALUE]\n\
Execute PROGRAM.\n\
\n\
Options:\n\
  -B, --reference-bail-out DUE	bail out references after DUE links\n\
  -D, --reference-bail-set SET	clear bailed out references when SET many\n\
  -M, --stat-storage		stat program memory storage requirements\n\
  -P, --print			does nothing\n\
  -S, --initial INDEX		set first line number as INDEX (default 1)\n\
  -W, --cpp COMMAND		set preprocessor command (default `cpp')\n\
  -a				make arguments available as argc()/argv()\n\
  -c				execute the PROGRAM program instead program\n\
				read from the PROGRAM file\n\
  -f, --framed			run function calls in a hierarchical fashion\n\
  -m, --stat-memory		stat memory operations\n\
  -o, --optimize		enable optimizations\n\
  -p, --preprocess		run source through the C preprocessor\n\
  -s, --sequenced		flatten call hierarchy (default)\n\
  -t, --trace			trace function calls\n\
      --binary			list available binary operators and exit\n\
      --dash			recognize dash for standard input in PROGRAM\n\
      --data			list defined constants and exit\n\
      --disallow-comments	disallow # led comments\n\
      --list			list available functions and exit\n\
      --help			display this help and exit\n\
      --version			output version information and exit");
}


int
main(int argc, char **argv)
{
    int list_functions = 0, status = 0;
    struct context_type context;

    context.bail_net = 1 << 13;
    context.bail_out = 4;
    context.bailer = NULL;

    context.disallow_comments = 0;

    context.do_memory = 0;
    context.do_optimize = 0;
    context.do_storage = 0;
    context.do_trace = 0;

    context.immediate = 0;

    context.initial = 1;

    context.pick_dash = 0;

    context.preprocess = 0;
    context.preprocessor = NULL;

    context.sequenced = 1;

    unsetenv("POSIXLY_CORRECT");

    {
	const char *bail_net_specs = NULL, *bail_out_specs = NULL,
	    *initial_specs = NULL;
	int fast = ~0, miss_fine = 0;

	while (1) {
	    char c;
	    static struct option long_options[] = {
/* *INDENT-OFF* */
		{   "binary",	    0x00,   NULL,   0x05    },
		{   "cpp",	    0x01,   NULL,   0x57    },
		{   "dash",	    0x00,   NULL,   0x04    },
		{   "data",	    0x00,   NULL,   0x02    },
		{   "disallow-comments",
				    0x00,   NULL,   0x03    },
		{   "framed",	    0x00,   NULL,   0x66    },
		{   "help",	    0x00,   NULL,   0x68    },
		{   "initial",	    0x01,   NULL,   0x53    },
		{   "list",	    0x00,   NULL,   0x01    },
		{   "optimize",	    0x00,   NULL,   0x6f    },
		{   "preprocess",   0x00,   NULL,   0x70    },
		{   "print",	    0x00,   NULL,   0x50    },
		{   "reference-bail-out",
				    0x01,   NULL,   0x42    },
		{   "reference-bail-set",
				    0x01,   NULL,   0x44    },
		{   "sequenced",    0x00,   NULL,   0x73    },
		{   "stat-memory",  0x00,   NULL,   0x6d    },
		{   "stat-storage", 0x00,   NULL,   0x4d    },
		{   "trace",	    0x00,   NULL,   0x74    },
		{   "version",	    0x00,   NULL,   0x76    },
		{   NULL,	    0x00,   NULL,   0x00    }
/* *INDENT-ON* */
	    };

	    c = getopt_long
		(argc, argv, "B:D:MPS:W:acfmopst", long_options, NULL);

	    if (!~c) {
		break;
	    }

	    switch (c) {
	    case 001:
		list_functions = 1;
		break;
	    case 002:
		list_functions = 2;
		break;
	    case 003:
		context.disallow_comments = 1;
		break;
	    case 004:
		context.pick_dash = 1;
		break;
	    case 005:
		list_functions = 3;
		break;
	    case 'B':
		bail_out_specs = optarg;
		break;
	    case 'D':
		bail_net_specs = optarg;
		break;
	    case 'M':
		context.do_storage = 1;
		break;
	    case 'P':
		break;
	    case 'S':
		initial_specs = optarg;
		break;
	    case 'W':
		context.preprocessor = optarg;
		break;
	    case 'a':
		miss_fine = 1;
		break;
	    case 'c':
		context.immediate = 1;
		break;
	    case 'f':
		context.sequenced = 0;
		break;
	    case 'h':
	    case 'v':
		if (!~fast) {
		    fast = c;
		}
		break;
	    case 'm':
		context.do_memory = 1;
		break;
	    case 'o':
		context.do_optimize = 1;
		break;
	    case 'p':
		context.preprocess = 1;
		break;
	    case 's':
		context.sequenced = 1;
		break;
	    case 't':
		context.do_trace = 1;
		break;
	    case '?':
		return 1;
	    }
	}

	if (~fast) {
	    switch (fast) {
	    case 'h':
		usage();
		break;
	    case 'v':
		printf("%s (%s) %s\n", argv[0], PACKAGE, VERSION);
	    }

	    return 0;
	}

	if (miss_fine) {
	    context.argc = argc - optind;
	    context.argv = argv + optind;
	} else {
	    context.argv = NULL;
	}

	if (bail_net_specs) {
	    if (x1f4_parse_wxcardinal
		(&context.bail_net, bail_net_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse reference bail out set siz"
			"especification: `%s'\n", argv[0], bail_net_specs);

		return 1;
	    }
	}

	if (bail_out_specs) {
	    if (x1f4_parse_wxcardinal
		(&context.bail_out, bail_out_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse reference bail out specifi"
			"cation: `%s'\n", argv[0], bail_out_specs);

		return 1;
	    }
	}

	if (initial_specs) {
	    if (x1f4_parse_wxcardinal
		(&context.initial, initial_specs, NULL, 0)) {
		fprintf(stderr, "%s: cannot parse first line number specific"
			"ation: `%s'\n", argv[0], initial_specs);

		return 1;
	    }
	}
    }

    if (list_functions) {
	if (list_functions == 1) {
	    list_0010(&context);
	} else {
	    if (list_functions == 2) {
		list_0220(&context);
	    } else {
		if (list_functions == 3) {
		    list_0230(&context);
		}
	    }
	}
    } else {
	if (context.argv ? !(argc - optind) : (argc - optind) % 3 != 1) {
	    fprintf(stderr, "%s: wrong number of arguments\nTry `%s --help' "
		    "for more information.\n", argv[0], argv[0]);

	    return 1;
	} else {
	    context.self = argv[0];

	    status = tempt_logic(&context, argc, argv);
	}
    }

    return status;
}
