/*
 * sshell.e.c
 * Copyright (C) 2007-2013, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>

#include <errno.h>
#include <string.h>

#include <sys/types.h>
#include <sys/wait.h>

#include <integral.v-d.h>
#include <sshell-names.h>
#include <sshell-types.h>
#include <sshell.e.h>

#define Record(record)			((struct record_type *) (record))

#define false(e)			0

static int type_call(void *, int (*) (void *, const char *, unsigned), void *,
		     const char *, unsigned);
static int type_exit(void *, int (*) (void *, const char *, unsigned), void *);
static int type_line(void *, int (*) (void *, const char *, unsigned), void *);

static int
type_call(void *look, int (*type) (void *, const char *, unsigned),
	  void *record, const char *call, unsigned size)
{
    int status = -1;
    
    do {
	if (type(look, call, size)) {
	    break;
	}
	if (type(look, " has failed for ", 16)) {
	    break;
	}
	if (type_line(look, type, record)) {
	    break;
	}
	if (type(look, ": ", 2)) {
	    break;
	}
	if (type(look, strerror(errno), strlen(strerror(errno)))) {
	    break;
	}
	
	status = 0;
    } while (0);
    
    return status;
}


static int
type_exit(void *look, int (*type) (void *, const char *, unsigned),
	  void *record)
{
    int status = -1;

    do {
	int effect;

	effect = Record(record)->address.collect.exit.status;

	if (WIFEXITED(effect)) {
	    if (WEXITSTATUS(effect)) {
		if (type(look, " has exited ", 12)) {
		    break;
		}
		if (x1f4_vprint_dintegral(look, type, WEXITSTATUS(effect))) {
		    break;
		}
	    }
	} else {
	    if (WIFSIGNALED(effect)) {
		if (type(look, " was killed by signal ", 22)) {
		    break;
		}
		if (x1f4_vprint_dintegral(look, type, WTERMSIG(effect))) {
		    break;
		}
	    }
	}
	
	status = 0;
    } while (0);
	
    return status;
}


static int
type_line(void *look, int (*type) (void *, const char *, unsigned),
	  void *record)
{
    int status = -1;

    do {
	const char *const *argv;

	if (type(look, "{ `", 3)) {
	    break;
	}
	if (type(look, Record(record)->command.path,
		 strlen(Record(record)->command.path))) {
	    break;
	}

	argv = Record(record)->command.argv;
	if (!argv) {
	    if (type(look, "', `", 4)) {
		break;
	    }
	    if (type(look, Record(record)->command.path,
		     strlen(Record(record)->command.path))) {
		break;
	    }
	} else {
	    while (*argv) {
		if (type(look, "', `", 4)) {
		    break;
		}
		if (type(look, *argv, strlen(*argv))) {
		    break;
		}

		argv++;
	    }
	    if (*argv) {
		break;
	    }
	}
	
	if (type(look, "' }", 3)) {
	    break;
	}

	status = 0;
    } while (0);

    return status;
}


int
x1f4_stat_sshell(void *record, int error)
{
    int status = -1;

    if (error == LINE_ERROR) {
	status = 0;
    } else {
	do {
	    int (*post) (void *), (*type) (void *, const char *, unsigned);
	    void *look;
	    
	    look = Record(record)->printer.look;
	    
	    post = Record(record)->printer.head;
	    if (post) {
		if (post(look)) {
		    break;
		}
	    }
	    
	    type = Record(record)->printer.type;
	    if (type) {
		if (false(error == -1)) {
		} else if (error == CELL_ERROR) {
		    if (type(look, "timeout expired for ", 20)) {
			break;
		    }
		    if (type_line(look, type, record)) {
			break;
		    }
		} else if (error == EXIT_ERROR) {
		    if (type_line(look, type, record)) {
			break;
		    }
		    if (type_exit(look, type, record)) {
			break;
		    }
		} else if (error == FLOW_ERROR) {
		    if (type_line(look, type, record)) {
			break;
		    }
		    if (type(look, " has overflood", 14)) {
			break;
		    }
		} else {
		    if (error == FORK_ERROR) {
			if (type_call(look, type, record, "fork", 4)) {
			    break;
			}
		    } else if (error == LOCK_ERROR) {
			if (type_call
			    (look, type, record, "fcntl.F_SETFL.O_NONBLOCK",
			     24)) {
			    break;
			}
#if defined HAVE_PSELECT
		    } else if (error == MASK_ERROR) {
			if (type_call(look, type, record, "sigprocmask", 11)) {
			    break;
			}
#endif				/* HAVE_PSELECT */
		    } else if (error == NEAR_ERROR) {
			if (type_call
			    (look, type, record, "fcntl.F_GETFD", 13)) {
			    break;
			}
		    } else if (error == PIPE_ERROR) {
			if (type_call(look, type, record, "pipe", 4)) {
			    break;
			}
		    } else if (error == POLL_ERROR) {
			if (type_call(look, type, record, "select", 6)) {
			    break;
			}
		    } else if (error == READ_ERROR) {
			if (type_call(look, type, record, "read", 4)) {
			    break;
			}
		    } else if (error == WAIT_ERROR) {
			if (type_call(look, type, record, "wait", 4)) {
			    break;
			}
		    }
		}
	    }
	    
	    post = Record(record)->printer.tail;
	    if (post) {
		if (post(look)) {
		    break;
		}
	    }

	    status = 0;
	} while (0);
    }

    return status;
}

