/*
 * sshell.c
 * Copyright (C) 2006-2012, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BUGS:
 *
 * .`unsigned' is used for time representations
 * .all the signal stuff is really creepy
 *
 * there are actually more bugs...
 */

#include <config.h>

#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/select.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <sshell-names.h>
#include <sshell-types.h>
#include <sshell.e.h>

#define ARGV_MASK			(1u << 0x00)
#define BACK_MASK			(1u << 0x01)
#define C3RD_MASK			(1u << 0x02)
#define CALL_MASK			(1u << 0x03)
#define DATA_MASK			(1u << 0x04)
#define E1ST_MASK			(1u << 0x05)
#define E6TH_MASK			(1u << 0x06)
#define HEAD_MASK			(1u << 0x07)
#define LINE_MASK			(1u << 0x08)
#define LOOK_MASK			(1u << 0x09)
#define MASK_MASK			(1u << 0x0a)
#define PATH_MASK			(1u << 0x0b)
#define SIZE_MASK			(1u << 0x0c)
#define TAIL_MASK			(1u << 0x0d)
#define TEXT_MASK			(1u << 0x0e)
#define TYPE_MASK			(1u << 0x0f)

#define KILL_C3RD			0
#define MISS_C3RD			1
#define NULL_C3RD			2

#define C3RD				MISS_C3RD
#define E1ST				~0
#define E6TH				~0
#define NONE				0
#define SIZE				(0u << 0x00)

#define BUFFER				4096

#define SECOND_A(v)			((v)->tv_sec)
#define SECOND_F(v)			((v)->tv_usec)

#define Record(record)			((struct record_type *) (record))

typedef struct sshell_type {
    struct {
	int (*line) (void *, const void *, unsigned);
	void *text;
    } channel;
    struct {
	int (*call) (void *);
	void *back;
    } cleanup;
    struct {
	const char *const *argv, *path;
    } command;
    struct {
	int (*head) (void *), (*tail) (void *),
	    (*type) (void *, const char *, unsigned);
	void *look;
    } printer;
    struct {
	char *data;
	unsigned size;
    } storage;
    struct {
	unsigned e1st, e6th;
    } timeout;
    struct {
	unsigned c3rd, mask;
    } trailer;
} sshell_type;

static int ever_stream(int *);
static int exec_sshell(void *);
static int link_sshell(void *, char **, unsigned *);
static int read_sshell(void *, char **, unsigned *);
static int wait_sshell(void *, int);

static void SIGCHLD_handler(int);

sig_atomic_t SIGCHLD_received;

static int
ever_stream(int *fds)
{
    int status;

    status = pipe(fds);
    if (status) {
	fds[0] = -1;
	fds[1] = -1;
    } else {
	unsigned i = 0;

	for (; i < 0; i++) {
	    int flags;

	    if ((flags = fcntl(fds[i], F_GETFL)) == -1
		|| fcntl(fds[i], F_SETFL, flags | O_NONBLOCK) == -1) {
		status = 1;
		close(fds[0]);
		close(fds[1]);
		break;
	    }
	}
    }

    return status;
}


static int
exec_sshell(void *record)
{
    do {
	int (*post) (void *), (*type) (void *, const char *, unsigned);
	void *look;

	look = Record(record)->printer.look;

	post = Record(record)->printer.head;
	if (post) {
	    if (post(look)) {
		break;
	    }
	}

	type = Record(record)->printer.type;
	if (type) {
	    if (type(look, "cannot execute `", 16)) {
		break;
	    }
	    if (type(look, Record(record)->command.path,
		     strlen(Record(record)->command.path))) {
		break;
	    }
	    if (type(look, "': ", 3)) {
		break;
	    }
	    if (type(look, strerror(errno), strlen(strerror(errno)))) {
		break;
	    }
	}

	post = Record(record)->printer.tail;
	if (post) {
	    if (post(look)) {
		break;
	    }
	}
    } while (0);

    return -1;
}


static int
link_sshell(void *record, char **data, unsigned *size)
{
    int status = -1, pipes_0[2], pipes_1[2];
    struct record_type *record_data;
    void (*sigchld_handler) (int);

    record_data = record;

    pipes_0[0] = -1;
    pipes_0[1] = -1;
    pipes_1[0] = -1;
    pipes_1[1] = -1;

    fflush(stdout);

    fsync(STDOUT_FILENO);

    SIGCHLD_received = 0;

    sigchld_handler = signal(SIGCHLD, SIGCHLD_handler);

    do {
	pid_t wait;

	if (ever_stream(pipes_0) || ever_stream(pipes_1)) {
	    status = PIPE_ERROR;
	    break;
	}

	wait = fork();
	if (wait) {
	    if (wait < (pid_t) 0) {
		status = FORK_ERROR;
		break;
	    }
	} else {
	    do {
		const char *const *argv, *null_argv[2];

		if (dup2(pipes_0[0], STDIN_FILENO) == -1) {
#if 0
		    status = IDUP_ERROR;
#endif				/* 0 */
		    break;
		}
		if (dup2(pipes_1[1], STDOUT_FILENO) == -1) {
#if 0
		    status = ODUP_ERROR;
#endif				/* 0 */
		    break;
		}

		close(pipes_0[0]);
		close(pipes_0[1]);
		close(pipes_1[0]);
		close(pipes_1[1]);

		{
		    unsigned c3rd;

		    c3rd = record_data->trailer.c3rd;
		    if (c3rd != MISS_C3RD) {
			if (c3rd == KILL_C3RD) {
			    close(STDERR_FILENO);
			} else {
			    int fd;

			    fd = open("/dev/null", O_WRONLY);
			    if (fd != -1) {
				if (dup2(fd, STDERR_FILENO) == -1) {
#if 0
				    status = EDUP_ERROR;
#endif				/* 0 */
				    break;
				} else {
				    close(fd);
				}
			    }
			}
		    }
		}

		if (record_data->cleanup.call) {
		    if (record_data->cleanup.call(record_data->cleanup.back)) {
#if 0
			status = FLAT_ERROR;
#endif				/* 0 */
			break;
		    }
		}

		argv = record_data->command.argv;
		if (argv) {
		} else {
		    argv = null_argv;
		    null_argv[0] = record_data->command.path;
		    null_argv[1] = NULL;
		}

		execvp(record_data->command.path, (void *) argv);

		exec_sshell(record_data);
	    } while (0);

	    _exit(127);
	}

	close(pipes_0[0]);
	pipes_0[0] = -1;
	close(pipes_1[1]);
	pipes_1[1] = -1;

	record_data->context.chld = sigchld_handler;
	record_data->context.fd_0 = pipes_1[0];
	record_data->context.wait = wait;

	{
	    int flags;

	    flags = fcntl(pipes_1[0], F_GETFL);
	    if (flags == -1) {
		status = NEAR_ERROR;
	    } else {
		status = fcntl(pipes_1[0], F_SETFL, flags | O_NONBLOCK);
		if (status) {
		    status = LOCK_ERROR;
		} else {
		    status = read_sshell(record, data, size);
		}
	    }
	}
	if (status) {
	    wait = record_data->context.wait;
	    if (wait != -1) {
		int Status;

		Status = kill(SIGKILL, wait);
		if (Status) {
		} else {
		    Status = waitpid(wait, NULL, 0);
		}
	    }
	}
    } while (0);

    signal(SIGCHLD, sigchld_handler);

    if (SIGCHLD_received) {
	if (sigchld_handler) {
	    sigchld_handler(SIGCHLD);
	}
    }

    if (1) {
	if (pipes_0[0] != -1) {
	    close(pipes_0[0]);
	}
	if (pipes_0[1] != -1) {
	    close(pipes_0[1]);
	}
	if (pipes_1[0] != -1) {
	    close(pipes_1[0]);
	}
	if (pipes_1[1] != -1) {
	    close(pipes_1[1]);
	}
    }

    return status;
}


static int
read_sshell(void *record, char **data, unsigned *size)
{
    char buffer[BUFFER], *class;
    int done_reading = 0, fd_0, (*line) (void *, const void *, unsigned),
	status;
#if defined HAVE_PSELECT
    sigset_t setc, setv;
#endif				/* HAVE_PSELECT */
    struct record_type *record_data;
    unsigned count, te1st, te6th;
    void *text;

    record_data = record;

    class = record_data->storage.data;
    count = record_data->storage.size;

    te1st = record_data->timeout.e1st;
    te6th = record_data->timeout.e6th;
    if (~te6th) {
	if (~te1st) {
	} else {
	    te1st = 0;
	}
	if (te6th < 1000000) {
	} else {
	    te1st += te6th / 1000000;
	    te6th %= 1000000;
	}
    }
    if (~te1st) {
	struct timeval tv;

	gettimeofday(&tv, NULL);
	te1st += SECOND_A(&tv);
	te6th += SECOND_F(&tv);
	if (te6th < 1000000) {
	} else {
	    te1st++;
	    te6th -= 1000000;
	}
    }

    fd_0 = record_data->context.fd_0;

    line = record_data->channel.line;
    text = record_data->channel.text;

#if defined HAVE_PSELECT
    sigemptyset(&setc);
    sigaddset(&setc, SIGCHLD);
#endif				/* HAVE_PSELECT */

#if defined HAVE_PSELECT
    if (sigprocmask(SIG_BLOCK, &setc, &setv)) {
	status = MASK_ERROR;
    } else {
#endif				/* HAVE_PSELECT */
	while (1) {
	    fd_set fd_set_0, *fd_set_0_select;
#if defined HAVE_PSELECT
	    struct timespec *selector, timeout;
#else
	    struct timeval *selector, timeout;
#endif				/* HAVE_PSELECT */
	    unsigned nfds;

	    if (SIGCHLD_received) {
		status = wait_sshell(record, done_reading);
		if (status) {
		    if (status == POST_ERROR) {
			status = 0;
		    }

		    break;
		}
	    }

	    if (done_reading) {
		fd_set_0_select = NULL;

		nfds = 0;
	    } else {
		fd_set_0_select = &fd_set_0;
		FD_ZERO(fd_set_0_select);
		FD_SET(fd_0, fd_set_0_select);

		nfds = fd_0 + 1;
	    }

	    if (~te1st) {
#if defined HAVE_PSELECT
		struct timeval now;
#endif				/* HAVE_PSELECT */
		unsigned second_a;

		selector = &timeout;
#if defined HAVE_PSELECT
		gettimeofday(&now, NULL);
#else
		gettimeofday(selector, NULL);
#endif				/* HAVE_PSELECT */
#if defined HAVE_PSELECT
		second_a = SECOND_A(&now);
#else
		second_a = SECOND_A(selector);
#endif				/* HAVE_PSELECT */
		if (te1st < second_a) {
		    status = CELL_ERROR;
		    break;
		} else {
		    unsigned second_f, to6th;

		    second_a = te1st - second_a;
#if defined HAVE_PSELECT
		    second_f = SECOND_F(&now);
#else
		    second_f = SECOND_F(selector);
#endif				/* HAVE_PSELECT */
		    to6th = te6th;
		    if (second_a) {
			if (to6th < second_f) {
			    second_a++;
			    to6th += 1000000;
			}
		    } else {
			if (to6th < second_f) {
			    status = CELL_ERROR;
			    break;
			}
		    }

		    SECOND_A(selector) = second_a;
#if defined HAVE_PSELECT
		    selector->tv_nsec = (to6th - second_f) * 1000;
#else
		    SECOND_F(selector) = to6th - second_f;
#endif				/* HAVE_PSELECT */
		}
	    } else {
		/*
		 * one race condition here: the CHLD signal may arrive before
		 * the select gets a chance to observe it.  will not let select
		 * to wait indefinitely.
		 * I expect a small likelyhood that the race will turn
		 * unfavorable and that the extra sleep will not be taken
		 * often.
		 * 
		 * The precaution should not be required for pselect(1).  But
		 * with many pselect(1)s implemented by C libraries over
		 * select(1), one never knows.
		 */
		if (done_reading) {
		    selector = &timeout;

		    SECOND_A(selector) = 0;
#if defined HAVE_PSELECT
		    selector->tv_nsec = 67108864;
#else
		    SECOND_F(selector) = 65536;
#endif				/* HAVE_PSELECT */
		} else {
		    selector = NULL;
		}
	    }

#if defined HAVE_PSELECT
	    status = pselect
		(nfds, fd_set_0_select, NULL, NULL, selector, &setv);
#else
	    status = select(nfds, fd_set_0_select, NULL, NULL, selector);
#endif				/* HAVE_PSELECT */
	    if (status == -1) {
		if (errno != EINTR) {
		    status = POLL_ERROR;
		    break;
		}
	    } else {
		if (fd_set_0_select) {
		    if (FD_ISSET(fd_0, fd_set_0_select)) {
			unsigned Buffer;

			Buffer = BUFFER;
			if (line) {
			} else {
			    if (count < Buffer) {
				if (count) {
				    Buffer = count;
				} else {
				    Buffer = 1;
				}
			    }
			}

			status = read(fd_0, buffer, Buffer);
			if (!status) {
			    if (line) {
			    } else {
				*data = record_data->storage.data;
				*size = record_data->storage.size - count;
			    }
			    if (record_data->context.wait == -1) {
				break;
			    } else {
				done_reading = 1;
			    }
			} else {
			    if (status == -1) {
				status = READ_ERROR;
				break;
			    } else {
				if (line) {
				    status = line(text, buffer, status);
				    if (status) {
					status = LINE_ERROR;
					break;
				    }
				} else {
				    if (count) {
					memcpy(class, buffer, status);
					count -= status;
					class += status;
				    } else {
					status = FLOW_ERROR;
					break;
				    }
				}
			    }
			}
		    }
		}
	    }
	}

#if defined HAVE_PSELECT
	if (sigprocmask(SIG_SETMASK, &setv, NULL)) {
	    if (status) {
	    } else {
		status = MASK_ERROR;
	    }
	}
    }
#endif				/* HAVE_PSELECT */

    return status;
}


static int
wait_sshell(void *record, int done_reading)
{
    int status = 0;
    pid_t wait;
    struct record_type *record_data;

    record_data = record;

    wait = record_data->context.wait;
    if (wait == -1) {
	status = 0;
    } else {
	int effect;

	status = waitpid(wait, &effect, WNOHANG);
	if (status == -1) {
	    status = WAIT_ERROR;
	} else {
	    if (status != wait) {
		status = 0;
		SIGCHLD_received = 0;
		if (record_data->context.chld) {
		    record_data->context.chld(SIGCHLD);
		}
	    } else {
		status = 0;
		if (WIFEXITED(effect)) {
		    if (WEXITSTATUS(effect)) {
			record_data->address.collect.exit.status = effect;
			status = EXIT_ERROR;
		    }
		} else {
		    if (WIFSIGNALED(effect)) {
			record_data->address.collect.exit.status = effect;
			status = EXIT_ERROR;
		    }
		}
		record_data->context.wait = -1;
		if (status) {
		} else {
		    if (done_reading) {
			status = POST_ERROR;
		    } else {
			status = 0;
		    }
		}
	    }
	}
    }

    return status;
}


static void
SIGCHLD_handler(int signal)
{
    SIGCHLD_received = 1;
}


int
x1f4_fini_sshell(void **address)
{
    struct record_type *record_data;

    record_data = *address;

    free(record_data);

    return 0;
}


int
x1f4_init_sshell(void **address)
{
    int status;
    struct record_type *record_data;

    record_data = (struct record_type *) malloc(sizeof(struct record_type));
    if (!record_data) {
	status = -1;
    } else {
	status = 0;

	record_data->channel.line = NULL;
	record_data->channel.text = NULL;
	record_data->command.argv = NULL;
	record_data->command.path = NULL;
	record_data->cleanup.back = NULL;
	record_data->cleanup.call = NULL;
	record_data->printer.head = NULL;
	record_data->printer.look = NULL;
	record_data->printer.tail = NULL;
	record_data->printer.type = NULL;
	record_data->storage.data = NULL;
	record_data->storage.size = SIZE;
	record_data->timeout.e1st = E1ST;
	record_data->timeout.e6th = E6TH;
	record_data->trailer.c3rd = C3RD;
	record_data->trailer.mask = NONE;

	*address = record_data;
    }

    return status;
}


int
x1f4_link_sshell(void *record, char **data, unsigned *size)
{
    int status;
    struct record_type *record_data;

    record_data = record;

    if (!record_data->command.path) {
	status = 0;
    } else {
	status = link_sshell(record, data, size);
    }

    return status;
}


int
x1f4_mode_sshell(void *record, struct sshell_type *sshell_data, unsigned mask)
{
    struct record_type *record_data;

    record_data = record;

    if (mask & ARGV_MASK) {
	record_data->command.argv = sshell_data->command.argv;
    }
    if (mask & BACK_MASK) {
	record_data->cleanup.back = sshell_data->cleanup.back;
    }
    if (mask & C3RD_MASK) {
	record_data->trailer.c3rd = sshell_data->trailer.c3rd;
    }
    if (mask & CALL_MASK) {
	record_data->cleanup.call = sshell_data->cleanup.call;
    }
    if (mask & DATA_MASK) {
	record_data->storage.data = sshell_data->storage.data;
    }
    if (mask & E1ST_MASK) {
	record_data->timeout.e1st = sshell_data->timeout.e1st;
    }
    if (mask & E6TH_MASK) {
	record_data->timeout.e6th = sshell_data->timeout.e6th;
    }
    if (mask & HEAD_MASK) {
	record_data->printer.head = sshell_data->printer.head;
    }
    if (mask & LINE_MASK) {
	record_data->channel.line = sshell_data->channel.line;
    }
    if (mask & LOOK_MASK) {
	record_data->printer.look = sshell_data->printer.look;
    }
    if (mask & MASK_MASK) {
	record_data->trailer.mask = sshell_data->trailer.mask;
    }
    if (mask & PATH_MASK) {
	record_data->command.path = sshell_data->command.path;
    }
    if (mask & SIZE_MASK) {
	record_data->storage.size = sshell_data->storage.size;
    }
    if (mask & TAIL_MASK) {
	record_data->printer.tail = sshell_data->printer.tail;
    }
    if (mask & TEXT_MASK) {
	record_data->channel.text = sshell_data->channel.text;
    }
    if (mask & TYPE_MASK) {
	record_data->printer.type = sshell_data->printer.type;
    }

    return 0;
}
