/*
 * qsce.a.c
 * Copyright (C) 2013, 2014, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stddef.h>
#include <string.h>

#include <qscc-defs.h>
#include <qscc-inter.h>
#include <qscc-names.h>
#include <qscc-types.h>

#define M_TEXT(___v, ___t) \
    ((void *) ((char *) (___v) + (___t)))

#define qsrate(qsrate)			((struct qsrate_type *) (qsrate))

static int call_node(struct qsrate_type *, unsigned, unsigned *,
		     struct fpnode_type *, struct fpnode_type **, unsigned *,
		     unsigned *, unsigned);
static int dash_node(struct qsrate_type *, unsigned, unsigned, unsigned,
		     unsigned, unsigned, unsigned, void **, unsigned *, int *,
		     unsigned *);
static int fail_node(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned, unsigned,
		     unsigned);
static int fail_rate(struct qsrate_type *, unsigned, unsigned,
		     struct fpnode_type *);
static int file_node(struct qsrate_type *, unsigned, void **, unsigned *,
		     unsigned);
static int flat_node(struct fpnode_type *, unsigned, unsigned);
static int fold_dash(struct fpnode_type *, unsigned, unsigned, unsigned);
static int join_ever(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned);
static int join_lead(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned);
static int join_miss(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned);
static int last_call(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned, unsigned,
		     unsigned, unsigned *);
static int land_node(struct qsrate_type *, unsigned, unsigned *,
		     struct fpnode_type *, struct fpnode_type *);
static int last_node(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned, unsigned,
		     unsigned, unsigned, unsigned *, unsigned *);
static int last_slip(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned, unsigned,
		     unsigned, unsigned, unsigned *);
static int lock_head(struct fpnode_type **, struct fpnode_type **, unsigned *,
		     unsigned *, unsigned, unsigned, unsigned, void *);
static int lock_tail(struct fpnode_type **, struct fpnode_type **, unsigned *,
		     unsigned *, unsigned, unsigned, unsigned, void *);
static int long_node(struct qsrate_type *, unsigned, unsigned *,
		     struct fpnode_type *, struct fpnode_type **, unsigned,
		     unsigned, unsigned *);
static int miss_rate(struct qsrate_type *);
static int part_node(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned *);
static int post_call(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned, unsigned,
		     unsigned, unsigned *);
static int post_node(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned, unsigned,
		     unsigned, unsigned, unsigned *);
static int post_slip(struct qsrate_type *, unsigned, struct fpnode_type *,
		     unsigned, unsigned, unsigned, unsigned, unsigned,
		     unsigned, unsigned, unsigned *);
static int rule_node(struct fpnode_type *, unsigned, unsigned *);
static int trap_dash(struct qsrate_type *, unsigned, unsigned,
		     struct fpnode_type *, unsigned, unsigned *);
static int zero_dash(struct fpnode_type *, unsigned, unsigned);

static int
call_node(struct qsrate_type *qsrate_data, unsigned lock, unsigned *near,
	  struct fpnode_type *fpnode_line, struct fpnode_type **fpnode,
	  unsigned *fast, unsigned *tile, unsigned count)
{
    int status;
    struct fpnode_type *fpnode_data, *fpnode_last, *fpnode_post;
    unsigned call, fail, last, line, post;

    line = 1 << lock;

    fpnode_data = *fpnode;

    call = fpnode_data - fpnode_line;

    fpnode_last = fpnode_data;

    fpnode_post = fpnode_data;

    last = call;
    while (last) {
	last--;
	fpnode_last--;
	if (fpnode_last->node) {
	    break;
	}
    }

    post = line - 1 - call;
    while (post) {
	post--;
	fpnode_post++;
	if (fpnode_post->node) {
	    break;
	}
    }

    fail = fpnode_data->node;

    fpnode_data->node = 0;

    last = fpnode_last->node;

    post = fpnode_post->node;

    if (post < last) {
	unsigned seek;

	seek = last + fail;
	if (seek < line) {
	    unsigned none;

	    status = last_node
		(qsrate_data, lock, fpnode_line, fpnode_last - fpnode_line,
		 last, call, fail, *fast, *tile, count, &seek, &none);
	    if (status) {
	    } else {
		*fpnode = fpnode_last;

		*fast -= none;

		fail = *near;
		fail--;
		*near = fail;
		if (fail == 1 && near == &qsrate_data->link_a.fplist.node) {
		} else {
		    last = fpnode_last - fpnode_line;

		    last++;
		    while (last < line) {
			(fpnode_line + last)->call += seek;
			last += last & ~(last - 1);
		    }

		    call++;
		    do {
			(fpnode_line + call)->call -= seek;
			call += call & ~(call - 1);
		    } while (call < line);
		}
	    }
	} else {
	    status = last_slip
		(qsrate_data, lock, fpnode_line, fpnode_last - fpnode_line,
		 last, call, fail, *fast, *tile, count, &seek);
	    if (status) {
	    } else {
		*fast -= seek;

		if (0) {
		} else {
		    last = fpnode_last - fpnode_line;

		    last++;
		    while (last < line) {
			(fpnode_line + last)->call -= seek;
			last += last & ~(last - 1);
		    }

		    call++;
		    do {
			(fpnode_line + call)->call += seek;
			call += call & ~(call - 1);
		    } while (call < line);
		}
	    }
	}
    } else {
	unsigned seek;

	seek = fail + post;
	if (seek < line) {
	    status = post_node
		(qsrate_data, lock, fpnode_line, call, fail,
		 fpnode_post - fpnode_line, post, *fast, *tile, count, &seek);
	    if (status) {
	    } else {
		*tile += seek;

		fail = *near;
		fail--;
		*near = fail;
		if (fail == 1 && near == &qsrate_data->link_a.fplist.node) {
		} else {
		    post = fpnode_post - fpnode_line;

		    post++;
		    while (post < line) {
			(fpnode_line + post)->call -= seek;
			post += post & ~(post - 1);
		    }

		    call++;
		    do {
			(fpnode_line + call)->call += seek;
			call += call & ~(call - 1);
		    } while (call < line);
		}
	    }
	} else {
	    status = post_slip
		(qsrate_data, lock, fpnode_line, call, fail,
		 fpnode_post - fpnode_line, post, *fast, *tile, count, &seek);
	    if (status) {
	    } else {
		*tile += seek;

		if (0) {
		} else {
		    post = fpnode_post - fpnode_line;

		    post++;
		    while (post < line) {
			(fpnode_line + post)->call -= seek;
			post += post & ~(post - 1);
		    }

		    call++;
		    do {
			(fpnode_line + call)->call += seek;
			call += call & ~(call - 1);
		    } while (call < line);
		}
	    }
	}
    }

    return status;
}


static int
dash_node(struct qsrate_type *qsrate_data, unsigned lock, unsigned rate,
	  unsigned mind, unsigned head, unsigned tail, unsigned slip,
	  void **star, unsigned *near, int *able, unsigned *part)
{
    int status;
    void *node;

    node = *star;

    if (rate) {
	struct fpnode_type *fpnode_ever, *fpnode_head, *fpnode_lead,
	    *fpnode_tail;
	unsigned count, fast, fold, left, rail, sort, tile;

	count = slip - mind;

	head -= mind;
	tail -= mind;

	if (head) {
	    lock_head
		(&fpnode_lead, &fpnode_head, &fast, &left, count, head,
		 lock, node);
	} else {
	    fpnode_lead = node;
	    fpnode_head = node;
	    fast = 0;
	    left = 0;
	}
	if (1) {
	    lock_tail
		(&fpnode_ever, &fpnode_tail, &tile, &rail, count, tail - 1,
		 lock, node);
	} else {
	    rail = count;
	    tile = count;
	    fpnode_tail = (struct fpnode_type *) node + (1 << lock);
	    fpnode_ever = fpnode_tail;
	}

	rate--;

	do {
	    int push = 0, slip;
	    unsigned ever = 0, lead = 0;

	    slip = fpnode_tail - fpnode_head;
	    if (fpnode_lead != fpnode_ever) {
		if (left < rail) {
		    status = trap_dash
			(qsrate_data, lock, rate, fpnode_head, slip, near);
		    if (status) {
			break;
		    }
		}
		if (head != left) {
		    status = dash_node
			(qsrate_data, lock, rate, fast, head, left, left,
			 &fpnode_lead->star, &fpnode_lead->node, &push, &lead);
		    if (status) {
			break;
		    }
		}
		if (tail != rail) {
		    int pull = 0;

		    status = dash_node
			(qsrate_data, lock, rate, rail, rail, tail, tile,
			 &fpnode_ever->star, &fpnode_ever->node, &pull, &ever);
		    if (status) {
			break;
		    }

		    if (pull) {
			push |= pull << 2;
		    }
		}
	    } else {
		status = dash_node
		    (qsrate_data, lock, rate, fast, head, tail, tile,
		     &fpnode_lead->star, &fpnode_lead->node, &push, &lead);
	    }

	    fold_dash(node, lock, head, tail);

	    status = 0;

	    count += head;
	    count -= tail;

	    zero_dash(node, lock, count);

	    switch (push) {
	    case 1:
		if (*near ^ 1) {
		    status = land_node
			(qsrate_data, lock, near, node, fpnode_lead);
		} else {
		    *part = 1;
		}
		break;
	    case 2:
		if (*near ^ 1) {
		    fold = left == head
			? tile + head - tail
			: left < tail ? head : left + head - tail;
		    status = call_node
			(qsrate_data, lock, near, node, &fpnode_lead, &fast,
			 &fold, count);
		    if (status) {
		    } else {
			if (lead) {
			    status = join_lead
				(qsrate_data, lock, fpnode_lead, lead, rate,
				 fold - fast);
			    if (status) {
			    } else {
				if (fpnode_lead->node << 1 < 1 << lock) {
				    status = call_node
					(qsrate_data, lock, near, node,
					 &fpnode_lead, &fast, &fold, count);
				}
			    }
			}
		    }
		} else {
		    *part = 1 + lead;
		}
		break;
	    case 4:
		if (*near ^ 1) {
		    status = land_node
			(qsrate_data, lock, near, node, fpnode_ever);
		} else {
		    *part = 1;
		}
		break;
	    case 5:
		if (fast) {
		    status = land_node
			(qsrate_data, lock, near, node, fpnode_lead);
		    if (status) {
		    } else {
			status = land_node
			    (qsrate_data, lock, near, node, fpnode_ever);
		    }
		} else {
		    if (*near ^ 2) {
			status = land_node
			    (qsrate_data, lock, near, node, fpnode_ever);
			if (status) {
			} else {
			    status = land_node
				(qsrate_data, lock, near, node, fpnode_lead);
			}
		    } else {
			status = land_node
			    (qsrate_data, lock, near, node, fpnode_ever);
			if (status) {
			} else {
			    if (count << 1 < qsrate_data->link_a.fplist.line) {
				*part = 1;
			    }
			}
		    }
		}
		break;
	    case 8:
		if (*near ^ 1) {
		    sort = tile + head - tail;
		    status = call_node
			(qsrate_data, lock, near, node, &fpnode_ever, &head,
			 &sort, count);
		    if (status) {
		    } else {
			if (ever) {
			    status = join_ever
				(qsrate_data, lock, fpnode_ever, ever, rate,
				 sort - head);
			    if (status) {
			    } else {
				if (fpnode_ever->node << 1 < 1 << lock) {
				    status = call_node
					(qsrate_data, lock, near, node,
					 &fpnode_ever, &head, &sort, count);
				}
			    }
			}
		    }
		} else {
		    *part = 1 + ever;
		}
		break;
	    case 10:
		if (fast) {
		    status = call_node
			(qsrate_data, lock, near, node, &fpnode_lead, &fast,
			 &head, count);
		    if (status) {
		    } else {
			if (lead) {
			    status = join_lead
				(qsrate_data, lock, fpnode_lead, lead, rate,
				 head - fast);
			    if (status) {
			    } else {
				if (fpnode_lead->node << 1 < 1 << lock) {
				    status = call_node
					(qsrate_data, lock, near, node,
					 &fpnode_lead, &fast, &head, count);
				    if (status) {
					break;
				    }
				}
			    }
			}
			sort = tile + head - tail;
			status = call_node
			    (qsrate_data, lock, near, node, &fpnode_ever,
			     &head, &sort, count);
			if (status) {
			} else {
			    if (ever) {
				status = join_ever
				    (qsrate_data, lock, fpnode_ever, ever,
				     rate, sort - head);
				if (status) {
				} else {
				    if (fpnode_ever->node << 1 < 1 << lock) {
					status = call_node
					    (qsrate_data, lock, near, node,
					     &fpnode_ever, &head, &sort,
					     count);
				    }
				}
			    }
			}
		    }
		} else {
		    if (*near ^ 2) {
			sort = tile + head - tail;
			status = call_node
			    (qsrate_data, lock, near, node, &fpnode_ever,
			     &head, &sort, count);
			if (status) {
			} else {
			    if (ever) {
				status = join_ever
				    (qsrate_data, lock, fpnode_ever, ever,
				     rate, sort - head);
				if (status) {
				} else {
				    if (fpnode_ever->node << 1 < 1 << lock) {
					status = call_node
					    (qsrate_data, lock, near, node,
					     &fpnode_ever, &head, &sort,
					     count);
					if (status) {
					    break;
					}
				    }
				}
			    }
			    status = call_node
				(qsrate_data, lock, near, node, &fpnode_lead,
				 &fast, &head, count);
			    if (status) {
			    } else {
				if (lead) {
				    status = join_lead
					(qsrate_data, lock, fpnode_lead, lead,
					 rate, head - fast);
				    if (status) {
				    } else {
					if (fpnode_lead->node << 1
					    < 1 << lock) {
					    status = call_node
						(qsrate_data, lock, near, node,
						 &fpnode_lead, &fast, &head,
						 count);
					}
				    }
				}
			    }
			}
		    } else {
			status = call_node
			    (qsrate_data, lock, near, node, &fpnode_lead,
			     &fast, &head, count);
			if (status) {
			} else {
			    if (ever && lead) {
				status = join_miss
				    (qsrate_data, lock, fpnode_lead, lead,
				     ever, rate, count);
			    } else {
				if (lead) {
				    status = join_lead
					(qsrate_data, lock, fpnode_lead, lead,
					 rate, head - fast);
				}
				if (ever) {
				    status = join_ever
					(qsrate_data, lock, fpnode_lead, ever,
					 rate, head - fast);
				}
			    }
			    if (fpnode_lead->node << 1 < 1 << lock) {
				*part = 1;
				if (lead | ever) {
				    if (near
					!= &qsrate_data->link_a.fplist.node) {
					part_node
					    (qsrate_data, lock, fpnode_lead,
					     rate, part);
				    }
				}
			    }
			}
		    }
		}
	    }

	    if (status) {
		break;
	    }

	    if (*near << 1 < 1 << lock) {
		*able = 2;
	    } else {
		if (near != &qsrate_data->link_a.fplist.node) {
		    status = file_node(qsrate_data, lock, star, near, count);
		}
	    }
	} while (0);
    } else {
	unsigned miss;

	miss = slip - tail;

	status = 0;

	*near = slip + head - mind - tail;

	if (miss) {
	    unsigned mall;

	    mall = qsrate_data->link_a.fplist.mall;
	    node = M_TEXT(node, (head - mind) * mall);
	    memmove(node, M_TEXT(node, (tail - head) * mall), miss * mall);
	}

	if (*near < qsrate_data->link_a.fplist.line >> 1) {
	    *able = 1;
	}
    }

    return status;
}


static int
fail_node(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_line, unsigned call, unsigned lane,
	  unsigned sail, unsigned post, unsigned meta, unsigned only)
{
    int status;
    struct fpnode_type *fpnode_data, *fpnode_post, *fpnode_text;
    unsigned down, i;

    fpnode_data = fpnode_line + call;
    fpnode_post = fpnode_line + post;

    fpnode_data->node = lane + meta;
    fpnode_post->node = 0;

    fpnode_data = fpnode_data->star;
    fpnode_post = fpnode_post->star;

    flat_node(fpnode_data, lock, sail);
    flat_node(fpnode_post, lock, only);

    down = lane + meta;

    fpnode_text = (void *) qsrate_data->link_e.fplink_data;
    if (fpnode_text) {
	struct fplink_type *fplink_data;

	fplink_data = ((struct fplink_type *) fpnode_text)->fplink_data;

	if (down << 1 < 1 << lock) {
	    i = down;
	    for (; i; i--) {
		if (lane) {
		    while (!fpnode_data->node) {
			fpnode_data++;
		    }
		    {
			*fpnode_text++ = *fpnode_data++;
			lane--;
		    }
		} else {
		    while (!fpnode_post->node) {
			fpnode_post++;
		    }
		    {
			*fpnode_text++ = *fpnode_post++;
			meta--;
		    }
		}

		fpnode_text->call = 0;
		fpnode_text->node = 0;
		fpnode_text++;
	    }

	    i = (1 << lock) - (down << 1);
	    for (; i; i--) {
		fpnode_text->call = 0;
		fpnode_text->node = 0;
		fpnode_text++;
	    }
	} else {
	    i = (down << 1) - (1 << lock);
	    for (; i; i--) {
		if (lane) {
		    while (!fpnode_data->node) {
			fpnode_data++;
		    }
		    {
			*fpnode_text++ = *fpnode_data++;
			lane--;
		    }
		} else {
		    while (!fpnode_post->node) {
			fpnode_post++;
		    }
		    {
			*fpnode_text++ = *fpnode_post++;
			meta--;
		    }
		}
	    }

	    i = (1 << lock) - down;
	    for (; i; i--) {
		if (lane) {
		    while (!fpnode_data->node) {
			fpnode_data++;
		    }
		    {
			*fpnode_text++ = *fpnode_data++;
			lane--;
		    }
		} else {
		    while (!fpnode_post->node) {
			fpnode_post++;
		    }
		    {
			*fpnode_text++ = *fpnode_post++;
			meta--;
		    }
		}

		fpnode_text->call = 0;
		fpnode_text->node = 0;
		fpnode_text++;
	    }
	}

	fpnode_text -= 1 << lock;

	rule_node(fpnode_text, lock, &i);

	status = qsrate_data->link_m.free
	    (qsrate_data->link_m.data, (fpnode_line + post)->star);
	if (status) {
	    status = FREE_ERROR;
	} else {
	    struct fplink_type *fplink_text;

	    fplink_text = (fpnode_line + call)->star;

	    qsrate_data->link_e.fplink_data = fplink_text;

	    fplink_text->fplink_data = fplink_data;

	    (fpnode_line + call)->star = fpnode_text;
	}
    } else {
	status = LINK_ERROR;
    }

    return status;
}


static int
fail_rate(struct qsrate_type *qsrate_data, unsigned lock, unsigned rate,
	  struct fpnode_type *fpnode_data)
{
    int status;
    unsigned node;
    void *star;

    for (;;) {
	node = fpnode_data->node;
	if (node) {
	    break;
	}

	fpnode_data++;
    }

    star = fpnode_data->star;

    status = qsrate_data->link_m.free
	(qsrate_data->link_m.data, qsrate_data->link_a.fplist.fpnode);
    if (status) {
	status = FREE_ERROR;
    } else {
	if (rate) {
	    qsrate_data->link_a.fplist.node = node;

	    qsrate_data->link_a.fplist.rate = rate;

	    qsrate_data->link_a.fplist.fpnode = star;

	    if (node == 1) {
		status = fail_rate(qsrate_data, lock, rate - 1, star);
	    }
	} else {
	    status = qsrate_data->link_m.mode
		(qsrate_data->link_m.data, &star,
		 qsrate_data->link_a.fplist.line * 3
		 * qsrate_data->link_a.fplist.mall >> 1);
	    if (status) {
		status = MODE_ERROR;
	    }

	    qsrate_data->link_a.fplist.rate = 0;

	    qsrate_data->link_a.fplist.fpnode = star;

	    star = qsrate_data->link_e.fplink_text;
	    if (star) {
		int excess;

		excess = qsrate_data->link_m.free
		    (qsrate_data->link_m.data, star);
		if (excess) {
		    if (status) {
		    } else {
			status = FREE_ERROR;
		    }
		} else {
		    qsrate_data->link_e.fplink_text = NULL;
		}
	    }
	}

	if (1) {
	    struct fplink_type *fplink_data;

	    fplink_data = (void *) qsrate_data->link_e.fplink_data;
	    if (fplink_data) {
		int excess;

		qsrate_data->link_e.fplink_data = fplink_data->fplink_data;

		excess = qsrate_data->link_m.free
		    (qsrate_data->link_m.data, fplink_data);
		if (excess) {
		    qsrate_data->link_e.fplink_data = fplink_data;
		    if (status) {
		    } else {
			status = FREE_ERROR;
		    }
		}
	    }
	}
    }

    return status;
}


static int
file_node(struct qsrate_type *qsrate_data, unsigned lock, void **star,
	  unsigned *near, unsigned park)
{
    int status;
    struct fpnode_type *fpnode_data, *fpnode_text;

    fpnode_data = *star;

    flat_node(fpnode_data, lock, park);

    fpnode_text = (void *) qsrate_data->link_e.fplink_data;
    if (fpnode_text) {
	struct fplink_type *fplink_data;
	unsigned down, i;

	fplink_data = ((struct fplink_type *) fpnode_text)->fplink_data;

	down = *near;

	i = (down << 1) - (1 << lock);
	for (; i; i--) {
	    if (1) {
		while (!fpnode_data->node) {
		    fpnode_data++;
		}
		{
		    *fpnode_text++ = *fpnode_data++;
		}
	    }
	}

	i = (1 << lock) - down;
	for (; i; i--) {
	    if (1) {
		while (!fpnode_data->node) {
		    fpnode_data++;
		}
		{
		    *fpnode_text++ = *fpnode_data++;
		}
	    }

	    fpnode_text->call = 0;
	    fpnode_text->node = 0;
	    fpnode_text++;
	}

	fpnode_text -= 1 << lock;

	{
	    struct fplink_type *fplink_text;

	    fplink_text = *star;

	    qsrate_data->link_e.fplink_data = fplink_text;

	    fplink_text->fplink_data = fplink_data;

	    *star = fpnode_text;
	}

	rule_node(fpnode_text, lock, &i);

	status = 0;
    } else {
	status = LINK_ERROR;
    }

    return status;
}


static int
flat_node(struct fpnode_type *fpnode_data, unsigned lock, unsigned park)
{
    if (lock) {
	struct fpnode_type *fpnode_miss;
	unsigned near;

	lock--;
	fpnode_miss = fpnode_data + (1 << lock);
	near = fpnode_miss->call;
	if (near) {
	    flat_node(fpnode_data, lock, near);
	    park -= near;
	}
	if (park) {
	    flat_node(fpnode_miss, lock, park);
	}
    } else {
	fpnode_data->call = park;
    }

    return 0;
}


static int
fold_dash(struct fpnode_type *fpnode_data, unsigned lock, unsigned head,
	  unsigned tail)
{
    unsigned *cell, half, slip;

    slip = tail - head;

    half = lock;

    tail--;

    cell = &fpnode_data->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (tail < news) {
	    *call = news - slip;
	} else {
	    if (head < news) {
		unsigned ever, fast, *rate;

		ever = news - head;

		slip = slip - ever;

		tail -= news;

		*call = news - ever;

		fast = half;

		while (fast) {
		    fast--;

		    rate = (void *)
			((char *) cell + (sizeof(struct fpnode_type) << fast));

		    news = *rate;
		    if (head < news) {
			ever = news - head;

			*rate = news - ever;
		    } else {
			head -= news;
			cell = rate;
		    }
		}

		fast = half;

		while (fast) {
		    fast--;

		    rate = (void *)
			((char *) call + (sizeof(struct fpnode_type) << fast));

		    news = *rate;
		    if (tail < news) {
			*rate = news - slip;
		    } else {
			*rate = 0;
			slip -= news;
			tail -= news;
			call = rate;
		    }
		}

		break;
	    } else {
		head -= news;
		tail -= news;
		cell = call;
	    }
	}
    }

    return 0;
}


static int
land_node(struct qsrate_type *qsrate_data, unsigned lock, unsigned *near,
	  struct fpnode_type *fpnode_line, struct fpnode_type *fpnode_data)
{
    int status = 0;
    struct fpnode_type *fpnode_last, *fpnode_post;
    unsigned call, fail, last, line, mall, miss, post;
    void *fpnode, *fqnode;

    miss = qsrate_data->link_a.fplist.line;

    line = 1 << lock;

    call = fpnode_data - fpnode_line;

    fpnode_last = fpnode_data;

    fpnode_post = fpnode_data;

    last = call;
    while (last) {
	last--;
	fpnode_last--;
	if (fpnode_last->node) {
	    break;
	}
    }

    post = line - 1 - call;
    while (post) {
	post--;
	fpnode_post++;
	if (fpnode_post->node) {
	    break;
	}
    }

    fail = fpnode_data->node;

    fpnode_data->node = 0;

    last = fpnode_last->node;

    post = fpnode_post->node;

    mall = qsrate_data->link_a.fplist.mall;

    fpnode = fpnode_data->star;

    if (post < last) {
	unsigned seek;

	fqnode = fpnode_last->star;

	seek = last + fail;
	if (seek < miss) {
	    if (0) {
	    } else {
		memcpy(M_TEXT(fqnode, mall * last), fpnode, fail * mall);
	    }

	    fpnode_data->node = 0;
	    fpnode_last->node = seek;

	    last = fpnode_last - fpnode_line;

	    seek = *near;
	    seek--;
	    if (seek == 1 && near == &qsrate_data->link_a.fplist.node) {
		status = _libx1f4l2_bill_qsrate
		    (qsrate_data, &qsrate_data->link_a.fplist.node,
		     fpnode_line, fpnode_last, last, fpnode_data, call);
	    } else {
		*near = seek;

		seek = fail;

		last = fpnode_last - fpnode_line;

		last++;
		do {
		    (fpnode_line + last)->call += seek;
		    last += last & ~(last - 1);
		} while (last < line);

		call++;
		while (call < line) {
		    (fpnode_line + call)->call -= seek;
		    call += call & ~(call - 1);
		}

		status = qsrate_data->link_m.free
		    (qsrate_data->link_m.data, fpnode);
		if (status) {
		    status = FREE_ERROR;
		}
	    }
	} else {
	    seek >>= 1;
	    seek -= fail;

	    if (2) {
		memmove(M_TEXT(fpnode, seek * mall), fpnode, fail * mall);
		last -= seek;
		memcpy(fpnode, M_TEXT(fqnode, last * mall), seek * mall);
	    }

	    fpnode_last->node = last;

	    fpnode_data->node = fail + seek;

	    if (1) {
		last = fpnode_last - fpnode_line;

		last++;
		do {
		    (fpnode_line + last)->call -= seek;
		    last += last & ~(last - 1);
		} while (last < line);

		call++;
		while (call < line) {
		    (fpnode_line + call)->call += seek;
		    call += call & ~(call - 1);
		}
	    }
	}
    } else {
	unsigned seek;

	fqnode = fpnode_post->star;

	seek = fail + post;
	if (seek < miss) {
	    if (0) {
	    } else {
		memcpy(M_TEXT(fpnode, fail * mall), fqnode, post * mall);
	    }

	    fpnode_data->node = seek;
	    fpnode_post->node = 0;

	    seek = *near;
	    seek--;
	    if (seek == 1 && near == &qsrate_data->link_a.fplist.node) {
		status = _libx1f4l2_mail_qsrate
		    (qsrate_data, &qsrate_data->link_a.fplist.node,
		     fpnode_line, fpnode_data, call, fpnode_post);
	    } else {
		*near = seek;

		seek = post;

		post = fpnode_post - fpnode_line;

		post++;
		while (post < line) {
		    (fpnode_line + post)->call -= seek;
		    post += post & ~(post - 1);
		}

		call++;
		do {
		    (fpnode_line + call)->call += seek;
		    call += call & ~(call - 1);
		} while (call < line);

		status = qsrate_data->link_m.free
		    (qsrate_data->link_m.data, fqnode);
		if (status) {
		    status = FREE_ERROR;
		}
	    }
	} else {
	    seek >>= 1;
	    seek -= fail;

	    fpnode = M_TEXT(fpnode, fail * mall);
	    if (2) {
		memcpy(fpnode, fqnode, seek * mall);
		post -= seek;
		memmove(fqnode, M_TEXT(fqnode, seek * mall), post * mall);
	    }

	    fpnode_post->node = post;

	    fpnode_data->node = fail + seek;

	    post = fpnode_post - fpnode_line;

	    if (1) {
		post++;
		while (post < line) {
		    (fpnode_line + post)->call -= seek;
		    post += post & ~(post - 1);
		}

		call++;
		do {
		    (fpnode_line + call)->call += seek;
		    call += call & ~(call - 1);
		} while (call < line);
	    }
	}
    }

    return status;
}


static int
join_ever(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_data, unsigned ever, unsigned rate,
	  unsigned park)
{
    int status;
    struct fpnode_type *fpnode_miss;
    void *node;

    node = fpnode_data->star;

    fpnode_miss = node;

    rate--;
    if (rate) {
	unsigned down, fall, line;

	line = 1 << lock >> 1;
	if (fpnode_data->node < line) {
	    down = 0;

	    fpnode_miss += 1 << lock;
	    do {
		unsigned fast;

		fpnode_miss--;
		fast = fpnode_miss->node;
		if (fast && fast < line) {
		    break;
		}

		down += fast;
	    } while (1);
	    if (down) {
		down = 0;
	    } else {
		down = park - 1;
	    }
	} else {
	    if (fpnode_miss->node < line) {
		down = 0;
	    } else {
		down = park - 1;
		fpnode_miss += (1 << lock) - 2;
	    }
	}

	status = long_node
	    (qsrate_data, lock, &fpnode_data->node, node, &fpnode_miss, down,
	     park, &fall);
	if (status) {
	} else {
	    ever--;
	    if (ever) {
		status = join_ever
		    (qsrate_data, lock, fpnode_miss, ever, rate, fall);
		if (status) {
		} else {
		    if (fpnode_miss->node < line) {
			status = long_node
			    (qsrate_data, lock, &fpnode_data->node, node,
			     &fpnode_miss, down, park, &fall);
		    }
		}
	    }
	}
    } else {
	unsigned line;

	fpnode_miss += 1 << lock;

	line = qsrate_data->link_a.fplist.line >> 1;
	do {
	    unsigned fast;

	    fpnode_miss--;
	    fast = fpnode_miss->node;
	    if (fast && fast < line) {
		break;
	    }
	} while (1);

	status = land_node
	    (qsrate_data, lock, &fpnode_data->node, node, fpnode_miss);
    }

    return status;
}


static int
join_lead(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_data, unsigned lead, unsigned rate,
	  unsigned park)
{
    int status;
    struct fpnode_type *fpnode_miss;
    void *node;

    node = fpnode_data->star;

    fpnode_miss = node;

    rate--;
    if (rate) {
	unsigned down, fall, line;

	line = 1 << lock >> 1;
	if (fpnode_data->node < line) {
	    down = 0;
	    do {
		unsigned fast;

		fast = fpnode_miss->node;
		if (fast && fast < line) {
		    break;
		}

		down += fast;

		fpnode_miss++;
	    } while (1);
	    if (down) {
		down = park - 1;
	    }
	} else {
	    if (fpnode_miss->node < line) {
		down = 0;
	    } else {
		down = park - 1;
		fpnode_miss += (1 << lock) - 2;
	    }
	}

	status = long_node
	    (qsrate_data, lock, &fpnode_data->node, node, &fpnode_miss, down,
	     park, &fall);
	if (status) {
	} else {
	    lead--;
	    if (lead) {
		status = join_lead
		    (qsrate_data, lock, fpnode_miss, lead, rate, fall);
		if (status) {
		} else {
		    if (fpnode_miss->node < line) {
			status = long_node
			    (qsrate_data, lock, &fpnode_data->node, node,
			     &fpnode_miss, down, park, &fall);
		    }
		}
	    }
	}
    } else {
	unsigned line;

	line = qsrate_data->link_a.fplist.line >> 1;
	do {
	    unsigned fast;

	    fast = fpnode_miss->node;
	    if (fast && fast < line) {
		break;
	    }

	    fpnode_miss++;
	} while (1);

	status = land_node
	    (qsrate_data, lock, &fpnode_data->node, node, fpnode_miss);
    }

    return status;
}


static int
join_miss(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_data, unsigned lead, unsigned ever,
	  unsigned rate, unsigned park)
{
    int status;
    struct fpnode_type *fpnode_miss;
    void *node;

    node = fpnode_data->star;

    fpnode_miss = node;
    while (!fpnode_miss->node) {
	fpnode_miss++;
    }

    rate--;
    if (rate) {
	unsigned fall;

	status = long_node
	    (qsrate_data, lock, &fpnode_data->node, node, &fpnode_miss, 0,
	     park, &fall);
	if (status) {
	} else {
	    lead--;
	    ever--;
	    if (ever && lead) {
		status = join_miss
		    (qsrate_data, lock, fpnode_miss, lead, ever, rate, park);
	    } else {
		if (lead) {
		    status = join_lead
			(qsrate_data, lock, fpnode_miss, lead, rate, park);
		} else {
		    if (ever) {
			status = join_ever
			    (qsrate_data, lock, fpnode_miss, ever, rate, park);
		    }
		}
	    }
	}
    } else {
	status = land_node
	    (qsrate_data, lock, &fpnode_data->node, node, fpnode_miss);
    }

    return status;
}


static int
last_call(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_line, unsigned call, unsigned lane,
	  unsigned sail, unsigned post, unsigned meta, unsigned only,
	  unsigned *seek)
{
    int status;
    struct fpnode_type *fpnode_data, *fpnode_post, *fpnode_text;
    unsigned i, part;

    fpnode_data = fpnode_line + call;
    fpnode_post = fpnode_line + post;

    fpnode_data = fpnode_data->star;
    fpnode_post = fpnode_post->star;

    flat_node(fpnode_data, lock, sail);
    flat_node(fpnode_post, lock, only);

    part = lane + meta;

    fpnode_text = (void *) qsrate_data->link_e.fplink_data;
    if (fpnode_text) {
	struct fplink_type *fplink_data;
	unsigned down, land = 0;

	down = part >> 1;

	part -= down;

	lane -= part;

	(fpnode_line + call)->node = part;
	(fpnode_line + post)->node = down;

	fplink_data = ((struct fplink_type *) fpnode_text)->fplink_data;

	fpnode_text += 1 << lock;
	fpnode_data += (1 << lock) - 1;
	fpnode_post += (1 << lock) - 1;

	i = (down << 1) - (1 << lock);
	for (; i; i--) {
	    if (meta) {
		while (!fpnode_post->node) {
		    fpnode_post--;
		}
		{
		    *--fpnode_text = *fpnode_post--;
		    meta--;
		}
	    } else {
		while (!fpnode_data->node) {
		    fpnode_data--;
		}
		{
		    land += fpnode_data->call;
		    *--fpnode_text = *fpnode_data--;
		}
	    }
	}

	i = (1 << lock) - down;
	for (; i; i--) {
	    fpnode_text--;
	    fpnode_text->call = 0;
	    fpnode_text->node = 0;

	    if (meta) {
		while (!fpnode_post->node) {
		    fpnode_post--;
		}
		{
		    *--fpnode_text = *fpnode_post--;
		    meta--;
		}
	    } else {
		while (!fpnode_data->node) {
		    fpnode_data--;
		}
		{
		    land += fpnode_data->call;
		    *--fpnode_text = *fpnode_data--;
		}
	    }
	}

	*seek = land;

	{
	    struct fplink_type *fplink_text;

	    fplink_text = (fpnode_line + post)->star;

	    qsrate_data->link_e.fplink_data = fplink_text;

	    fplink_text->fplink_data = fplink_data;

	    (fpnode_line + post)->star = fpnode_text;
	}

	fpnode_post = (fpnode_line + call)->star;

	fpnode_post += 1 << lock;

	fpnode_data++;

	i = 1 << lock;
	do {
	    fpnode_post--;
	    fpnode_post->call = 0;
	    fpnode_post->node = 0;

	    i -= 2;

	    if (0) {
	    } else {
		do {
		    fpnode_data--;
		} while (!fpnode_data->node);
		{
		    *--fpnode_post = *fpnode_data;
		}
	    }
	} while (fpnode_post != fpnode_data);
	for (; i; i--) {
	    fpnode_post--;
	    if (fpnode_post->node) {
	    } else {
		fpnode_post->call = 0;
	    }
	}

	rule_node(fpnode_text, lock, &i);

	rule_node(fpnode_post, lock, &i);

	status = 0;
    } else {
	status = LINK_ERROR;
    }

    return status;
}


static int
last_node(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_line, unsigned call, unsigned lane,
	  unsigned post, unsigned meta, unsigned fast, unsigned tile,
	  unsigned park, unsigned *seek, unsigned *none)
{
    unsigned *cell, half, only, sail;

    sail = tile - fast;

    only = park;

    half = lock;

    fast--;

    cell = &fpnode_line->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (fast < news) {
	    only = news;
	} else {
	    fast -= news;
	    only -= news;
	    cell = call;
	}
    }

    *none = only;

    *seek = sail;

    return fail_node
	(qsrate_data, lock, fpnode_line, call, lane, only, post, meta, sail);
}


static int
last_slip(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_line, unsigned call, unsigned lane,
	  unsigned post, unsigned meta, unsigned fast, unsigned tile,
	  unsigned park, unsigned *seek)
{
    unsigned *cell, half, only, sail;

    sail = tile - fast;

    only = park;

    half = lock;

    fast--;

    cell = &fpnode_line->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (fast < news) {
	    only = news;
	} else {
	    fast -= news;
	    only -= news;
	    cell = call;
	}
    }

    return last_call
	(qsrate_data, lock, fpnode_line, call, lane, only, post, meta, sail,
	 seek);
}


static int
lock_head(struct fpnode_type **fqnode, struct fpnode_type **fpnode,
	  unsigned *sale, unsigned *call, unsigned park, unsigned dash,
	  unsigned lock, void *node)
{
    struct fpnode_type *fpnode_data;
    unsigned *cell, half, left;

    left = dash;

    half = lock;

    cell = &((struct fpnode_type *) node)->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (dash < news) {
	    park = news;
	} else {
	    dash -= news;
	    park -= news;
	    cell = call;
	}
    }

    fpnode_data = (void *)
	((char *) cell - offsetof(struct fpnode_type, call));

    if (dash) {
	*fqnode = fpnode_data;

	fpnode_data++;
    } else {
	park = 0;

	*fqnode = fpnode_data;
    }

    *fpnode = fpnode_data;

    *sale = left - dash;

    *call = left - dash + park;

    return 0;
}


static int
lock_tail(struct fpnode_type **fqnode, struct fpnode_type **fpnode,
	  unsigned *sale, unsigned *call, unsigned park, unsigned dash,
	  unsigned lock, void *node)
{
    struct fpnode_type *fpnode_data;
    unsigned *cell, half, rail;

    rail = dash;

    half = lock;

    cell = &((struct fpnode_type *) node)->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (dash < news) {
	    park = news;
	} else {
	    dash -= news;
	    park -= news;
	    cell = call;
	}
    }

    fpnode_data = (void *)
	((char *) cell - offsetof(struct fpnode_type, call));

    *fqnode = fpnode_data;

    if (park - dash != 1) {
    } else {
	fpnode_data++;

	rail += park;
	park = 0;
    }

    *call = rail - dash;

    *sale = rail - dash + park;

    *fpnode = fpnode_data;

    return 0;
}


static int
long_node(struct qsrate_type *qsrate_data, unsigned lock, unsigned *near,
	  struct fpnode_type *fpnode_line, struct fpnode_type **fpnode,
	  unsigned down, unsigned park, unsigned *fall)
{
    int status;
    unsigned *cell, fast = 0, half, only, tile;

    only = park;

    half = lock;

    fast = down;

    cell = &fpnode_line->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (down < news) {
	    only = news;
	} else {
	    down -= news;
	    only -= news;
	    cell = call;
	}
    }

    fast -= down;

    tile = fast + only;

    status = call_node
	(qsrate_data, lock, near, fpnode_line, fpnode, &fast, &tile, park);

    *fall = tile - fast;

    return status;
}


static int
miss_rate(struct qsrate_type *qsrate_data)
{
    int status;

    status = qsrate_data->link_m.free
	(qsrate_data->link_m.data, qsrate_data->link_a.fplist.fpnode);
    if (status) {
	status = FREE_ERROR;
    } else {
	void *node;

	node = qsrate_data->link_e.fplink_text;

	qsrate_data->link_e.fplink_text = NULL;

	qsrate_data->link_a.fplist.rate = 0;

	if (node) {
	    status = qsrate_data->link_m.mode
		(qsrate_data->link_m.data, &node,
		 qsrate_data->link_a.fplist.line * 3
		 * qsrate_data->link_a.fplist.mall >> 1);
	    if (status) {
		status = MODE_ERROR;
	    }
	} else {
	    status = qsrate_data->link_m.mode
		(qsrate_data->link_m.data, &node,
		 qsrate_data->link_a.fplist.line * 3
		 * qsrate_data->link_a.fplist.mall >> 1);
	    if (status) {
		status = MODE_ERROR;
	    }
	}

	qsrate_data->link_a.fplist.fpnode = node;

	while (1) {
	    struct fplink_type *fplink_data;

	    fplink_data = (void *) qsrate_data->link_e.fplink_data;
	    if (fplink_data) {
		int excess;

		qsrate_data->link_e.fplink_data = fplink_data->fplink_data;

		excess = qsrate_data->link_m.free
		    (qsrate_data->link_m.data, fplink_data);
		if (excess) {
		    qsrate_data->link_e.fplink_data = fplink_data;
		    if (status) {
		    } else {
			status = FREE_ERROR;
		    }

		    break;
		}
	    } else {
		break;
	    }
	}
    }

    return status;
}


static int
part_node(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_data, unsigned rate, unsigned *part)
{
    struct fpnode_type *fpnode_miss;

    fpnode_miss = fpnode_data->star;

    rate--;
    if (rate) {
	unsigned line;

	line = 1 << lock >> 1;
	for (;;) {
	    unsigned node;

	    node = fpnode_miss->node;
	    if (node) {
		if (node < line) {
		    (*part)++;
		    part_node(qsrate_data, lock, fpnode_miss, rate, part);
		}

		break;
	    }

	    fpnode_miss++;
	}
    } else {
	unsigned line;

	line = qsrate_data->link_a.fplist.line >> 1;
	for (;;) {
	    unsigned node;

	    node = fpnode_miss->node;
	    if (node) {
		if (node < line) {
		    (*part)++;
		}

		break;
	    }

	    fpnode_miss++;
	}
    }

    return 0;
}


static int
post_call(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_line, unsigned call, unsigned lane,
	  unsigned sail, unsigned post, unsigned meta, unsigned only,
	  unsigned *seek)
{
    int status;
    struct fpnode_type *fpnode_data, *fpnode_post, *fpnode_text;
    unsigned i, part;

    fpnode_data = fpnode_line + call;
    fpnode_post = fpnode_line + post;

    fpnode_data = fpnode_data->star;
    fpnode_post = fpnode_post->star;

    flat_node(fpnode_data, lock, sail);
    flat_node(fpnode_post, lock, only);

    part = lane + meta;

    fpnode_text = (void *) qsrate_data->link_e.fplink_data;
    if (fpnode_text) {
	struct fplink_type *fplink_data;
	unsigned down, land = 0;

	down = part >> 1;

	part -= down;

	meta -= part;

	(fpnode_line + call)->node = down;
	(fpnode_line + post)->node = part;

	fplink_data = ((struct fplink_type *) fpnode_text)->fplink_data;

	i = (down << 1) - (1 << lock);
	for (; i; i--) {
	    if (lane) {
		while (!fpnode_data->node) {
		    fpnode_data++;
		}
		{
		    *fpnode_text++ = *fpnode_data++;
		    lane--;
		}
	    } else {
		while (!fpnode_post->node) {
		    fpnode_post++;
		}
		{
		    land += fpnode_post->call;
		    *fpnode_text++ = *fpnode_post++;
		}
	    }
	}

	i = (1 << lock) - down;
	for (; i; i--) {
	    if (lane) {
		while (!fpnode_data->node) {
		    fpnode_data++;
		}
		{
		    *fpnode_text++ = *fpnode_data++;
		    lane--;
		}
	    } else {
		while (!fpnode_post->node) {
		    fpnode_post++;
		}
		{
		    land += fpnode_post->call;
		    *fpnode_text++ = *fpnode_post++;
		}
	    }

	    fpnode_text->call = 0;
	    fpnode_text->node = 0;
	    fpnode_text++;
	}

	*seek = land;

	fpnode_text -= 1 << lock;

	{
	    struct fplink_type *fplink_text;

	    fplink_text = (fpnode_line + call)->star;

	    qsrate_data->link_e.fplink_data = fplink_text;

	    fplink_text->fplink_data = fplink_data;

	    (fpnode_line + call)->star = fpnode_text;
	}

	fpnode_data = (fpnode_line + post)->star;

	{
	    {
		while (!fpnode_post->node) {
		    fpnode_post++;
		}
	    }
	}

	i = 1 << lock;
	while (fpnode_data != fpnode_post) {
	    if (0) {
	    } else {
		{
		    *fpnode_data++ = *fpnode_post++;
		}
		while (!fpnode_post->node) {
		    fpnode_post++;
		}
	    }

	    i -= 2;

	    fpnode_data->call = 0;
	    fpnode_data->node = 0;
	    fpnode_data++;
	}
	for (; i; i--) {
	    fpnode_data++;
	    if (fpnode_data->node) {
	    } else {
		fpnode_data->call = 0;
	    }
	}

	fpnode_data -= 1 << lock;

	rule_node(fpnode_text, lock, &i);

	rule_node(fpnode_data, lock, &i);

	status = 0;
    } else {
	status = LINK_ERROR;
    }

    return status;
}


static int
post_node(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_line, unsigned call, unsigned lane,
	  unsigned post, unsigned meta, unsigned fast, unsigned tile,
	  unsigned park, unsigned *seek)
{
    unsigned *cell, half, only, sail;

    sail = tile - fast;

    only = park;

    half = lock;

    cell = &fpnode_line->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (tile < news) {
	    only = news;
	} else {
	    tile -= news;
	    only -= news;
	    cell = call;
	}
    }

    *seek = only;

    return fail_node
	(qsrate_data, lock, fpnode_line, call, lane, sail, post, meta, only);
}


static int
post_slip(struct qsrate_type *qsrate_data, unsigned lock,
	  struct fpnode_type *fpnode_line, unsigned call, unsigned lane,
	  unsigned post, unsigned meta, unsigned fast, unsigned tile,
	  unsigned park, unsigned *seek)
{
    unsigned *cell, half, only, sail;

    sail = tile - fast;

    only = park;

    half = lock;

    cell = &fpnode_line->call;
    while (half) {
	unsigned *call, news;

	half--;

	call = (void *) ((char *) cell + (sizeof(struct fpnode_type) << half));

	news = *call;
	if (tile < news) {
	    only = news;
	} else {
	    tile -= news;
	    only -= news;
	    cell = call;
	}
    }

    return post_call
	(qsrate_data, lock, fpnode_line, call, lane, sail, post, meta, only,
	 seek);
}


static int
rule_node(struct fpnode_type *fpnode_data, unsigned lock, unsigned *rule)
{
    if (lock) {
	struct fpnode_type *fpnode_miss;
	unsigned call, feat;

	lock--;
	fpnode_miss = fpnode_data + (1 << lock);
	rule_node(fpnode_data, lock, &feat);
	rule_node(fpnode_miss, lock, &call);
	fpnode_miss->call = feat;
	*rule = call + feat;
    } else {
	*rule = fpnode_data->call;
    }

    return 0;
}


static int
trap_dash(struct qsrate_type *qsrate_data, unsigned lock, unsigned rate,
	  struct fpnode_type *fpnode_data, unsigned trap, unsigned *read)
{
    int status = 0;
    unsigned node;

    node = *read;

    for (; trap; trap--) {
	if (fpnode_data->node) {
	    void *star;

	    star = fpnode_data->star;
	    status = _libx1f4l2_park_qsrate
		(qsrate_data, star, rate, 1 << lock);
	    if (status) {
		break;
	    } else {
		node--;
		fpnode_data->node = 0;
	    }
	}

	fpnode_data++;
    }

    *read = node;

    return status;
}


static int
zero_dash(struct fpnode_type *fpnode_data, unsigned lock, unsigned size)
{
    if (lock) {
	lock--;
	if (size) {
	    unsigned call;

	    fpnode_data += 1 << lock;
	    call = fpnode_data->call;
	    zero_dash(fpnode_data - (1 << lock), lock, call);
	    zero_dash(fpnode_data, lock, size - call);
	    fpnode_data->call = call;
	} else {
	    zero_dash(fpnode_data, lock, 0);
	    fpnode_data += 1 << lock;
	    zero_dash(fpnode_data, lock, 0);
	}
    } else {
	fpnode_data->call = size;
    }

    return 0;
}


int
x1f4_dash_qsrate(void *qsrate, unsigned head, unsigned tail)
{
    int status;
    unsigned count;

    count = qsrate(qsrate)->link_a.fplist.size;

    if (head < count && head < tail + 1 && tail < count) {
	int able;
	unsigned lock, part, rate;
	void *node;

	node = qsrate(qsrate)->link_a.fplist.fpnode;

	rate = qsrate(qsrate)->link_a.fplist.rate;

	tail++;

	lock = qsrate(qsrate)->link_a.fplist.lock;

	status = dash_node
	    (qsrate, lock, rate, 0, head, tail, count,
	     &qsrate(qsrate)->link_a.fplist.fpnode,
	     &qsrate(qsrate)->link_a.fplist.node, &able, &part);
	if (status) {
	} else {
	    qsrate(qsrate)->link_a.fplist.size = count - tail + head;
	    if (rate) {
		if (qsrate(qsrate)->link_a.fplist.node < 2) {
		    if (qsrate(qsrate)->link_a.fplist.node) {
			status = fail_rate(qsrate, lock, rate - 1, node);
		    } else {
			status = miss_rate(qsrate);
		    }
		}
	    }
	}
    } else {
	status = DECK_ERROR;
    }

    return status;
}
