/*
 * ix.0.c
 * Copyright (C) 2010-2014, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stddef.h>
#include <string.h>

#include <c1.h>
#include <cv.h>
#include <dt.h>
#include <e2line.h>
#include <e2list.h>
#include <e4-m.0.h>
#include <e4.h>
#include <e42.h>
#include <e4fine.h>
#include <e4less.h>
#include <ix-defs.h>
#include <ix-inter.h>
#include <ix-types.h>
#include <lxdata.h>
#include <lxfile.h>
#include <lxnear.h>
#include <lxnote.h>
#include <lxwide.h>
#include <m.h>
#include <mxdeck.h>
#include <nt.h>
#include <tp.h>

typedef struct effect_type {
    int (*fini) (struct indexset_type *);
} effect_type;

typedef struct vxslip_type {
    const struct x1f4_lxnote_type *lxnote_data;
    void *slip;
} vxslip_type;

typedef struct vxmind_type {
    struct {
	struct vxslip_type *vxslip_data;
	unsigned miss;
    } lead_i;
    struct {
	struct x1f4_lxwide_type lxwide;
	unsigned bits;
    } lead_w;
    struct {
	struct x1f4_datatype_type *datatype_data;
	unsigned miss;
	void *data;
    } link_t;
    struct {
	int (*free) (void *, void *), (*link) (void *, void **, unsigned),
	    (*mode) (void *, void **, unsigned);
	void *data;
    } link_w;
} vxmind_type;

typedef struct window_type {
    int (*fini) (struct indexset_type *, struct vxmind_type *),
	(*init) (struct indexset_type *, struct vxmind_type *, unsigned,
		 struct x1f4_textport_type *);
} window_type;

extern const struct x1f4_function_type *const x1f4_e4_computables,
    x1f4_e4_defaults[];

static int call_dtlist(struct indexset_type *);
static int call_e4fine(struct indexset_type *);
static int call_e4hash(struct indexset_type *);
static int call_edlist(struct indexset_type *);
static int call_etlist(struct indexset_type *);
static int call_fsdeck(struct indexset_type *);
static int call_lxwide(struct indexset_type *);
static int call_o1list(struct indexset_type *);
static int call_o2list(struct indexset_type *);
static int case_window(struct indexset_type *, struct vxmind_type *);
static int cfix_lxwide(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int cfix_vxslip(struct vxmind_type *, void *, unsigned,
		       int (*) (void *, unsigned, void *),
		       const struct x1f4_lxtile_type *);
static int copy_e4hash(struct indexset_type *, struct vxmind_type *,
		       struct x1f4_textport_type *);
static int copy_vxmind(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int dfix_lxnear(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int fast_fllist(struct indexset_type *, struct vxmind_type *,
		       unsigned, struct x1f4_textport_type *);
static int ffix_lxnear(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int fini_dtlist(struct indexset_type *, struct vxmind_type *);
static int fini_e4fine(struct indexset_type *, struct vxmind_type *);
static int fini_e4hash(struct indexset_type *, struct vxmind_type *);
static int fini_edlist(struct indexset_type *, struct vxmind_type *);
static int fini_etlist(struct indexset_type *, struct vxmind_type *);
static int fini_fsdeck(struct indexset_type *, struct vxmind_type *);
static int fini_lxwide(struct indexset_type *, struct vxmind_type *);
static int fini_o1list(struct indexset_type *, struct vxmind_type *);
static int fini_o2list(struct indexset_type *, struct vxmind_type *);
static int flat_fllist(struct indexset_type *, struct vxmind_type *);
static int init_dtlist(struct indexset_type *, struct vxmind_type *,
		       unsigned, struct x1f4_textport_type *);
static int init_e4fine(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_e4hash(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_e4less(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_edlist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_eflist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_etlist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_fsdeck(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_lxwide(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_o1list(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int init_o2list(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int link_dtlist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int link_etlist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int lose_fllist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int miss_fllist(struct indexset_type *);
static int rule_fllist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int rule_fsdeck(struct indexset_type *,
		       const struct x1f4_linetext_type *);
static int rule_jsdeck(struct indexset_type *,
		       const struct x1f4_function_type *);
static int rule_lookup(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int rule_vxtext(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int seek_dtlist(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int tile_lxwide(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int type_fsdeck(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);
static int type_lxwide(struct indexset_type *, struct vxmind_type *, unsigned,
		       struct x1f4_textport_type *);

static const struct effect_type violet[] = {
/* *INDENT-OFF* */
    {	call_fsdeck						},
    {	call_e4hash						},
    {	miss_fllist						},
    {	call_dtlist						},
    {	call_e4fine						},
    {	call_lxwide						},
    {	call_edlist						},
    {	call_o1list						},
    {	call_o2list						},
    {	call_etlist						}
/* *INDENT-ON* */
};
static const struct window_type riddle[] = {
/* *INDENT-OFF* */
    {	case_window,		copy_vxmind			},
    {	fini_fsdeck,		init_fsdeck			},
    {	fini_e4hash,		init_e4hash			},
    {	flat_fllist,		fast_fllist			},
    {	case_window,		rule_fllist			},
    {	case_window,		seek_dtlist			},
    {	fini_dtlist,		init_dtlist			},
    {	case_window,		link_dtlist			},
    {	case_window,		ffix_lxnear			},
    {	fini_e4fine,		init_e4fine			},
    {	case_window,		rule_vxtext			},
    {	case_window,		rule_lookup			},
    {	case_window,		init_lxwide			},
    {	fini_lxwide,		tile_lxwide			},
    {	case_window,		dfix_lxnear			},
    {	case_window,		cfix_lxwide			},
    {	fini_etlist,		init_etlist			},
    {	case_window,		link_etlist			},
    {	case_window,		type_lxwide			},
    {	fini_edlist,		init_edlist			},
    {	fini_o1list,		init_o1list			},
    {	fini_o2list,		init_o2list			},
    {	case_window,		init_eflist			},
    {	case_window,		type_fsdeck			},
    {	case_window,		lose_fllist			}
/* *INDENT-ON* */
};
static const struct x1f4_nodetype_type list_types[] = {
/* *INDENT-OFF* */
    {	"cardinal",	NULL,		X1f4_E4_BILL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,
	0,		8,			NULL		},
    {	"integer",	NULL,		X1f4_E4_MODE,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,
	0,		7,			NULL		},
    {	"real",		NULL,		X1f4_E4_REAL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,
	0,		4,			NULL		},
    {	"text",		NULL,		X1f4_E4_TEXT,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,		NULL,			NULL,
	NULL,
	0,		4,			NULL		}
/* *INDENT-ON* */
};

static int
call_dtlist(struct indexset_type *indexset_data)
{
    return indexset_data->codelink_set.free
	(indexset_data->codelink_set.data,
	 indexset_data->nodetype_set.nodetype_data);
}


static int
call_e4fine(struct indexset_type *indexset_data)
{
    return x1f4_fini_e4fine(&indexset_data->transfer_set.fine);
}


static int
call_e4hash(struct indexset_type *indexset_data)
{
    int status;
    void *less;

    less = indexset_data->autodeck_set.less;
    if (less) {
	status = x1f4_fini_e4ll(&less);
    } else {
	status = 0;
    }

    return status;
}


static int
call_edlist(struct indexset_type *indexset_data)
{
    void *text;

    text = indexset_data->variable_set.text;
    if (text) {
	x1f4_air_state(text);
    }

    return 0;
}


static int
call_etlist(struct indexset_type *indexset_data)
{
    return indexset_data->codelink_set.free
	(indexset_data->codelink_set.data,
	 indexset_data->datatype_set.datatype_data);
}


static int
call_fsdeck(struct indexset_type *indexset_data)
{
    return x1f4_fini_mxdeck(&indexset_data->function_set.text);
}


static int
call_lxwide(struct indexset_type *indexset_data)
{
    int status = 0;
    struct x1f4_lxslip_type *lxslip_data;
    unsigned miss;

    lxslip_data = indexset_data->sliplong_set.lxslip_data;

    miss = indexset_data->sliplong_set.miss;

    lxslip_data += miss;

    for (; miss; miss--) {
	int excess, (*note) (const struct x1f4_lxnote_type **);
	const struct x1f4_lxnote_type *lxnote_data;

	lxslip_data--;

	note = lxslip_data->note;

	note(&lxnote_data);

	excess = lxnote_data->fini(&lxslip_data->slip);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}


static int
call_o1list(struct indexset_type *indexset_data)
{
    int status;
    void *operator1s;

    operator1s = (void *) indexset_data->operator_set.operator1s;
    if (operator1s) {
	const struct x1f4_operator_type *const *operator0s;

	x1f4_llink_operator1s(&operator0s);
	if (operator1s != operator0s) {
	    status = indexset_data->codelink_set.free
		(indexset_data->codelink_set.data, operator1s);
	} else {
	    status = 0;
	}
    } else {
	status = 0;
    }

    return status;
}


static int
call_o2list(struct indexset_type *indexset_data)
{
    int status;
    void *operator2s;

    operator2s = (void *) indexset_data->operator_set.operator2s;
    if (operator2s) {
	const struct x1f4_operator_type *const *operator0s;

	x1f4_llink_operator2s(&operator0s);
	if (operator2s != operator0s) {
	    x1f4_ilink_operator2s(&operator0s);
	    if (operator2s != operator0s) {
		status = indexset_data->codelink_set.free
		    (indexset_data->codelink_set.data, operator2s);
	    } else {
		status = 0;
	    }
	} else {
	    status = 0;
	}
    } else {
	status = 0;
    }

    return status;
}


static int
case_window(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return 0;
}


static int
cfix_lxwide(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    struct vxslip_type *vxslip_data;
    unsigned j, miss;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    if (textport_bits & X1f4_TEXTPORT_NODELIST) {
	j = textport_data->nodetype_set.miss;
    } else {
	j = 0;
    }

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*cfix) (void *, unsigned, void *);

	cfix = vxslip_data->lxnote_data->cfix;
	if (cfix) {
	    cfix_vxslip
		(vxmind_data, vxslip_data->slip, j, cfix,
		 textport_data->tilelong_set.lxtile_data);
	}

	vxslip_data++;
    }

    return 0;
}


static int
cfix_vxslip(struct vxmind_type *vxmind_data, void *note, unsigned j,
	    int (*cfix) (void *, unsigned, void *),
	    const struct x1f4_lxtile_type *lxtile_data)
{
    struct vxslip_type *vxslip_data;
    unsigned miss;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*seek) (const void *);

	seek = vxslip_data->lxnote_data->seek;
	if (seek) {
	    unsigned count;
	    void *slip;

	    slip = vxslip_data->slip;

	    count = seek(lxtile_data->data);
	    for (; count; count--) {
		cfix(note, LIST_TYPES + j, slip);
		j++;
	    }
	}

	vxslip_data++;

	lxtile_data++;
    }

    return 0;
}


static int
copy_e4hash(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data,
	    struct x1f4_textport_type *textport_data)
{
    indexset_data->autodeck_set.less = NULL;

    vxmind_data->lead_w.lxwide.autolink_set.free =
	textport_data->autolink_set.free;
    vxmind_data->lead_w.lxwide.autolink_set.link =
	textport_data->autolink_set.link;
    vxmind_data->lead_w.lxwide.autolink_set.mode =
	textport_data->autolink_set.mode;
    vxmind_data->lead_w.lxwide.autolink_set.pick =
	textport_data->autolink_set.pick;
    vxmind_data->lead_w.lxwide.autolink_set.slip =
	textport_data->autolink_set.slip;

    vxmind_data->lead_w.lxwide.autolink_set.data =
	textport_data->autolink_set.data;

    indexset_data->autolink_set.free = textport_data->autolink_set.free;
    indexset_data->autolink_set.link = textport_data->autolink_set.link;
    indexset_data->autolink_set.mode = textport_data->autolink_set.mode;
    indexset_data->autolink_set.pick = textport_data->autolink_set.pick;
    indexset_data->autolink_set.slip = textport_data->autolink_set.slip;

    indexset_data->autolink_set.data = textport_data->autolink_set.data;

    return 0;
}


static int
copy_vxmind(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    indexset_data->codelink_set.data = vxmind_data->link_w.data;
    indexset_data->codelink_set.free = vxmind_data->link_w.free;
    indexset_data->codelink_set.link = vxmind_data->link_w.link;
    indexset_data->codelink_set.mode = vxmind_data->link_w.mode;

    return 0;
}


static int
dfix_lxnear(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    struct vxslip_type *vxslip_data;
    const struct x1f4_lxnear_type *lxnear_data = NULL;
    const struct x1f4_lxtile_type *lxtile_data;
    unsigned miss;
    void *lxdata = NULL, *lxfile = NULL, *lxnear = NULL;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    lxtile_data = textport_data->tilelong_set.lxtile_data;

    miss = textport_data->tilelong_set.miss;
    for (; miss; miss--) {
	int (*note)(const struct x1f4_lxnote_type **);

	note = lxtile_data->note;
	if (note == x1f4_note_lxdata) {
	    lxdata = vxslip_data->slip;
	} else {
	    if (note == x1f4_note_lxfile) {
		lxfile = vxslip_data->slip;
	    } else {
		if (note == x1f4_note_lxnear) {
		    if (lxtile_data->bits & X1f4_LXNEAR_LONGPIPE) {
			lxnear = vxslip_data->slip;
			lxnear_data = lxtile_data->data;
		    }
		}
	    }
	}

	lxtile_data++;

	vxslip_data++;
    }

    if (lxnear) {
	struct x1f4_datapipe_type *datapipe_type;

	datapipe_type = (struct x1f4_datapipe_type *)
	    lxnear_data->longpipe_set.datapipe_data;
	miss = lxnear_data->longpipe_set.miss;
	for (; miss; miss--) {
	    int (*copy) (void *, const char *, unsigned);

	    copy = datapipe_type->copy;
	    if (copy == x1f4_pipe_lxdata) {
		if (lxdata) {
		    x1f4_dfix_lxnear
			(lxnear, lxnear_data->longpipe_set.miss - miss,
			 lxdata);
		}
	    } else {
		if (copy == x1f4_pipe_lxfile) {
		    if (lxfile) {
			x1f4_dfix_lxnear
			    (lxnear, lxnear_data->longpipe_set.miss - miss,
			     lxfile);
		    }
		}
	    }

	    datapipe_type++;
	}
    }

    return 0;
}


static int
fast_fllist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;
    unsigned miss;
    void *lxslip, *vxslip;

    miss = textport_data->tilelong_set.miss;

    status = vxmind_data->link_w.link
	(vxmind_data->link_w.data, &lxslip,
	 sizeof(struct x1f4_lxslip_type) * miss);
    if (status) {
    } else {
	status = vxmind_data->link_w.link
	    (vxmind_data->link_w.data, &vxslip,
	     sizeof(struct vxslip_type) * miss);
	if (status) {
	    vxmind_data->link_w.free(vxmind_data->link_w.data, lxslip);
	} else {
	    indexset_data->sliplong_set.lxslip_data = lxslip;
	    indexset_data->sliplong_set.miss = miss;
	    vxmind_data->lead_i.miss = miss;
	    vxmind_data->lead_i.vxslip_data = vxslip;
	}
    }

    return status;
}


static int
ffix_lxnear(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int data = -1, file = -1, type;
    struct vxslip_type *vxslip_data;
    const struct x1f4_lxnear_type *lxnear_data = NULL;
    const struct x1f4_lxtile_type *lxtile_data;
    unsigned miss;

    type = textport_data->lasttype_set.type + 1;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    lxtile_data = textport_data->tilelong_set.lxtile_data;

    miss = textport_data->tilelong_set.miss;
    for (; miss; miss--) {
	int (*note)(const struct x1f4_lxnote_type **), (*seek)(const void *);

	note = lxtile_data->note;
	if (note == x1f4_note_lxdata) {
	    data = type;
	} else {
	    if (note == x1f4_note_lxfile) {
		file = type;
	    } else {
		if (note == x1f4_note_lxnear) {
		    if (lxtile_data->bits & X1f4_LXNEAR_LONGPIPE) {
			lxnear_data = lxtile_data->data;
		    }
		}
	    }
	}

	seek = vxslip_data->lxnote_data->seek;
	if (seek) {
	    type += seek(lxtile_data->data);
	}

	lxtile_data++;

	vxslip_data++;
    }

    if (lxnear_data != NULL) {
	struct x1f4_datapipe_type *datapipe_type;

	datapipe_type = (struct x1f4_datapipe_type *)
	    lxnear_data->longpipe_set.datapipe_data;
	miss = lxnear_data->longpipe_set.miss;
	for (; miss; miss--) {
	    int (*copy) (void *, const char *, unsigned);

	    copy = datapipe_type->copy;
	    if (copy == x1f4_pipe_lxdata) {
		if (data ^ -1) {
		    datapipe_type->type = data;
		}
	    } else {
		if (copy == x1f4_pipe_lxfile) {
		    if (file ^ -1) {
			datapipe_type->type = file;
		    }
		}
	    }

	    datapipe_type++;
	}
    }

    return 0;
}


static int
fini_dtlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_dtlist(indexset_data);
}


static int
fini_e4fine(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_e4fine(indexset_data);
}


static int
fini_e4hash(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_e4hash(indexset_data);
}


static int
fini_edlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_edlist(indexset_data);
}


static int
fini_etlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_etlist(indexset_data);
}


static int
fini_o1list(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_o1list(indexset_data);
}


static int
fini_o2list(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_o2list(indexset_data);
}


static int
fini_fsdeck(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    return call_fsdeck(indexset_data);
}


static int
fini_lxwide(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    int status = 0;
    struct vxslip_type *vxslip_data;
    unsigned miss;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    miss = vxmind_data->lead_i.miss;

    vxslip_data += miss;

    for (; miss; miss--) {
	int excess;

	vxslip_data--;
	excess = vxslip_data->lxnote_data->fini(&vxslip_data->slip);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}
    }

    return status;
}


static int
flat_fllist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data)
{
    int status;

    status = miss_fllist(indexset_data);

    if (vxmind_data->lead_i.vxslip_data) {
	int excess;

	excess = vxmind_data->link_w.free
	    (vxmind_data->link_w.data, vxmind_data->lead_i.vxslip_data);
	if (excess) {
	} else {
	    status = excess;
	}
    }

    return status;
}


static int
init_dtlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;
    unsigned miss;
    void *data;

    miss = vxmind_data->link_t.miss + LIST_TYPES;
    if (textport_bits & X1f4_TEXTPORT_NODELIST) {
	miss += textport_data->nodetype_set.miss;
    }

    status = vxmind_data->link_w.link
	(vxmind_data->link_w.data, &data,
	 sizeof(struct x1f4_nodetype_type) * miss);
    if (status) {
    } else {
	indexset_data->nodetype_set.nodetype_data = data;
	indexset_data->nodetype_set.miss = miss;
	vxmind_data->lead_w.lxwide.datatype_set.nodetype_data = data;
	vxmind_data->lead_w.lxwide.datatype_set.miss = miss;
	vxmind_data->link_t.data = data;
    }

    return status;
}


static int
init_e4fine(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    return x1f4_init_e4fine(&indexset_data->transfer_set.fine, 0, NULL);
}


static int
init_e4hash(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;

    if (textport_bits & X1f4_TEXTPORT_AUTOLINK) {
	status = copy_e4hash(indexset_data, vxmind_data, textport_data);
    } else {
	status = init_e4less
	    (indexset_data, vxmind_data, textport_bits, textport_data);
    }

    return status;
}


static int
init_e4less(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;
    struct x1f4_e2line_type *e2line_data;
    struct x1f4_e4less_type e4line;
    void *e4less;

    e2line_data = &indexset_data->autodeck_set.e2line;

    e2line_data->bits = X1f4_E2LIST_FLATLINE;

    e2line_data->e2list.flatline_set.class = textport_data->autodeck_set.class;

    if (textport_bits & X1f4_TEXTPORT_RESOURCE) {
	e2line_data->bits |= X1f4_E2LIST_MISSLINK;
	e2line_data->e2list.misslink_set.data =
	    textport_data->resource_set.data;
	e2line_data->e2list.misslink_set.free =
	    textport_data->resource_set.free;
	e2line_data->e2list.misslink_set.link =
	    textport_data->resource_set.link;
	e2line_data->e2list.misslink_set.mode =
	    textport_data->resource_set.mode;
    }

    e4line.misslink_set.data = e2line_data;

    x1f4_type_e2list(&e4line.misslink_set);

    status = x1f4_init_e4ll(&e4less, X1f4_E4LL_MISSLINK, &e4line);
    if (status) {
    } else {
	struct x1f4_frame_type frame;

	indexset_data->autodeck_set.less = e4less;

	x1f4_deck_e4ll
	    (e4less, &indexset_data->autodeck_set.text,
	     &indexset_data->autodeck_set.deck);

	x1f4_copy_e4ll(indexset_data->autodeck_set.less, &frame);

	indexset_data->autolink_set.free = frame.free;
	indexset_data->autolink_set.link = frame.link;
	indexset_data->autolink_set.mode = frame.mode;
	indexset_data->autolink_set.pick = frame.pick;
	indexset_data->autolink_set.slip = frame.slip;

	indexset_data->autolink_set.data = frame.data;

	vxmind_data->lead_w.lxwide.autolink_set.free = frame.free;
	vxmind_data->lead_w.lxwide.autolink_set.link = frame.link;
	vxmind_data->lead_w.lxwide.autolink_set.mode = frame.mode;
	vxmind_data->lead_w.lxwide.autolink_set.pick = frame.pick;
	vxmind_data->lead_w.lxwide.autolink_set.slip = frame.slip;

	vxmind_data->lead_w.lxwide.autolink_set.data = frame.data;
    }

    return status;
}


static int
init_edlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status = 0;
    struct vxslip_type *vxslip_data;
    unsigned miss;
    void *text = NULL;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*vset) (void *, void **);

	vset = vxslip_data->lxnote_data->vset;
	if (vset) {
	    status = vset(vxslip_data->slip, &text);
	    if (status) {
		break;
	    }
	}

	vxslip_data++;
    }

    indexset_data->variable_set.text = text;

    return 0;
}


static int
init_eflist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;
    void *text;

    text = indexset_data->variable_set.text;

    status = x1f4_push_intrinsics(&text);
    if (status) {
    } else {
	status = x1f4_push_typelist(&text, vxmind_data->link_t.datatype_data);
    }

    indexset_data->variable_set.text = text;

    return status;
}


static int
init_etlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;
    unsigned f;
    void *data;

    if (textport_bits & X1f4_TEXTPORT_TYPELIST) {
	const struct x1f4_datatype_type *datatype_data, *datatype_link;

	datatype_link = textport_data->datatype_set.miss;

	datatype_data = datatype_link;
	while (datatype_data->name) {
	    datatype_data++;
	}

	f = datatype_data - datatype_link;
    } else {
	f = 0;
    }

    status = vxmind_data->link_w.link
	(vxmind_data->link_w.data, &data,
	 sizeof(struct x1f4_datatype_type)
	 * (vxmind_data->link_t.miss + f + 1));
    if (status) {
    } else {
	indexset_data->datatype_set.datatype_data = data;

	if (textport_bits & X1f4_TEXTPORT_TYPELIST) {
	    f *= sizeof(struct x1f4_datatype_type);

	    memcpy(data, textport_data->datatype_set.miss, f);

	    data = (char *) data + f;
	}

	vxmind_data->link_t.datatype_data = data;
    }

    return status;
}


static int
init_fsdeck(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    return x1f4_init_mxdeck(&indexset_data->function_set.text, 0, NULL);
}


static int
init_lxwide(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    struct x1f4_lxwide_type *lxwide_data;
    unsigned bits = 0;

    lxwide_data = &vxmind_data->lead_w.lxwide;

    if (textport_bits & X1f4_TEXTPORT_CASTTYPE) {
	bits |= X1f4_LXWIDE_CASTTYPE;
    }

    if (textport_bits & X1f4_TEXTPORT_CODELINK) {
	bits |= X1f4_LXWIDE_CODELINK;
	lxwide_data->codelink_set.data = textport_data->codelink_set.data;
	lxwide_data->codelink_set.free = textport_data->codelink_set.free;
	lxwide_data->codelink_set.link = textport_data->codelink_set.link;
	lxwide_data->codelink_set.mode = textport_data->codelink_set.mode;
    }

    if (1) {
	lxwide_data->eelookup_set.eelookup_data =
	    &indexset_data->eelookup_set.eelookup;
    }

    if (textport_bits & X1f4_TEXTPORT_MISSBAIL) {
	bits |= X1f4_LXWIDE_MISSBAIL;
	lxwide_data->missbail_set.call = textport_data->missbail_set.call;
	lxwide_data->missbail_set.fine = textport_data->missbail_set.fine;
	lxwide_data->missbail_set.miss = textport_data->missbail_set.miss;
	lxwide_data->missbail_set.text = textport_data->missbail_set.text;
    }

    if (textport_bits & X1f4_TEXTPORT_RESOURCE) {
	bits |= X1f4_LXWIDE_RESOURCE;
	lxwide_data->resource_set.data = textport_data->resource_set.data;
	lxwide_data->resource_set.free = textport_data->resource_set.free;
	lxwide_data->resource_set.link = textport_data->resource_set.link;
	lxwide_data->resource_set.mode = textport_data->resource_set.mode;
    }

    if (textport_bits & X1f4_TEXTPORT_TEXTFLAT) {
	bits |= X1f4_LXWIDE_TEXTFLAT;
	lxwide_data->textflat_set.data = textport_data->textflat_set.data;
	lxwide_data->textflat_set.line = textport_data->textflat_set.line;
	lxwide_data->textflat_set.post = textport_data->textflat_set.post;
	lxwide_data->textflat_set.push = textport_data->textflat_set.push;
    }

    if (1) {
	lxwide_data->transfer_set.fine = indexset_data->transfer_set.fine;
    }

    if (1) {
	lxwide_data->widetype_set.type = textport_data->lasttype_set.type;
    }

    vxmind_data->lead_w.bits = bits;

    return 0;
}


static int
init_o1list(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status = 0;
    const struct x1f4_operator_type *const *operator1s,
	*const *const *operator1s_set;
    unsigned count = 0;

    if (textport_bits & X1f4_TEXTPORT_RUN1LIST) {
	operator1s_set = textport_data->operator_set.operator1s_set;
	if (operator1s_set) {
	    operator1s = *operator1s_set;
	    while (operator1s) {
		for (; *operator1s; count++, operator1s++);
		operator1s_set++;
		operator1s = *operator1s_set;
	    }
	}
    } else {
	operator1s_set = NULL;
    }

    if (count) {
	unsigned trans = 0;
	void *list;

	x1f4_llink_operator1s(&operator1s);
	for (; *operator1s; trans++, operator1s++);

	status = vxmind_data->link_w.link
	    (vxmind_data->link_w.data, &list,
	     sizeof(struct x1f4_operator_type *) * (trans + count + 1));
	if (status) {
	} else {
	    const struct x1f4_operator_type **operator0s;
	    unsigned shift;

	    indexset_data->operator_set.operator1s = list;

	    shift = trans * sizeof(struct x1f4_operator_type *);

	    memcpy(list, operator1s - trans, shift);

	    operator0s = (void *) ((char *) list + shift);

	    if (operator1s_set) {
		operator1s_set = textport_data->operator_set.operator1s_set;

		operator1s = *operator1s_set;
		while (operator1s) {
		    while (*operator1s) {
			*operator0s++ = *operator1s++;
		    }
		    operator1s_set++;
		    operator1s = *operator1s_set;
		}
	    }

	    *operator0s = NULL;
	}
    } else {
	x1f4_llink_operator1s(&indexset_data->operator_set.operator1s);
    }

    return status;
}


static int
init_o2list(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status = 0;
    struct vxslip_type *vxslip_data;
    const struct x1f4_operator_type *const *operator2s,
	*const *const *operator2s_set;
    unsigned class = 0, count = 0, miss;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    if (textport_bits & X1f4_TEXTPORT_RUN2LIST) {
	operator2s_set = textport_data->operator_set.operator2s_set;
	if (operator2s_set) {
	    operator2s = *operator2s_set;
	    while (operator2s) {
		for (; *operator2s; count++, operator2s++);
		operator2s_set++;
		operator2s = *operator2s_set;
	    }
	}
    } else {
	operator2s_set = NULL;
    }

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*near) (void *);
	const struct x1f4_lxnote_type *lxnote_data;

	lxnote_data = vxslip_data->lxnote_data;

	near = lxnote_data->near;
	if (near) {
	    class += near(vxslip_data->slip);
	}

	vxslip_data++;
    }

    if (class | count) {
	unsigned trans = 0;
	void *list;

	if (textport_bits & X1f4_TEXTPORT_SUB2LIST) {
	    x1f4_ilink_operator2s(&operator2s);
	} else {
	    x1f4_llink_operator2s(&operator2s);
	}

	for (; *operator2s; trans++, operator2s++);

	status = vxmind_data->link_w.link
	    (vxmind_data->link_w.data, &list,
	     sizeof(struct x1f4_operator_type *)
	     * (trans + class + count + 1));
	if (status) {
	} else {
	    const struct x1f4_operator_type **operator0s;
	    unsigned shift;

	    indexset_data->operator_set.operator2s = list;

	    shift = trans * sizeof(struct x1f4_operator_type *);

	    memcpy(list, operator2s - trans, shift);

	    operator0s = (void *) ((char *) list + shift);

	    vxslip_data = vxmind_data->lead_i.vxslip_data;

	    if (operator2s_set) {
		operator2s_set = textport_data->operator_set.operator2s_set;

		operator2s = *operator2s_set;
		while (operator2s) {
		    while (*operator2s) {
			*operator0s++ = *operator2s++;
		    }
		    operator2s_set++;
		    operator2s = *operator2s_set;
		}
	    }

	    miss = vxmind_data->lead_i.miss;
	    for (; miss; miss--) {
		int (*bset) (void *, const struct x1f4_operator_type **);
		const struct x1f4_lxnote_type *lxnote_data;

		lxnote_data = vxslip_data->lxnote_data;

		bset = lxnote_data->bset;
		if (bset) {
		    operator0s += bset(vxslip_data->slip, operator0s);
		}

		vxslip_data++;
	    }

	    *operator0s = NULL;
	}
    } else {
	if (textport_bits & X1f4_TEXTPORT_SUB2LIST) {
	    x1f4_ilink_operator2s(&indexset_data->operator_set.operator2s);
	} else {
	    x1f4_llink_operator2s(&indexset_data->operator_set.operator2s);
	}
    }

    return status;
}


static int
link_dtlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int type;
    struct vxslip_type *vxslip_data;
    const struct x1f4_lxtile_type *lxtile_data;
    struct x1f4_nodetype_type *nodetype_data;
    unsigned miss;

    nodetype_data = vxmind_data->link_t.data;

    memcpy(nodetype_data, list_types,
	   LIST_TYPES * sizeof(struct x1f4_nodetype_type));

    nodetype_data += LIST_TYPES;

    if (textport_bits & X1f4_TEXTPORT_NODELIST) {
	miss = textport_data->nodetype_set.miss;
	memcpy(nodetype_data, textport_data->nodetype_set.nodetype_data,
	       miss * sizeof(struct x1f4_nodetype_type));
	nodetype_data += miss;
    }

    lxtile_data = textport_data->tilelong_set.lxtile_data;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    type = textport_data->lasttype_set.type + 1;

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*clip) (const void *, struct x1f4_nodetype_type **, int *);

	clip = vxslip_data->lxnote_data->clip;
	if (clip) {
	    clip(lxtile_data->data, &nodetype_data, &type);
	}

	vxslip_data++;

	lxtile_data++;
    }

    return 0;
}


static int
link_etlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    struct vxslip_type *vxslip_data;
    struct x1f4_datatype_type *datatype_slip;
    const struct x1f4_lxtile_type *lxtile_data;
    unsigned miss;

    lxtile_data = textport_data->tilelong_set.lxtile_data;

    datatype_slip = vxmind_data->link_t.datatype_data;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*seek) (const void *);
	const struct x1f4_lxnote_type *lxnote_data;

	lxnote_data = vxslip_data->lxnote_data;

	seek = lxnote_data->seek;
	if (seek) {
	    const struct x1f4_datatype_type *datatype_data;
	    unsigned count;

	    count = seek(lxtile_data->data);

	    lxnote_data->dset(vxslip_data->slip, &datatype_data);

	    for (; count; count--) {
		*datatype_slip++ = *datatype_data++;
	    }
	}

	vxslip_data++;

	lxtile_data++;
    }

    datatype_slip->name = NULL;

    return 0;
}


static int
lose_fllist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status;

    status = vxmind_data->link_w.free
	(vxmind_data->link_w.data, vxmind_data->lead_i.vxslip_data);
    if (status) {
    } else {
	vxmind_data->lead_i.vxslip_data = NULL;
    }

    return status;
}


static int
miss_fllist(struct indexset_type *indexset_data)
{
    return indexset_data->codelink_set.free
	(indexset_data->codelink_set.data,
	 (void *) indexset_data->sliplong_set.lxslip_data);
}


static int
rule_fllist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    struct vxslip_type *vxslip_data;
    struct x1f4_lxslip_type *lxslip_data;
    const struct x1f4_lxtile_type *lxtile_data;
    unsigned miss;

    lxtile_data = textport_data->tilelong_set.lxtile_data;

    lxslip_data = indexset_data->sliplong_set.lxslip_data;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    miss = textport_data->tilelong_set.miss;
    for (; miss; miss--) {
	int (*note) (const struct x1f4_lxnote_type **);
	const struct x1f4_lxnote_type *lxnote_data;

	note = lxtile_data->note;

	lxtile_data++;

	note(&lxnote_data);

	lxslip_data->note = note;

	lxslip_data++;

	vxslip_data->lxnote_data = lxnote_data;

	vxslip_data++;
    }

    return 0;
}


static int
rule_fsdeck(struct indexset_type *indexset_data,
	    const struct x1f4_linetext_type *linetext_data)
{
    const char *name;
    int status;

    name = linetext_data->function.name;
    if (name) {
	void *fsdeck;

	fsdeck = indexset_data->function_set.text;

	do {
	    status = x1f4_post_mxpath
		(fsdeck, name, linetext_data->function.length,
		 &linetext_data->function);
	    if (status) {
		break;
	    } else {
		linetext_data++;
	    }

	    name = linetext_data->function.name;
	} while (name);
    } else {
	status = 0;
    }

    return status;
}


static int
rule_jsdeck(struct indexset_type *indexset_data,
	  const struct x1f4_function_type *function_data)
{
    const char *name;
    int status;

    name = function_data->name;
    if (name) {
	void *fsdeck;

	fsdeck = indexset_data->function_set.text;

	do {
	    status = x1f4_post_mxpath
		(fsdeck, name, function_data->length, function_data);
	    if (status) {
		break;
	    } else {
		function_data++;
	    }

	    name = function_data->name;
	} while (name);
    } else {
	status = 0;
    }

    return status;
}


static int
rule_lookup(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    indexset_data->eelookup_set.eelookup.type_l.context = indexset_data;
    indexset_data->eelookup_set.eelookup.type_l.fix = x1f4_look_indexset;
    indexset_data->eelookup_set.eelookup.type_q.e4fine =
	indexset_data->transfer_set.fine;

    return 0;
}


static int
rule_vxtext(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
#if 0
    const struct x1f4_nodetype_type *nodetype_data;
    unsigned count, i;

    nodetype_data = vxmind_data->link_t.data;

    count = vxmind_data->link_t.miss + LIST_TYPES;
    if (textport_bits & X1f4_TEXTPORT_NODELIST) {
	miss += textport_data->nodetype_set.miss;
    }

    i = count;
    for (; i; i--) {
	if (nodetype_data->code == X1f4_E4_TEXT) {
	    i = count - i;
	    vxmind_data->lead_w.lxwide.textnode_set.node = i;
	    break;
	} else {
	    nodetype_data++;
	}
    }
#else
    vxmind_data->lead_w.lxwide.textnode_set.node = 3;
#endif				/* 0 */

    return 0;
}


static int
seek_dtlist(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    struct vxslip_type *vxslip_data;
    const struct x1f4_lxtile_type *lxtile_data;
    unsigned miss, type_count = 0;

    lxtile_data = textport_data->tilelong_set.lxtile_data;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*seek) (const void *);

	seek = vxslip_data->lxnote_data->seek;
	if (seek) {
	    type_count += seek(lxtile_data->data);
	}

	vxslip_data++;

	lxtile_data++;
    }

    vxmind_data->link_t.miss = type_count;

    return 0;
}


static int
tile_lxwide(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    int status = 0;
    struct vxslip_type *vxslip_data;
    struct x1f4_lxslip_type *lxslip_data;
    const struct x1f4_lxtile_type *lxtile_data;
    struct x1f4_lxwide_type *lxwide_data;
    unsigned bits, miss;

    lxtile_data = textport_data->tilelong_set.lxtile_data;

    lxslip_data = indexset_data->sliplong_set.lxslip_data;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    lxwide_data = &vxmind_data->lead_w.lxwide;

    bits = vxmind_data->lead_w.bits;

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	const struct x1f4_lxnote_type *lxnote_data;
	void *slip;

	lxnote_data = vxslip_data->lxnote_data;

	status = lxnote_data->tile
	    (&slip, bits, lxwide_data, lxtile_data->bits, lxtile_data->data);
	if (status) {
	    break;
	} else {
	    const struct x1f4_linetext_type *linetext_data;

	    lxnote_data->fset(slip, &linetext_data);

	    rule_fsdeck(indexset_data, linetext_data);

	    lxtile_data++;

	    lxslip_data->slip = slip;

	    lxslip_data++;

	    vxslip_data->slip = slip;

	    vxslip_data++;
	}
    }
    if (status) {
	miss = vxmind_data->lead_i.miss - miss;
	for (; miss; miss--) {
	    vxslip_data--;
	    vxslip_data->lxnote_data->fini(&vxslip_data->slip);
	}
    }

    return status;
}


static int
type_fsdeck(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    const struct x1f4_function_type *functions;

    if (textport_bits & X1f4_TEXTPORT_SUBFLIST) {
	functions = x1f4_e4_computables;
    } else {
	functions = x1f4_e4_defaults;
    }

    return rule_jsdeck(indexset_data, functions);
}


static int
type_lxwide(struct indexset_type *indexset_data,
	    struct vxmind_type *vxmind_data, unsigned textport_bits,
	    struct x1f4_textport_type *textport_data)
{
    struct vxslip_type *vxslip_data;
    const struct x1f4_datatype_type *datatype_data;
    unsigned miss;

    datatype_data = vxmind_data->link_t.datatype_data;

    vxslip_data = vxmind_data->lead_i.vxslip_data;

    miss = vxmind_data->lead_i.miss;
    for (; miss; miss--) {
	int (*jack) (void *, const struct x1f4_datatype_type *);

	jack = vxslip_data->lxnote_data->jack;
	if (jack) {
	    jack(vxslip_data->slip, datatype_data);
	}

	vxslip_data++;
    }

    return 0;
}


int
x1f4_fast_indexset(struct indexset_type *indexset_data,
		   unsigned textport_bits,
		   struct x1f4_textport_type *textport_data)
{
    int status;
    struct vxmind_type vxmind;
    static const struct window_type *window_data;
    unsigned access;

    if (textport_bits & X1f4_TEXTPORT_CODELINK) {
	vxmind.link_w.data = textport_data->codelink_set.data;
	vxmind.link_w.free = textport_data->codelink_set.free;
	vxmind.link_w.link = textport_data->codelink_set.link;
	vxmind.link_w.mode = textport_data->codelink_set.mode;
    } else {
	vxmind.link_w.free = _x1f4_e4_free_data;
	vxmind.link_w.link = _x1f4_e4_link_data;
	vxmind.link_w.mode = _x1f4_e4_mode_data;
    }

    access = sizeof(riddle) / sizeof(struct window_type);

    window_data = riddle;

    do {
	status = window_data->init
	    (indexset_data, &vxmind, textport_bits, textport_data);
	if (status) {
	    break;
	} else {
	    window_data++;
	}

	access--;
    } while (access);

    if (status) {
	access = sizeof(riddle) / sizeof(struct window_type) - access;

	for (; access; access--) {
	    window_data--;
	    window_data->fini(indexset_data, &vxmind);
	}
    }

    return status;
}


int
x1f4_flat_indexset(struct indexset_type *indexset_data)
{
    int status = 0;
    static const struct effect_type *effect_data;
    unsigned access;

    access = sizeof(violet) / sizeof(struct effect_type);

    effect_data = violet + access;

    do {
	int excess;

	effect_data--;
	excess = effect_data->fini(indexset_data);
	if (excess) {
	    if (status) {
	    } else {
		status = excess;
	    }
	}

	access--;
    } while (access);

    return status;
}
