/*
 * e4-e.2.c
 * Copyright (C) 2006-2014, Ciprian Niculescu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stddef.h>
#include <string.h>

#include <e4-byte.h>
#include <e4-defs.h>
#include <e4-inter.h>
#include <e4-types.h>

#define EXPRESSION(expression) \
    ((e4_expression_type *) expression)

#define E_LINK(mcontext, mdata, size) \
    (EXPRESSION(mcontext)->m.link					      \
	(EXPRESSION(mcontext)->m.data, (void *) (mdata), (size)))

static int copy_atom(void *, struct e4_atom_type *);
static int copy_case(void *, int, struct e4_base_type *,
		     const struct e4_last_type *, unsigned, unsigned);
static int copy_data(struct e4_expression_type *, struct e4_expression_type *);
static int copy_ever(void *, struct e4_atom_type *);
static int copy_hack(void *, struct e4_hack_type *, struct e4_hack_type *);
static int copy_last(void *, struct e4_atom_type *);
static int copy_list(void *, struct e4_hack_type *, struct e4_hack_type *,
		     unsigned);
static int copy_lock(void *, struct e4_atom_type *);
static int copy_news(void *, int, struct e4_hack_type *, struct e4_hack_type *,
		     unsigned);
static int copy_odb1(void *, struct e4_atom_type *);
static int copy_odb2(void *, struct e4_atom_type *);
static int copy_pick(void *, struct e4_atom_type *);
static int copy_text(void *, struct e4_atom_type *);
static int copy_this(void *, struct e4_atom_type *);

static int
copy_atom(void *expression, struct e4_atom_type *atom_data)
{
    int status = 0;

    do {
	int type;

	if (atom_data->odb2._2nd) {
	    status = copy_odb2(expression, atom_data);
	    if (status) {
		atom_data->odb1 = NULL;
		atom_data->odb2._2nd = NULL;
		atom_data->type = MODE;
		break;
	    }
	}
	if (atom_data->odb1) {
	    status = copy_odb1(expression, atom_data);
	    if (status) {
		atom_data->odb1 = NULL;
		atom_data->type = MODE;
		break;
	    }
	}

	type = atom_data->type;

	if (type == EVER) {
	    status = copy_ever(expression, atom_data);
	} else if (type == LAST) {
	    status = copy_last(expression, atom_data);
	} else if (type == LOCK) {
	    status = copy_lock(expression, atom_data);
	} else if (type == PICK) {
	    status = copy_pick(expression, atom_data);
	} else if (type == TEXT) {
	    status = copy_text(expression, atom_data);
	} else if (type == THIS) {
	    status = copy_this(expression, atom_data);
	}
    } while (0);

    return status;
}


static int
copy_case(void *expression, int type, struct e4_base_type *base_data,
	  const struct e4_last_type *last_data, unsigned bits, unsigned count)
{
    int status;
    struct e4_flat_type *flat_data, *flat_text;
    const struct e4_last_type *last_slip;
    unsigned class, copy;

    flat_text = (void *) last_data;

    last_slip = (void *) (flat_text - 2)->flat_data;

    if (count == last_slip->count && !(last_slip->bits & SLIP_LIST)) {
	class = 0;
    } else {
	class = count;
	if (last_slip->bits & (LINK_PASS | POST_TYPE)) {
	    class <<= 1;
	}
    }

    copy = sizeof(struct e4_last_type) + class * sizeof(int);

    status = E_LINK
	(expression, &flat_data, sizeof(struct e4_flat_type) * 3 + copy);
    if (status) {
	status = ALLOC_ERROR;
    } else {
	struct e4_last_type *last_text;

	flat_data->flat_data = EXPRESSION(expression)->flat_data;
	flat_data->read = DATA_FLAT;

	EXPRESSION(expression)->flat_data = flat_data;

	flat_data[1].flat_data = (void *) last_slip;
	flat_data[2].flat_data = (void *) (flat_text - 1)->flat_data;

	last_text = (void *) (flat_data + 3);

	memcpy(last_text, last_data, copy);
	if (class) {
	    last_text->args = (void *) (last_text + 1);
	}

	base_data->last_data = last_text;

	if (type == LAST) {
	    void *context;

	    if (last_slip->bits & TEXT_LINK) {
		context = ((struct e4_flat_type *) last_data - 1)->flat_data;
	    } else {
		context = NULL;
	    }

	    status = last_slip->last(context, last_text, &base_data->text);
	}
    }

    return status;
}


static int
copy_data(struct e4_expression_type *expression_data,
	  struct e4_expression_type *expression_text)
{
    int status;

    status = copy_hack(expression_data, &expression_data->hack_data,
		       &expression_text->hack_data);
    if (status) {
    } else {
	unsigned size;

	size = expression_text->post_size;
	if (expression_data->bits & EXTERN_STACKS_MESS) {
	} else {
	    status = E_LINK(expression_data, &expression_data->post_data,
			    sizeof(struct e4_post_type) * size);
	}
	if (status) {
	} else {
	    expression_data->post_size = size;
	    size = expression_text->list_size;
	    expression_data->list_size = size;
	    if (size) {
		if (expression_data->bits & EXTERN_STACKS_MESS) {
		} else {
		    status = E_LINK(expression_data, &expression_data->list,
				    sizeof(void *) * size);
		}
	    }
	}
    }

    return status;
}


static int
copy_ever(void *expression, struct e4_atom_type *atom_data)
{
    int status;
    const struct e4_ever_type *ever_data;

    ever_data = atom_data->data.ever.ever_data;
    if (ever_data->bits & FLAT_LINK) {
	struct e4_flat_type *flat_data;

	status = E_LINK(expression, &flat_data, __news_alloc_size__);
	if (status) {
	} else {
	    struct e4_ever_type *ever_miss;
	    struct e4_news_type *news_data;
	    void **record, *state;

	    flat_data->flat_data = EXPRESSION(expression)->flat_data;
	    flat_data->read = NEWS_FLAT;

	    EXPRESSION(expression)->flat_data = flat_data;

	    ever_miss = (void *) (flat_data + 1);

	    news_data = (void *) (ever_miss + 1);

	    state = news_data + 1;

	    record = state;

	    *record = (void *) &news_data->line;

	    *ever_miss = *ever_data;

	    *news_data = *(struct e4_news_type *) (ever_data + 1);
	    news_data->call = 1;

	    atom_data->data.ever.state = state;
	    atom_data->data.ever.ever_data = ever_miss;
	}
    } else {
	status = 0;
    }

    return status;
}


static int
copy_hack(void *expression, struct e4_hack_type *hack_data,
	  struct e4_hack_type *hack_text)
{
    int status;
    struct e4_atom_type *atom_data;
    unsigned count;

    count = hack_text->count;

    status =
	E_LINK(expression, &atom_data, count * sizeof(struct e4_atom_type));
    if (status) {
    } else {
	struct e4_atom_type *atom_text;
	unsigned i;

	hack_data->atoms = atom_data;

	atom_text = hack_text->atoms;

	memcpy(atom_data, atom_text, count * sizeof(struct e4_atom_type));

	i = count;
	for (; i; i--) {
	    status = copy_atom(expression, atom_data);
	    if (status) {
		i--;
		break;
	    } else {
		atom_data++;
		atom_text++;
	    }
	}

	hack_data->count = count - i;
    }

    return status;
}


static int
copy_last(void *expression, struct e4_atom_type *atom_data)
{
    int status;
    struct e4_hack_type *hack_data, *hack_text;
    const struct e4_last_type *last_data;
    unsigned count, offset;

    hack_text = atom_data->data.last.base;
    last_data = hack_base(hack_text)->last_data;

    offset = sizeof(struct e4_base_type);

    count = last_data->count;
    status = E_LINK
	(expression, &hack_data,
	 offset + (sizeof(struct e4_hack_type)
		   + sizeof(int (*) (x1f4_e4_LOAD_ARGS_0))) * count);
    if (status) {
	atom_data->type = MODE;
    } else {
	hack_data = (struct e4_hack_type *) ((char *) hack_data + offset);

	atom_data->data.last.base = hack_data;

	status = copy_news(expression, LAST, hack_data, hack_text, count);
    }

    return status;
}


static int
copy_list(void *expression, struct e4_hack_type *hack_data,
	  struct e4_hack_type *hack_text, unsigned count)
{
    int status;
    unsigned i;

    *hack_base(hack_data) = *hack_base(hack_text);

    if (count) {
	for (i = count; i; i--) {
	    hack_data->count = 0;
	    hack_data->atoms = NULL;
	    hack_data++;
	}

	hack_data -= count;

	for (i = count; i; i--) {
	    status = copy_hack(expression, hack_data, hack_text);
	    if (status) {
		break;
	    } else {
		hack_data++;
		hack_text++;
	    }
	}
    } else {
	status = 0;
    }
    if (status) {
    } else {
	memcpy(hack_data, hack_text,
	       sizeof(int (*) (x1f4_e4_LOAD_ARGS_0)) * count);
    }

    return status;
}


static int
copy_lock(void *expression, struct e4_atom_type *atom_data)
{
    int status;
    struct e4_hack_type *hack_data, *hack_text;
    const struct e4_last_type *last_data;
    unsigned count, offset;

    hack_text = atom_data->data.last.base;
    last_data = hack_base(hack_text)->last_data;

    offset = sizeof(struct e4_base_type) + sizeof(struct e4_fine_type);

    count = last_data->count;
    status = E_LINK
	(expression, &hack_data,
	 offset + (sizeof(struct e4_hack_type)
		   + sizeof(int (*) (x1f4_e4_LOAD_ARGS_0))) * count);
    if (status) {
	atom_data->type = MODE;
    } else {
	hack_data = (struct e4_hack_type *) ((char *) hack_data + offset);

	*hack_fine(hack_data) = *hack_fine(hack_text);

	atom_data->data.last.base = hack_data;

	status = copy_news(expression, LOCK, hack_data, hack_text, count);
	if (status) {
	} else {
	    const struct e4_ever_type *ever_data;
	    struct e4_fine_type *fine_data;

	    fine_data = hack_fine(hack_data);

	    ever_data = fine_data->ever_data;
	    if (ever_data->bits & FLAT_LINK) {
		struct e4_flat_type *flat_data;

		status = E_LINK(expression, &flat_data, __news_alloc_size__);
		if (status) {
		} else {
		    struct e4_base_type *base_data;
		    struct e4_ever_type *ever_miss;
		    struct e4_news_type *news_data;
		    void *state;
		    const void **record;

		    flat_data->flat_data = EXPRESSION(expression)->flat_data;
		    flat_data->read = NEWS_FLAT;

		    EXPRESSION(expression)->flat_data = flat_data;

		    ever_miss = (void *) (flat_data + 1);

		    news_data = (void *) (ever_miss + 1);

		    state = news_data + 1;

		    record = state;

		    *record = &news_data->line;

		    *ever_miss = *ever_data;

		    *news_data = *(struct e4_news_type *) (ever_data + 1);
		    news_data->call = 1;

		    base_data = (void *) hack_data;
		    base_data--;
		    if (base_data->last_data->bits & FLAT_LINE) {
		    } else {
			base_data->last_data = &news_data->line.last;
		    }

		    fine_data->line = state;
		    fine_data->ever_data = ever_miss;
		}
	    }
	}
    }

    return status;
}


static int
copy_news(void *expression, int type, struct e4_hack_type *hack_data,
	  struct e4_hack_type *hack_text, unsigned count)
{
    int status;

    status = copy_list(expression, hack_data, hack_text, count);
    if (status) {
    } else {
	struct e4_base_type *base_data;
	const struct e4_last_type *last_data;
	unsigned bits;

	base_data = (void *) hack_data;
	base_data--;
	last_data = base_data->last_data;
	bits = last_data->bits;
	if (bits & FLAT_LINE) {
	    status = copy_case
		(expression, type, base_data, last_data, bits, count);
	}
    }

    return status;
}


static int
copy_odb1(void *expression, struct e4_atom_type *atom_data)
{
    int status;
    struct e4_odb1_type *odb1_data = NULL, *odb1_text;

    odb1_text = atom_data->odb1;
    do {
	struct e4_odb1_type *odb1_miss;

	status = E_LINK(expression, &odb1_miss, sizeof(struct e4_odb1_type));
	if (status) {
	    break;
	} else {
	    *odb1_miss = *odb1_text;
	    if (odb1_data) {
		odb1_data->odb1_data = odb1_miss;
	    } else {
		atom_data->odb1 = odb1_miss;
	    }
	    odb1_data = odb1_miss;
	    odb1_text = odb1_text->odb1_data;
	}
    } while (odb1_text);

    if (odb1_data) {
	odb1_data->odb1_data = NULL;
    } else {
	atom_data->odb1 = NULL;
    }

    return status;
}


static int
copy_odb2(void *expression, struct e4_atom_type *atom_data)
{
    int status;
    struct e4_odb2_type *odb2_data = NULL, *odb2_text;

    odb2_text = atom_data->odb2._2nd;
    do {
	struct e4_odb2_type *odb2_miss;

	status = E_LINK(expression, &odb2_miss, sizeof(struct e4_odb2_type));
	if (status) {
	    break;
	} else {
	    *odb2_miss = *odb2_text;
	    if (odb2_data) {
		odb2_data->odb2_data = odb2_miss;
	    } else {
		atom_data->odb2._2nd = odb2_miss;
	    }
	    odb2_data = odb2_miss;
	    odb2_text = odb2_text->odb2_data;
	}
    } while (odb2_text);

    if (odb2_data) {
	odb2_data->odb2_data = NULL;
    } else {
	atom_data->odb2._2nd = NULL;
    }

    return status;
}


static int
copy_pick(void *expression, struct e4_atom_type *atom_data)
{
    int status;
    struct e4_hack_type *hack_data, *hack_text;
    const struct e4_last_type *last_data;
    unsigned count, offset;

    hack_text = atom_data->data.last.base;
    last_data = hack_base(hack_text)->last_data;

    offset = sizeof(struct e4_base_type) + sizeof(struct e4_near_type);

    count = last_data->count;
    status = E_LINK
	(expression, &hack_data,
	 offset + (sizeof(struct e4_hack_type)
		   + sizeof(int (*) (x1f4_e4_LOAD_ARGS_0))) * count);
    if (status) {
	atom_data->type = MODE;
    } else {
	hack_data = (struct e4_hack_type *) ((char *) hack_data + offset);

	*hack_near(hack_data) = *hack_near(hack_text);

	atom_data->data.last.base = hack_data;

	status = copy_news(expression, PICK, hack_data, hack_text, count);
    }

    return status;
}


static int
copy_text(void *expression, struct e4_atom_type *atom_data)
{
    e4_text_type miss, text;
    int status;
    unsigned size;

    text = *(C_TEXT *) &atom_data->data.lead.data;
    size = strlen(text) + 1;
    status = E_LINK(expression, &miss, size);
    if (status) {
	atom_data->type = MODE;
    } else {
	C_TEXT *slip;

	memcpy(miss, text, size);
	slip = (C_TEXT *) &atom_data->data.lead.data;
	*slip = miss;
    }

    return status;
}


static int
copy_this(void *expression, struct e4_atom_type *atom_data)
{
    int status;
    struct e4_hack_type *hack_data;

    status = E_LINK(expression, &hack_data, sizeof(struct e4_hack_type));
    if (status) {
	atom_data->type = MODE;
    } else {
	struct e4_hack_type *hack_text;

	hack_text = atom_data->data.hack.data;
	hack_data->count = 0;
	hack_data->atoms = NULL;
	atom_data->data.hack.data = hack_data;
	status = copy_hack(expression, hack_data, hack_text);
    }

    return status;
}


int
x1f4_copy_expression(struct e4_expression_type **expression,
		     struct e4_expression_type *expression_text)
{
    int status;
    struct e4_expression_type *expression_data;

    status = expression_text->m.link
	(expression_text->m.data, (void *) &expression_data,
	 sizeof(struct e4_expression_type));
    if (status) {
	status = ALLOC_ERROR;
    } else {
#if __COPY_E4_POST__
	expression_data->fast = expression_text->fast;
#endif				/* __COPY_E4_POST__ */
	expression_data->bits = expression_text->bits;
	expression_data->flat_data = NULL;
	expression_data->hack_data.count = 0;
	expression_data->hack_data.atoms = NULL;
	expression_data->list = NULL;
	expression_data->load = expression_text->load;
	expression_data->m = expression_text->m;
	expression_data->state = 1;
	expression_data->post_data = NULL;
#if __LINE_E4_POST__
	expression_data->post = expression_text->post;
#endif				/* __LINE_E4_POST__ */
	expression_data->type = expression_text->type;
	status = copy_data(expression_data, expression_text);
	if (status) {
	    x1f4_fini_expression(&expression_data);
	} else {
	    *expression = expression_data;
	}
    }

    return status;
}
