##################################################
# SPYCE - Python-based HTML Scripting
# Copyright (c) 2002 Rimon Barr.
#
# Refer to spyce.py
# CVS: $Id: spyceParser.py,v 1.5 2002/11/26 16:52:03 batripler Exp $
##################################################

import lex, yacc
import spyceException

##################################################
# Spyce Tokenizer
#

tokens = (
  'STMT_OPEN', 'CHUNK_OPEN', 'CHUNK_GLOBAL_OPEN', 'EVAL_OPEN', 'DIRECT_OPEN', 'LAMBDA_OPEN', 
  'COMMENT_OPEN', 'COMMENT_CLOSE', 'PY_CLOSE',
  'OPEN_ESC', 'CLOSE_ESC',
  'OPEN', 'CLOSE', 
  'ESCAPE', 'MINUS', 'NEWLINE', 
  'OTHER',
)

def t_OPEN_ESC(t):
  r'\\\[\['
  return t

def t_CLOSE_ESC(t):
  r'\\\]\]'
  return t

def t_CHUNK_OPEN(t):
  r'\[\[\\'
  return t

def t_CHUNK_GLOBAL_OPEN(t):
  r'\[\[\\\\'
  return t

def t_EVAL_OPEN(t):
  r'\[\[='
  return t

def t_DIRECT_OPEN(t):
  r'\[\[\.'
  return t

def t_LAMBDA_OPEN(t):
  r'\[\[spy'
  return t

def t_COMMENT_OPEN(t):
  r'\[\[--'
  return t

def t_COMMENT_CLOSE(t):
  r'--\]\]'
  return t

def t_STMT_OPEN(t):
  r'\[\['
  return t

def t_PY_CLOSE(t):
  r'\]\]'
  return t

def t_OPEN(t):
  r'\['
  return t

def t_CLOSE(t):
  r'\]'
  return t

def t_ESCAPE(t):
  r'\\'
  return t

def t_MINUS(t):
  r'-'
  return t

def t_NEWLINE(t):
  r'\n+'
  return t

def t_OTHER(t):
  r'[^][\-\\\n]+'
  return t

def t_error(t):
  raise spyceException.spyceSyntaxError('Untokenizable character: %s'%`t.value[0]`, ((t.lineno, t.colno), (t.lineno, t.colno), '', None))

##################################################
# Spyce Parser
#

def p_spyce_plus(t):
  '''spyce_plus : spyce_plus spyce
                | spyce'''
  try:
    t[0] = t[1] + [t[2]]
  except:
    t[0] = [t[1]]

def p_spyce(t):
  '''spyce : html
           | stmt
           | eval
           | chunk
           | chunk_global
           | directive
           | comment'''
  t[0] = 'spyce', t.posspan(0), t[1]

def p_html(t):
  '''html : html_string_plus'''
  t[0] = 'html', t.posspan(0), t[1]

def p_html_string_plus(t):
  '''html_string_plus : html_string_plus html_string
                      | html_string'''
  try:
    t[0] = t[1] + [t[2]]
  except:
    t[0] = [t[1]]

def p_html_string(t):
  '''html_string : OPEN
                 | CLOSE
                 | ESCAPE
                 | MINUS
                 | NEWLINE
                 | OTHER'''
  t[0] = 'html_string', t.posspan(0), t[1]

def p_html_string_esc(t):
  '''html_string : OPEN_ESC
                 | CLOSE_ESC'''
  t[0] = 'html_string', t.posspan(0), t[1][1:]

def p_stmt(t):
  '''stmt : STMT_OPEN code_plus PY_CLOSE'''
  t[0] = 'stmt', t.posspan(0), t[2]

def p_chunk(t):
  '''chunk : CHUNK_OPEN code_plus PY_CLOSE'''
  t[0] = 'chunk', t.posspan(0), t[2]

def p_chunk_global(t):
  '''chunk_global : CHUNK_GLOBAL_OPEN code_plus PY_CLOSE'''
  t[0] = 'gchunk', t.posspan(0), t[2]

def p_eval(t):
  '''eval : EVAL_OPEN code_plus PY_CLOSE'''
  t[0] = 'eval', t.posspan(0), t[2]

def p_lambda(t):
  '''lambda : LAMBDA_OPEN spyce_plus PY_CLOSE'''
  t[0] = 'lambda', t.posspan(0), t[2]

def p_directive(t):
  '''directive : DIRECT_OPEN inside_plus PY_CLOSE'''
  t[0] = 'directive', t.posspan(0), t[2]

def p_comment(t):
  '''comment : COMMENT_OPEN comment_string_plus COMMENT_CLOSE'''
  t[0] = 'comment', t.posspan(0), t[2]

def p_comment_string(t):
  '''comment_string : STMT_OPEN
                    | CHUNK_OPEN
                    | CHUNK_GLOBAL_OPEN
                    | EVAL_OPEN
                    | DIRECT_OPEN
                    | LAMBDA_OPEN
                    | COMMENT_OPEN
                    | PY_CLOSE
                    | OPEN_ESC
                    | CLOSE_ESC
                    | OPEN
                    | CLOSE 
                    | ESCAPE
                    | MINUS
                    | NEWLINE 
                    | OTHER'''
  t[0] = t[1]

def p_comment_string_plus(t):
  '''comment_string_plus : comment_string_plus comment_string
                         | comment_string'''
  try:
    t[0] = t[1] + [t[2]]
  except:
    t[0] = [t[1]]

def p_inside(t):
  '''inside : OPEN
            | CLOSE
            | ESCAPE
            | MINUS
            | NEWLINE
            | OTHER'''
  t[0] = 'inside', t.posspan(0), t[1]

def p_inside_plus(t):
  '''inside_plus : inside_plus inside
                 | inside '''
  try:
    t[0] = t[1] + [t[2]]
  except:
    t[0] = [t[1]]

def p_code(t):
  '''code : OPEN
          | CLOSE
          | ESCAPE
          | MINUS
          | NEWLINE
          | OTHER'''
  t[0] = 'code', t.posspan(0), t[1]

def p_code_lambda(t):
  '''code : lambda'''
  t[0] = t[1]

def p_code_plus(t):
  '''code_plus : code_plus code
               | code'''
  try:
    t[0] = t[1] + [t[2]]
  except:
    t[0] = [t[1]]

def p_error(t):
  if t:
    if 'posspan' in dir(t):
      begin, end = t.posspan()
    else:
      begin, end = (t.lineno, t.colno), (t.endlineno, t.endcolno)
    raise spyceException.spyceSyntaxError('Error parsing token: %s'%`t.value`, (begin, end, '', None))
  else:
    raise spyceException.spyceSyntaxError('Unexpected EOF while parsing', (('EOF', None), ('EOF', None), '', None))


def docStrings():
  'LEAVE THIS DOC_STRING IN PLACE'
  try: 
    return docStrings.__doc__
  except AttributeError: return 0

##################################################
# External functions
#

def getSpyceTokens(buf):
  lexer = lex.lex()
  lexer.input(buf)
  tokens = []
  while 1:
    tok = lexer.token()
    if not tok: break
    tokens.append(tok)
  return tokens

def _getParser():
  return yacc.yacc(debug=0, tabmodule="spyceParserTable")

def parseSpyce(buf, lock=None):
  if lock: lock.lock()
  try:
    try: import spyceParserTable
    except: 
      raise spyceException.spyceParserError('Please compile spyce to generate spyceParserTable.py')
    try: lexer = lex.lex(optimize=1)
    except:
      raise spyceException.spyceParserError('unable to initialize tokenizer. Ensure Python is not running in optimized mode.')
    try: parser = _getParser()
    except:
      raise spyceException.spyceParserError('unable to initialize parser. Ensure Python is not running in optimized mode.')
    lexer.input(buf)
    result = parser.parse(debug=0,lexer=lexer)
    return result
  finally:
    if lock: lock.unlock()

def genParserTable():
  print 'Generating spyce parser table: spyceParserTable.py'
  parser = _getParser()
  import spyceParserTable
  
##################################################
# Testing
#

def testTokenizer(buf):
  tokens = getSpyceTokens(buf)
  for t in tokens:
    print t

def testParser(buf):
  import pprint
  pprint.pprint(parseSpyce(buf))

def test():
  import sys
  buf = sys.stdin.read()
  print '*** Testing tokenizer'
  testTokenizer(buf)
  print '*** Testing parser'
  testParser(buf)



if __name__ == '__main__':
  genParserTable()
  #test()

