##################################################
# SPYCE - Python-based HTML Scripting
# Copyright (c) 2002 Rimon Barr.
#
# Refer to spyce.py
# CVS: $Id: spyceCompile.py,v 1.55 2002/12/01 21:13:23 batripler Exp $
##################################################

#try:
#  exec('import sre as re')  # due to stack limitations of sre
#  # exec to be backwards compatible with Python 1.5
#except:
#  import re
import re  # otherwise apache 2.0 pcre library conflicts
           # we just can't win! either stack limits (sre), or 
           # library conflicts (pre)! :)

from cStringIO import StringIO
import sys, string, token, tokenize, os
import spyceParser, spyceTag, spyceException, spyceUtil

__doc__ = '''Compile Spyce files into Python code.'''

##################################################
# Special method names
#

SPYCE_CLASS = 'spyceImpl'
SPYCE_INIT_FUNC = 'spyceStart'
SPYCE_DESTROY_FUNC = 'spyceFinish'
SPYCE_PROCESS_FUNC = 'spyceProcess'
SPYCE_GLOBAL_CODE = '__SPYCE_GLOBAL_CODE_CONSTANT'
SPYCE_WRAPPER = '_spyceWrapper'
DEFAULT_CODEPOINT = [SPYCE_PROCESS_FUNC]

##################################################
# Misc. Utility functions
#

DIRECTIVE_NAME = re.compile('[a-zA-Z][-a-zA-Z0-9_:]*')
DIRECTIVE_ATTR = re.compile(
    r'\s*([a-zA-Z_][-.:a-zA-Z_0-9]*)(\s*=\s*'
    r'(\'[^\']*\'|"[^"]*"|[-a-zA-Z0-9./:;+*%?!&$\(\)_#=~]*))?')
def parseDirective(text):
  "Parse a Spyce directive into name and an attribute list."
  attrs = {}
  match = DIRECTIVE_NAME.match(text)
  if not match: return None, {}
  name = string.lower(text[:match.end()])
  text = string.strip(text[match.end()+1:])
  while text:
    match = DIRECTIVE_ATTR.match(text)
    if not match: break
    attrname, rest, attrvalue = match.group(1, 2, 3)
    if not rest: attrvalue = None
    elif attrvalue[:1] == "'" == attrvalue[-1:] or \
        attrvalue[:1] == '"' == attrvalue[-1:]:
      attrvalue = attrvalue[1:-1]
    attrs[string.lower(attrname)] = attrvalue
    text = text[match.end()+1:]
  return name, attrs

##################################################
# Build Pre-Python AST
#

# ast node types
AST_PY      = 0
AST_PYEVAL  = 1
AST_TEXT    = 2
AST_COMPACT = 3

# compacting modes
COMPACT_OFF   = 0
COMPACT_LINE  = 1
COMPACT_SPACE = 2
COMPACT_FULL  = 3

class ppyAST:
  "Generate a pre-Python AST"
  def __init__(self):
    "Initialise parser data structures, AST, token handlers, ..."
    # set up ast
    self._code = { 
      'elements': {}, 
      'leafs': [], 
    }
    self._codepoint = self._code
    self._codepointname = []
    self._mods = []
    self._taglibs = {}
  def getCode(self):
    return self._code
  def getModules(self):
    return self._mods
  def getTaglibs(self):
    return self._taglibs
  # routines to navigate AST
  def selectCodepoint(self, codepointname):
    code = self._code
    for point in codepointname:
      code = code['elements'][point]
    self._codepoint = code
    self._codepointname = codepointname
  def getCodepoint(self):
    return self._codepointname
  def descendCodepoint(self, codepointSuffix):
    self._codepointname.append(codepointSuffix)
    self.selectCodepoint(self._codepointname)
  def ascendCodepoint(self):
    suffix = self._codepointname.pop()
    self.selectCodepoint(self._codepointname)
    return suffix
  # routines that modify the ast
  def appendCodepoint(self, codepointSuffix, firstline, ref=None):
    self._codepoint['elements'][codepointSuffix] = {
      'elements': {},
      'leafs': [],
    }
    self.descendCodepoint(codepointSuffix)
    self.addCode(string.strip(firstline), ref) # note: firstline is not indented
  def addCode(self, code, ref=None):
    self._codepoint['leafs'].append((AST_PY, code, ref))
  def addGlobalCode(self, code, ref=None):
    codepoint = self.getCodepoint()
    self.selectCodepoint([SPYCE_GLOBAL_CODE])
    self.addCode(code+'\n', ref)
    self.selectCodepoint(codepoint)
    pass
  def addEval(self, eval, ref=None):
    self._codepoint['leafs'].append((AST_PYEVAL, eval, ref))
  def addCodeIndented(self, code, ref=None, globalcode=0):
    # rimtodo: multiline quotes throws off line calculations below
    code = removeMultiLineQuotes(code)
    # funky hack: put in NULLs to preserve indentation
    #   NULLs don't appear in code, and the BraceConverter will
    #   turn them back into spaces. If we leave them as spaces,
    #   BraceConverter is just going to ignore them and pay attention
    #   only to the braces. (not the best compile-time performance!)
    code = string.split(code, '\n')
    code = map(lambda l: (len(l)-len(string.lstrip(l)), l), code)
    code = map(lambda (indent, l): chr(0)*indent + l, code)
    code.append('')
    # split code lines
    (brow, bcol), (erow, ecol), text, file = ref
    row = brow
    for l in code:
      cbcol = 0
      cecol = len(l)
      if row==brow: cbcol = bcol
      if row==erow: cecol = ecol
      ref = (row, cbcol), (row, cecol), l, file
      if globalcode: self.addGlobalCode(l, ref)
      else: self.addCode(l, ref)
      row = row + 1
  def addText(self, text, ref=None):
    self._codepoint['leafs'].append((AST_TEXT, text, ref))
  def addCompact(self, compact, ref):
    self._codepoint['leafs'].append((AST_COMPACT, compact, ref))
  def addModule(self, modname, modfrom, modas):
    self._mods.append((modname, modfrom, modas))
  def addTaglib(self, libname, libfrom=None, libas=None):
    if not libas: libas=libname
    self._taglibs[libas] = libname, libfrom

def processMagic(buf):
  if buf[:2]=='#!':
    buf = string.join(string.split(buf, '\n')[1:], '\n')
  return buf

RE_LIB_TAG = re.compile(r'''
  <                                    # beginning of tag
  (?P<end>/?)                          # ending tag
  (?P<lib>[a-zA-Z][-.a-zA-Z0-9_]*):    # lib name
  (?P<name>[a-zA-Z][-.a-zA-Z0-9_]*)    # tag name
  (?P<attrs>(?:\s+                     # attributes
    (?:[a-zA-Z_][-.:a-zA-Z0-9_]*       # attribute name
      (?:\s*=\s*                       # value indicator
        (?:'[^']*'                     # LITA-enclosed value
          |"[^"]*"                     # LIT-enclosed value
          |[^'">\s]+                   # bare value
        )
      )?
    )
  )*)
  \s*                                  # trailing whitespace
  (?P<single>/?)                       # single / unpaired tag
  >                                    # end of tag
''', re.VERBOSE)
class spyceParser2:
  def __init__(self, server):
    self._server = server
    self._tagChecker = spyceTag.spyceTagChecker(server)
  def finish(self):
    self._tagChecker.finish()
  def processFile(self, buf, filename, sig):
    try:
      buf = processMagic(buf)
      ast = ppyAST()
      self._curdir, self._curfile = os.getcwd(), '<string>'
      if filename:
        self._curdir, self._curfile = os.path.split(filename)
      if not self._curdir:
        self._curdir = os.getcwd()
      self.moduleSetup(ast, sig)
      self.functionPre(ast)
      self.processSpyceList(ast, spyceParser.parseSpyce(buf, self._server.lock))
      self.functionPost(ast)
      return ast.getCode(), ast.getModules()
    except spyceException.spyceSyntaxError, e:
      if e.info:
        begin, end, text, _ = e.info
        e.info = begin, end, text, self._curfile
      raise e
  def moduleSetup(self,ast, sig):
    # define global code location
    ast.selectCodepoint([])
    ast.appendCodepoint(SPYCE_GLOBAL_CODE, '')
    ast.addGlobalCode('''
try:
  exec('from __future__ import nested_scopes')
except: pass
from spyceException import spyceDone, spyceRedirect, spyceRuntimeException
''')
    ## define spyceInit
    #ast.selectCodepoint([])
    #ast.appendCodepoint(SPYCE_INIT_FUNC, 'def '+SPYCE_INIT_FUNC+'()')
    #ast.addCode('pass', None)
    ## define spyceDestroy
    #ast.selectCodepoint([])
    #ast.appendCodepoint(SPYCE_DESTROY_FUNC, 'def '+SPYCE_DESTROY_FUNC+'()')
    #ast.addCode('pass', None)
    # define spyceProcess
    ast.selectCodepoint([])
    ast.appendCodepoint(SPYCE_PROCESS_FUNC, 'def '+SPYCE_PROCESS_FUNC+'('+sig+')')
    # get ready for the code
    ast.selectCodepoint(DEFAULT_CODEPOINT)
  def functionPre(self, ast):
    ast.addCode('try:{', None)
    ast.addCode('pass', None)
  def functionPost(self, ast):
    ast.addCode('} except spyceDone: pass', None)
    ast.addCode('except spyceRedirect: raise', None)
    ast.addCode('except KeyboardInterrupt: raise', None)
    ast.addCode('except:{ raise spyceRuntimeException(%s) }'%SPYCE_WRAPPER, None)
  def processSpyceList(self, ast, nodelist):
    self.processListNode(ast, nodelist, self.processSpyceNode)
  def processListNode(self, ast, nodelist, f):
    if not type(nodelist) == type([]):
      nodelist = [nodelist]
    for node in nodelist:
      f(ast, node)
  def processSpyceNode(self, ast, node):
    "Process a Spyce element."
    try: type, (begin, end), info = node
    except: raise spyceException.spyceSyntaxError('invalid spyce node', (begin, end, `node`, self._curfile))
    try: subtype = info[0]
    except: raise spyceException.spyceSyntaxError('invalid spyce node info field', (begin, end, `info`, self._curfile))
    try:
      result = {
        'html': self.processHtmlNode,
        'stmt': self.processStmtNode,
        'chunk': self.processChunkNode,
        'gchunk': self.processGlobalChunkNode,
        'eval': self.processEvalNode,
        'directive': self.processDirectiveNode,
        'comment': self.processCommentNode,
      }[subtype](ast, info)
    except KeyError:
      raise spyceException.spyceSyntaxError('unknown spyce subnode type', (begin, end, `subtype`, self._curfile))
  def processHtmlNode(self, ast, node):
    "Process HTML (possibly with some active tags)"
    try: type, (begin, end), htmllist = node
    except: raise spyceException.spyceSyntaxError('invalid html node', (begin, end, `node`, self._curfile))
    html = self.getNodeList(htmllist)
    m = RE_LIB_TAG.search(html)
    while m:
      plain = html[:m.start()]
      if plain:
        plain_end, tag_begin = calcEndPos(begin, plain)
        self.processHtmlText(ast, plain, begin, plain_end)
      else: tag_begin = begin
      tag = m.group(0)
      tag_end, begin = calcEndPos(tag_begin, tag)
      self.processHtmlTag(ast, tag, 
        not not m.group('end'), m.group('lib'), m.group('name'), 
        m.group('attrs'), not m.group('single'),
        tag_begin, tag_end)
      html = html[m.end():]
      m = RE_LIB_TAG.search(html)
    self.processHtmlText(ast, html, begin, end)
  def processHtmlTag(self, ast, tag, tagend, taglib, tagname, tagattrs, tagpair, begin, end):
    "Process HTML tags"
    # make sure prefix belongs to loaded taglibrary
    if not ast._taglibs.has_key(taglib):
      self.processHtmlText(ast, tag, begin, end)
      return
    # parse process tag attributes
    _, tagattrs = parseDirective('x '+tagattrs)
    # syntax check
    if not tagend: # start tag
      self._tagChecker.startTag(ast._taglibs[taglib], tagname, tagattrs, tagpair, (begin, end, tag, self._curfile))
    else: # end tag
      self._tagChecker.endTag(ast._taglibs[taglib], tagname, (begin, end, tag, self._curfile))
    # add tag python code
    if tagpair:  # paired tag
      if not tagend:  # open tag
        ast.addCode('try: {', (begin, end, tag, self._curfile))
        ast.addCode('taglib.tagPush(%s, %s, %s, %s)' % (
            repr(taglib), repr(tagname), repr(tagattrs), repr(tagpair)), 
          (begin, end, tag, self._curfile))
        ast.addCode('try: {', (begin, end, tag, self._curfile))
        ast.addCode('if taglib.tagBegin(): {',
          (begin, end, tag, self._curfile))
        ast.addCode('try: {', (begin, end, tag, self._curfile))
        ast.addCode('while 1: {', (begin, end, tag, self._curfile))
      else:  # close tag
        ast.addCode('if not taglib.tagBody(): break', (begin, end, tag, self._curfile))
        ast.addCode('}', (begin, end, tag, self._curfile))
        ast.addCode('} finally: taglib.tagEnd()', (begin, end, tag, self._curfile))
        ast.addCode('}', (begin, end, tag, self._curfile))
        ast.addCode('} except: taglib.tagCatch()', (begin, end, tag, self._curfile))
        ast.addCode('} finally: taglib.tagPop()', (begin, end, tag, self._curfile))
    else: # singleton
      ast.addCode('try: {', (begin, end, tag, self._curfile))
      ast.addCode('taglib.tagPush(%s, %s, %s, %s)' % (
          repr(taglib), repr(tagname), repr(tagattrs), repr(tagpair)), 
        (begin, end, tag, self._curfile))
      ast.addCode('try: {', (begin, end, tag, self._curfile))
      ast.addCode('taglib.tagBegin()',
        (begin, end, tag, self._curfile))
      ast.addCode('} except: taglib.tagCatch()', (begin, end, tag, self._curfile))
      ast.addCode('} finally: taglib.tagPop()', (begin, end, tag, self._curfile))
  def processHtmlText(self, ast, text, begin, end):
    "Process plain HTML text"
    if text:
      ast.addText(text, (begin, end, '<html string>', self._curfile))
  def processStmtNode(self, ast, node):
    "Process a Spyce Python statement element."
    try: type, ((beginrow, begincol), (endrow, endcol)), stmtlist = node
    except: raise spyceException.spyceSyntaxError('invalid stmt node', (begin, end, `node`, self._curfile))
    stmt = self.getNodeList(stmtlist)
    currow = beginrow
    lines = string.split(stmt, '\n')
    for l in lines:
      if currow == beginrow: curcolbegin = begincol
      else: curcolbegin = 0
      if currow == endrow: curcolend = endcol
      else: curcolend = len(l)
      l = string.strip(l)
      if l:
        ast.addCode(l, ((currow, curcolbegin), (currow, curcolend), l, self._curfile))
      currow = currow + 1
  def processChunkNode(self, ast, node, globalChunk=0):
    "Process a Spyce Python chunk element."
    try: type, (begin, end), chunklist = node
    except: raise spyceException.spyceSyntaxError('invalid chunk node', (begin, end, `node`, self._curfile))
    chunk = self.getNodeList(chunklist)
    chunk = string.split(chunk, '\n')
    # eliminate initial blank lines
    brow, bcol = begin
    while chunk and not string.strip(chunk[0]):
      chunk = chunk[1:]
      brow = brow + 1
    begin = brow, bcol
    if not chunk: return
    # outdent chunk based on first line
    # note: modifies multi-line strings having more spaces than first line outdent
    #    by removing outdent number of spaces at the beginning of each line.
    #    -- difficult to deal with efficiently (without parsing) so just 
    #    don't do this!
    outdent = len(chunk[0]) - len(string.lstrip(chunk[0]))
    for i in range(len(chunk)):
      if string.strip(chunk[i][:outdent]):
        chunk[i] = ' '*outdent + chunk[i]
    chunk = map(lambda l, outdent=outdent: l[outdent:], chunk)
    chunk = string.join(chunk, '\n')
    # add entire block
    if chunk:
      try:
        ast.addCodeIndented(chunk, (begin, end, chunk, self._curfile), globalChunk)
      except tokenize.TokenError, e:
        msg, (row, col) = e
        raise spyceException.spyceSyntaxError(msg, (begin, end, chunk, self._curfile) )
  def processGlobalChunkNode(self, ast, node):
    "Process a Spyce Python global chunk element."
    self.processChunkNode(ast, node, 1)
  def processEvalNode(self, ast, node):
    "Process a Spyce Python expression element."
    try: type, (begin, end), evallist = node
    except: raise spyceException.spyceSyntaxError('invalid eval node', (begin, end, `node`, self._curfile))
    expr = self.getNodeList(evallist)
    if expr:
      ast.addEval(string.strip(expr), (begin, end, '='+expr, self._curfile))
  def processCommentNode(self, ast, node):
    pass
  def processDirectiveNode(self, ast, node):
    "Process a Spyce directive element."
    try: type, (begin, end), directlist = node
    except: raise spyceException.spyceSyntaxError('invalid directive node', (begin, end, `node`, self._curfile))
    directive = self.getNodeList(directlist)
    name, attrs = parseDirective(directive)
    if name=='compact':
      compact_mode = COMPACT_FULL
      if attrs.has_key('mode'):
        mode = string.lower(attrs['mode'])
        if mode=='off':
          compact_mode = COMPACT_OFF
        elif mode=='line':
          compact_mode = COMPACT_LINE
        elif mode=='space':
          compact_mode = COMPACT_SPACE
        elif mode=='full':
          compact_mode = COMPACT_FULL
        else:
          raise spyceException.spyceSyntaxError('invalid compacting mode "%s" specified'%mode, (begin, end, directive, self._curfile))
      ast.addCompact(compact_mode, (begin, end, '<spyce compact directive>', self._curfile))
    elif name in ('module', 'import'):
      if not attrs.has_key('name') and not attrs.has_key('names'):
        raise spyceException.spyceSyntaxError('name or names attribute required', (begin, end, directive, self._curfile) )
      if attrs.has_key('names'):
        mod_names = filter(None, map(string.strip, string.split(attrs['names'],',')))
        for mod_name in mod_names:
          ast.addModule(mod_name, None, None)
          ast.addCode('%s.init()'%mod_name, (begin, end, directive, self._curfile))
      else:
        mod_name = attrs['name']
        mod_from = spyceUtil.extractValue(attrs, 'from')
        mod_as = spyceUtil.extractValue(attrs, 'as')
        mod_args = spyceUtil.extractValue(attrs, 'args', '')
        if mod_as: theName=mod_as
        else: theName=mod_name
        ast.addModule(mod_name, mod_from, mod_as)
        ast.addCode('%s.init(%s)'%(theName,mod_args), (begin, end, directive, self._curfile))
    elif name in ('taglib',):
      if not attrs.has_key('name') and not attrs.has_key('names'):
        raise spyceException.spyceSyntaxError('name or names attribute required', (begin, end, directive, self._curfile) )
      fullfile = os.path.join(self._curdir, self._curfile)
      if attrs.has_key('names'):
        taglib_names = filter(None, map(string.strip, string.split(attrs['names'],',')))
        for taglib_name in taglib_names:
          self._tagChecker.loadLib(taglib_name, None, None, fullfile, (begin, end, directive, self._curfile))
          ast.addTaglib(taglib_name)
          ast.addCode('taglib.load(%s)'%repr(taglib_name), (begin, end, directive, self._curfile))
      else:
        taglib_name = attrs['name']
        taglib_from = spyceUtil.extractValue(attrs, 'from')
        taglib_as = spyceUtil.extractValue(attrs, 'as')
        self._tagChecker.loadLib(taglib_name, taglib_from, taglib_as, fullfile, (begin, end, directive, self._curfile))
        ast.addTaglib(taglib_name, taglib_from, taglib_as)
        ast.addCode('taglib.load(%s, %s, %s)'%(repr(taglib_name), repr(taglib_from), repr(taglib_as)), (begin, end, directive, self._curfile))
    elif name=='include':
      if not attrs.has_key('file'):
        raise spyceException.spyceSyntaxError('file attribute missing', (begin, end, directive, self._curfile) )
      filename = os.path.join(self._curdir, attrs['file'])
      f = None
      try:
        try:
          f = open(filename)
          buf = f.read()
        finally:
          if f: f.close()
      except KeyboardInterrupt: raise
      except:
        raise spyceException.spyceSyntaxError('unable to open included file: %s'%filename, (begin, end, directive, self._curfile) )
      prevdir, prevfile = self._curdir, self._curfile
      self._curdir, self._curfile = os.path.dirname(filename), filename
      self.processSpyceList(ast, spyceParser.parseSpyce(processMagic(buf), self._server.lock))
      self._curdir, self._curfile = prevdir, prevfile 
    else:
      raise spyceException.spyceSyntaxError('invalid spyce directive', (begin, end, directive, self._curfile) )
  def getNodeList(self, nodelist):
    s = ''
    for n in nodelist:
      try: subtype = n[0]
      except: raise spyceException.spyceSyntaxError('unable to find node info field: %s' % `n`)
      try:
        result = {
          'html_string': self.getTextNode,
          'code': self.getTextNode,
          'inside': self.getTextNode,
          'lambda': self.getLambdaNode,
          'spyce': self.getSpyceNode,
        }[subtype](n)
        s = s + result
      except KeyError:
        raise spyceException.spyceSyntaxError('unknown subnode type: %s' % `subtype`)
    return s
  def getTextNode(self, node):
    try: type, (begin, end), text = node
    except: raise spyceException.spyceSyntaxError('invalid textual node: %s' % `node`, (begin, end, text, self._curfile))
    return text
  def getLambdaNode(self, node):
    try: type, (begin, end), lambdalist = node
    except: raise spyceException.spyceSyntaxError('invalid lambda-code node: %s' % `node`, (begin, end, `lambdalist`, self._curfile))
    lamb = string.split(self.getNodeList(lambdalist), ':')
    try:
      params = lamb[0]
      memoize = 0
      if params and params[0]=='!':
        params = params[1:]
        memoize = 1
      lamb = string.join(lamb[1:],':')
    except:
      raise spyceException.spyceSyntaxError('invalid spyce lambda', (begin, end, lamb, self._curfile))
    lamb = 'spylambda.define(%s, %s, %d)' % (`params`, `lamb`, memoize)
    return lamb
  def getSpyceNode(self, node):
    try: type, (begin, end), info = node
    except: raise spyceException.spyceSyntaxError('invalid spyce node', (begin, end, `node`, self._curfile))
    try: subtype = info[0]
    except: raise spyceException.spyceSyntaxError('invalid spyce node info field', (begin, end, `info`, self._curfile))
    try:
      return {
        'html': self.getHtmlNode,
        'stmt': self.getStmtNode,
        'chunk': self.getChunkNode,
        'gchunk': self.getGlobalChunkNode,
        'eval': self.getEvalNode,
        'directive': self.getDirectiveNode,
        'comment': self.processCommentNode,
      }[subtype](info)
    except KeyError:
      raise spyceException.spyceSyntaxError('unknown spyce subnode type', (begin, end, subtype, self._curfile))
  def getHtmlNode(self, node):
    return self.getElementNode(node, '', '')
  def getStmtNode(self, node):
    return self.getElementNode(node, '[[', ']]')
  def getChunkNode(self, node):
    return self.getElementNode(node, '[[\\', ']]')
  def getGlobalChunkNode(self, node):
    return self.getElementNode(node, '[[\\\\', ']]')
  def getEvalNode(self, node):
    return self.getElementNode(node, '[[=', ']]')
  def getDirectiveNode(self, node):
    return self.getElementNode(node, '[[.', ']]')
  def getCommentNode(self, node):
    return ''
  def getElementNode(self, node, start, finish):
    try: type, (begin, end), ellist = node
    except: raise spyceException.spyceSyntaxError('invalid element node', (begin, end, `node`, self._curfile))
    el = self.getNodeList(ellist)
    return start+el+finish

RE_MULTI_LINE_QUOTE_BEGIN = re.compile(r'r?'+"(''')|"+'(""")')
def removeMultiLineQuotes(s):
  def findMultiLineQuote(s):
    quotelist = []
    def eatToken(type, string, begin, end, _, quotelist=quotelist):
      if type == token.STRING and RE_MULTI_LINE_QUOTE_BEGIN.match(string):
        quotelist.append((string, begin,end))
    tokenize.tokenize(StringIO(s).readline, eatToken)
    if quotelist: return quotelist[0]
  def replaceRegionWithLine(s, begin, end, s2):
    (beginrow, begincol), (endrow, endcol) = begin, end
    beginrow, endrow = beginrow-1, endrow-1
    s = string.split(s, '\n')
    s1, s3 = s[:beginrow], s[endrow+1:]
    s2 = s[beginrow][:begincol] + s2 + s[endrow][endcol:]
    return string.join(s1 + [s2] + s3, '\n')
  match = findMultiLineQuote(s)
  while match:
    s2, begin, end = match
    s = replaceRegionWithLine(s, begin, end, `eval(s2)`)
    match = findMultiLineQuote(s)
  return s

def calcEndPos(begin, str):
  if not str: raise 'empty string'
  beginrow, begincol = begin
  eol = 0
  if str[-1]=='\n': 
    str = str[:-1]+' '
    eol = 1
  lines = string.split(str, '\n')
  endrow = beginrow + len(lines)-1
  if endrow!=beginrow:
    begincol = 0
  endcol = begincol + len(lines[-1]) - 1
  beginrow, begincol = endrow, endcol + 1
  if eol:
    begincol = 0
    beginrow = beginrow + 1
  return (endrow, endcol), (beginrow, begincol)

##################################################
# Peep-hole optimizer
#

class spyceOptimize:
  def __init__(self, ast):
    self.compaction(ast)
    self.sideBySideWrites(ast)
    #self.splitCodeLines(ast)
  def splitCodeLines(self, ast):
    elements, leafs = ast['elements'], ast['leafs']
    for el in elements.keys():
      self.splitCodeLines(elements[el])
    if leafs:
      i = 0
      while i<len(leafs):
        row = 1
        type, text, ref = leafs[i]
        if type == AST_PY and ref:
          (brow, bcol), (erow, ecol), code, file = ref
          lines = string.split(code, '\n')
          if code==text and len(lines)>1:
            del leafs[i]
            row = brow
            for l in lines:
              cbcol = 0
              cecol = len(l)
              if row==brow: cbcol = bcol
              if row==erow: becol = ecol
              leafs.insert(i+(brow-row), (AST_PY, l, ((row, cbcol), (row, cecol), l, file)))
              row = row + 1
        i = i + row

  def sideBySideWrites(self, ast):
    elements, leafs = ast['elements'], ast['leafs']
    for el in elements.keys():
      self.sideBySideWrites(elements[el])
    if leafs:
      i = 0
      while i+1<len(leafs):
        type1, text1, ref1 = leafs[i]
        type2, text2, ref2 = leafs[i+1]
        file1 = None
        file2 = None
        if ref1:
          _, _, _, file1 = ref1
        if ref2:
          _, _, _, file2 = ref2
        if type1==AST_TEXT and type2==AST_TEXT and file1==file2:
          text = text1 + text2
          begin, _, orig, _ = ref1
          _, end, _, _ = ref2
          leafs[i] = AST_TEXT, text, (begin, end, orig, file1)
          del leafs[i+1]
          i = i - 1
        i = i+1
  def compaction(self, ast):
    elements, leafs = ast['elements'], ast['leafs']
    compact = COMPACT_LINE
    for el in elements.keys():
      self.compaction(elements[el])
    if leafs:
      i = 0
      while i<len(leafs):
        type, text, ref = leafs[i]
        if type==AST_COMPACT:
          compact = text
        elif type==AST_TEXT:
          # line compaction
          if compact==COMPACT_LINE or compact==COMPACT_FULL:
            # remove any trailing whitespace
            text = string.split(text, '\n')
            for j in range(len(text)-1):
              text[j] = string.rstrip(text[j])
            text = string.join(text, '\n')
            # gobble the end of the line
            ((row, _), _, _, file) = ref
            rowtext = string.split(text, '\n')
            if rowtext: rowtext = string.strip(rowtext[0])
            crow = row ; cfile = file
            j = i
            while j and not rowtext:
              j = j - 1
              type2, text2, ref2 = leafs[j]
              if ref2: (_, (crow, _), _, cfile) = ref2
              if crow != row or file != cfile: break
              if type2 == AST_TEXT:
                text2 = string.split(text2, '\n')
                if text2: text2 = text2[-1]
                rowtext = rowtext + string.strip(text2)
              elif type2 == AST_PYEVAL:
                rowtext = 'x'
            if not rowtext:
              text = string.split(text, '\n')
              if text and not string.strip(text[0]):
                text = text[1:]
              text = string.join(text, '\n')
            # gobble beginning of the line
            (_, (row, _), _, file) = ref
            rowtext = string.split(text, '\n')
            if rowtext: rowtext = string.strip(rowtext[-1])
            crow = row ; cfile = file
            j = i + 1
            while j<len(leafs) and not rowtext:
              type2, text2, ref2 = leafs[j]
              if ref2: ((crow, _), _, _, cfile) = ref2
              if crow != row or file != cfile: break
              if type2 == AST_TEXT:
                text2 = string.split(text2, '\n')
                if text2: text2 = text2[0]
                rowtext = rowtext + string.strip(text2)
              elif type2 == AST_PYEVAL:
                rowtext = 'x'
              j = j + 1
            if not rowtext:
              text = string.split(text, '\n')
              if text: text[-1] = string.strip(text[-1])
              text = string.join(text, '\n')
          # space compaction
          if compact==COMPACT_SPACE or compact==COMPACT_FULL:
            text = spyceUtil.spaceCompact(text)
          # update text, if any
          if text: leafs[i] = type, text, ref
          else: 
            del leafs[i]
            i = i -1
        elif type in [AST_PY, AST_PYEVAL]:
          pass
        else:
          raise 'error: unknown AST node type'
        i = i + 1

##################################################
# Output classes
#

class LineWriter:
  "Output class that counts lines written."
  def __init__(self, f, initialLine = 1):
    self.f = f
    self.lineno = initialLine
  def write(self, s):
    self.f.write(s)
    self.lineno = self.lineno + len(string.split(s,'\n'))-1
  def writeln(self, s):
    self.f.write(s+'\n')
  def close(self):
    self.f.close()
  def getLineNumber(self):
    "Return the current line number."
    return self.lineno

class IndentingWriter:
  "Output class that helps with indentation of code."
  # Note: this writer is line-oriented.
  def __init__(self, f, indentSize=2):
    self._f = f
    self._indentSize = indentSize
    self._indent = 0
    self._currentLine = ''
  def close(self):
    if self._indent > 0:
      raise 'unmatched open brace'
    self._f.close()
  def indent(self):
    self._indent = self._indent + 1
  def outdent(self):
    self._indent = self._indent - 1
    if self._indent<0: 
      raise 'unmatched close brace'
  def dumpLine(self, s):
    self._f.write(' '*(self._indent*self._indentSize))
    self._f.write(s)
    self._f.write('\n')
  def write(self, s):
    self._currentLine = self._currentLine + s
    lines = string.split(self._currentLine, '\n')
    for l in lines[:-1]:
      self.dumpLine(l)
    self._currentLine=lines[-1]
  def writeln(self, s=''):
    self.write(s+'\n')
  # remaining methods are defined in terms of writeln(), indent(), outdent()
  def pln(self, s=''):
    self.writeln(s)
  def pIln(self, s=''):
    self.indent(); self.pln(s)
  def plnI(self, s=''):
    self.pln(s); self.indent()
  def pOln(self, s=''):
    self.outdent(); self.pln(s)
  def plnO(self, s=''):
    self.pln(s); self.outdent()
  def pOlnI(self, s=''):
    self.outdent(); self.pln(s); self.indent()
  def pIlnO(self, s=''):
    self.indent(); self.pln(s); self.outdent()

##################################################
# Print out Braced Python
#

class emitBracedPython:
  def __init__(self, out, ast):
    out = LineWriter(out)
    self._spyceRefs = {}
    # text compaction
    self.compact = COMPACT_LINE
    self._gobblelineNumber = 1
    self._gobblelineText = ''
    # do the deed!
    self.emitSpyceRec(out, self._spyceRefs, None, ast['elements'], ast['leafs'][1:])
  def getSpyceRefs(self):
    return self._spyceRefs
  def emitSpyceRec(self, out, spyceRefs, header, elements, leafs):
    if header:
      out.write(header+':{\n')
    def processLevel(el, out=out, spyceRefs=spyceRefs, self=self):
      self.emitSpyceRec(out, spyceRefs, el['leafs'][0][1], el['elements'], el['leafs'][1:])
    try:
      processLevel(elements[SPYCE_GLOBAL_CODE])
      del elements[SPYCE_GLOBAL_CODE]
    except KeyError: pass
    for el in elements.keys():
      processLevel(elements[el])
    if leafs:
      for type, text, ref in leafs:
        line1 = out.getLineNumber()
        if type==AST_TEXT:
          out.write('response.writeStatic(%s)\n' % `text`)
        elif type==AST_PY:
          out.write(text+'\n')
        elif type==AST_PYEVAL:
          out.write('response.writeExpr(%s)\n' % text)
        elif type==AST_COMPACT:
          self.compact = text
        else:
          raise 'error: unknown AST node type'
        line2 = out.getLineNumber()
        if ref:
          for l in range(line1, line2):
            spyceRefs[l] = ref
    if not leafs and not elements:
      out.write('pass\n')
    if header:
      out.write('}\n')

##################################################
# Print out regular Python
#

class BraceConverter:
  "Convert Python with braces into indented (normal) Python code."
  def __init__(self, out):
    self.out = IndentingWriter(out)
    self.prevname = 0
    self.prevstring = 0
    self.dictlevel = 0
  def emitToken(self, type, string):
    if type==token.NAME:
      if self.prevname: self.out.write(' ')
      if self.prevstring: self.out.write(' ')
      self.out.write(string)
    elif type==token.OP:
      if string=='{': 
        if self.prevcolon and not self.dictlevel:
          self.out.plnI()
        else:
          self.dictlevel = self.dictlevel + 1
          self.out.write(string)
      elif string=='}':
        if not self.dictlevel:
          self.out.plnO()
        else:
          self.dictlevel = self.dictlevel - 1
          self.out.write(string)
      else:
        self.out.write(string)
    elif type==token.ERRORTOKEN and string==chr(0):
      self.out.write(' ')
    elif type==token.STRING:
      if self.prevname: self.out.write(' ')
      string  = `eval(string)`  # get rid of multi-line strings
      self.out.write(string)
    elif type==token.NUMBER:
      if self.prevname: self.out.write(' ')
      self.out.write(string)
    else:
      #print type, token.tok_name[type], `string`
      self.out.write(string)
    self.prevname = type==token.NAME
    self.prevstring = type==token.STRING
    self.prevcolon = type==token.OP and string==':'

def emitPython(out, bracedPythonCode, spyceRefs):
  out = LineWriter(out)
  spyceRefs2 = {}
  braceConv = BraceConverter(out)
  def eatToken(type, string, begin, end, _, out=out, braceConv=braceConv, spyceRefs=spyceRefs, spyceRefs2=spyceRefs2):
    try:
      beginrow, _ = begin
      line1 = out.getLineNumber()
      braceConv.emitToken(type, string)
      line2 = out.getLineNumber()
      if spyceRefs.has_key(beginrow):
        for l in range(line1, line2):
          spyceRefs2[l] = spyceRefs[beginrow]
    except:
      raise spyceException.spyceSyntaxError(sys.exc_info()[0])
  try:
    tokenize.tokenize(StringIO(bracedPythonCode).readline, eatToken)
  except tokenize.TokenError, e:
    msg, (row, col) = e
    raise spyceException.spyceSyntaxError(msg)
  return spyceRefs2

# rimtodo: need to also update refs
RE_EMPTY_LINES = re.compile('\n\s+\n', re.M)
RE_CONSECUTIVE_NEWLINE = re.compile('\n\n\n', re.M)
RE_NEWLINE_AFTER_COLON = re.compile(':\n\n', re.M)
def removeConsecutiveNewlines(str):
  str = RE_EMPTY_LINES.sub('\n\n', str)
  str = RE_CONSECUTIVE_NEWLINE.sub('\n\n', str)
  str = RE_NEWLINE_AFTER_COLON.sub(':\n', str)
  return str

def calcRowCol(str, pos):
  lines = string.split(str, '\n')
  row = 1
  while pos > len(lines[0]):
    pos = pos - len(lines[0]) - 1
    del lines[0]
    row = row + 1
  return row, pos

RE_BRACES = re.compile('{|}')
def checkBalancedParens(str, refs):
  m = RE_BRACES.search(str)
  stack = []
  try:
    while m:
      if m.group(0)=='{': stack.append(m)
      else: stack.pop()
      m = RE_BRACES.search(str, m.end())
  except IndexError: 
    row, _ = calcRowCol(str, m.start())
    try: info = refs[row]
    except KeyError: info =None
    raise spyceException.spyceSyntaxError("unbalanced open brace '{'", info)
  if stack:
    m = stack[-1]
    row, _ = calcRowCol(str, m.start())
    try: info = refs[row]
    except KeyError: info =None
    raise spyceException.spyceSyntaxError("unbalanced close brace '}'", info)

##################################################
# Compile spyce files
#

def spyceCompile(buf, filename, sig, server):
  # parse 
  parser = spyceParser2(server)
  ast, libs = parser.processFile(buf, filename, sig)
  parser.finish()
  # optimize the ast
  spyceOptimize(ast)
  # generate braced code
  out = StringIO()
  refs = emitBracedPython(out, ast).getSpyceRefs()
  # then, generate regular python code
  bracedPythonCode = out.getvalue()
  checkBalancedParens(bracedPythonCode, refs)
  out = StringIO()
  refs = emitPython(out, bracedPythonCode, refs)
  return out.getvalue(), refs, libs

if __name__ == '__main__':
  import spyce
  f = open(sys.argv[1])
  spycecode = f.read()
  f.close()
  pythoncode, refs, libs = spyceCompile(spycecode, sys.argv[1], '', spyce.getServer())
  print pythoncode

