#############################################################################
#
#	naming.py - Pyro Naming Service
#
#	This is part of "Pyro" - Python Remote Objects
#	which is (c) Irmen de Jong - irmen@bigfoot.com.
#
#############################################################################


import sys, time
import socket, select
import SocketServer
from Pyro.core import *
import Pyro.errors
import Pyro.protocol
from Pyro.util import Log


PYRONS_VERSION = '1.1'

#############################################################################
#
# The Pyro NameServer Locator.
# Use a broadcast mechanism to find the broadcast server of the NS which
# can provide us with the URI of the NS.
#
#############################################################################

class NameServerLocator:

	def sendSysCommand(self,request,host=None,port=None,trace=0):
		if not port:
			# select the default broadcast port
			port = Pyro.config.PYRO_NS_BC_PORT
		# We must discover the location of the naming service.
		# Pyro's NS can answer to broadcast requests.
		s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
		s.bind(('', 0))
		if not host:
			# use a broadcast
			destination = ('<broadcast>', port)
			s.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
		else:
			# use unicast
			destination = (host, port)
			
		if trace:
			print 'LOCATOR: Searching Pyro Naming Service...'
		try:
			for i in range(Pyro.config.PYRO_BC_RETRIES+1):
				s.sendto(request, destination)  # send to Pyro NS's bc port
				ins,outs,exs = select.select([s],[],[s],Pyro.config.PYRO_BC_TIMEOUT)
				if s in ins:
					reply, fromaddr = s.recvfrom(1000)
					return reply
				if trace and i<Pyro.config.PYRO_BC_RETRIES:
					print 'LOCATOR: Retry',i+1
		except socket.error,e:
			Log.error('NameServerLocator','network error:',e)
			if trace:
				print 'LOCATOR: network error:',e
		Log.error('NameServerLocator','Naming Service not responding')
		raise Pyro.errors.PyroError('Naming Service not responding')
	
	def getNS(self,host=None,port=None,trace=0):
		reply = self.sendSysCommand('location',host,port,trace)
		Log.msg('NameServerLocator','Naming Service found:',reply)
		return NameServerProxy(Pyro.core.PyroURI(reply))


# NOTE: The NameServerProxy class below is hand crafted.
# This is because we want to enforce the default group name on all name
# arguments that are not absolute.
# In doing so, we make sure that each object name is passed as an
# absolute name (from the name space root) too. This is needed because
# the NS has no state and we can't really 'cd' to a group in the NS.

class NameServerProxy:  # originally subclassed from Pyro.core.ObjBase
	def __init__(S,URI):
		S.URI = URI
		S.objectID = URI.objectID
		S.adapter = Pyro.protocol.getProtocolAdapter(S.URI.protocol)
		S.adapter.bindToURI(URI)
		S.sep = Pyro.config.PYRO_NS_GROUPSEP
		S.rootchar = Pyro.config.PYRO_NS_ROOTCHAR
	def ping(S):
		return S.adapter.remoteInvocation('ping',0)
	def register(S,name,URI):
		return S.adapter.remoteInvocation('register',0,S.__expand(name),URI)
	def resolve(S,name):
		return S.adapter.remoteInvocation('resolve',0,S.__expand(name))
	def flatlist(S):
		return S.adapter.remoteInvocation('flatlist',0)
	def unregister(S,name):
		return S.adapter.remoteInvocation('unregister',0,S.__expand(name))
	def createGroup(S,gname):
		return S.adapter.remoteInvocation('createGroup',0,S.__expand(gname))
	def deleteGroup(S,gname):
		return S.adapter.remoteInvocation('deleteGroup',0,S.__expand(gname))
	def list(S,gname):
		return S.adapter.remoteInvocation('list',0,S.__expand(gname))
	def fullName(S,name):
		return S.__expand(name)
	def __expand(S,name):
		if Pyro.config.PYRO_NS_DEFAULTGROUP[0]!=Pyro.config.PYRO_NS_ROOTCHAR:
			raise Pyro.errors.NamingError('default group name is not absolute')
		if name:
			if name[0]==S.rootchar:
				return name
			return Pyro.config.PYRO_NS_DEFAULTGROUP+S.sep+name
		else:
			return Pyro.config.PYRO_NS_DEFAULTGROUP


#############################################################################
#
#	The Name Server (a Pyro Object).
#
#	It has more methods than the ones available in the proxy (above)
#	but that is because the other methods are private and for internal
#	use only.
#
#############################################################################

class NameServer(Pyro.core.ObjBase):
	def __init__(self):
		Pyro.core.ObjBase.__init__(self)
		self.groupsep = Pyro.config.PYRO_NS_GROUPSEP
		self.rootchar = Pyro.config.PYRO_NS_ROOTCHAR
		self.root=NamedTree('<root>',self.groupsep)
		# fix up our default name and groupname
		Pyro.config.PYRO_NS_NAME = string.replace(string.replace(Pyro.config.PYRO_NS_NAME,'.',self.groupsep), ':', self.rootchar)
		Pyro.config.PYRO_NS_DEFAULTGROUP = string.replace(string.replace(Pyro.config.PYRO_NS_DEFAULTGROUP,'.',self.groupsep), ':', self.rootchar)
		# create default groups
		self.createGroup(self.rootchar+'Pyro')
		self.createGroup(Pyro.config.PYRO_NS_DEFAULTGROUP)

	def register(self,name,URI):
		self.validateName(name)
		self.validateURI(URI)
		(group, name)=self.locateGrpAndName(name)
		if isinstance(group,NameValue):
			raise Pyro.errors.NamingError('parent is no group')
		try:
			group.newleaf(name,URI)
			Log.msg('NameServer','registered',name,'with URI',str(URI))
		except KeyError:
			Log.warn('NameServer','name already exists:',name)
			raise Pyro.errors.NamingError('name already exists')

	def unregister(self,name):
		self.validateName(name)
		(group, name)=self.locateGrpAndName(name)
		try:
			group.cutleaf(name)
			Log.msg('NameServer','unregistered',name)
		except KeyError:
			raise Pyro.errors.NamingError('name not found')

	def resolve(self,name):
		self.validateName(name)
		try:
			branch=self.getBranch(name)
			if isinstance(branch,NameValue):
				return branch.value
			else:
				Log.msg('NameServer','attempt to resolve groupname:',name)
				raise Pyro.errors.NamingError('attempt to resolve groupname')
		except KeyError:
			raise Pyro.errors.NamingError('name not found')
		except AttributeError:
			raise Pyro.errors.NamingError('group not found')

	def flatlist(self):
		# return a dump
		r=self.root.flatten()
		for i in range(len(r)):
			r[i]=(self.rootchar+r[i][0], r[i][1])
		return r

	def ping(self):
		pass  # just accept a remote invocation.

	# --- hierarchical naming support
	def createGroup(self,groupname):
		self.validateName(groupname)
		(parent,name)=self.locateGrpAndName(groupname)
		try:
			parent.newbranch(name)
			Log.msg('NameServer','created group',groupname)
		except KeyError,x:
			raise Pyro.errors.NamingError(x)

	def deleteGroup(self,groupname):
		self.validateName(groupname)
		(parent,name)=self.locateGrpAndName(groupname)
		try:
			parent.cutbranch(name)
			Log.msg('NameServer','deleted group',name)
		except KeyError:
			raise Pyro.errors.NamingError('group not found')
		except ValueError:
			Log.msg('NameServer','attempt to deleteGroup a non-group:',groupname)
			raise Pyro.errors.NamingError('is no group')
			
	def list(self,groupname):
		if not groupname:
			groupname=self.rootchar
		self.validateName(groupname)
		try:
			return self.getBranch(groupname).list()
		except KeyError:
			raise Pyro.errors.NamingError('group not found')
		except AttributeError:
			Log.msg('NameServer','attempt to list a non-group:',groupname)
			raise Pyro.errors.NamingError('is no group')
			

	# --- private methods follow
	def locateGrpAndName(self,name):
		# ASSUME name is absolute (from root) (which is required here)
		idx=string.rfind(name,self.groupsep)
		if idx>=0:
			# name is hierarchical
			grpname=name[:idx]
			name=name[idx+1:]
			try:
				return (self.getBranch(grpname), name)
			except KeyError:
				raise Pyro.errors.NamingError('(parent)group not found')
		else:
			# name is in root
			return (self.root, name[1:])

	def getBranch(self,name):
		# ASSUME name is absolute (from root) (which is required here)
		name=name[1:]
		if name:
			return reduce(lambda x,y: x[y], string.split(name,self.groupsep), self.root)
		else:
			return self.root

	def validateName(self,name):
		if len(name)>=1 and name[0]==self.rootchar:
			if ('' not in string.split(name,self.groupsep)):
				for i in name:
					if ord(i)<33 or ord(i)>126:
						raise Pyro.errors.NamingError('invalid name')
				return
			else:
				raise Pyro.errors.NamingError('invalid name')
		else:
			raise Pyro.errors.NamingError('name is not absolute')

	def validateURI(self,URI):
		if isinstance(URI, Pyro.core.PyroURI):
			return
		try:
			u = Pyro.core.PyroURI('')
			u.reinitFromString(URI)
		except:
			raise Pyro.errors.NamingError('invalid URI')

	def publishURI(self, uri):
		uri=str(uri)
		try:
			f=open(Pyro.config.PYRO_NS_URIFILE,'w')
			f.write(uri+'\n'); f.close()
			print 'URI written to:',Pyro.config.PYRO_NS_URIFILE
			print 'URI is:',uri
			Log.msg('NameServer','URI written to',Pyro.config.PYRO_NS_URIFILE)
		except:
			Log.warn('NameServer','Couldn\'t write URI to',Pyro.config.PYRO_NS_URIFILE)

#############################################################################
#
#	NamedTree data type. Used for the hierarchical naming service.
#
#############################################################################
class NameValue:
	def __init__(self, name,value=None):
		self.name=name
		self.value=value

class NamedTree:
	def __init__(self, name, sep):
		self.name=name
		self.branches={}
		self.sep=sep
	def newbranch(self,name):
		if name in self.branches.keys():
			raise KeyError,'duplicate name'
		t = NamedTree(name,self.sep)
		self.branches[name]=t
		return t
	def newleaf(self,name,value=None):
		if name in self.branches.keys():
			raise KeyError,'duplicate name'
		l = NameValue(name,value)
		self.branches[name]=l
		return l
	def cutleaf(self,name):
		if isinstance(self.branches[name], NameValue):
			del self.branches[name]
		else:
			raise ValueError,'not a leaf'
	def cutbranch(self,name):
		if isinstance(self.branches[name], NamedTree):
			del self.branches[name]
		else:
			raise ValueError,'not a branch'
	def __getitem__(self,name):
		return self.branches[name]
	def list(self):
		l=[]
		for (k,v) in self.branches.items():
			if isinstance(v, NamedTree):
				l.append( (k,0) )	# tree
			elif isinstance(v, NameValue):
				l.append( (k,1) )	# leaf
			else:
				raise ValueError('corrupt tree')
		return l
	def flatten(self,prefix=''):
		flat=[]
		for (k,v) in self.branches.items():
			if isinstance(v, NameValue):
				flat.append( (prefix+k, v.value) )
			elif isinstance(v, NamedTree):
				flat.extend(v.flatten(prefix+k+self.sep))
		return flat
				

		
#############################################################################
#
#	The Persistent Name Server (a Pyro Object).
#	NOTE: This implementation is LIKELY TO CHANGE because it's EXPERIMENTAL.
#	NOTE: That's why I changed the name to Beta...
#	NOTE: Oompf, this code is getting OLD. It's way out of date with the
#	      regular NS above (which recently got hierarchical too...)
#
#############################################################################

class BetaPersistentNameServer(NameServer):

	def __init__(self,dbfile=None,name=None):
		if not name:
			name=Pyro.config.PYRO_NS_NAME
		NameServer.__init__(self)
		# change to a persistent naming service
		import anydbm,os,marshal
		self.dbfile = os.path.join(Pyro.config.PYRO_STORAGE,dbfile or 'NS.database')
		self.database = anydbm.open(self.dbfile,'c')
		self.nsname = name
		try:
			# Try to figure our persistent GUID by looking in the database
			uri=Pyro.core.PyroURI(self.database[self.nsname])
			self.setPersistentGUID(uri.objectID)
		except KeyError:
			pass  # Keep the default GUID
		if hasattr(self.database,'sync'):
			self.dbsync = self.database.sync
		else:
			self.dbsync = lambda:None
		self.dbsync()
	def checkForInitialCreate(self):
		# Check if the name is already in the DB. If it isn't, this is
		# the first time the DB is used... (probably just created)
		try:
			uri = self.database[self.nsname]
			return 0
		except KeyError:
			return 1
	def register(self,name,URI):
		NameServer.register(self,name,str(URI))
		self.dbsync()
	def unregister(self,name):
		NameServer.unregister(self,name)
		self.dbsync()
	def resolve(self,name):
		try:
			return Pyro.core.PyroURI(self.database[name])
		except KeyError:
			Log.msg('BetaPersistentNameServer','name not found:',name)
			raise Pyro.errors.NamingError('name not found')
	def flatlist(self):
		# return the contents of our database
		db = {}
		for k in self.database.keys():
			db[k]=self.database[k]
		return db


#############################################################################
#
# The broadcast server which listens to broadcast requests of clients who
# want to discover our location, or send other system commands.
#
#############################################################################


class BroadcastServer(SocketServer.UDPServer):

	nameServerURI = ''	# the Pyro URI of the Name Server

	def server_activate(self):
		self.shutdown=0				# should the server loop stop?
		self.preferredTimeOut=3.0	# preferred timeout for the server loop
			
	def setNS_URI(self,URI):
		self.nameServerURI=str(URI)
	def keepRunning(self, keep):
		self.ignoreShutdown = keep	# ignore shutdown requests (i.e. keep running?)

	def bcCallback(self,ins):
		for i in ins:
			i.handle_request()
	
		
class bcRequestHandler(SocketServer.BaseRequestHandler):

	def handle(self):
		Log.msg('BroadcastServer','incoming request from',str(self.client_address[0]))
		# request is a simple string
		cmd = self.request[0]
		if cmd=='location':
			# somebody wants to know our location, give them our URI
			self.request[1].sendto(self.server.nameServerURI,self.client_address)
		elif cmd=='shutdown':
			# we should die!?
			if self.server.ignoreShutdown:
				Log.msg('BroadcastServer','Shutdown ignored.')
				self.request[1].sendto('Shutdown request denied',self.client_address)
			else:
				Log.msg('BroadcastServer','Shutdown received.')
				print 'BroadcastServer received shutdown request... will shutdown shortly...'
				self.request[1].sendto('Will shut down shortly, when no more requests arrive',self.client_address)
				self.server.shutdown=1
		else:
			Log.warn('BroadcastServer','Invalid command ignored:',cmd)


#############################################################################

def startServer(nsport=0, bcport=0, keep=0, persistent=0, dbfile=None):
	if not nsport:
		nsport=Pyro.config.PYRO_NS_PORT
	if not bcport:
		bcport=Pyro.config.PYRO_NS_BC_PORT
	Pyro.core.initServer()
	PyroDaemon = Pyro.core.Daemon(port=nsport,norange=1)
	if persistent:
		ns=BetaPersistentNameServer(dbfile,Pyro.config.PYRO_NS_NAME)
		PyroDaemon.useNameServer(ns)
		if ns.checkForInitialCreate():
			# NS database has just been created,
			# should enter the NS itself in the database
			NS_URI=PyroDaemon.connect(ns,Pyro.config.PYRO_NS_NAME)
		else:
			# NS is already in the database
			NS_URI=PyroDaemon.connectPersistent(ns,Pyro.config.PYRO_NS_NAME)
	
	else:
		ns=NameServer()
		PyroDaemon.useNameServer(ns)
		NS_URI=PyroDaemon.connect(ns,Pyro.config.PYRO_NS_NAME)
	
	bcserver = BroadcastServer(('',bcport),bcRequestHandler)
	bcserver.keepRunning(keep)
	if keep:
		print 'Will ignore shutdown requests.'
	else:
		print 'Will accept shutdown requests.'

	ns.publishURI(NS_URI)
	bcserver.setNS_URI(NS_URI)
	Log.msg('NS daemon','This is Pyro Naming Service V'+PYRONS_VERSION+'.')
	if persistent:
		Log.msg('NS daemon','Persistent mode, database is in',ns.dbfile)
	Log.msg('NS daemon','Starting on',PyroDaemon.hostname,'port',
	  PyroDaemon.port, ' broadcast server on port',bcport)

	print 'Naming Service started.'

	# I use a timeout here otherwise you can't break gracefully on Windoze
	while not bcserver.shutdown:
		try:
			PyroDaemon.handleRequests(bcserver.preferredTimeOut,[bcserver],bcserver.bcCallback)
		except KeyboardInterrupt:
			Log.warn('NS daemon','shutdown on user break signal')
			print 'Shutting down on user break signal.'
			bcserver.shutdown=1
		except:
			import traceback
			(exc_type, exc_value, exc_trb) = sys.exc_info()
			tb = traceback.format_exception(exc_type, exc_value, exc_trb)[-5:]
			out=''
			for line in tb:
				out=out+line
			Log.error('NS daemon', 'Unexpected exception, type',exc_type,
				'\n--- partial traceback of this exception follows:\n',
				out,'\n--- end of traceback')
			print '*** Exception occured!!! Partial traceback:'
			print out
			print '*** Resuming operations...'

	Log.msg('NS daemon','Shut down gracefully.')
	print 'Naming Service gracefully stopped.'

def main():
	if __name__!='__main__':
		del sys.argv[0]	# remove '-c' arg
	import Pyro.util
	Args = Pyro.util.ArgParser()
	Args.parse(sys.argv,'hkp:b:d:')
	try:
		Args.getOpt('h')
		print 'Usage: ns [-h] [-k] [-p port] [-b port] [-d [databasefile]]'
		print '  where -p  = NS server port'
		print '        -b = NS broadcast port'
		print '        -k = keep, do not respond to shutdown requests'
		print '        -d = use persistent database, provide optional filename'
		print '        -h = print this help'
		raise SystemExit
	except KeyError:
		pass
	port = int(Args.getOpt('p',Pyro.config.PYRO_NS_PORT))
	bcport = int(Args.getOpt('b',Pyro.config.PYRO_NS_BC_PORT))
	try:
		dbfile = Args.getOpt('d')
		persistent = 1
	except KeyError:
		persistent = 0
		dbfile = None

	try:
		Args.getOpt('k')
		keep=1
	except KeyError:
		keep=0


	Args.printIgnored()
	if Args.args:
		print 'Ignored arguments:',string.join(Args.args)
	startServer(port,bcport,keep,persistent,dbfile)

