#############################################################################
#
#	core.py - Pyro Core Library
#
#	This is part of "Pyro" - Python Remote Objects
#	which is (c) Irmen de Jong - irmen@bigfoot.com.
#
#############################################################################

import sys, time, string
import socket
import traceback
import Pyro.protocol
import Pyro.util
from Pyro.errors import *
from types import MethodType, BuiltinMethodType

Log=Pyro.util.Log

#### Remote Invocation Flags (bit flags) ####

RIF_Varargs  = (1<<0)		# for '*args' syntax
RIF_Keywords = (1<<1)		# for '**keywords' syntax


#############################################################################
#
#	ObjBase		- Server-side object implementation base class
#	              or master class with the actual object as delegate
#
#############################################################################

class ObjBase:
	def __init__(self):
		self.objectGUID=Pyro.util.getGUID()
		self.delegate=self
	def GUID(self):
		return self.objectGUID
	def delegateTo(self,delegate):
		self.delegate=delegate
	def setPersistentGUID(self,GUID):
		# only to be used for persistent objects which should get a fixed GUID
		self.objectGUID=GUID
	def setDaemon(self, daemon):
		self.daemon=daemon
	def getDaemon(self):
		return self.daemon
	def getProxy(self):
		return self.daemon.getProxyForObj(self)
	def getAttrProxy(self):
		return self.daemon.getAttrProxyForObj(self)
	def Pyro_dyncall(self, method, flags, args):
		# find the method in this object, and call it with the supplied args.
		keywords={}
		if flags & RIF_Keywords:
			# reconstruct the varargs from a tuple like
			#  (a,b,(va1,va2,va3...),{kw1:?,...})
			keywords=args[-1]
			args=args[:-1]
		if flags & RIF_Varargs:
			# reconstruct the varargs from a tuple like (a,b,(va1,va2,va3...))
			args=args[:-1]+args[-1]
		return apply(getattr(self.delegate,method),args,keywords)
	# remote getattr/setattr support:
	def remote_hasattr(self, attr):
		try:
			attr = getattr(self,attr)
			if type(attr) in (MethodType, BuiltinMethodType):
				return 'method'
		except:
			pass
		return 'attr'
	def remote_getattr(self, attr):
		return getattr(self, attr)
	def remote_setattr(self, attr, value):
		return setattr(self, attr, value)


#############################################################################
#
#	PyroURI		- Pyro Universal Resource Identifier
#
#	This class represents a Pyro URI (which consists of four parts,
#	a protocol identifier, a hostname, a portnumber, and an object ID.
#	
#	The URI can be converted to a string representation (str converter).
#	The URI can also be read back from such a string (reinitFromString).
#	The URI can be initialised from its parts (init).
#	The URI can be initialised from a string directly, if the init
#	 code detects a ':' and '/' in the host argument (which is then
#        assumed to be a string URI, not a hostname).
#
#############################################################################

class PyroURI:
	def __init__(self,host,objectID=0,port=0,protocol='PYRO'):
		# If the 'host' arg contains ':' and '/', assume it's a URI string.
		if string.find(host,':')>=0 and string.find(host,'/')>=0:
			self.reinitFromString(host)
		else:
			self.host=host
			if port:
				self.port=port
			else:
				self.port=Pyro.config.PYRO_PORT
			self.protocol=protocol
			self.objectID=objectID
	def __str__(self):
		return self.protocol+'://'+self.host+':'+str(self.port)+  \
							'/'+str(self.objectID)
	def __repr__(self):
		return '<PyroURI \''+str(self)+'\'>'
	def init(self,host,objectID,port=0,protocol='PYRO'):
		self.host=host
		self.objectID=objectID
		if port:
			self.port=port
		else:
			self.port=Pyro.config.PYRO_PORT
		self.protocol=protocol
	def optimizeLocalhost(self):
		if self.host == Pyro.protocol.getHostname():
			self.host = 'localhost'
	def reinitFromString(self,arg):
		arg=string.strip(arg)
		_argsave=arg
		colon = string.find(arg,':')
		try:
			if colon>=1:
				self.protocol = arg[:colon]
				if arg[colon+1:colon+3]=='//':
					arg=arg[colon+3:]
					colon = string.find(arg,':')
					if colon>=1:
						self.host=arg[:colon]
						arg=arg[colon+1:]
						slash=string.find(arg,'/')
						if slash>=1:
							self.port=int(arg[:slash])
							self.objectID=arg[slash+1:]
							return
					else:
						slash=string.find(arg,'/')
						if slash>=1:
							self.host=arg[:slash]
							self.port=Pyro.config.PYRO_PORT
							self.objectID=arg[slash+1:]
							return
		except ValueError, IndexError:
			pass
		Log.error('PyroURI','illegal URI format passed: '+_argsave)
		raise URIError('illegal URI format')


#############################################################################
#
#	DynamicProxy	- dynamic Pyro proxy
#
#	Can be used by clients to invoke objects for which they have no
#	precompiled proxy.
#
#############################################################################

def getProxyForURI(URI):
	return DynamicProxy(URI)
def getAttrProxyForURI(URI):
	return DynamicProxyWithAttrs(URI)

class DynamicProxy:
	def __init__(self, URI):
		self.URI = URI
		self.objectID = URI.objectID
		self.adapter = None				# delay adapter binding
		self._name=[]
	def __getattr__(self, name):
		# allows it to be safely pickeled
		if name != "__getinitargs__":
			self._name.append(name)
			return self.__invokePYRO__
		raise AttributeError()
	def __repr__(self):
		return "<"+self.__class__.__name__+" instance at "+hex(id(self))+">"
	def __str__(self):
		return repr(self)

	# Note that a slightly faster way of calling is this:
	#  instead of proxy.method(args...) use proxy('method',args...)
	def __call__(self, method, *vargs, **kargs):
		self._name.append(method)
		return self.__invokePYRO__
	def __invokePYRO__(self, *vargs, **kargs):
		if self.adapter is None:
			# delayed adapter binding.
			self.adapter = Pyro.protocol.getProtocolAdapter(self.URI.protocol)
			self.adapter.bindToURI(self.URI)
		name = self._name[-1]
		del self._name[-1]
		return self.adapter.remoteInvocation(name,
						RIF_Varargs|RIF_Keywords, vargs, kargs)
	# pickling support:
	def __getstate__(self):
		temp = {}
		for key in self.__dict__.keys():
			if key != "adapter":
				temp[key] = self.__dict__[key]
		return temp
	def __setstate__(self, value):
		for key in value.keys():
			self.__dict__[key] = value[key]
		self.adapter = None

	
class DynamicProxyWithAttrs(DynamicProxy):
	def __init__(self, URI):
		# first set the list of 'local' attrs for __setattr__
		self.__dict__["_local_attrs"] = ("_local_attrs","URI", "objectID", "adapter", "_name", "attr_cache")
		DynamicProxy.__init__(self, URI)
		self.attr_cache = {}
	def remote_getattr(self, attr, value=0):
		if value: self._name.append("remote_getattr")
		else: self._name.append("remote_hasattr")
		return self.__invokePYRO__(attr)
	def findattr(self, attr):
		if self.attr_cache.has_key(attr):
			return self.attr_cache[attr]
		# look it up and cache the value
		self.attr_cache[attr] = self.remote_getattr(attr)
		return self.attr_cache[attr]
	def __setattr__(self, attr, value):
		if attr in self.__dict__["_local_attrs"]:
			self.__dict__[attr]=value
		else:
			result = self.findattr(attr)
			if result=="attr":
				self._name.append("remote_setattr")
				return self.__invokePYRO__(attr,value)
			else:
				raise AttributeError('not an attribute')
	def __getattr__(self, attr):
		# allows it to be safely pickeled
		if attr != "__getinitargs__":
			result=self.findattr(attr)
			if result=="method":
				self._name.append(attr)
				return self.__invokePYRO__
			elif result:
				return self.remote_getattr(attr, 1)
		raise AttributeError()
		

#############################################################################
#
#	Daemon		- server-side Pyro daemon
#
#	Accepts and dispatches incoming Pyro method calls.
#
#############################################################################

class Daemon(Pyro.protocol.TCPServer):
	def __init__(self,protocol='PYRO',port=0,norange=0):
		self.hostname = Pyro.protocol.getHostname()	
		if port:
			self.port = port
		else:
			self.port = Pyro.config.PYRO_PORT
		self.implementations={}
		self.NameServer = None
		self.protocol = protocol
		self.adapter = Pyro.protocol.getProtocolAdapter(protocol)
		self.adapter.setDaemon(self)
		if norange:
			portrange=1
		else:
			portrange=Pyro.config.PYRO_PORT_RANGE
		for i in range(portrange):
			try:
				Pyro.protocol.TCPServer.__init__(self, DaemonSlave(), self.port)
				return
			except socket.error:
				self.port = self.port+1
		Log.error('Daemon','Couldn\'t start Pyro daemon')
		raise DaemonError('Couldn\'t start Pyro daemon')
	
	def __del__(self):
		# server shutting down, unregister all known objects in the NS
		for o in self.implementations.keys():
			self.NameServer.unregister(self.implementations[o][1])

	def __str__(self):
		return '<Pyro Daemon on '+self.hostname+':'+str(self.port)+'>'

	def useNameServer(self,NS):
		self.NameServer=NS

	def connectPersistent(self,object,name):
		# enter the (object,name) in the known implementations dictionary
		self.implementations[object.GUID()]=(object,name)
		object.setDaemon(self)
		# Notice: we don't register the object with the NS;
		# it's already persistent there!

	def disconnectPersistent(self,object):
		# Don't unregister with naming service, only remove it from the impl. directory
		try:
			del self.implementations[object.GUID()]
		finally:
			object.setDaemon(None)

	def getProxyForObj(self, obj):
		return DynamicProxy( PyroURI(self.hostname, obj.GUID(),
					  protocol=self.protocol, port=self.port) )
	def getAttrProxyForObj(self, obj):
		return DynamicProxyWithAttrs ( PyroURI(self.hostname, obj.GUID(),
					  protocol=self.protocol, port=self.port) )
	
	def connect(self, object, name=None):
		# enter the (object,name) in the known implementations dictionary
		self.implementations[object.GUID()]=(object,name)
		object.setDaemon(self)
		URI = PyroURI(self.hostname, object.GUID(),
					  protocol=self.protocol, port=self.port)
		# if not transient, register the object with the NS
		if name:
			if self.NameServer:
				self.NameServer.register(name, URI)
			else:
				Log.warn('Daemon','connecting object without naming service specified:',name)
		return URI

	def disconnect(self,object):
		try:
			if self.implementations[object.GUID()][1]:
				# only unregister with NS if it had a name (was not transient)
				self.NameServer.unregister(self.implementations[object.GUID()][1])
			del self.implementations[object.GUID()]
		finally:
			object.setDaemon(None)

	def handleError(self,conn):			# overridden from TCPServer
		(exc_type, exc_value, exc_trb) = sys.exc_info()
		if exc_type in(Pyro.errors.ProtocolError, socket.error):
			# ProtocolError or socket.error, shut down the connection
			# XXX is this wat we want???
			Log.error('Daemon','Due to network error: shutting down connection with',conn.addr)
			self.removeConnection(conn)
		else:
			# This code used to distinguish between 'accidents' and
			# 'on purpose' exceptions. No longer. Every exception
			# is treated the same.
			tb = traceback.format_exception(exc_type, exc_value, exc_trb)[-5:]
			out=''
			for line in tb:
				out=out+line
			Log.warn('Daemon', 'Exception during processing of request from',
				conn.addr,' type',exc_type,
				'\n--- partial traceback of this exception follows:\n',
				out,'\n--- end of traceback')
			self.adapter.returnException(conn.sock,exc_value)
	
class DaemonSlave:
	def handleRequest(self,conn):
		try:
			self.daemon.adapter.handleInvocation(conn.sock)
		except SocketClosedError:
			# other side has closed connection
			self.daemon.removeConnection(conn)


#############################################################################
#
#	Client/Server Init code
#
#############################################################################

import time, string

def _initGeneric_pre():
	if Pyro.config.PYRO_TRACELEVEL == 0: return
	try:
		out='\n'+'-'*60+' NEW SESSION\n'+time.ctime(time.time())+ \
			'   Pyro Initializing, version '+Pyro.PYRO_VERSION+'\n'
		Log.raw(out)
	except IOError,e:
		sys.stderr.write('PYRO: Can\'t write the tracefile '+Pyro.config.PYRO_LOGFILE.name+'\n'+e)

def _initGeneric_post():
	if Pyro.config.PYRO_TRACELEVEL == 0: return
	try:
		out='Configuration settings are as follows:\n'
		for item in dir(Pyro.config):
			if item[0:4]=='PYRO':
				out=out+item+' = '+str(Pyro.config.__dict__[item])+'\n'
		out=out+'Init done.\n'+'-'*70+'\n'
		Log.raw(out)
	except IOError,e:
		pass

def initClient(banner=1):
	_initGeneric_pre()
	if Pyro.config.PYRO_TRACELEVEL >0: Log.raw('This is initClient.\n')
	_initGeneric_post()
	if banner:
		print 'Pyro Client Initialized. Using Pyro V'+Pyro.PYRO_VERSION
	
def initServer(banner=1):
	_initGeneric_pre()
	if Pyro.config.PYRO_TRACELEVEL >0: Log.raw('This is initServer.\n')
	_initGeneric_post()
	if banner:
		print 'Pyro Server Initialized. Using Pyro V'+Pyro.PYRO_VERSION

