#!/usr/local/bin/python
"""MIME control module -
Copyright (C) 1999-2000 Michael P. Reilly,  All rights reserved

The purpose of this module is to make a clear merge between
mimetools.Message and MimeWriter.MimeWriter.  Unfortunately, the two
are not compatible.

There are two published classes in this module: Field and MIME_document.
The MIME_document class encapsulates all the data for a single MIME
file, including multipart documents.  The Field class encapsulates a
header field.

Usage:
Loading a MIME document from a file
  f = MIME_document()
  f.load(open('mail.mime'))
  f.dump(open('mail.mime', 'w'))

Creating a new MIME document
  f = MIME_document('''\
  Hi there, mom.  I just wanted to tell you that I enjoyed Christmas.  Thank
  you for the mittens, they fit well.
  ''',
    type='text/plain',
    From='arcege@shore.net', To='my.mom@home.net'
  )
  print f.read()
  f['subject'] = 'Many thanks and holiday wishes'  # add a header
  f.write('Love,\n\tMichael')  # I forgot the signature
  from_addr = str(f['from'])   # who is sending it?
  majortype = f['content-type'].majortype()  # 'text'

Making a multipart (and using the MIME_recoder subclass):
  # send a picture
  imagefilename = 'card.gif'
  imagedata = open(imagefilename).read()
  image = MIME_recoder_f(
      imagedata,
      ( MIMEField('content-type', 'image/gif', name=imagefilename),
        MIMEField('content-length', len(imagedata)),
        MIMEField('content-disposition', 'attachment', filename=imagefilename),
      )
    )
  # now we encode it
  image = image.encode('base64')
  # make a new enclosing document instance
  # get f's headers to put into the new document
  f_fields = f.values()
  # we need to remove the content-type header field
  f_fields.remove('content-type')
  f_fields.remove('content-length')
  f_fields.remove('content-disposition')
  g = MIME_document( ( f, image ), fields=f_fields )

To prevent MIME_document from adding a "MIME-Version" field (in not
present in the data), then set NoVersion=1 (to the constructor or
load() method).

If the document is a multipart file, then the instance mimics UserList,
otherwise it mimics a file object (without a fileno method).  All
instances act like a dictionary to access the header fields.

"""

__version__ = '1.5.1'

import UserList, mimetools, string, types
from UserDict import UserDict
try:
  from cStringIO import StringIO
except:
  from StringIO import StringIO
  try:
    StringIO.truncate
  except AttributeError:
    # add truncate method to the class (now available everywhere in the appl)
    def truncate(self, size=0):
      if self.buflist:
        self.buf = self.buf + string.joinfields(self.buflist, '')
        self.buflist = []
      buf = self.buf[:size]
      self.__init__(buf)
    StringIO.truncate = truncate
    del truncate

No_Encoding = (None, 'None', '7bit', '8bit', 'binary')

class MimeError(Exception):
  """Signals that an error occured in the operation of the MIME document.
For example, accessing a multipart doc as a file."""

class TempFile:
  import os
  __os_remove = os.remove
  del os

  def __init__(self, mode='wb+'):
    import tempfile
    self.name = name = tempfile.mktemp()
    self.mode = mode
    self.fp = open(name, mode)
    if not hasattr(self.fp, 'truncate'):
      self.truncate = self._truncate

  def __del__(self):
    TempFile.close(self)
    TempFile.remove(self)

  def __getattr__(self, attrname):
    if attrname == 'fp':
      return self.__dict__.get(attrname, None)
    elif self.__dict__.has_key('fp'):
      return getattr(self.__dict__.get('fp'), attrname)
    else:
      raise AttributeError, attrname

  def close(self):
    fp = self.fp
    if fp:
      fp.close()
      self.fp = None

  def remove(self):
    try:
      self.__os_remove(self.name)
    except:
      pass

  # add truncate method to the class (now available everywhere in the appl)
  def _truncate(self, size=0):
    self.fp.seek(0)
    if size == 0:
      buf = ''
    else:
      buf = self.fp.read(size)
    self.close()
    self.fp = open(self.name, self.mode)
    if buf:
      self.fp.write(buf)
      self.fp.seek(0)

def _isnumber(value):
  """Is the value a Python number?"""
  return (
    isinstance(value, types.IntType) or
    isinstance(value, types.LongType)
  )

def capfield(name):
  """Capitalize the words (separated by hyphens) of a field."""
  parts = map(string.capitalize, string.splitfields(name, '-'))
  return string.joinfields(parts, '-')

class Field(UserDict):
  """Represent a rfc822 header.  The constructor can take three forms of
input:
  1) a single argument which is an instance of the class or subclass.
  2) a single string which is the formatted output of the field:
    Field('Content-type: text/plain; charset=us-ascii')
  3) a field name and data with attributes:
    Field('content-type', 'text/plain', charset='us-ascii')
Field name is case independant for all operations.
Field attributes are accessed through keys, as a dictionary.
Operations:
  int(self)         - return the field data as an integer or
                      raise ValueError
  str(self)         - return the field data as a string
  repr(self)        - return the formatted output as in (2) above
  hash(self)        - hash based on the field name
  cmp(self, o)      - compare based on the field name
Methods
  add_attr(self, d) - add a dictionary of addtributes to the field

If attribute value is None, then just print the attribute key, for example:
`Field('content-name', 'mimecntl.py', inline=None)` yields
  'Content-Name: mimecntl.py; inline'
"""
  maxcol = 72  # column wrap
  def __init__(self, _fieldname, _fielddata=None, fieldsep=', ', **attrs):
    UserDict.__init__(self)
    self.sep = fieldsep
    if isinstance(_fieldname, Field):
      o_attrs = _fieldname.copy()
      o_attrs.update(attrs)
      (_fieldname, _fielddata, attrs) = \
        (_fieldname.name, _fieldname.field_data, o_attrs)
    elif _fieldname in (None, '') and _fielddata is None:
      _fieldname = ''
    elif (isinstance(_fielddata, types.ListType) or
          isinstance(_fielddata, types.TupleType)):
      _fielddata = tuple(_fielddata)
    elif _fielddata == None and isinstance(_fieldname, types.StringType):
      if string.lower(_fieldname[:5]) == 'from ':
        _fieldname, _fielddata = string.splitfields(_fieldname, ' ', 1)
        _fieldname = '_from'
      else:
        _fieldname, _fielddata = string.splitfields(_fieldname, ':', 1)
      _fieldname = string.strip(_fieldname)
      _fielddata = string.strip(_fielddata)
      _fielddata, o_attrs = self.parse_data(_fielddata)
      o_attrs.update(attrs)
      attrs = o_attrs
    else:
      _fielddata = string.strip( str(_fielddata) )
      _fielddata, o_attrs = self.parse_data(_fielddata)
      o_attrs.update(attrs)
      attrs = o_attrs
    self.name = string.lower(_fieldname)
    self.field_data = _fielddata
    self.add_attr(attrs)

  def parse_data(self, data):
    return data, {}
  def __repr__(self):
    if self.name == '_from':
      return 'From %s' % self.field_data
    else:
      return '%s: %s' % (capfield(self.name), self.field_data)

  def __int__(self):
    return int(self.field_data)
  def __str__(self):
    if isinstance(self.field_data, types.TupleType):
      data = string.join(map(str, self.field_data), self.sep)
    else:
      data = str(self.field_data)
    return data

  # these are for "list(field)" and "tuple(field)" calls
  def __len__(self):
    if isinstance(self.field_data, types.TupleType):
      return len(self.field_data)
    else:
      return 1
  def __getitem__(self, index):
    if isinstance(index, types.IntType):
      if isinstance(self.field_data, types.TupleType):
        return self.field_data[index]
      elif index == 0:
        return self.field_data
      else:
        raise IndexError, index
    else:
      return UserDict.__getitem__(self, index)

  def __hash__(self):
    return hash(self.name)
  def __cmp__(self, other):
    if isinstance(other, Field):
      return cmp(self.name, other.name)
    else:
      return cmp(self.name, string.lower(str(other)))
    return cmp(self.name, other)
  def __rcmp__(self, other):
    if isinstance(other, Field):
      return cmp(other.name, self.name)
    else:
      return cmp(string.lower(str(other)), self.name)

  def match(self, other):
    if isinstance(other, Field):
      return cmp(self.field_data, other.field_data)
    else:
      data, attr = self.parse_data(other)
      return cmp(self.field_data, data)
  def contained_in(self, list):
    for item in list:
      if self.match(item) == 0:
        break
    else:
      return 0
    return 1
  def add_attr(self, attrs):
    for name, value in attrs.items():
      if ' ' in name or '\t' in name:
        raise ValueError, 'no whitespace allowed in names'
      self[string.lower(name)] = value

class MIMEField(Field):
  """MIMEField(...) - a subclass of Field(...) with the same arguments.
Methods:
  majortype(self)   - assume field is content-type and return the
                      string before the '/'
  minortype(self)   - assume field is content-type and return the
                      string after the '/'
"""

  def parse_data(self, data):
    attrs = {}
    if ';' in data:
      parts = string.splitfields(data, ';')
      data = parts[0]
      for i in parts[1:]:
        try:
          key, datum = string.splitfields(i, '=', 1)
          try:
            datum = eval(datum, {'__builtins__': {}} )
          except:
            pass
        except ValueError:
          key, datum = i, None
        attrs[ string.strip(key) ] = datum
      del parts
    parts = string.split(data, self.sep)
    if len(parts) >= 2:
      data = tuple(parts)
    return data, attrs

  def __repr__(self):
    if self.field_data == None:
      return ''
    if self.name == '_from':
      current_line = 'From %s' % str(self)
    else:
      current_line = '%s: %s' % (capfield(self.name), str(self))
    data = []
    # add the attributes
    for name, value in self.items():
      if value == None:
        piece = str(name)
      elif isinstance(value, types.IntType):
        piece = '%s=%d' % (name, int(value))
      else:
        piece = '%s="%s"' % (name, str(value))
      if len(current_line) + len(piece) + 2 > self.maxcol:
        data.append(current_line + ';')
        current_line = '\t%s' % piece
      else:
        current_line = '%s; %s' % (current_line, piece)
    data.append(current_line)
    return string.joinfields(data, '\n')

  def majortype(self):
    if self.field_data == None:
      return ''
    pos = string.find(self.field_data, '/')
    if pos >= 0:
      return string.lower(self.field_data[:pos])
    else:
      return ''
  def minortype(self):
    if self.field_data == None:
      return ''
    pos = string.find(self.field_data, '/')
    if pos >= 0:
      return string.lower(self.field_data[pos+1:])
    else:
      return ''

dummyfield = MIMEField(None, None)

class MIME_document:
  """A class to encapsulate a MIME document.  Allows parsing and
recreation (some whitespace differences from original).
Instance construction from the following means:
  1) another instance of the class or a subclass
  2) a file object containing a MIME document
  3) data parts, as follows:
    a) if data is a sequence other than a string, then make the
document multipart, if the items in the sequence are not MIME documents
then make them instance of the same class
    b) otherwise the data is assumed to be the message body
In data mode (3), header fields are entered either as keyword arguments
or as Field instances in the second argument.

Constructor arguments:
  data      - (described above)
  fields    - sequence of Field instances or formatted header fields
  type      - content-type value
  encoding  - content-transfer-encoding value
  NoVersion - if 1, do not add MIME-Version field, default is 0
  **kwargs  - other header fields

By default, if no "MIME-Version" header exists, one is added.  This can
be prevented by passing NoVersion=1 to the constructor or load() method.

Data Access:
Headers: accessed as a dictionary.
Multipart data: accessed as a list, with the exception of:
  sort() and count() and index() methods.
Non-multipart data: accessed as a file object with exception of:
  fileno(), close(), flush(), isatty()
The non-multipart data is writable as well as readable.

Operatins:
  if self:  - always true
  len(self)
    composite  - number of items (message/* types are always one)
    other      - byte length of body
  self[index]
    index is string - return header or KeyError
    composite  - returns item in list
    other      - raise MimeError
  self[index] = data
    index is string - set header
    index is Field instance - set header (with index modified by data)
    composite  - replace item in list
    other      - raise MimeError
  del self[index]
    index is string - remove header
    index is Field instance - remove header
    composite  - remove item in list
    other      - raise MimeError
  self[lo:hi]
    composite  - return items in list
    other      - raise MimeError
  self[lo:hi] = [data]
    composite  - replace items in list
    other      - raise MimeError
  del self[lo:hi]
    composite  - remove items in list
    other      - raise MimeError
  str(str)     - return string representation (see dump method below)

Methods
  self.append(data)
    multipart     - append data (as instance) to list
    message       - call method on enclosing instance
    other         - raise MimeError
  self.insert(pos, data)
    multipart     - insert data (as instance) into list
    message       - call method on enclosing instance
    other         - raise MimeError
  self.remove(data)
    multipart     - remove data from list
    message       - call method on enclosing instance
    other         - raise MimeError
  self.items()           - return (ordered) name, field pairs of headers
  self.keys()            - return (ordered) header names 
  self.values()          - return (ordered) header fields
  self.has_key(name)     - return true if name is a header
  self.tell()
    multipart     - raise MimeError
    message       - call method on enclosed instance
    other         - return current position in body
  self.seek(offset, whence=0)
    multipart     - raise MimeError
    message       - call method on enclosed instance
    other         - change current position in body
  self.read(size=-1)
    multipart     - raise MimeError
    message       - call method on enclosed instance
    other         - read from body
  self.readline(size=-1)
    multipart     - raise MimeError
    message       - call method on enclosed instance
    other         - read line from body
  self.readlines()
    multipart     - raise MimeError
    message       - call method on enclosed instance
    other         - read lines from body
  self.write(data)
    multipart     - raise MimeError
    message       - call method on enclosed instance
    other         - write data into body at current position
  self.writelines(lines)
    multipart     - raise MimeError
    message       - call method on enclosed instance
    other         - write lines into body at current position
  self.convert(type=None, encoding=None) -
    Change document type based on content_type and transfer_encoding fields
  self.dump(file=None) -
    Display the entire document recursively.  If a multipart and no
    content boundary exists, create one.  If a file object is given,
    write the document string to the file.  Otherwise return the string.
    [str(self) is the same as self.dump()]
  self.load(data, NoVersion=0) -
    Parse the data into a MIME document and replace the existing
    instance data with the parsed data.  If data is a file instance,
    read the file for the data.
"""

  leader = """\
This is a multimedia message in MIME format.  If you are reading this
prefix, your mail reader does not understand MIME. You may wish to look
into upgrading to a newer version of your mail reader.
"""
  normfield_class = Field
  mimefield_class = MIMEField

  def __init__( self,
                data='',
                fields=[],
                type=None,
                encoding=None,
                NoVersion=0,
                **kwargs):

    if self.check_is_file(data):
      self.headers = []
      self.multipart = self.composite = 0
      self.data = None
      self.load(data, NoVersion=NoVersion)  # load from the file object
      return

    # copy the data
    if isinstance(data, MIME_document):
      # if the data's leader value is the Class's leader,
      # don't replicate it
      if data.leader is not self.__class__.leader:
        self.leader = data.leader
      fields = data.headers[:]
      if data.multipart:
        data = data.data[:]               # this is a list
      elif data.composite == 1:
        data = self.__class__(data.data)  # copy the instance
      else:
        data.data.seek(0)
        data = data.data.read()

    # add the header fields
    self.headers = []
    for value in list(fields) + kwargs.items():
      if not isinstance(value, Field):
        if isinstance(value, types.TupleType):
          value = apply(self.field_class, value)
        else:
          value = self.field_class(value)
      self.headers.append(value)

    # the type=... and encoding=... arguments override values in fields
    # if the desired field isn't in the headers, then use the dummyfield
    # to allow the majortype method to be called
    if type is not None:
      if isinstance(type, Field):
        ctype = MIMEField(type)
      else:
        ctype = MIMEField('content-type', type)
      try:    pos = self.headers.index('content-type')
      except: self.headers.append( ctype )
      else:   self.headers[pos] = ctype
    else:
      try:    pos = self.headers.index('content-type')
      except: ctype = dummyfield
      else:   ctype = self.headers[pos]
    if encoding is not None:
      if isinstance(type, Field):
        ttype = MIMEField(type)
      else:
        ttype = MIMEField('content-transfer-encoding', type)
      try:    pos = self.headers.index('content-transfer-encoding')
      except: self.headers.append( ttype )
      else:   self.headers[pos] = ttype
    else:
      try:    pos = self.headers.index('content-transfer-encoding')
      except: ttype = dummyfield
      else:   ttype = self.headers[pos]

    isseq = (isinstance(data, types.TupleType) or
             isinstance(data, types.ListType) or
             isinstance(data, UserList.UserList))
    if isseq and ctype is dummyfield:
      ctype = self.field_class('content-type', 'multipart/mixed')
      self.headers.append( ctype )
    elif isseq and ctype.majortype() != 'multipart':
      raise ValueError, 'content-type must be multipart'

    # populate self.data at least
    multipart, composite = self.convert(ctype, ttype)
    if data:
      if multipart == composite == 1:
        data = list(data)
        for i in xrange(len(data)):
          if not isinstance(data[i], MIME_document):
            data[i] = self.__class__(data[i])
        self.data = data

      elif composite == 1:
        if isinstance(data, MIME_document):
          self.data = data
        elif data:
          self.data.load(data)
      else:
        self.data.write(data)
        self.data.seek(0)

    if not NoVersion:  # we want to add a MIME-Version: field by default
      self.normalize_version()

  def field_class(self, name, data=None):
    if string.lower(str(name)[:8]) == 'content-':
      fieldata = self.mimefield_class(name, data)
    else:
      fieldata = self.normfield_class(name, data)
    return fieldata

  def normalize_version(self, default_version=1.0):
    howmany = self.headers.count('mime-version')
    if howmany == 0:
      # don't put it first since there may be a 'From <username>' header
      self.headers.insert(1,
        self.field_class('mime-version', default_version)
      )
    else:
      # remove all the MIME-Version: headers and only add the first
      try:
        p = []
        while 1:
          pos = self.headers.index('mime-version')
          p.append ( self.headers[pos] )
          del self.headers[pos]
      except:
        self.headers.insert(0, p[0])

  # method callbacks to access headers as a dictionary and
  # composite body as a list
  # if the index is a python number, assume list access
  def __nonzero__(self):  # to override __len__ method
    return 1
  def __len__(self):
    if self.composite:
      return len(self.data)
    else:
      pos = self.data.tell()
      self.data.seek(0, 2)
      flen = self.data.tell()
      self.data.seek(pos)
      return flen
  def __getitem__(self, index):
    if _isnumber(index):
      if not self.composite:
        raise MimeError, 'use file operations'
      return self.data[index]
    elif isinstance(index, types.StringType) or isinstance(index, Field):
      try:
        pos = self.headers.index(index)
      except:
        raise KeyError, index
      else:
        return self.headers[pos]
    else:
      raise TypeError, index
  def __setitem__(self, index, data):
    if _isnumber(index):
      if not self.composite:
        raise MimeError, 'use file operations'
      if not isinstance(data, MIME_document):
        data = self.__class__(data)
      self.data[index] = data
      return
    elif isinstance(index, types.StringType):
      if not isinstance(data, Field):
        data = self.field_class(index, data)
      try:
        pos = self.headers.index(index)
      except:
        self.headers.append( data )
      else:
        self.headers[pos] = data
    elif isinstance(index, Field):
      if data is None:
        data = self.field_class(index)
      elif not isinstance(data, Field):
        data = self.field_class(index.name, data)
      try:
        pos = self.headers.index(index)
      except:
        self.headers.append( data )
      else:
        self.headers[pos] = data
    else:
      raise TypeError, index
  def __delitem__(self, index):
    if _isnumber(index):
      if not self.composite:
        raise MimeError, 'use file operation'
      del self.data[index]
    elif isinstance(index, types.StringType) or isinstance(index, Field):
      self.headers.remove(index)
    else:
      raise TypeError, index

  def __getslice__(self, lo, hi):
    if self.composite != 1:
      raise MimeError, 'use file operations'
    return self.data[lo:hi]
  def __setslice__(self, lo, hi, sequence):
    sequence = list(sequence)
    if self.composite != 1:
      raise MimeError, 'use file operations'
    elif self.multipart:
      for i in xrange(len(sequence)):
        if not isinstance(sequence[i], MIME_document):
          sequence[i] = self.__class__(sequence[i])
      self.data[lo:hi] = sequence
    else:  # message/*
      if len(sequence) != 1:
        raise IndexError, 'only one item allowed'
      if not isinstance(sequence[0], MIME_document):
        sequence[0] = self.__class__(sequence[0])
      self.data[:] = sequence
  def __delslice__(self, lo, hi):
    if self.composite != 1:
      raise MimeError, 'use file operations'
    del self.data[lo:hi]
  def append(self, data):
    if self.composite != 1:
      raise MimeError, 'use file operations'
    if not isinstance(data, MIME_document):
      data = self.__class__(data)
    self.data.append( data )
  def insert(self, position, data):
    if self.composite != 1:
      raise MimeError, 'use file operations'
    if not isinstance(data, MIME_document):
      data = self.__class__(data)
    self.data.insert(position, data)
  def remove(self, data):
    if self.composite != 1:
      raise MimeError, 'use file operations'
    # only works with instance itself, since there is no cmp method
    self.data.remove(data)
  def reverse(self):
    if self.composite != 1:
      raise MimeError, 'use file operations'
    self.data.reverse()

  def items(self):
    return map(lambda h: (h.name, h), self.headers)
  def keys(self):
    return map(lambda h: h.name,      self.headers)
  def values(self):
    return map(None,                  self.headers)
  def has_key(self, key):
    try:    self.headers.index(key)
    except: return 0
    else:   return 1

  # methods to treat document body as a file
  def tell(self):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    return self.data.tell()
  def seek(self, offset, whence=0):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    return self.data.seek(offset, whence)
  def read(self, size=-1):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    return self.data.read(size)
  def readline(self, size=-1):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    return self.data.readline()
  def readlines(self):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    return self.data.readlines()
  def write(self, data):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    return self.data.write(data)
  def writelines(self, lines):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    return self.data.writelines(lines)
  def truncate(self, size=0):
    if self.multipart or (self.composite == 1 and self.data.multipart):
      raise MimeError, 'use sequence operations'
    self.seek(0)
    data = self.read(size)
    rc = self.data.truncate()
    self.seek(0)
    self.write(data)
    return rc

  def convert(self, type=dummyfield, encoding=dummyfield):
    if type is None:
      self.multipart = self.composite = 0
      self.data = self._create_internal_file()
    elif type.majortype() == 'multipart':
      self.multipart = self.composite = 1
      self.data = []
    elif (type.majortype() == 'message' and
          (encoding is None or encoding.contained_in(No_Encoding))
         ):
      self.multipart = 0
      self.composite = 1
      self.data = self.__class__(NoVersion=1)
    elif (type.majortype() == 'message'):
      self.multipart = 0
      self.composite = 2
      self.data = self._create_internal_file()
    else:
      self.multipart = self.composite = 0
      self.data = self._create_internal_file()
    return self.multipart, self.composite

  # to match the un/pickler interface
  def dump(self, file=None):
    """Display the entire document recursively.  If a multipart and
no content boundary exists, create one from mimetools.choose_boundary().
"""
    if self.multipart:  # find boundary
      try:
        ctype = self['content-type']
        if not ctype.has_key('boundary'):
          ctype['boundary'] = mimetools.choose_boundary()
      except ValueError:
        raise MimeError, 'no content-type on multipart'
    fields = map(repr, self.values())
    headers = string.joinfields(fields, '\n')
    if self.multipart:
      boundary = ctype['boundary']
      parts = [ self.leader ]
      for part in self.data:
        parts.append( part.dump() )
      body = self._join_on_boundary(parts, boundary)
      retstr = "%s\n\n%s" % (headers, body)
    else:
      if self.composite == 1:  # if composite == 2, then encoded
        data_text = self.data.dump()  # this is another instance
      else:
        pos = self.data.tell()
        self.data.seek(0)
        data_text = self.data.read()
        self.data.seek(pos)
      retstr = "%s\n\n%s" % (headers, data_text)
    if file is not None:
      file.write(retstr)
    else:
      return retstr
  # __str__ receives only the instance, no other argument, so the string
  # is always returned
  __str__ = dump

  def load(self, data, NoVersion=0):
    if self.check_is_file(data):
      # rewind if possible and get data from file
      try:    data.seek(0)
      except: pass
      data = data.read()
    try:
      headers, body = string.splitfields(data, '\n\n', 1)
    except ValueError:
      raise ValueError, 'not a MIME document'
    headers = string.splitfields(headers, '\n')
    if headers[0] == '':
      del headers[0]
    l = []
    # scan for line continuations and append to previous line
    for header in headers:
      if header[0] in string.whitespace:
        l[-1] = string.rstrip(l[-1]) + ' ' + string.lstrip(header)
      else:
        l.append(header)
    headers = map(self.field_class, l)
    del l
    try:    pos = headers.index('content-type')
    except: ctype = dummyfield
    else:   ctype = headers[pos]
    try:    pos = headers.index('content-transfer-encoding')
    except: ttype = dummyfield
    else:   ttype = headers[pos]
    multipart = composite = 0
    if ctype.majortype() == 'multipart':
      multipart = composite = 1
    elif ctype.majortype() == 'message':
      multipart = 0
      if ttype.contained_in(No_Encoding):
        composite = 1
      else:
        composite = 2

    if composite and multipart:  # a multipart
      # return a list of subparts, with codes for each
      parts = self._split_on_boundary(body, ctype['boundary'], '--')
      body = []
      endfound = 0
      self.leader = None
      # make the subparts into class instances too
      for (code, part) in parts:
        if code == -1:
          raise ValueError, 'must be a properly formatted MIME document'
        elif code == 0 and self.leader is None:
          self.leader = part
        elif code == 1 or code == 2:
          if code == 2:
            endfound = 1
          p = self.__class__(NoVersion=1)
          p.load('\n' + part)
          body.append(p)
      if not endfound:
        raise ValueError, 'must be a properly formatted MIME document'

    elif composite == 1 and not multipart:  # an encoded message
      subpart = self.__class__(NoVersion=1)
      subpart.load('\n' + body)
      body = subpart
    else:
      stream = self._create_internal_file()
      stream.write(body)
      body = stream
      if self.data:
        self._delete_internal_file(self.data)
    self.headers[:] = headers
    self.data = body
    self.multipart = multipart
    self.composite = composite
    if not NoVersion:
      self.normalize_version()
    if not multipart and not composite:
      self.data.seek(0)

  def check_is_file(self, obj):
    """Determine if the given object is a "file".  Because of how
the class works, the 'seek' method must exist.  It is possible,
and probable, that subclasses will want to use StringIO instances or
buffers."""
    return (hasattr(obj, 'fileno') and
            hasattr(obj, 'seek') and
            callable(getattr(obj, 'seek')))

  # internal storage processing
  def _create_internal_file(self, initialcontents=None):
    if initialcontents:
      return StringIO(initialcontents)
    else:
      return StringIO()
  def _delete_internal_file(self, file):
    pass

  def _join_on_boundary(self, parts, boundary, sep='--'):
    boundary = sep + boundary
    body = string.joinfields(parts, ('\n%s\n' % boundary))
    retstr = "%s\n%s--\n" % (body, boundary)
    return retstr

  def _split_on_boundary(self, body, boundary, sep='--'):
    """Unfortunately, the specs for MIME say that the boundary can have
whitespace before the end of boundary.  This splits on that case, without
resorting to regular expressions.  Return a list of tuples, each tuple
is a code number and the data in that section.  Code numbers are as
follow:
0 - outside boundary (front of or after)
1 - part between two boundaries
2 - last part (terminated by end boundary)
-1 - outside boundary, but no end boundary found; error condition
"""
    boundary = sep + boundary
    boundlen = len(boundary)
    parts = []
    lastpos = lastscan = 0
    foundend = 0

    body = '\n' + body
    while 1:
      pos = string.find(body, '\n' + boundary, lastscan)
      if pos == -1:
        if foundend:
          parts.append( (0,  body[lastpos:]) )
        else:
          parts.append( (-1, body[lastpos:]) )  # did not find end of bound
        break
      else:
        lastscan = pos + 1
        lpos     = pos + 1 + boundlen
        try:
          epos = string.find(body, '\n', lpos)
          f = string.strip(body[lpos:epos])
          if f not in (sep, ''):
            raise ValueError("bad boundary")
        except ValueError:  # ignore bad boundaries
          pass
        else:
          if foundend or lastpos == 0:  # leader or trailer
            parts.append( (0, body[lastpos:pos]) )
          elif f == sep:
            foundend = 1
            parts.append( (2, body[lastpos:pos]) )
          else:
            parts.append( (1, body[lastpos:pos]) )
          lastpos = epos + 1
    return parts

class MIME_document_f(MIME_document):
  def _create_internal_file(self, initialcontents=None):
    file = TempFile()
    if initialcontents:
      file.write(initialcontents)
      file.seek(0)
    return file
  def _delete_internal_file(self, file):
    if hasattr(file, 'remove'):
      file.remove()

class Recoder_mixin:
  """Allow for encoding and decoding of the document as per content-
transfer-encoding header.
"""

  def decode(self, encoding=None):
    retval = self.__class__( self, fields=self.values(), NoVersion=1 )
    if self.multipart:
      retval[:] = []
      try:
        code = self['content-transfer-encoding']
        if not code.contained_in(No_Encoding):
          encoding = str(code)
      except:
        pass
      for subpart in self:
        retval.append( subpart.decode(encoding) )

    else:  # even 'message' types are decoded
      retval.truncate()
      try:
        if self.has_key('content-transfer-encoding'):
          encoding = self['content-transfer-encoding']

        if str(encoding) not in No_Encoding:
          pos = self.tell()
          if self.composite == 2:
            self.seek(0)
            source = self._create_internal_file()
            dest = self._create_internal_file()
            block = self.read(512)
            while block:
              source.write(block)
              block = self.read(512)
            mimetools.decode(source, dest, string.lower(str(encoding)))
            dest.seek(0)
            retval.convert(self['content-type'], '8bit')
            retval.data.load(dest.read())
          elif self.composite == 1:
            raise SystemError, 'do not decode'
          else:
            self.seek(0)
            mimetools.decode(self, retval, string.lower(str(encoding)))
            retval.seek(0)
          self.seek(pos)  # reset the file pointer
          retval['content-transfer-encoding'] = '8bit'

        else:
          raise SystemError, 'do not decode'
      except SystemError, why:
        retval = self  # don't decode
    return retval

  def encode(self, encoding=None):
    retval = self.__class__( self, NoVersion=1 )
    if self.multipart:
      del retval[:]
      if encoding is None:
        try:
          encoding = str(self['content-transfer-encoding'])
        except:
          pass
      for subpart in self:
        retval.append( subpart.encode(encoding) )

    else:  # even 'message' types are encoded
      retval.truncate()
      try:
        try:
          code = self['content-transfer-encoding']
          if not code.contained_in(No_Encoding):
            raise SystemError, 'already encoded'
        except KeyError:
          pass

        if encoding not in No_Encoding:
          encoding_field = MIMEField('content-transfer-encoding', encoding)
          if self.composite == 2:
            raise SystemError, 'already encoded'  # should be an assert
          elif self.composite:
            source = self._create_internal_file(self.dump())
            dest = self._create_internal_file()
            mimetools.encode(source, dest, string.lower(str(encoding)))
            retval.convert(self['content-type'], encoding_field)
            block = dest.read(512)
            while block:
              retval.write(block)
              block = dest.read(512)
          else:
            pos = self.tell()
            self.seek(0)
            mimetools.encode(self, retval, string.lower(str(encoding)))
            retval.seek(0)
            self.seek(pos)
          retval[encoding_field] = None

        else:
          raise SystemError, 'do not encode'
      except SystemError, why:
        retval = self  # don't encode
    return retval

class MIME_recoder(Recoder_mixin, MIME_document):
  pass
class MIME_recoder_f(Recoder_mixin, MIME_document_f):
  pass


def _test():
  import tempfile
  f = MIME_document('Hi there\n')
  g = MIME_document("What's Up?\n",
          type=MIMEField('content-type', 'text/html') )
  h = MIME_document( (f, g) )
  i = MIME_document( ('hi\n', h) )
  print 'Combined document:'
  print i

  # load a new document into the existing f
  try:
    try:
      test_multi = TempFile()
      test_multi_fname = test_multi.name
      #test_multi_fname = tempfile.mktemp()
      #test_multi = open(test_multi_fname, 'w')
      test_multi.write( str(MIME_document('Hello there\n')) )
      test_multi.flush()
      test_multi.close(0)
    except:
      pass
    else:
      try:
        old_h_image = str(h)
        f.load(test_multi_fname)
        if str(h) == old_h_image:
          print 'Error: embedded document not changed correctly'
      except IOError:
        print 'Cannot continue test, file not available'
  finally:
    try:
      os.remove(test_multi_fname)
    except:
      pass

if __name__ == '__main__':
  _test()

