#!/usr/bin/env python
########################################################################
""" wing-install.py -- Simple installer for wing

Copyright (c) 1999-2001, Archaeopteryx Software, Inc.  All rights reserved.

Written by Stephan R.A. Deibel and John P. Ehresman

--------------------
Recent Modifications

$Log: wing-install.py,v $
Revision 1.9  2001/09/14 20:33:55  sdeibel
Bumped version info for 1 pt 1 b 7 r 2

Revision 1.8  2001/08/31 13:50:05  sdeibel
Bumped version numbers and added start at release instructions

Revision 1.7  2001/07/11 15:50:01  sdeibel
Truncated huge revision logs

Revision 1.6  2001/07/11 04:34:12  sdeibel
Added utility to update copyright dates and fixed the many files that
were outdated in this regard

Revision 1.5  2001/06/27 21:56:48  sdeibel
Prep for release 1 pt 1 b 6

Revision 1.4  2001/05/11 15:46:29  sdeibel
Bumped versions to 1 point 1 b 5

Revision 1.3  2001/04/20 22:53:59  sdeibel
Fixed installation on machines with tar 1.12 or earlier

Revision 1.2  2001/04/03 00:20:55  sdeibel
Now include py152 with the dist on Linux

Revision 1.1  2001/03/30 00:00:33  sdeibel
Moved all build files into new directory, added code to generate the
binary dist tar file using the new build scripts, and misc fixes

Revision 1.41  2001/03/28 16:45:44  sdeibel
Clarified need for and where to get a license, and updated all version
numbers and dates

Revision 1.40  2001/03/01 02:40:22  sdeibel
Bumped version numbers

Revision 1.39  2001/02/26 14:50:01  jpe
Changes to handle wing.py updates.

Revision 1.38  2001/02/20 20:30:11  sdeibel
Fixed debugger failure to prune stack in binary dist and prep
for release 1pt1 b2

"""
########################################################################

import sys
import os
import string
import stat
import grp

kDefaultPrefix = '/usr/local'
kTerminalWidth = 75
kVerbose = 0
kVersion = "1.1b7"
kBuild = "2"
kSourceDistFilename = 'source-package-%s-%s.tar' % (kVersion, kBuild)
kBinaryDistFilename = 'binary-package-%s-%s.tar' % (kVersion, kBuild)

kFilelistLocalName = 'file-list.txt'

kBuildRootOption = '--rpm-build-root'
kMultiUserGroupOption = '--multi-user-group'
kMultiUserDirOption = '--multi-user-dir'
kWinghomeOption = '--winghome'
kBinDirOption = '--bin-dir'

def AskYesOrNo(msg):
  """ Ask question with yes or no answer.  Returns true for yes, false for no.
  Message should not include '?'. """

  PrintMsg(msg + ' (Y/N)? ', add_newline = 0)
  line = sys.stdin.readline()
  if string.lower(line[0]) == 'y':
    return 1
  else:
    return 0

def FindWrapPos(text):
  """ Find a word wrap position.  Text must not contain a '\n'. """

  if len(text) < kTerminalWidth:
    return None

  # Find last space before kTerminalWidth
  pos = kTerminalWidth - 1
  while pos >= 0:
    if text[pos] in string.whitespace:
      return pos
    pos = pos - 1
    
  # No space before kTerminalWidth, so find first after it
  pos = kTerminalWidth
  while pos < len(text):
    if text[pos] in string.whitespace:
      return pos
    pos = pos + 1
   
  return None

def PrintMsg(msg, wrap = 1, add_newline = 1, ostream = sys.stdout, indent2 = '     '):
  """ Print message to ostream, wrapping and adding newline if requested. """

  line_seq = string.split(msg, '\n')
  for i in range(0, len(line_seq)):
    line = line_seq[i]
     
    wrap_pos = FindWrapPos(line)
    while wrap_pos != None and wrap_pos > len(indent2):
      ostream.write(line[:wrap_pos])
      ostream.write('\n')
      line = indent2 + line[wrap_pos + 1:]
      wrap_pos = FindWrapPos(line)

    ostream.write(line)
    if i < len(line_seq) - 1 or add_newline:
      ostream.write('\n')

def ErrorExit(msg, exit_code = -1):
  """ Display error message and exit. """

  PrintMsg(msg, ostream = sys.stderr)
  sys.exit(exit_code)

def AskForInputLine(msg, default = '', ending_punc = '?'):
  """ Ask for a directory with the given prompt / default.  Return result. """

  if default != '':
    msg = '%s (default = %s)' % (msg, default)
  msg = msg + ending_punc + ' '
  PrintMsg(msg, add_newline = 0)
  line = sys.stdin.readline()
  if string.strip(line) == '':
    return default
  else:
    return line

def AskForDirectory(msg, default = ''):
  """ Ask for a directory with the given prompt / default.  Return result. """

  line = AskForInputLine(msg, default, ending_punc = '?')
  dir = os.path.expanduser(string.strip(line))
  if dir == '':
    dir = default
  if not os.path.exists(dir):
    if AskYesOrNo('%s does not exist, create it' % dir):
      RecursiveMakeDir(dir)

  if not os.path.isdir(dir):
    ErrorExit('Required directory does not exist, please run wing-install again')
  return dir

def UnpackTar(tar_file, target_dir, gzipped = 0):
  """ Unpack tar file into given directory.  gzipped should be set to 1 if tar file
  is compressed with gzip.  Returns list of file names unpacked. """

  # Unpack archive
  uid = os.getuid()
  gid = os.getgid()
  options = ('-x -v -k --directory="%s" --file="%s"' % (target_dir, tar_file))
  if gzipped:
    options = '-z ' + options
  tar_cmd = 'tar ' + options
  redirect_cmd = 'sh -c "export UMASK=0022; %s 2>&1"' % tar_cmd
  if kVerbose:
    PrintMsg("Tar command is %s" % redirect_cmd)
  child_out = os.popen(redirect_cmd, 'r')

  # Set ownership and group on all files (we don't use tar --group and --owner
  # because that doesn't work on all versions of tar)
  cmd = 'chown -R %s.%s %s' % (str(uid), str(gid), target_dir)
  if os.system(cmd) != 0:
    PrintMsg("Warning: Failed to change file owner and group in installation")
  else:
    PrintMsg("Set ownership of installation to user=%s, group=%s" % (str(uid), str(gid)))

  # Get list of files in the archive  
  file_list = []
  try:
    line = child_out.readline()
    while line != '':
      if line[:len('tar:')] == 'tar:':
        fields = string.split(line, ':')
        if len(fields) == 4 and string.strip(fields[3]) == 'File exists':
          PrintMsg("The file '%s' already exists and won't be overwritten" 
                   % string.strip(fields[1]))
      else:
        name = os.path.join(target_dir, string.strip(line))
        if kVerbose:
          PrintMsg('Writing %s' % name)
        file_list.append(name)
      
      line = child_out.readline()

  except OSError:
    ErrorExit("Failed to expand %s" % tar_file)
  else:
    if uid == 0:
      for name in file_list:
        try:
          os.chown(name, uid, gid)
        except OSError:
          pass
        
    return file_list
  
def CreateSymLink(target, alias, relative = 1):
  """ Creates symlink; ask if we should overwrite any current file or link with name 
  of the alias. """

  if os.path.exists(alias):
    if os.path.islink(alias):
      refd = os.readlink(alias)
      if not os.path.isabs(refd):
        refd = os.path.abspath(os.path.join(os.path.dirname(alias), refd))
      if os.path.exists(refd) and os.path.samefile(refd, target):
        if kVerbose:
          PrintMsg('%s symbolic link already exists' % alias)
        return
      msg = 'A link named %s (-> %s) exists, overwrite' % (alias, refd)
    elif os.path.isdir(alias):
      if len(os.listdir(alias)) != 0:
        ErrorExit("Unable to create sym link %s because a non-empty"
                  "directory with the same name exists" % alias)
      msg = 'An empty directory named %s exists, overwrite' % alias
    else:
      msg = 'A file named %s exists, overwrite' % (alias)
      
    res = AskYesOrNo(msg)
    if not res:
      return
    if kVerbose:
      PrintMsg('Removing %s' % alias)
    if os.path.isdir(alias):
      os.rmdir(alias)
    else:
      os.remove(alias)

  # Figure out the relative alias if one is requested and target is an abs path
  if relative and os.path.isabs(target):
    abs_alias = os.path.abspath(alias)
    common = os.path.commonprefix((target, abs_alias))
    while len(common) != 0 and common[-1] != os.sep:
      common = common[:-1]
    rel_target = target[len(common):]
    head, tail = os.path.split(abs_alias[len(common):])
    while head != '':
      rel_target = os.path.join(os.pardir, rel_target)
      head, tail = os.path.split(head)

    target = rel_target

  if kVerbose:
    PrintMsg('Creating symbolic link %s (-> %s)' % (alias, target))
  os.symlink(target, alias)

def SetWingHomeInit(wing_dir, filename):
  """ Setup winghome initialization in given script file. """

  if kVerbose:
    PrintMsg('Setting up WINGHOME in %s' % filename)

  kXformMap = {
    'WINGHOME=$DEVEL_WINGHOME': ('WINGHOME=%s' % wing_dir),
    "WINGHOME=os.environ['DEVEL_WINGHOME']": ("WINGHOME='%s'" % wing_dir),
    'WINGVERSION=$DEVEL_WINGVERSION': ('WINGVERSION=%s' % kVersion),
    'WINGPYTHON=$DEVEL_WINGPYTHON': ('WINGPYTHON=%s/bin/PyCore/python' % wing_dir),
    '#PYTHONHOME=SET_BY_INSTALLER': ('PYTHONHOME=%s/bin/PyCore' % wing_dir),
  }

  file = open(filename, 'ra')
  lines = file.readlines()
  file.close()

  for i in range(0, len(lines)):
    sline = string.strip(lines[i])
    if kXformMap.has_key(sline):
      prefix_len = string.find(lines[i], sline)
      lines[i] = prefix_len * ' ' + kXformMap[sline] + '\n'
 
  file = open(filename, 'wa')
  file.writelines(lines)
  file.close()

def FindArg(argv, name, remove = 1):
  """ Find value for a two arg command line sequence.  Args are removed from
  argv if remove is true. """

  for i in range(0, len(argv) - 1):
    if argv[i] == name:
      val = argv[i + 1]
      if remove:
        del argv[i:i + 2]
      return val

  return None
  
def RecursiveMakeDir(dir_name):
  """ Recursively make all directories.  Exits with an error if a something other
  than a directory already exists for any component. """

  assert dir_name != ''
  # Ensure lib dir exists and if it's not chack to see if it's a file and the create
  if not os.path.isdir(dir_name):
    if os.path.exists(dir_name):
      ErrorExit("A file named %s exists, please move or choose a different install-directory"
                % dir_name)
    parent_dir, local_name = os.path.split(dir_name)
    RecursiveMakeDir(parent_dir)
    try:
      os.mkdir(dir_name, 0755)
    except OSError:
      ErrorExit('Unable to create directory %s, check permissions' % dir_name)

def GetFileInfoTuple(filename):
  """ Returns filename, type, size, & mtime of file on disk. """

  filename = os.path.abspath(filename)
  st = os.stat(filename)
  mtime = str(st[stat.ST_MTIME])
  size = str(st[stat.ST_SIZE])
  if os.path.islink(filename):
    ftype = 'symlink'
  elif os.path.isdir(filename):
    ftype = 'directory'
  else:
    ftype = 'file'

  return filename, ftype, size, mtime


def WriteFilelist(filelist_name, version, option_list, file_info_list):
  """ Write the first two lines of the filelist file. """

  file = open(filelist_name, 'wa')

  file.write("Wing IDE file list version %s\n" % version)
  file.write("Options: %s\n" % string.join(option_list, ' '))
  for info in file_info_list:
    file.write(string.join(info, ' ') + '\n')

  file.close()

def ReadFilelist(filelist_name):
  """ Attempts to read a filelist file with the given name.  If sucessful,
  a tuple of (version, option-list, file-list) is
  returned; each entry in file-list if a (filename, file-type, size, mtime)
  tuple.  If no such file exists, (kVersion, [], []) is returned. """

  empty_info = (kVersion, [], [])
  try:
    filelist = open(filelist_name, 'ra')
  except IOError:
    return empty_info

  line1 = filelist.readline()
  if line1 == '':
    return empty_info
  fields = string.split(string.strip(line1), ' ')
  if len(fields) == 0:
    return empty_info
  version = fields[-1]

  line2 = filelist.readline()
  if line2 == '':
    return empty_info
  option_list = string.split(string.strip(line2), ' ')
  if option_list[0] == 'Options:':
    del option_list[0]

  files = []
  line = filelist.readline()
  while line != '':
    info = string.split(string.strip(line), ' ')
    if len(info) > 4:  # Space(s) in the filename
      filename = string.join(info[:-3], ' ')
      info = (filename, info[-3], info[-2], info[-1])
    files.append(info)

    line = filelist.readline()
    
  filelist.close()

  return version, option_list, files

def RemoveInstalledFiles(winghome, filelist_name):
  """ Removes all files listed as installed in given filelist.txt file,
  including the filelist.txt file itself.  Returns list of files that were
  not deleted. """

  kept_files = []

  info = ReadFilelist(filelist_name)
  if info == None:
    ErrorExit('No %s file found.  This file is required for the uninstall script.'
              ' See the manual for instructions on how to remove an installation'
              ' by hand.' % filelist_name)
  version, option_list, file_list = info

  for filename, ftype, size, mtime in file_list:
    print filename
    if os.path.exists(filename) or os.path.islink(filename):
      if os.path.islink(filename) and not os.path.exists(filename):
        do_delete = 1
      else:
        info = GetFileInfoTuple(filename)
        if info[1] == ftype and (ftype == 'directory'
                                 or (info[2] == size and info[3] == mtime)):
          do_delete = 1
        else:
          do_delete = AskYesOrNo('%s has changed since installation.\n'
                                 'Should it be deleted' % filename)
      if not do_delete:
        kept_files.append(filename)
      else:
        if kVerbose:
          PrintMsg('Deleting %s' % filename)
        if os.path.isdir(filename):
          if len(os.listdir(filename)) == 0:
            os.rmdir(filename)
        else:        
          os.remove(filename)

        # Delete dir if empty and under winghome, but not winghome
        dirname = os.path.dirname(filename)
        while dirname != winghome and string.find(dirname, winghome) == 0 \
              and len(os.listdir(dirname)) == 0:
          parent = os.path.dirname(dirname)
          if kVerbose:
            PrintMsg('Removing %s directory' % dirname)
          os.rmdir(dirname)
          if parent == dirname:
            break
          dirname = parent

  # Remove filelist.txt
  if kVerbose:
    PrintMsg('Deleting %s' % filelist_name) 
  os.remove(filelist_name)

  return kept_files

def DoUninstall(argv):
  """ Unistall. """

  winghome = FindArg(argv, kWinghomeOption)
  if winghome == None:
    ErrorExit(kWinghomeOption + " must be specified when uninstalling")
  winghome = os.path.expanduser(winghome)

  if kVerbose:
    PrintMsg('Removing installation in %s' % winghome)

  filelist_name = os.path.join(winghome, kFilelistLocalName)
  RemoveInstalledFiles(winghome, filelist_name)

  if len(os.listdir(winghome)) == 0:
    if kVerbose:
      PrintMsg('Removing %s directory' % winghome)
    os.rmdir(winghome)
  else:
    PrintMsg('Files remain in %s.  You may want to remove them and remove'
             ' the directory.' % winghome)

  PrintMsg('Removal of installation successful.  Thank you for using'
           ' the Wing IDE.\n')

def SetupMultiUser(winghome, argv):
  """ Setup dir for multi user license lock files. """

  filelist_name = os.path.join(winghome, kFilelistLocalName)
  wing_lock_dir = os.path.join(winghome, 'floating-locks')
  wing_lock_dir = os.path.abspath(wing_lock_dir)

  version, option_list, file_info_list = ReadFilelist(filelist_name)

  # Find & create real dir to put lock files in
  real_dir = FindArg(argv, kMultiUserDirOption)
  if real_dir == None:
    msg = ("An installation supporting a multi-user license requires"
           " a shared directory for lock files.  Where should this"
           " directory be located")
    real_dir = AskForDirectory(msg, wing_lock_dir)
  real_dir = os.path.abspath(real_dir)
  
  RecursiveMakeDir(real_dir)    
  file_info_list.append(GetFileInfoTuple(real_dir))

  if not os.path.exists(wing_lock_dir) \
     or not os.path.samefile(real_dir, wing_lock_dir):
    CreateSymLink(real_dir, wing_lock_dir)
    file_info_list.append(GetFileInfoTuple(wing_lock_dir))

  group = FindArg(argv, kMultiUserGroupOption)
  if group == None:
    msg = ("Enter the name of the user group that will be able to run Wing using"
           " the multi-user license.  This group will be given write permission"
           " to the %s directory. Enter '<everyone>' to make this directory writable"
           " by all users on the system" % real_dir)
    group = AskForInputLine(msg, '<everyone>', ':')
  group = string.strip(group)
  if group == '' or ' ' in group:
    ErrorExit("The group name must be non empty and not contain any spaces")
  elif group == '<everyone>':
    os.chmod(real_dir, 0777)
  else:
    os.chmod(real_dir, 0775)
    try:
      gr_name, gr_passwd, gr_gid, gr_mem = grp.getgrnam(group)
    except KeyError:
      ErrorExit("Group %s does not exist.  Please create it and run again.")

    try:
      res = os.system('chgrp %s %s' % (group, real_dir))
    except OSError:
      res = -1
    if res != 0:
      ErrorExit("Unable to change group on %s to %s.  Please check that the %s"
                " group exists, that chgrp command is available, & that the active user"
                " ID is a member of the group %s."
                % (real_dir, group, real_dir, group))

    try:
      os.system('chmod g+s %s' % real_dir)
    except OSError:
      res = -1
    if res != 0:
      PrintMsg('Unable to set the sticky bit on %s.' % real_dir)

  WriteFilelist(filelist_name, version, option_list, file_info_list)
  PrintMsg("Installation is now configured for use with a multi-user license.")

def DoInstall(argv):
  """ Perform an installation. """

  build_root = FindArg(argv, kBuildRootOption, remove = 0)

  # Find dir for winghome
  winghome = FindArg(argv, kWinghomeOption)
  if winghome == None:
    winghome = AskForDirectory('What directory do you want to install the'
                               ' support files for Wing IDE into',
                               '/usr/local/lib/wingide')

  # Adjust for build root
  if build_root != None:
    if not os.path.isabs(winghome):
      ErrorExit("The winghome directory must be absolute path when using %s." 
                % kBuildRootOption)

  filelist_name = os.path.join(winghome, 'file-list.txt')
  files = []

  # Detect which install to do
  install_binary = '--install-binary' in argv
  install_source = '--install-source' in argv
  if not install_binary and not install_source and not kBuildRootOption in argv:
    src_exists = os.path.exists(kSourceDistFilename) 
    bin_exists = os.path.exists(kBinaryDistFilename)
    if src_exists and bin_exists:
      install_binary = AskYesOrNo("Install binary distribution")
      install_source = AskYesOrNo("Install source distribution")
    else:
      install_source = src_exists
      install_binary = bin_exists

  # Make winghome & bin dirs
  RecursiveMakeDir(winghome)

  # If existing files, ask if we should continue
  if len(os.listdir(winghome)) != 0 and not os.path.exists(filelist_name):
    res = AskYesOrNo("Files exist in %s, overwrite" % winghome)
    if not res:
      ErrorExit("Existing files found in %s" % winghome)

  # Read file info & see what's already installed
  version, option_list, file_info_list = ReadFilelist(filelist_name)

  # if versions are different, we're upgrading so remove old version
  if version != kVersion:
    msg = ('Upgrading from version %s: removing its files.  You will be given a chance to'
           ' keep any file that has been modified; files kept will be renamed to'
           ' <filename>.oldsaved.' % version)
    PrintMsg(msg)
    kept_files = RemoveInstalledFiles(winghome, filelist_name)
    for filename in kept_files:
      os.rename(filename, filename + '.oldsaved')
    #files.extend(kept_files)
    version, option_list, file_info_list = ReadFilelist(filelist_name)

  # Check for already installed packages
  if 'binary' in option_list:
    PrintMsg("Not installing binary distribution because it's already installed")
    install_binary = 0
  if 'source' in option_list:
    PrintMsg("Not installing source distribution because it's already installed")
    install_source = 0

  if install_binary:
    bin_dir = FindArg(argv, kBinDirOption)
    if bin_dir == None:
      bin_dir = AskForDirectory('What directory do you want to install links to the Wing IDE'
                                ' startup scripts into', '/usr/local/bin')
    RecursiveMakeDir(bin_dir)
  else:
    bin_dir = None

  if install_source:
    PrintMsg('Installing source')
    files.extend(UnpackTar(kSourceDistFilename, winghome))
    option_list.append('source')

  if install_binary:
    # Unpack the tar archive
    PrintMsg('Installing binaries')
    files.extend(UnpackTar(kBinaryDistFilename, winghome))
    option_list.append('binary')

    # Set up the WINGHOME inits
    if build_root == None:
      real_winghome = winghome
    else:
      real_winghome = winghome[len(build_root):]
    SetWingHomeInit(real_winghome, os.path.join(winghome, 'wing'))
    SetWingHomeInit(real_winghome, os.path.join(winghome, 'bin', 'wing.py'))
    SetWingHomeInit(real_winghome, os.path.join(winghome, 'wingdb'))
    SetWingHomeInit(real_winghome, os.path.join(winghome, 'bin', 'wingdb.py'))
    SetWingHomeInit(real_winghome, os.path.join(winghome, 'wingdbstub.py'))
    SetWingHomeInit(real_winghome, os.path.join(winghome, 'wing-uninstall'))

    # Create symlinks if binary dir was specified
    if bin_dir != None:
      wing_symlink = os.path.abspath(os.path.join(bin_dir, 'wing'))
      CreateSymLink(os.path.abspath(os.path.join(winghome, 'wing')), wing_symlink)
      files.append(wing_symlink)

      #wingdb_symlink = os.path.join(bin_dir, 'wingdb')
      #CreateSymLink(os.path.join(winghome, 'wingdb'), wingdb_symlink)
      #files.append(wingdb_symlink)

  # Write file-list.txt
  for name in files:
    file_info_list.append(GetFileInfoTuple(name))
  WriteFilelist(filelist_name, version, option_list, file_info_list)
  
  if install_binary:
    PrintMsg('Done installing.  Make sure that %s is in your path and type'
             ' "wing" to start the Wing IDE.\n' % bin_dir)
  elif install_source:\
    PrintMsg('Done installing.')

  if '--multi-user' in argv:
    SetupMultiUser(winghome, argv)
  
if __name__ == '__main__':
  if '--help' in sys.argv:
    ErrorExit("Usage %s [--verbose] [--uninstall] [--install-source]"
              " [--multi-user]"
              " [--winghome <install directory>]"
              " [--bin-dir <bin dir for symlinks>]"
              " [--rpm-build-root <dir rpm is building in]"
              " [--multi-user-dir <dir to use for license lock files]"
              % sys.argv[0])

  if '--verbose' in sys.argv:
    global kVerbose
    kVerbose = 1

  # Set up umask so files created don't have write permissions
  os.umask(0022)
  os.environ['UMASK'] = '0022'
  
  if '--uninstall' in sys.argv:
    DoUninstall(sys.argv)
  else:
    DoInstall(sys.argv)

