# stest.py = system-test for easibox.
#
#############################################################
# Copyright (c) 2003 Philip Hunt. 
# You may use this software under the terms of the GNU
# General Public License (GPL). See file COPYING for details.
#############################################################

# Last altered: 12-Jan-2004
# History:
# 26-Dec-2003 PH: created
# 12-Jan-2004 PH: added testing of .tar.bz2 files

"""***

TODO: 

in test_ver2(), check that the archive files contain <aaaa> and 
<bbbb>.

in test_ver2(), check that the archive files *particularly* 
<tproj-4.5.6.zip>, do not contain <bbbb>. This is a known bug
in easibox-0.1.5, which is not currently being picked up.

***"""

#----- Python standard library
import sys, os
import StringIO


sys.path.append('..') #so utility can be imported
import utility

import unbox

#---------------------------------------------------------------------

debug = 1
#verbose = 0
profiling = 0 # producing profiling output?

passed = 0 # number of tests passed

def passedTest(msg):
   global passed
   passed += 1
   print "%s - PASSED (%d)" % (msg, passed)
        
def assertSame(r, sb, comment=""):
   if comment != "": comment += "; "
   msg = "%sr=%s (should be %s)" % (comment, repr(r), repr(sb))
   if r == sb:
      passedTest(msg)
   else:
      raise AssertionError, "Failed: %s" % msg  
      
def assertBool(bool, comment=""):
   if bool:
      passedTest("%s; true" % comment)
   else:
      raise AssertionError, "Failed: %s" % comment 
       
def assertFileExists(filename):
   "test that a file exists and is readable"
   if not os.access(filename, os.F_OK):
      raise AssertionError, "File <%s> doesn't exist" % filename
   if not os.access(filename, os.R_OK):
      raise AssertionError, "File <%s> isn't readable" % filename      
   passedTest("file <%s> exists" % filename) 
   
def assertFileIs(filename, data):
   "test that file (filename) contains (data)"
   r = utility.readFile(filename)
   dataShort = repr(data)
   if len(data)>10: dataShort = repr(data[:10])+'..'
   if not r == data:
      raise AssertionError, "File <%s> should be %s" \
         % (filename, dataShort)
   passedTest("file <%s> is %s" % (filename, dataShort)) 
   
def assertNoFile(filename):
   "test that file (filename) doesn't exist"
   if os.access(filename, os.F_OK):
      raise AssertionError, "File <%s> incorrectly exists" % filename
   passedTest("file <%s> correctly doesn't exist" % filename) 

def run(comment, command):
   if comment: 
      comment = comment + " - " 
   print "%sCMD { %s }..." % (comment, command)
   os.system(command)

#---------------------------------------------------------------------

def test_test():
   print "\n>>>>> test_test: testing the test system"
   r = 1
   assertSame(r, 1, "asserting same")
   assertBool(1, "asserting boolean")

#---------------------------------------------------------------------
# data for testing

aaaa = """aaaa
this is file aaaa, for testing
"""

bbbb = """bbbb
this is file bbbb, for testing
"""

rc1 = """# .easiboxrc file for tproj1, version 1
boxFormats = ['zip', 'tar', 'tgz', 'tar.gz', 'tar.bz2']
archiveDestination = "../tbox/"
include("*")
"""

rc2_nobbbb = """# .easiboxrc file for tproj1, version 2: tests exclude()
boxFormats = ['zip', 'tar', 'tgz', 'tar.gz', 'tar.bz2']
archiveDestination = "../tbox/"
include("*")
exclude("bbbb")
"""

#---------------------------------------------------------------------

def test_tproj1():
   print "\n----> test_tproj1: making tproj1"
   run("remove ./tproj1/", "rm -r tproj1")
   run("remove ./tbox/", "rm -r box")
   utility.writeFile("tproj1/aaaa", aaaa)
   assertFileExists("tproj1/aaaa")
   utility.writeFile("tproj1/bbbb", bbbb)
   assertFileExists("tproj1/bbbb")
   utility.writeFile("tproj1/.easiboxrc", rc1)
   assertFileExists("tproj1/.easiboxrc")
  
def test_ver():
   # requires test_tproj1() to have been run
   print "\n>>>>> test_ver: test making packages with --ver"
   run("should archive tproj1", "cd tproj1; easibox --ver=4.5.6")
   assertFileExists("tbox/tproj1-4.5.6.zip")
   assertFileExists("tbox/tproj1-4.5.6.tar")
   assertFileExists("tbox/tproj1-4.5.6.tgz")
   assertFileExists("tbox/tproj1-4.5.6.tar.gz")
   assertFileIs("tbox/tproj1-4.5.6/aaaa", aaaa)
   assertFileIs("tbox/tproj1-4.5.6/bbbb", bbbb)
   
   unbox.unbox("ubox", "tbox/tproj1-4.5.6.zip")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertFileIs("ubox/tproj1-4.5.6/bbbb", bbbb)
   
   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tar")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertFileIs("ubox/tproj1-4.5.6/bbbb", bbbb)

   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tgz")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertFileIs("ubox/tproj1-4.5.6/bbbb", bbbb)

   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tar.gz")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertFileIs("ubox/tproj1-4.5.6/bbbb", bbbb)

   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tar.bz2")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertFileIs("ubox/tproj1-4.5.6/bbbb", bbbb)
     
def test_packver():   
   # requires test_ver() to have been run
   print "\n>>>>> test_packver: test making packages with --pack --ver"
   run("should archive tproj1 as newname", 
      "cd tproj1; easibox --pack=newname --ver=0.7")
   assertFileExists("tbox/newname-0.7.zip")
   assertFileExists("tbox/newname-0.7.tar")
   assertFileExists("tbox/newname-0.7.tgz")
   assertFileExists("tbox/newname-0.7.tar.gz")
   assertFileIs("tbox/newname-0.7/aaaa", aaaa)
   assertFileIs("tbox/newname-0.7/bbbb", bbbb)
     
def test_ver2():
   # requires test_ver() to have been run
   print "\n>>>>> test_ver2: test --ver, part 2"
   utility.writeFile("tproj1/.easiboxrc", rc2_nobbbb)
   assertFileExists("tproj1/.easiboxrc")   
   run("should archive tproj1", "cd tproj1; easibox --ver=4.5.6")
   assertFileExists("tbox/tproj1-4.5.6.zip")
   assertFileExists("tbox/tproj1-4.5.6.tar")
   assertFileExists("tbox/tproj1-4.5.6.tgz")
   assertFileExists("tbox/tproj1-4.5.6.tar.gz")
   assertFileIs("tbox/tproj1-4.5.6/aaaa", aaaa)
   assertNoFile("tbox/tproj1-4.5.6/bbbb")
   
   unbox.unbox("ubox", "tbox/tproj1-4.5.6.zip")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertNoFile("ubox/tproj1-4.5.6/bbbb")
   
   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tar")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertNoFile("ubox/tproj1-4.5.6/bbbb")

   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tgz")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertNoFile("ubox/tproj1-4.5.6/bbbb")

   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tar.gz")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertNoFile("ubox/tproj1-4.5.6/bbbb")

   unbox.unbox("ubox", "tbox/tproj1-4.5.6.tar.bz2")
   assertFileIs("ubox/tproj1-4.5.6/aaaa", aaaa)
   assertNoFile("ubox/tproj1-4.5.6/bbbb")


#---------------------------------------------------------------------

def all_tests():
   global passed
   print "===== TESTING EASIBOX ====="
   
   test_test()
   test_tproj1()
   test_ver()
   test_packver()
   test_ver2()
   
   passmsg = "***** PASSED: %d tests completed successfully *****" % passed
   stars = "*" * len(passmsg)
   print "\n%s\n%s\n%s" % (stars, passmsg, stars)

#---------------------------------------------------------------------

if __name__ == "__main__":
   if profiling:
      import profile
      import pstats
      profile.run('all_tests()', 'stest.prof')
      p = pstats.Stats('stest.prof')
      p.strip_dirs()
      p.sort_stats('time','cum').print_stats(0.5)
   else:   
      all_tests()

#end
