"""Framework for getting file information, including filetype-specific metadata.

Instantiate appropriate class with filename to get metadata.  Returned object
acts just like a dictionary, with key-value pairs for each piece of metadata.
    import fileinfo
    info = fileinfo.MP3FileInfo("/music/ap/mahadeva.mp3")
    print "\\n".join(["%s=%s" % (k, info[k]) for k in info.keys()])

Or use listDirectory function to get info on all files in a directory.
    infoList = fileinfo.listDirectory("/music/ap/", [".mp3"])
    for info in infoList:
        ...

Framework can be extended by adding classes for particular file types, e.g.
HTMLFileInfo, MPGFileInfo, DOCFileInfo.  Each class is completely responsible for
parsing its files appropriately; see MP3FileInfo for example.
"""
import os
from UserDict import UserDict

class FileInfo(UserDict):
    "base class for file info"
    def __init__(self, filename=None):
        UserDict.__init__(self)
        self["name"] = filename
    
class MP3FileInfo(FileInfo):
    "class for MP3 file info, including ID3v1.0 tags if found"
    def __normalize(self, data):
        "strip whitespace and nulls"
        return data.replace("\00", " ").strip()
    
    def __parse(self, filename):
        "parse ID3v1.0 tags from MP3 file"
        self.clear()
        try:
            fsock = open(filename, "rb", 0)
            try:
                fsock.seek(-128, 2)
                tagdata = fsock.read(128)
            finally:
                fsock.close()
            if tagdata[:3] == 'TAG':
                self["title"]   = self.__normalize(tagdata[3:33])
                self["artist"]  = self.__normalize(tagdata[33:63])
                self["album"]   = self.__normalize(tagdata[63:93])
                self["year"]    = self.__normalize(tagdata[93:97])
                self["comment"] = self.__normalize(tagdata[97:126])
                self["genre"]   = ord(tagdata[127])
        except IOError:
            pass

    def __setitem__(self, key, item):
        if key == "name" and item:
            self.__parse(item)
        FileInfo.__setitem__(self, key, item)

def __getFileInfoObjectClass(filename):
    "get file info class from filename extension, returns class"
    subclass = "%sFileInfo" % os.path.splitext(filename)[1].upper()[1:]
    if globals().has_key(subclass):
        return eval(subclass)
    else:
        return FileInfo
    
def listDirectory(directory, fileExtList):
    "return list of file info objects for files of particular extensions"
    fileExtList = [ext.upper() for ext in fileExtList]
    fileList = [os.path.join(directory, f) for f in os.listdir(directory) \
                if os.path.splitext(f)[1].upper() in fileExtList]
    return [__getFileInfoObjectClass(f)(f) for f in fileList]

if __name__ == "__main__":
    for info in listDirectory("/music/_singles/", [".mp3"]):
        for k, v in info.items():
            print "%s=%s" % (k, v)
        print
