package joker

import (
	"testing"
)

type testPair struct {
	cards       []*Card
	arrangement []*Card
	ranking     Ranking
	description string
}

var tests = []testPair{
	{
		[]*Card{
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Ace, Spades),
			NewCard(Nine, Diamonds),
		},
		[]*Card{
			NewCard(Ace, Spades),
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Nine, Diamonds),
		},
		HighCard,
		"high card ace high",
	},
	{
		[]*Card{
			NewCard(King, Spades),
			NewCard(Queen, Hearts),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Nine, Diamonds),
		},
		[]*Card{
			NewCard(Queen, Hearts),
			NewCard(Queen, Spades),
			NewCard(King, Spades),
			NewCard(Jack, Spades),
			NewCard(Nine, Diamonds),
		},
		Pair,
		"pair of queens",
	},
	{
		[]*Card{
			NewCard(Two, Spades),
			NewCard(Queen, Hearts),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Two, Diamonds),
		},
		[]*Card{
			NewCard(Queen, Hearts),
			NewCard(Queen, Spades),
			NewCard(Two, Spades),
			NewCard(Two, Diamonds),
			NewCard(Jack, Spades),
		},
		TwoPair,
		"two pair queens and twos",
	},
	{
		[]*Card{
			NewCard(Six, Spades),
			NewCard(Queen, Hearts),
			NewCard(King, Spades),
			NewCard(Six, Hearts),
			NewCard(Six, Diamonds),
		},
		[]*Card{
			NewCard(Six, Spades),
			NewCard(Six, Hearts),
			NewCard(Six, Diamonds),
			NewCard(King, Spades),
			NewCard(Queen, Hearts),
		},
		ThreeOfAKind,
		"three of a kind sixes",
	},
	{
		[]*Card{
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Ace, Spades),
			NewCard(Ten, Diamonds),
		},
		[]*Card{
			NewCard(Ace, Spades),
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Ten, Diamonds),
		},
		Straight,
		"straight ace high",
	},
	{
		[]*Card{
			NewCard(Two, Spades),
			NewCard(Three, Spades),
			NewCard(Four, Spades),
			NewCard(Ace, Spades),
			NewCard(Five, Diamonds),
		},
		[]*Card{
			NewCard(Five, Diamonds),
			NewCard(Four, Spades),
			NewCard(Three, Spades),
			NewCard(Two, Spades),
			NewCard(Ace, Spades),
		},
		Straight,
		"straight five high",
	},
	{
		[]*Card{
			NewCard(Seven, Spades),
			NewCard(Four, Spades),
			NewCard(Five, Spades),
			NewCard(Three, Spades),
			NewCard(Two, Spades),
		},
		[]*Card{
			NewCard(Seven, Spades),
			NewCard(Five, Spades),
			NewCard(Four, Spades),
			NewCard(Three, Spades),
			NewCard(Two, Spades),
		},
		Flush,
		"flush seven high",
	},
	{
		[]*Card{
			NewCard(Seven, Spades),
			NewCard(Three, Hearts),
			NewCard(Seven, Clubs),
			NewCard(Three, Diamonds),
			NewCard(Seven, Hearts),
		},
		[]*Card{
			NewCard(Seven, Spades),
			NewCard(Seven, Clubs),
			NewCard(Seven, Hearts),
			NewCard(Three, Hearts),
			NewCard(Three, Diamonds),
		},
		FullHouse,
		"full house sevens full of threes",
	},
	{
		[]*Card{
			NewCard(Seven, Spades),
			NewCard(Seven, Clubs),
			NewCard(Seven, Diamonds),
			NewCard(Three, Diamonds),
			NewCard(Seven, Hearts),
		},
		[]*Card{
			NewCard(Seven, Spades),
			NewCard(Seven, Clubs),
			NewCard(Seven, Diamonds),
			NewCard(Seven, Hearts),
			NewCard(Three, Diamonds),
		},
		FourOfAKind,
		"four of a kind sevens",
	},
	{
		[]*Card{
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Ten, Spades),
			NewCard(Nine, Spades),
		},
		[]*Card{
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Ten, Spades),
			NewCard(Nine, Spades),
		},
		StraightFlush,
		"straight flush king high",
	},
	{
		[]*Card{
			NewCard(Ace, Spades),
			NewCard(Five, Spades),
			NewCard(Four, Spades),
			NewCard(Three, Spades),
			NewCard(Two, Spades),
		},
		[]*Card{
			NewCard(Five, Spades),
			NewCard(Four, Spades),
			NewCard(Three, Spades),
			NewCard(Two, Spades),
			NewCard(Ace, Spades),
		},
		StraightFlush,
		"straight flush five high",
	},
	{
		[]*Card{
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Ten, Spades),
			NewCard(Ace, Spades),
		},
		[]*Card{
			NewCard(Ace, Spades),
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Jack, Spades),
			NewCard(Ten, Spades),
		},
		RoyalFlush,
		"royal flush",
	},
	{
		[]*Card{
			NewCard(Ace, Spades),
			NewCard(King, Spades),
			NewCard(Queen, Spades),
			NewCard(Two, Spades),
			NewCard(Two, Clubs),
			NewCard(Two, Hearts),
			NewCard(Two, Diamonds),
		},
		[]*Card{
			NewCard(Two, Spades),
			NewCard(Two, Clubs),
			NewCard(Two, Hearts),
			NewCard(Two, Diamonds),
			NewCard(Ace, Spades),
		},
		FourOfAKind,
		"four of a kind twos",
	},
}

func TestHands(t *testing.T) {
	for _, test := range tests {
		h, _ := NewHand(test.cards)
		if h.Ranking() != test.ranking {
			t.Errorf("expected %v got %v", test.ranking, h.Ranking())
		}
		for i := 0; i < 5; i++ {
			actual, expected := h.Cards()[i], test.arrangement[i]
			if actual.Rank() != expected.Rank() || actual.Suit() != expected.Suit() {
				t.Errorf("expected %v got %v", expected, actual)
			}
		}
		if test.description != h.Description() {
			t.Errorf("expected \"%v\" got \"%v\"", test.description, h.Description())
		}
	}
}

type equality int

const (
	greaterThan equality = iota
	lessThan
	equalTo
)

type testEquality struct {
	cards1 []*Card
	cards2 []*Card
	e      equality
}

var equalityTests = []testEquality{
	{
		[]*Card{
			NewCard(Ace, Spades),
			NewCard(Five, Spades),
			NewCard(Four, Spades),
			NewCard(Three, Spades),
			NewCard(Two, Spades),
		},
		[]*Card{
			NewCard(King, Spades),
			NewCard(King, Clubs),
			NewCard(King, Hearts),
			NewCard(Jack, Diamonds),
			NewCard(Jack, Spades),
		},
		greaterThan,
	},
	{
		[]*Card{
			NewCard(Ten, Spades),
			NewCard(Nine, Hearts),
			NewCard(Eight, Diamonds),
			NewCard(Seven, Clubs),
			NewCard(Six, Spades),
			NewCard(Two, Hearts),
			NewCard(Three, Spades),
		},
		[]*Card{
			NewCard(Ten, Spades),
			NewCard(Nine, Hearts),
			NewCard(Eight, Diamonds),
			NewCard(Seven, Clubs),
			NewCard(Six, Spades),
			NewCard(Ace, Hearts),
			NewCard(King, Spades),
		},
		equalTo,
	},
}

func TestCompareHands(t *testing.T) {
	for _, test := range equalityTests {
		h1, _ := NewHand(test.cards1)
		h2, _ := NewHand(test.cards2)
		compareTo := h1.CompareTo(h2)

		switch test.e {
		case greaterThan:
			if compareTo <= 0 {
				t.Errorf("expected %v to be greater than %v", h1, h2)
			}
		case lessThan:
			if compareTo >= 0 {
				t.Errorf("expected %v to be less than %v", h1, h2)
			}
		case equalTo:
			if compareTo != 0 {
				t.Errorf("expected %v to be equal to %v", h1, h2)
			}
		}
	}
}

func TestDeck(t *testing.T) {
	deck := NewDeck(true)
	cards := deck.PopMulti(52)
	if len(cards) != 52 {
		t.Error("Not fifty two cards in the deck")
	}
}
