package joker

import (
	"errors"
	"strings"
)

// A Rank represents the rank of a card.
type Rank string

const (
	Two   Rank = "2"
	Three Rank = "3"
	Four  Rank = "4"
	Five  Rank = "5"
	Six   Rank = "6"
	Seven Rank = "7"
	Eight Rank = "8"
	Nine  Rank = "9"
	Ten   Rank = "T"
	Jack  Rank = "J"
	Queen Rank = "Q"
	King  Rank = "K"
	Ace   Rank = "A"
)

// String returns a string in the format "2"
func (r Rank) String() string {
	return string(r)
}

// singularName returns the name of the rank in singular form such as "two" for Two.
func (r Rank) singularName() string {
	return singularNames[r]
}

// pluralName returns the name of the rank in plural form such as "twos" for Two.
func (r Rank) pluralName() string {
	return pluralNames[r]
}

// Valid returns true if the rank is valid
func (r Rank) valid() bool {
	return r.indexOf() != -1
}

// IndexOf returns the index of the rank in the ascending order of ranks.
// IndexOf returns -1 if the rank is not found.
func (r Rank) indexOf() int {
	for i, rank := range allRanks() {
		if r == rank {
			return i
		}
	}
	return -1
}

// ByHighRank is a slice of ranks that when sorted will be ordered in ascending rank
type byHighRank []Rank

// Len implements the sort.Interface interface.
func (a byHighRank) Len() int { return len(a) }

// Swap implements the sort.Interface interface.
func (a byHighRank) Swap(i, j int) { a[i], a[j] = a[j], a[i] }

// Less implements the sort.Interface interface.
func (a byHighRank) Less(i, j int) bool {
	iRank, jRank := a[i], a[j]
	iIndex, jIndex := iRank.indexOf(), jRank.indexOf()
	return iIndex < jIndex
}

var (
	singularNames = map[Rank]string{
		Two:   "two",
		Three: "three",
		Four:  "four",
		Five:  "five",
		Six:   "six",
		Seven: "seven",
		Eight: "eight",
		Nine:  "nine",
		Ten:   "ten",
		Jack:  "jack",
		Queen: "queen",
		King:  "king",
		Ace:   "ace",
	}

	pluralNames = map[Rank]string{
		Two:   "twos",
		Three: "threes",
		Four:  "fours",
		Five:  "fives",
		Six:   "sixes",
		Seven: "sevens",
		Eight: "eights",
		Nine:  "nines",
		Ten:   "tens",
		Jack:  "jacks",
		Queen: "queens",
		King:  "kings",
		Ace:   "aces",
	}
)

// A Suit represents the suit of a card.
type Suit string

const (
	Spades   Suit = "♠"
	Hearts   Suit = "♥"
	Diamonds Suit = "♦"
	Clubs    Suit = "♣"
)

// String returns a string in the format "♠"
func (s Suit) String() string {
	return string(s)
}

func (s Suit) valid() bool {
	return strings.Contains("♠♥♦♣", string(s))
}

// A Card represents a playing card in the game of poker.  It is composed of a rank and suit.
type Card struct {
	rank Rank
	suit Suit
}

// NewCard returns a new card and panics if r or s are invalid.
func NewCard(r Rank, s Suit) *Card {
	if !r.valid() {
		panic("card: invalid rank " + string(r))
	} else if !s.valid() {
		panic("card: invalid suit " + string(s))
	}
	return &Card{r, s}
}

// Rank returns the rank of the card.
func (c *Card) Rank() Rank {
	return c.rank
}

// Suit returns the suit of the card.
func (c *Card) Suit() Suit {
	return c.suit
}

// String returns a string in the format "4♠"
func (c *Card) String() string {
	return string(c.Rank()) + string(c.Suit())
}

// MarshalText implements the encoding.TextMarshaler interface.
// The text format is "4♠".
func (c *Card) MarshalText() ([]byte, error) {
	return []byte(c.String()), nil
}

// UnmarshalText implements the encoding.TextUnmarshaler interface.
// The card is expected to be in the format "4♠".
func (c *Card) UnmarshalText(text []byte) error {
	s := string(text)
	if len(s) == 2 {
		rank, suit := Rank(s[0]), Suit(s[1])
		if rank.valid() && suit.valid() {
			c.rank = rank
			c.suit = suit
			return nil
		}
	}
	return errors.New(`card: serialization should be of the format "4♠"`)
}

// ByHighCard is a slice of cards that when sorted will be ordered in ascending rank
type byHighCard []*Card

// Len implements the sort.Interface interface.
func (a byHighCard) Len() int { return len(a) }

// Swap implements the sort.Interface interface.
func (a byHighCard) Swap(i, j int) { a[i], a[j] = a[j], a[i] }

// Less implements the sort.Interface interface.
func (a byHighCard) Less(i, j int) bool {
	iCard, jCard := a[i], a[j]
	iIndex, jIndex := iCard.Rank().indexOf(), jCard.Rank().indexOf()
	return iIndex < jIndex
}

func allRanks() []Rank {
	return []Rank{Two, Three, Four, Five, Six, Seven, Eight,
		Nine, Ten, Jack, Queen, King, Ace}
}

func allSuits() []Suit {
	return []Suit{Spades, Hearts, Diamonds, Clubs}
}
