#ifndef ESCAPE_HEADER_DEFINED
#define ESCAPE_HEADER_DEFINED

// escape
//
// This is an iostream manipulator similar to width() or endl
// See April 2000 C++ Report, "Effective Standard C++ Library"
// for full description of manipulators and how to implement them.
//
// There are doubtless many improvements that can be done to this
// program, some of which I point out along the way.
//
// A sample driver program is included at the end of this file.
//
// Inserts escape characters into a std::ostream.  This is
// useful, for example, to put backslashes before quotation
// marks and other backslashes.
//
// explicit escape(const std::string &inputStr,
//                 char escapeChar='\\',
//                 char surroundChar='"',
//                 const char *charsToEscape=NULL)
//
// Parameters:
//    inputStr - the original string to escape
//    escapeChar - the character to precede the characters that need escaping
//    surroundChar - If this is '\0' it is ignored. Otherwise, the entire string
//                   will be preceded and followed by this character.
//    charsToEscape - a NULL-terminated string containing the characters
//                    that need to be escaped.  The escape character itself
//                    is implicitly escaped.
//
//
// Description:
//   The constructor allows you to specify a list of one or more
//   characters to escape (i.e. prefix with the escape character).  A
//   ZModem transfer, for example, will escape XON, XOFF, and CAN with
//   a CTRL-X (CAN) character.  The escape character, i.e. the character
//   that becomes the prefix, is implicitly escaped.
//
//   The default parameters will escape quotation marks and backslashes
//   with a backslash character.
//
//   Example:
//   int main()
//   {
//     const char * escStr1="Hello, \"MY\" backslash \\ world\n";
//     std::cout << escStr1;
//     std::cout << escape(escStr1);
//     const char * escStr2="With percents (%)\n";
//     std::cout << escStr2;
//     std::cout << escape(escStr2, '%');
//     const char * escStr3="Escape A, B, and C with *\n";
//     std::cout << escStr3;
//     std::cout << escape(escStr3, '*', '\0', "ABC");
//   }
//   will print:
//   Hello, "MY" backslash \ world
//   Hello, \"MY\" backslash \\ world
//   With percents (%)
//   With percents (%%)
//   Escape A, B, and C with *
//   Escape *A, *B, and *C with **
//
#include <iostream>
#include <string>

// The ostream manipulator...
class escape
{
    const char *inputStr;
    const char *escStr;
    char escapeChar;
    char surroundChar;
public:
    explicit escape(const std::string &inputStr,
                char escapeChar='\\',
                char surroundChar='"',
                const char *charsToEscape=NULL)
    :
    inputStr(inputStr.c_str()),
    escStr(charsToEscape),
    escapeChar(escapeChar),
    surroundChar(surroundChar)
    {
    }
    explicit escape(const char *inputStr,
                char escapeChar='\\',
                char surroundChar='"',
                const char *charsToEscape=NULL)
    :
    inputStr(inputStr),
    escStr(charsToEscape),
    escapeChar(escapeChar),
    surroundChar(surroundChar)
    {
    }
    std::ostream &doEsc(std::ostream &os) const
    {
        if (surroundChar != '\0') {
            os.put(surroundChar);
        }
        if (inputStr) {
            int charPos=0;
            while (inputStr[charPos]) {
                bool escapeIt=false;
                if (inputStr[charPos] == escapeChar ||
                    inputStr[charPos] == surroundChar) {
                    escapeIt=true;
                } else if(escStr != NULL ) {
                    // Possible enhancement: use strchr here instead
                    // of the for loop.
                    for (int srchPos=0; escStr[srchPos]; srchPos++) {
                        if (inputStr[charPos] == escStr[srchPos]) {
                            escapeIt=true;
                            break;
                        }
                    }
                }
                if (escapeIt) {
                    os.put(escapeChar);
                }
                os.put(inputStr[charPos++]);
            }
        }
        if (surroundChar != '\0') {
            os.put(surroundChar);
        }
        return os;
    }

};

std::ostream& operator << (std::ostream &os, const escape &e)
{
    return e.doEsc(os);
}

// and the istream manipulator
class unescape
{
    char escapeChar;
    char terminator;
    std::string &outStr;
public:
    explicit unescape(
                std::string &outStr,
                char escapeChar='\\',
                char terminator='"'
                )
    :
    escapeChar(escapeChar),
    terminator(terminator),
    outStr(outStr)
    {
    }
    std::istream &Retrieve(std::istream &is)
    {
        outStr="";
        char nextChar;

        // Use extraction operator to skip white spaces
        is >> nextChar;
        if (nextChar == terminator) {
            nextChar=is.get();
        }

        // Keep reading characters until there is an error, we
        // hit the end of file, or we hit the terminator.
        while (is.good() && !is.eof()
               && nextChar != terminator) 
        {
            // simply read each character one at a time,
            // and un-escape them.
            if (nextChar == escapeChar) {
                nextChar=is.get();
                if (is.eof()) {
                    break;
                }
            }
            outStr += nextChar;
            nextChar=is.get();
        }
        return is;
    }
};

std::istream& operator >> (std::istream &is, unescape &e)
{
    return e.Retrieve(is);
}

// Note: there is no corresponding .cpp file to this header.

// Un-comment the following line to enable the test driver
//#define ADD_TEST_DRIVER_MAIN

#ifdef ADD_TEST_DRIVER_MAIN
#include <fstream>
// A test driver for the streams:
int main()
{
    std::ofstream of("esc.txt");
    const char * escStr1="Hello, \"MY\" backslash \\ world";
    std::cout << escStr1 << "\n";
    of << escape(escStr1) << "\n";

    std::string escStr2("and another string \\\"with quotes\\\"");
    std::cout << escStr2 << "\n";;
    of << escape(escStr2) << "\n";

    escStr2="And now, we'll do percents (%)";
    std::cout << escStr2 << "\n";;
    of << escape(escStr2, '%', '\0') << "\n";

    const char * escStr3="Escape A, B, and C with *";
    std::cout << escStr3;
    of << escape(escStr3, '*', '\0', "ABC");
    of.close();

    std::ifstream ifile("esc.txt");
    std::string s1, s2;
    ifile >> unescape(s1) >> unescape(s2);
    std::cout << "Retrieved [" << s1 << "]\n";
    std::cout << "Retrieved [" << s2 << "]\n";
    ifile >> unescape(s1, '%', '\n');
    std::cout << "Retrieved [" << s1 << "]\n";
    ifile >> unescape(s1, '*', '\n');
    std::cout << "Retrieved [" << s1 << "]\n";
}
#endif // test driver

#endif // include sentry

