/***********************************************************
	Stack.h

	Stack class by Herb Sutter
	Appears in Items 12 and 14 of
	"Exceptional C++"
	by Herb Sutter,
	published by Addison Wesley, 2000.

	This is actually the third of a set of 3 stack classes 
	to appear in the book.

	Entered by Ben Stanley 20000816

	Part of CUJ article on Exception Testing.
************************************************************/
#ifndef STACK_H
#define STACK_H

#include <cassert>
#include <algorithm> // for swap
#include <cstdlib> // for size_t

// Tell test harness which interface to expect to test.
#define STACK_HAS_TOP

// my_construct() constructs a new object in
// a given location using an initial value
// 
template <class T1, class T2>
void my_construct( T1* p, const T2& value )
{
	new (p) T1(value);
}

// my_destroy() destroys an object or a range 
// of objects
// 
template <class T>
void my_destroy( T* p )
{
	p->~T();
}

template <class FwdIter>
void my_destroy( FwdIter first, FwdIter last )
{
	while( first != last )
	{
		my_destroy( first );
		++first;
	}
}

// swap() just exchanges two values
// This comes from <algorithm>



template <class T> 
class StackImpl
{
public:
	StackImpl( std::size_t size );
	~StackImpl();
	void Swap( StackImpl& other ) throw();

	T*			v_; // ptr to a memory area big enough for 'vsize_' Ts
	std::size_t vsize_;
	std::size_t vused_; // # of T's actually in use

private:
	// private and undefined: no copying allowed
	StackImpl( const StackImpl& );
	StackImpl& operator=( const StackImpl& );
};

template <class T>
StackImpl<T>::StackImpl( std::size_t size )
:	v_( static_cast<T*>( size == 0 ? 0 : operator new( sizeof(T)*size ) ) ),
	vsize_(size),
	vused_(0)
{}

template <class T>
StackImpl<T>::~StackImpl()
{
	my_destroy( v_, v_+vused_ ); // this can't throw
	operator delete( v_ );
}

template <class T>
void StackImpl<T>::Swap(StackImpl& other) throw()
{
	swap( v_,     other.v_     );
	swap( vsize_, other.vsize_ );
	swap( vused_, other.vused_ );
}

template <class T>
class Stack
{
public:
	Stack( size_t size = 0 )
	:	impl_( size )
	{}
	
	Stack( const Stack& other )
	:	impl_( other.impl_.vused_ )
	{
		while( impl_.vused_ < other.impl_.vused_ ) {
			my_construct( impl_.v_+impl_.vused_,
				other.impl_.v_[impl_.vused_] );
			++impl_.vused_;
		}
	}
	
	Stack& operator=( const Stack& other )
	{
		Stack temp( other );
		impl_.Swap( temp.impl_); // this can't throw
		return *this;
	}
	
	size_t Size() const
	{
		return impl_.vused_;
	}
	
	void Push( const T& t)
	{
		if( impl_.vused_ == impl_.vsize_ ) {
			Stack temp( impl_.vsize_*2+1 );
			while( temp.Size() < impl_.vused_ ) {
				temp.Push( impl_.v_[temp.Size()] );
			}
			temp.Push( t );
			impl_.Swap( temp.impl_ );
		} else {
			my_construct( impl_.v_+impl_.vused_, t );
			++impl_.vused_;
		}
	}

	T& Top()
	{
		if( impl_.vused_ == 0 ) {
			throw "empty stack";
		}
		return impl_.v_[impl_.vused_-1];
	}
	
	void Pop()
	{
		if( impl_.vused_ == 0 ) {
			throw "pop from empty stack";
		} else {
			--impl_.vused_;
			my_destroy( impl_.v_+impl_.vused_ );
		}
	}

	bool Consistent() const
	{
		// Weak check.
		return impl_.vused_ <= impl_.vsize_ and 
			(impl_.v_ != 0 or impl_.vsize_ == 0);
	}


private:
	StackImpl<T> impl_; // private implementation
};

#endif // STACK_H
