/*****************************************************
Counter.cpp
Class for counting tests passed, tests failed,
number of exception points, which exception point to
fail on.

Also provides simple debugging memory manager which allows
you to detect memory leaks. Only works for new/delete.
However, both array/non-array versions are supported.
The debugging memory manager is an important part of the
automated testing process.

Copyright Ben Stanley 20000815 bstanley@uow.edu.au
Permission is hereby granted to re-use at your own risk.
No warranty is provided.
Original author must be acknowledged.

Part of CUJ article on Exception Testing.
******************************************************/

#include "Counter.h"
#include <new>
#include <cstdlib>
#include <cassert>

using namespace std;

int Counter::sExceptionPointCount = 0;
int Counter::sThrowCount = -1;
bool Counter::sDontThrow = false;
int Counter::sNewCount = 0;
int Counter::sPassCount = 0;
int Counter::sFailCount = 0;
ostream* Counter::mOs = &cout;

void Counter::CouldThrow() throw ( TestException )
{
	if( !sDontThrow &&
		++sExceptionPointCount == sThrowCount ) 
	{
		throw TestException();
	}
}

void Counter::SetThrowCount( int inCount ) throw()
{
	sExceptionPointCount = 0;
	sThrowCount = inCount;
	sDontThrow = false;
}

void Counter::DontThrow() throw()
{
	sDontThrow = false;
}

bool Counter::HasThrown() throw()
{
	return sExceptionPointCount >= sThrowCount;
}

int Counter::GetAllocationCount() throw()
{
	return sNewCount;
}

void Counter::Pass( const char* testName ) throw()
{
	++sPassCount;
	//(*mOs) << "        Passed test " << testName << endl;
}

void Counter::Fail( const char* testName ) throw()
{
	++sFailCount;
	(*mOs) << "****    Failed test " << testName 
		<< " at exception count " << sThrowCount
		<< "." << endl;
}

void Counter::Test( bool result, const char* testName ) throw()
{
	if( result ) {
		Pass( testName );
	} else {
		Fail( testName );
	}
}

void Counter::PrintTestSummary()
{
	(*mOs) << "Test Results:" << std::endl;
	(*mOs) << "Total Tests: " << sPassCount + sFailCount << std::endl;
	(*mOs) << "Passed     : " << sPassCount << std::endl;
	(*mOs) << "Failed     : " << sFailCount << std::endl;
}

void Counter::SetOutputStream( std::ostream* os )
{
	assert( os );
	mOs = os;
}

/******************************************************
	Rudimentary Debugging Memory Manager
	
	The code below implements a very simple debugging 
	memory manager. It can detect:
		* memory leaks
		* delete of unallocated pointers
	It also allows the Test Driver function
	to cause an exception to be thrown from certain
	memory allocations.


	Note on non-throwing version of operator-new:
	We have to implement all versions of new and delete so that
	they all operate consistently.

	This provides all 8 standard variants defined in 
	section 18.4 of the C++ standad.

	I have not implemented the placement forms. 
	I figure that they can't throw and that their behaviour does not
	need to be changed to use malloc/free.
	
	Implementation:
	
	When you request a block from new, this memory manager
	requests a block from malloc. The format of the block 
	is shown below:
	
	+-----------+------------------------------------------+
	| DListNode | User Area                                |
	+-----------+------------------------------------------+
	
	The DListNode is a doubly linked list containing all
	allocated blocks. This list is searched when a block
	is de-allocated to check that the pointer is valid.
	new returns the address of the user area.
	
	When the block is deleted, the beginning of the malloc
	block is calculated by subtracting the size of DListNode
	from the user area pointer.
*******************************************************/

struct DListNode {
	void Append( DListNode* other ) {
		other->next = this->next;
		other->prev = this;
		this->next->prev = other;
		this->next = other;
	}

	void Remove() {
		next->prev = this->prev;
		prev->next = this->next;
		this->prev = this;
		this->next = this;
	}

	struct DListNode *next, *prev;
};

static DListNode sAllocatedList = { &sAllocatedList, &sAllocatedList };

void* Counter::MemAllocated( void* rawMem ) {
	DListNode* node = reinterpret_cast<DListNode*>( rawMem );
	sAllocatedList.Append( node );
	++sNewCount;
	return node + 1; // Adds size of a DListNode
}

void Counter::DoDeallocate( void* userMem ) {
	DListNode* search = sAllocatedList.next;
	DListNode* target = reinterpret_cast<DListNode*>( userMem ) - 1;
	// Linear search. This may be slow, but it's adequate for small tests.
	while( search != &sAllocatedList ) {
		if( search == target ) break;
		search = search->next;
	}
	if( search != target ) {
		(*mOs) << "Deleted unallocated pointer " << userMem 
			<< " at exception count " << sThrowCount << "." << endl;
		abort();
	}
	target->Remove();
	--sNewCount;
	free( target );
}

void* operator new( size_t size ) throw (bad_alloc)
{
	try {
		Counter::CouldThrow();
		// standard new behaviour
		while(1) {
			void* mem = malloc( size + sizeof( DListNode ) );
			if( mem ) {
				return Counter::MemAllocated( mem );
			}
			new_handler current_handler = set_new_handler(0);
			set_new_handler( current_handler );
			if( current_handler ) {
				current_handler();
			} else {
				return 0;
			}
		}
	}
	catch( const TestException& ) {
		//	(*mOs) << "Testing new failing." << endl;
		throw bad_alloc();
	}
}

void* operator new( size_t size, const nothrow_t& ) throw ()
{
	void* mem = malloc( size + sizeof( DListNode ) );
	if( mem ) return Counter::MemAllocated( mem );
	return mem;
}

void operator delete( void* ptr ) throw ()
{
	if( ptr ) {
		Counter::DoDeallocate( ptr );
	}
}

void operator delete( void* ptr, const nothrow_t& ) throw ()
{
	if( ptr ) {
		Counter::DoDeallocate( ptr );
	}
}

void* operator new[]( size_t size ) throw (bad_alloc)
{
	try {
		Counter::CouldThrow();
		// standard new behaviour
		while(1) {
			void* mem = malloc( size + sizeof( DListNode ) );
			if( mem ) {
				return Counter::MemAllocated( mem );
			}
			new_handler current_handler = set_new_handler(0);
			set_new_handler( current_handler );
			if( current_handler ) {
				current_handler();
			} else {
				return 0;
			}
		}
	}
	catch( const TestException& ) {
		//	(*mOs) << "Testing new failing." << std::endl;
		throw bad_alloc();
	}
}

void* operator new[]( size_t size, const nothrow_t& ) throw ()
{
	void* mem = malloc( size + sizeof( DListNode ) );
	if( mem ) return Counter::MemAllocated( mem );
	return mem;
}

void operator delete[]( void* ptr ) throw()
{
	if( ptr ) {
		Counter::DoDeallocate( ptr );
	}
}

void operator delete[]( void* ptr, const std::nothrow_t& ) throw()
{
	if( ptr ) {
		Counter::DoDeallocate( ptr );
	}
}

