/********************************************************************************
*                                                                               *
*                      T e x t   E d i t o r   C o n t r o l                    *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* $Id: textedit.cpp,v 1.26 2000/04/27 05:54:39 jeroen Exp $                     *
********************************************************************************/
#include "fx.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <signal.h>
#ifndef WIN32
#include <unistd.h>
#endif




/*******************************************************************************/


// Mini application object
class TextWindow : public FXMainWindow {
  FXDECLARE(TextWindow)
protected:
  FXMenuBar         *menubar;                 // Menu bar
  FXMenuPane        *filemenu;                // File menu
  FXMenuPane        *helpmenu;                // Help menu
  FXToolbar         *toolbar;                 // Tool bar
  FXText            *editor;                  // Multiline text widget
  FXColorWell       *textwell;                // Text color
  FXColorWell       *backwell;                // Text background color
  FXColorWell       *seltextwell;             // Selected text color
  FXColorWell       *selbackwell;             // Selected text background color
  FXStatusbar       *statusbar;               // Status bar
  FXVerticalFrame   *treebox;                 // Tree box containing directories/files
  FXDirList         *dirlist;                 // Directory view
  FXSplitter        *splitter;                // Splitter
  FXTextField       *filter;                  // Filter for tree list
  FXRecentFiles      mrufiles;                // Recent files
  FXString           filename;                // File being edited
  FXbool             namechanged;             // Save becomes save as if no name given yet
  FXbool             modified;                // Modified state
protected:
  FXbool loadfile(const FXString& file);
  FXbool savefile(const FXString& file);
protected:
  TextWindow(){}
public:
  long onCmdAbout(FXObject*,FXSelector,void*);
  long onCmdRevert(FXObject*,FXSelector,void*);
  long onUpdRevert(FXObject*,FXSelector,void*);
  long onCmdOpen(FXObject*,FXSelector,void*);
  long onCmdSave(FXObject*,FXSelector,void*);
  long onUpdSave(FXObject*,FXSelector,void*);
  long onCmdSaveAs(FXObject*,FXSelector,void*);
  long onCmdNew(FXObject*,FXSelector,void*);
  long onCmdFont(FXObject*,FXSelector,void*);
  long onCmdQuit(FXObject*,FXSelector,void*);
  long onCmdPrint(FXObject*,FXSelector,void*);
  long onUpdTitle(FXObject*,FXSelector,void*);
  long onCmdTreeList(FXObject*,FXSelector,void*);
  long onCmdBackColor(FXObject*,FXSelector,void*);
  long onUpdBackColor(FXObject*,FXSelector,void*);
  long onCmdTextColor(FXObject*,FXSelector,void*);
  long onUpdTextColor(FXObject*,FXSelector,void*);
  long onCmdSelBackColor(FXObject*,FXSelector,void*);
  long onUpdSelBackColor(FXObject*,FXSelector,void*);
  long onCmdSelTextColor(FXObject*,FXSelector,void*);
  long onUpdSelTextColor(FXObject*,FXSelector,void*);
  long onCmdRecentFile(FXObject*,FXSelector,void*);
  long onCmdWrap(FXObject*,FXSelector,void*);
  long onUpdWrap(FXObject*,FXSelector,void*);
public:
  enum{
    ID_ABOUT=FXMainWindow::ID_LAST,
    ID_OPEN,
    ID_REVERT,
    ID_SAVE,
    ID_SAVEAS,
    ID_NEW,
    ID_TITLE,
    ID_FONT,
    ID_QUIT,
    ID_PRINT,
    ID_TREELIST,
    ID_BACKCOLOR,
    ID_TEXTCOLOR,
    ID_SELBACKCOLOR,
    ID_SELTEXTCOLOR,
    ID_RECENTFILE,
    ID_TOGGLE_WRAP,
    ID_LAST
    };
public:
  TextWindow(FXApp* a);
  virtual void create();
  void start(int& argc,char** argv);
  };


/*******************************************************************************/
  

// Save as
const unsigned char filesaveas[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf3,0x00,0x00,0x80,0x80,0x80,
  0xb2,0xc0,0xdc,0xff,0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,0x80,0x80,0x00,0x00,
  0x00,0xff,0x00,0xff,0xff,0x00,0x00,0x80,0xc0,0xc0,0xc0,0x00,0x80,0x80,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x2c,0x00,0x00,
  0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x04,0x5a,0x10,0xc8,0x19,0xaa,0xbd,0x40,0x0c,
  0x91,0xc1,0xfd,0xc0,0x36,0x10,0x24,0xf1,0x59,0x99,0x58,0x4c,0x2c,0xba,0x19,0x5a,
  0x26,0xcc,0x9c,0x17,0x64,0xc6,0x01,0xcb,0x74,0x77,0xe7,0x07,0x9d,0x0c,0xc6,0x99,
  0x49,0x04,0x40,0x1d,0x12,0xd1,0x29,0x85,0x90,0x41,0xdd,0x81,0x39,0x4b,0x24,0x08,
  0x29,0x68,0x94,0x4a,0xc3,0x8a,0xb4,0x53,0x9e,0x00,0x5b,0x4a,0x86,0x69,0xe3,0x89,
  0x01,0xc1,0x6e,0x8b,0x4d,0x16,0x85,0x02,0x3b,0x41,0xc3,0x31,0x68,0xfb,0x09,0x50,
  0xea,0x93,0x22,0x00,0x3b
  };

// File Save
const unsigned char filesave[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf2,0x00,0x00,0xb2,0xc0,0xdc,
  0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,0x00,0x80,0x80,0x80,0xff,0xff,0xff,0x00,
  0x00,0x00,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x03,
  0x42,0x08,0xaa,0xb1,0xfe,0x4c,0x34,0x06,0xa3,0x90,0x20,0xcc,0x1a,0xae,0xc4,0x95,
  0x75,0x6d,0x8f,0xa6,0x79,0x64,0xa6,0x0d,0x44,0x87,0x9a,0x0c,0xc1,0x9a,0xb4,0x3c,
  0x05,0x33,0xad,0xd7,0x43,0x50,0x0c,0xbf,0x20,0x10,0xb8,0x6a,0x0d,0x85,0x42,0x8d,
  0x11,0x79,0xec,0x65,0x7a,0x4c,0x64,0x23,0xb0,0x6c,0x26,0x55,0xd6,0xe6,0x6d,0xc7,
  0x55,0x24,0x00,0x00,0x3b
  };


// File Open
const unsigned char fileopen[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf2,0x00,0x00,0xb2,0xc0,0xdc,
  0x00,0x00,0x00,0x00,0xff,0x00,0xff,0xff,0x00,0x80,0x80,0x80,0xff,0xff,0xff,0x00,
  0x00,0x00,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x03,
  0x45,0x08,0xba,0x1c,0xfe,0x2c,0xae,0x20,0x44,0x90,0x91,0xd6,0x8b,0x95,0xb6,0x1c,
  0x00,0x79,0x95,0x25,0x3e,0x03,0x31,0x3c,0xa5,0x73,0xaa,0xf0,0xda,0xae,0x5e,0xfa,
  0xdc,0xb8,0xa3,0x06,0xc5,0xd0,0xff,0x3e,0x9f,0xc3,0x16,0x04,0x02,0x75,0x81,0xa2,
  0xb2,0x77,0x49,0xf2,0x96,0xc5,0x26,0x21,0x69,0x54,0x36,0xa9,0x50,0xa6,0xe7,0x59,
  0xfd,0x71,0x72,0x60,0x4e,0x02,0x00,0x3b
  };

// Print
const unsigned char print[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf2,0x00,0x00,0xb2,0xc0,0xdc,
  0x00,0x00,0x00,0xff,0xff,0xff,0xc0,0xc0,0xc0,0x80,0x80,0x80,0x00,0x80,0x80,0x00,
  0x00,0x00,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x03,
  0x49,0x08,0x0a,0xd1,0xbd,0x10,0x06,0x41,0x5f,0x94,0x62,0xea,0xbb,0x66,0x75,0x1c,
  0x93,0x09,0x43,0x19,0x84,0x1e,0x55,0x5d,0x8e,0x40,0x04,0x6f,0x66,0x29,0x41,0x91,
  0xaa,0xf5,0x49,0x3b,0x7c,0x7f,0x06,0x25,0x92,0x70,0x30,0x24,0x35,0x0a,0xc8,0xa4,
  0x52,0x79,0xb4,0x39,0x99,0x35,0x5b,0x74,0x49,0x95,0xfa,0xae,0x8f,0x1a,0xa1,0xb0,
  0xed,0x72,0xb9,0xba,0x17,0xd6,0x41,0x00,0x24,0x00,0x00,0x3b
  };

// File new
const unsigned char filenew[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf2,0x00,0x00,0xb2,0xc0,0xdc,
  0xff,0xff,0x00,0xff,0xff,0xff,0x80,0x80,0x80,0x00,0x00,0x00,0xc0,0xc0,0xc0,0x00,
  0x00,0x00,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x03,
  0x48,0x08,0x0a,0xb1,0xfe,0xab,0x41,0x28,0x18,0xa8,0xd3,0x85,0x3d,0xfa,0x80,0x5b,
  0x28,0x0e,0x44,0x16,0x16,0x82,0x30,0x08,0xe5,0xa3,0x06,0x68,0xba,0xb2,0xd0,0x10,
  0x08,0x31,0x49,0xec,0x8f,0x9d,0xfe,0x85,0x20,0xe1,0xb3,0xf0,0xfd,0x8e,0x43,0xc7,
  0xec,0xf8,0x4b,0x16,0x99,0x48,0xa2,0x62,0x09,0x75,0x4e,0xa1,0x4d,0x29,0x40,0xb7,
  0xeb,0x76,0xb5,0x5b,0x8f,0xd8,0x03,0x48,0x00,0x00,0x3b
  };

// Cut
const unsigned char cut[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf1,0x00,0x00,0xb2,0xc0,0xdc,
  0x00,0x00,0x00,0x00,0x00,0x80,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,
  0x10,0x00,0x00,0x02,0x29,0x84,0x8f,0xa9,0x10,0xd7,0xeb,0x22,0x34,0x8f,0x4d,0xda,
  0x42,0x9d,0xf9,0x3a,0xed,0x49,0xa1,0x10,0x08,0x21,0x60,0x0a,0xea,0xa4,0xa6,0x86,
  0xa9,0xc0,0x2e,0xba,0xc8,0xf4,0x9d,0xd8,0x28,0x5c,0xf3,0xe7,0x54,0x00,0x00,0x3b
  };
  
// Copy
const unsigned char copy[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf1,0x00,0x00,0xb2,0xc0,0xdc,
  0x00,0x00,0x00,0xff,0xff,0xff,0x00,0x00,0x80,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,
  0x10,0x00,0x00,0x02,0x38,0x84,0x8f,0x69,0xc1,0xad,0x18,0x84,0x14,0xec,0xad,0x49,
  0xa9,0x8d,0x8c,0x8e,0x0f,0x42,0x98,0x30,0x60,0xdf,0xc1,0x35,0xe5,0x54,0x0e,0x97,
  0x49,0x45,0xa0,0x9b,0x32,0xeb,0x48,0xe3,0x71,0x43,0x02,0x8d,0x3a,0x92,0xb8,0x14,
  0xa5,0x9f,0xec,0x71,0x33,0x21,0x67,0x33,0x8b,0xf3,0x99,0x28,0x00,0x00,0x3b
  };
  
  
// Paste
const unsigned char paste[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf2,0x00,0x00,0xb2,0xc0,0xdc,
  0x00,0x00,0x00,0xff,0xff,0x00,0xc0,0xc0,0xc0,0x80,0x80,0x00,0x80,0x80,0x80,0x00,
  0x00,0x80,0xff,0xff,0xff,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x03,
  0x4b,0x08,0xba,0xdc,0x1e,0x30,0x38,0x15,0xa1,0x10,0x55,0x52,0x52,0x48,0xc0,0x5f,
  0xd0,0x15,0x9a,0x48,0x0e,0x68,0x2a,0x7a,0x1b,0x99,0x41,0x05,0x49,0x8d,0x5c,0x4d,
  0x97,0x36,0x61,0xec,0xbc,0x51,0xd2,0x05,0xc3,0x61,0x38,0xdc,0xb5,0x6a,0x42,0xe2,
  0x41,0x68,0x00,0xac,0x46,0xc2,0xc8,0x92,0x17,0xc8,0x25,0x95,0xc5,0x27,0x27,0x9a,
  0x59,0x3a,0x2b,0x57,0x6c,0xb3,0xd1,0x2b,0x8f,0x27,0x93,0x04,0x00,0x3b
  };

  
// Penguin
const unsigned char penguin[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x12,0x00,0xf2,0x00,0x00,0xb2,0xc0,0xdc,
  0x80,0x80,0x80,0x00,0x00,0x00,0xc0,0xc0,0xc0,0x10,0x10,0x10,0xff,0xff,0xff,0xe0,
  0xa0,0x08,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x12,0x00,0x00,0x03,
  0x53,0x08,0xba,0x21,0x12,0x2b,0xc6,0xe6,0x9e,0x94,0x62,0x64,0x77,0xa3,0x20,0x4e,
  0x21,0x74,0x8b,0x60,0x9c,0x1a,0xa9,0x98,0xa8,0x45,0xb2,0x85,0x38,0x76,0x4f,0x6c,
  0xbb,0x93,0x60,0xdb,0x0d,0xe4,0xd9,0x83,0x1d,0xe7,0x57,0x18,0x04,0x6f,0xb8,0x4c,
  0xec,0x88,0x9c,0x01,0x0c,0x47,0x66,0xac,0xa2,0x38,0x19,0x76,0x36,0x83,0xc3,0xf0,
  0xb4,0x5e,0x77,0x03,0xaf,0xf8,0x7b,0x13,0x77,0xad,0xd3,0xad,0x75,0x61,0xa5,0x54,
  0x02,0x27,0x45,0x02,0x00,0x3b
  };


// Palette of colors
const unsigned char colorpal[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf2,0x00,0x00,0x80,0x80,0x80,
  0x00,0x00,0x00,0xff,0xff,0x00,0xff,0x00,0xff,0x00,0xff,0x00,0xff,0x00,0x00,0x00,
  0x00,0xff,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x03,
  0x47,0x08,0x0a,0xd1,0xfb,0x30,0x88,0xe9,0xe0,0x92,0x33,0x07,0xcb,0xf2,0xf8,0xd4,
  0xd3,0x64,0xc2,0x37,0x48,0x55,0x83,0x11,0x04,0x83,0x09,0x9b,0x23,0xb1,0x2d,0x49,
  0x6d,0xdd,0x42,0xc6,0x68,0x67,0xdf,0x23,0x58,0xa7,0x40,0x7c,0x05,0x85,0x12,0x62,
  0xe1,0x65,0xc3,0xa1,0x24,0x86,0x68,0x54,0x73,0xc9,0x48,0x0d,0x54,0xd1,0xf1,0x66,
  0x91,0xa9,0x70,0x11,0x30,0x07,0x90,0x00,0x00,0x3b
  };


// Big time logo
extern const unsigned char bigpenguin[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x30,0x00,0x39,0x00,0xf3,0x00,0x00,0xb2,0xc0,0xdc,
  0x5a,0x52,0x41,0x0e,0x0b,0x0b,0x47,0x3b,0x26,0x71,0x6e,0x67,0xb7,0xb5,0xb0,0xd4,
  0xd2,0xce,0xf9,0xf7,0xf7,0x99,0x8d,0x77,0xa3,0x77,0x1b,0xee,0xba,0x12,0xbe,0xa2,
  0x14,0x8b,0x5e,0x07,0xb3,0x8b,0x27,0xee,0xc7,0x1c,0xd2,0x98,0x0f,0x2c,0x00,0x00,
  0x00,0x00,0x30,0x00,0x39,0x00,0x00,0x04,0xfe,0x10,0xc8,0x49,0xab,0x0d,0x22,0x8b,
  0x61,0xbb,0xff,0x20,0xa6,0x8d,0x1c,0x68,0x9e,0xd3,0x30,0xae,0x19,0x81,0xbe,0x9d,
  0xca,0x66,0x81,0xe8,0xc2,0x38,0x31,0x6b,0x84,0x28,0xe0,0x38,0xd9,0x6e,0x15,0x00,
  0xbe,0x84,0xc3,0x91,0x11,0x95,0x9c,0xdd,0x96,0x9f,0xe1,0x20,0x30,0x40,0x16,0xa1,
  0x51,0x01,0xa1,0xe0,0x1b,0x14,0x0a,0x06,0x9d,0xe6,0x8a,0xed,0x6c,0x0c,0x85,0x83,
  0xcc,0x70,0x40,0x18,0x0c,0xa4,0xb2,0x67,0x5a,0x2d,0x00,0x32,0x07,0x4c,0x40,0xad,
  0x29,0xc9,0x2b,0x74,0x5a,0x77,0x02,0x06,0x6b,0x5d,0x7f,0x16,0x2a,0x05,0x02,0x09,
  0x0a,0x0a,0x0b,0x0b,0x0c,0x1b,0x7c,0x19,0x7e,0x88,0x12,0x2a,0x08,0x0d,0x0a,0x0e,
  0x9d,0x9d,0x0a,0x12,0x2b,0x97,0x14,0x18,0x09,0x0f,0x9c,0x9e,0xa9,0xa2,0xa3,0x12,
  0x54,0x9b,0xa8,0x9e,0x0a,0x0f,0x0f,0x2b,0x4f,0x97,0x18,0x0c,0x8e,0xb1,0x0d,0xb3,
  0xb4,0x1b,0x34,0xac,0x00,0x18,0x08,0x0c,0x0f,0x90,0x09,0xa6,0xc6,0x00,0x2a,0x45,
  0x19,0xc1,0x3a,0x05,0x08,0xc9,0x0f,0x0d,0x0d,0x60,0x68,0x2d,0x95,0xac,0x19,0x6f,
  0xd2,0xc8,0x0d,0x00,0x6c,0x07,0x94,0x3c,0xb7,0x93,0x68,0x5f,0x5f,0xe2,0xe3,0x8b,
  0x71,0x88,0xdc,0xec,0xea,0xe3,0xf3,0xe3,0x48,0x82,0x72,0x2d,0xf4,0xeb,0xf4,0xec,
  0x2c,0x72,0x32,0xfc,0x02,0xf2,0x63,0x61,0x09,0x88,0x88,0x3d,0x02,0x13,0xda,0xdb,
  0xb0,0x44,0xcc,0x8f,0x7d,0x01,0xdf,0x48,0x44,0xe0,0xc4,0x88,0x10,0x30,0x09,0xe7,
  0x49,0x4c,0xb3,0xf0,0x07,0x90,0x3e,0xfe,0x10,0x33,0x1e,0xe8,0x56,0x11,0x86,0x8f,
  0x34,0x22,0x05,0xc2,0x99,0x81,0x63,0x4c,0xca,0x8c,0x77,0xec,0xd9,0x02,0xe1,0x30,
  0xe4,0xcb,0x71,0x2b,0x7f,0x69,0x43,0x21,0x03,0xe1,0x4d,0x81,0x3b,0x5e,0x68,0x40,
  0xf9,0x33,0x60,0xc7,0x99,0x1d,0x1c,0x16,0x55,0x38,0x83,0x4c,0x16,0x3c,0x4b,0x03,
  0x3a,0x74,0xf7,0x41,0xc8,0x80,0xa8,0x52,0x83,0x82,0xe8,0x73,0x15,0x2b,0xbd,0x41,
  0xfe,0xb6,0x56,0xf2,0xe9,0xf5,0xc0,0xd4,0x3e,0x26,0x04,0x30,0x48,0xa0,0xa5,0xec,
  0x3c,0x1f,0x95,0x18,0x82,0x38,0xe5,0x48,0x00,0xd1,0xa8,0x9d,0xe0,0xfe,0x72,0x6a,
  0x81,0x97,0x23,0x59,0x02,0x10,0x60,0x35,0xc0,0xa9,0x01,0x8b,0x04,0x0b,0x3e,0x20,
  0x98,0xf5,0x57,0x01,0xdb,0xae,0x45,0x09,0x2b,0x90,0x34,0x82,0x41,0xa7,0x0e,0xd2,
  0xfc,0x7e,0xe2,0xa4,0x02,0xf2,0x4d,0x00,0x8e,0x12,0x08,0xc9,0xe5,0x88,0x82,0xa6,
  0x5e,0x8d,0x1b,0x2f,0xa8,0xf4,0xd3,0x00,0xe8,0x59,0xc8,0xd6,0xd2,0x55,0x80,0x60,
  0x82,0xdf,0xd4,0x8d,0x3b,0xb1,0xcd,0x70,0x37,0xa1,0xeb,0x53,0xbd,0x18,0x73,0x72,
  0x54,0x1b,0x00,0x02,0x5d,0xa9,0x1d,0xa4,0x76,0xf9,0xb2,0xc0,0x2b,0x47,0xb3,0x1b,
  0x4b,0x58,0x3c,0x3c,0x77,0xee,0xdd,0x64,0x33,0x32,0x78,0x8e,0xbb,0x31,0x02,0xea,
  0xdd,0xab,0x73,0x4a,0xd0,0x5b,0xe5,0x86,0xe8,0x7f,0x95,0xa7,0x06,0x8f,0x5b,0x7d,
  0x63,0x9b,0xf4,0xda,0x05,0xa0,0xeb,0xde,0xba,0x72,0xf6,0xe1,0xab,0x27,0x10,0xe9,
  0xb0,0x58,0xfa,0xff,0x8d,0xf1,0x67,0x82,0x1e,0x72,0xb8,0x89,0x16,0xce,0x1b,0xd1,
  0x10,0x60,0x8f,0x7f,0xf9,0xc9,0x52,0x5b,0x70,0xe1,0xb9,0xf7,0x80,0x68,0x4d,0x64,
  0xc0,0x60,0x7d,0x8e,0x34,0x50,0x9c,0x71,0xdf,0x04,0x27,0x5c,0x63,0x13,0xea,0xb5,
  0xc3,0x76,0xe8,0x51,0x83,0x94,0x04,0x04,0x10,0x80,0xcc,0x8a,0x02,0xce,0xd2,0x40,
  0x02,0x0c,0x58,0x55,0xc5,0x8c,0x55,0xd0,0x90,0x40,0x8b,0xd4,0x30,0x70,0xa2,0x05,
  0x29,0x32,0xb0,0xd6,0x8a,0x3e,0x56,0xb1,0x23,0x05,0x03,0xfc,0x08,0xe4,0x86,0x2f,
  0xf4,0xe0,0x63,0x90,0x7c,0x7d,0x40,0xc5,0x92,0x0c,0x34,0x39,0x41,0x04,0x00,0x3b
  };


// Fonts
const unsigned char fonts[]={
  0x47,0x49,0x46,0x38,0x37,0x61,0x10,0x00,0x10,0x00,0xf0,0x00,0x00,0xb2,0xc0,0xdc,
  0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x10,0x00,0x10,0x00,0x00,0x02,0x25,0x84,
  0x8f,0x09,0xc1,0xed,0x8c,0x9a,0x81,0x8b,0xce,0x70,0xac,0x55,0x5b,0x45,0xfc,0x74,
  0xd5,0x32,0x95,0xc9,0x86,0x79,0x63,0xa6,0xae,0xa6,0x87,0xb6,0xb1,0xaa,0x89,0x61,
  0x8b,0xe7,0x6d,0x01,0x00,0x3b
  };
  
  
// Patterns
const FXchar *patterns[]={
  "All Files",          "*",
  "C++ Source Files",   "*.cpp",
  "C Source Files",     "*.c",
  "C++ Header Files",   "*.h",
  "Text Files",         "*.txt",
  "HTML Header Files",  "*.html",
  NULL
  };

/*******************************************************************************/
  
// Map
FXDEFMAP(TextWindow) TextWindowMap[]={
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_ABOUT,        TextWindow::onCmdAbout),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_REVERT,       TextWindow::onCmdRevert),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_REVERT,       TextWindow::onUpdRevert),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_OPEN,         TextWindow::onCmdOpen),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_SAVE,         TextWindow::onCmdSave),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_SAVE,         TextWindow::onUpdSave),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_SAVEAS,       TextWindow::onCmdSaveAs),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_NEW,          TextWindow::onCmdNew),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_TITLE,        TextWindow::onUpdTitle),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_FONT,         TextWindow::onCmdFont),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_QUIT,         TextWindow::onCmdQuit),
  FXMAPFUNC(SEL_SIGNAL,  TextWindow::ID_QUIT,         TextWindow::onCmdQuit),
  FXMAPFUNC(SEL_CLOSE,   TextWindow::ID_TITLE,        TextWindow::onCmdQuit),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_PRINT,        TextWindow::onCmdPrint),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_TREELIST,     TextWindow::onCmdTreeList),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_BACKCOLOR,    TextWindow::onCmdBackColor),
  FXMAPFUNC(SEL_CHANGED, TextWindow::ID_BACKCOLOR,    TextWindow::onCmdBackColor),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_BACKCOLOR,    TextWindow::onUpdBackColor),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_SELBACKCOLOR, TextWindow::onCmdSelBackColor),
  FXMAPFUNC(SEL_CHANGED, TextWindow::ID_SELBACKCOLOR, TextWindow::onCmdSelBackColor),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_SELBACKCOLOR, TextWindow::onUpdSelBackColor),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_TEXTCOLOR,    TextWindow::onCmdTextColor),
  FXMAPFUNC(SEL_CHANGED, TextWindow::ID_TEXTCOLOR,    TextWindow::onCmdTextColor),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_TEXTCOLOR,    TextWindow::onUpdTextColor),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_SELTEXTCOLOR, TextWindow::onCmdSelTextColor),
  FXMAPFUNC(SEL_CHANGED, TextWindow::ID_SELTEXTCOLOR, TextWindow::onCmdSelTextColor),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_SELTEXTCOLOR, TextWindow::onUpdSelTextColor),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_RECENTFILE,   TextWindow::onCmdRecentFile),
  FXMAPFUNC(SEL_UPDATE,  TextWindow::ID_TOGGLE_WRAP,  TextWindow::onUpdWrap),
  FXMAPFUNC(SEL_COMMAND, TextWindow::ID_TOGGLE_WRAP,  TextWindow::onCmdWrap),
  };


// Object implementation
FXIMPLEMENT(TextWindow,FXMainWindow,TextWindowMap,ARRAYNUMBER(TextWindowMap))


// Make some windows
TextWindow::TextWindow(FXApp* a):FXMainWindow(a,"FOX Text Editor: - untitled",NULL,NULL,DECOR_ALL,0,0,850,600,0,0){
  FXIcon *bigp=new FXGIFIcon(getApp(),bigpenguin);
  FXIcon *tinyp=new FXGIFIcon(getApp(),penguin);
  
  setIcon(bigp);
  setMiniIcon(tinyp);
  
  // Make main window; set myself as the target
  setTarget(this);
  setSelector(ID_TITLE);
  
  // Make color dialog
  FXColorDialog *colordlg=new FXColorDialog(this,"Color Dialog");
  
  // Make menu bar
  menubar=new FXMenuBar(this,FRAME_RAISED|LAYOUT_SIDE_TOP|LAYOUT_FILL_X);
  
  // Tool bar
  FXToolbarShell* dragshell=new FXToolbarShell(this,0,100,100);
  toolbar=new FXToolbar(this,dragshell,FRAME_RAISED|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|PACK_UNIFORM_WIDTH|PACK_UNIFORM_HEIGHT,0,0,0,0, 2,2,2,2, 0,0);
  
  // Status bar
  statusbar=new FXStatusbar(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X|STATUSBAR_WITH_DRAGCORNER);
  
  // The good old penguin, what would we be without it?
  new FXButton(statusbar,"\tHello, I'm Tux...\nThe symbol for the Linux Operating System.\nAnd all it stands for.",tinyp,NULL,0,LAYOUT_FILL_Y|LAYOUT_RIGHT);
  
  // File menu
  filemenu=new FXMenuPane(this);
  new FXMenuTitle(menubar,"&File",NULL,filemenu);
    
  // Edit Menu
  FXMenuPane *editmenu=new FXMenuPane(this);
  new FXMenuTitle(menubar,"&Edit",NULL,editmenu);
  
  // Options Menu
  FXMenuPane *optionsmenu=new FXMenuPane(this);
  new FXMenuTitle(menubar,"&Options",NULL,optionsmenu);
  
  // View menu
  FXMenuPane *viewmenu=new FXMenuPane(this);
  new FXMenuTitle(menubar,"&View",NULL,viewmenu);
    
  // Help menu
  helpmenu=new FXMenuPane(this);
  new FXMenuTitle(menubar,"&Help",NULL,helpmenu,LAYOUT_RIGHT);

  // Splitter
  splitter=new FXSplitter(this,LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y|SPLITTER_TRACKING);
  
  // Sunken border for tree
  treebox=new FXVerticalFrame(splitter,LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
  
  // Make tree
  FXHorizontalFrame* treeframe=new FXHorizontalFrame(treebox,FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
  dirlist=new FXDirList(treeframe,0,this,ID_TREELIST,DIRLIST_SHOWFILES|TREELIST_BROWSESELECT|TREELIST_SHOWS_LINES|TREELIST_SHOWS_BOXES|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  FXHorizontalFrame* filterframe=new FXHorizontalFrame(treebox,LAYOUT_FILL_X);
  new FXLabel(filterframe,"Filter:");
  filter=new FXTextField(filterframe,10,dirlist,FXDirList::ID_SET_PATTERN,JUSTIFY_RIGHT|FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X);
  
  // Sunken border for text widget
  FXHorizontalFrame *textbox=new FXHorizontalFrame(splitter,FRAME_SUNKEN|FRAME_THICK|LAYOUT_FILL_X|LAYOUT_FILL_Y,0,0,0,0, 0,0,0,0);
  
  // Make editor window
  editor=new FXText(textbox,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y);
  
  // Show column number in status bar
  new FXTextField(statusbar,4,editor,FXText::ID_CURSOR_COLUMN,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,1,1,0,0);

  // Caption before number  
  new FXLabel(statusbar,"Col:",NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);
  
  // Show line number in status bar
  new FXTextField(statusbar,4,editor,FXText::ID_CURSOR_LINE,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,1,1,0,0);

  // Caption before number  
  new FXLabel(statusbar,"Line:",NULL,LAYOUT_RIGHT|LAYOUT_CENTER_Y);
  
  // Text color
  textwell=new FXColorWell(statusbar,FXRGB(0,0,0),this,ID_TEXTCOLOR,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);
  backwell=new FXColorWell(statusbar,FXRGB(0,0,0),this,ID_BACKCOLOR,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);
  seltextwell=new FXColorWell(statusbar,FXRGB(0,0,0),this,ID_SELTEXTCOLOR,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);
  selbackwell=new FXColorWell(statusbar,FXRGB(0,0,0),this,ID_SELBACKCOLOR,FRAME_SUNKEN|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y);
  
  textwell->setTipText("Text");
  backwell->setTipText("Background");
  seltextwell->setTipText("Selection");
  selbackwell->setTipText("Selection Background");
  
  textwell->setHelpText("Change text color.");
  backwell->setHelpText("Change background color.");
  seltextwell->setHelpText("Change selection color.");
  selbackwell->setHelpText("Change selection background color.");
  
  // Make some icons
  FXIcon *filenewicon=new FXGIFIcon(getApp(),filenew);
  FXIcon *fileopenicon=new FXGIFIcon(getApp(),fileopen);
  FXIcon *filesaveicon=new FXGIFIcon(getApp(),filesave);
  FXIcon *filesaveasicon=new FXGIFIcon(getApp(),filesaveas,0,IMAGE_ALPHAGUESS);
  FXIcon *printicon=new FXGIFIcon(getApp(),print);
  FXIcon *cuticon=new FXGIFIcon(getApp(),cut);
  FXIcon *copyicon=new FXGIFIcon(getApp(),copy);
  FXIcon *pasteicon=new FXGIFIcon(getApp(),paste);
  
  // Toobar buttons: File manipulation
  new FXButton(toolbar,"&New\tNew\tCreate new document.",filenewicon,this,ID_NEW,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  new FXButton(toolbar,"&Open\tOpen\tOpen document file.",fileopenicon,this,ID_OPEN,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  new FXButton(toolbar,"&Save\tSave\tSave document.",filesaveicon,this,ID_SAVE,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  new FXButton(toolbar,"S&ave As\tSave As\tSave document to another file.",filesaveasicon,this,ID_SAVEAS,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  
  // Toobar buttons: Print
  new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,20,20);
  new FXButton(toolbar,"&Print\tPrint\tPrint document.",printicon,this,ID_PRINT,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  
  // Toobar buttons: Editing
  new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,20,20);
  new FXButton(toolbar,"Cut\tCut",cuticon,editor,FXText::ID_CUT_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  new FXButton(toolbar,"Copy\tCopy",copyicon,editor,FXText::ID_COPY_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  new FXButton(toolbar,"Paste\tPaste",pasteicon,editor,FXText::ID_PASTE_SEL,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  
  // Color
  new FXFrame(toolbar,LAYOUT_TOP|LAYOUT_LEFT|LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT,0,0,20,20);
  new FXButton(toolbar,"&Colors\tColors\tDisplay color dialog.",new FXGIFIcon(getApp(),colorpal),colordlg,FXWindow::ID_SHOW,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  new FXButton(toolbar,"Fon&ts\tFonts\tDisplay font dialog.",new FXGIFIcon(getApp(),fonts),this,ID_FONT,ICON_ABOVE_TEXT|BUTTON_TOOLBAR|FRAME_RAISED|LAYOUT_TOP|LAYOUT_LEFT);
  
  // File Menu entries
  new FXMenuCommand(filemenu,"&Open...        \tCtl-O\tOpen document file.",fileopenicon,this,ID_OPEN);
  new FXMenuCommand(filemenu,"&Revert...\t\tRevert to saved.",NULL,this,ID_REVERT);
  new FXMenuCommand(filemenu,"&New...\tCtl-N\tCreate new document.",filenewicon,this,ID_NEW);
  new FXMenuCommand(filemenu,"&Save\tCtl-S\tSave changes to file.",filesaveicon,this,ID_SAVE);
  new FXMenuCommand(filemenu,"Save &As...\t\tSave document to another file.",filesaveasicon,this,ID_SAVEAS);
  new FXMenuCommand(filemenu,"&Print...\tCtl-P\tPrint document.",printicon,this,ID_PRINT);
  new FXMenuCommand(filemenu,"&Editable\t\tDocument editable.",NULL,editor,FXText::ID_TOGGLE_EDITABLE);
  new FXMenuCommand(filemenu,"&Iconify...\t\tIconify editor.",NULL,this,FXTopWindow::ID_ICONIFY);
  
  // Recent file menu; this automatically hides if there are no files
  FXMenuSeparator* sep1=new FXMenuSeparator(filemenu);
  sep1->setTarget(&mrufiles);
  sep1->setSelector(FXRecentFiles::ID_ANYFILES);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_1);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_2);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_3);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_4);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_5);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_6);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_7);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_8);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_9);
  new FXMenuCommand(filemenu,NULL,NULL,&mrufiles,FXRecentFiles::ID_FILE_10);
  new FXMenuCommand(filemenu,"&Clear Recent Files",NULL,&mrufiles,FXRecentFiles::ID_CLEAR);
  FXMenuSeparator* sep2=new FXMenuSeparator(filemenu);
  sep2->setTarget(&mrufiles);
  sep2->setSelector(FXRecentFiles::ID_ANYFILES);
  new FXMenuCommand(filemenu,"&Quit\tCtl-Q",NULL,this,ID_QUIT,MENU_DEFAULT);
  
  // Edit Menu entries
  new FXMenuCommand(editmenu,"&Undo\tCtl-Z\tUndo last change.");
  new FXMenuCommand(editmenu,"&Redo\tCtl-R\tRedo last undo.");
  new FXMenuCommand(editmenu,"&Copy\tCtl-C\tCopy selection to clipboard.",copyicon,editor,FXText::ID_COPY_SEL);
  new FXMenuCommand(editmenu,"C&ut\tCtl-X\tCut selection to clipboard.",cuticon,editor,FXText::ID_CUT_SEL);
  new FXMenuCommand(editmenu,"&Paste\tCtl-V\tPaste from clipboard.",pasteicon,editor,FXText::ID_PASTE_SEL);
  new FXMenuCommand(editmenu,"&Delete\t\tDelete selection.",NULL,editor,FXText::ID_DELETE_SEL);
  
  // Options menu
  new FXMenuCommand(optionsmenu,"Font...\t\tChange text font.",NULL,this,ID_FONT);
  new FXMenuCommand(optionsmenu,"Colors...\t\tChange text colors.",NULL,colordlg,FXWindow::ID_SHOW);
  new FXMenuCommand(optionsmenu,"Word Wrap\t\tToggle word wrap.",NULL,this,ID_TOGGLE_WRAP);
  
  // View Menu entries
  new FXMenuCommand(viewmenu,"File list\t\tDisplay file list.",NULL,treebox,FXWindow::ID_TOGGLESHOWN);
  new FXMenuCommand(viewmenu,"Show hidden files\t\tShow hidden files and directories.",NULL,dirlist,FXDirList::ID_TOGGLE_HIDDEN);
  new FXMenuCommand(viewmenu,"Toolbar\t\tDisplay toolbar.",NULL,toolbar,FXWindow::ID_TOGGLESHOWN);
  new FXMenuCommand(viewmenu,"Status line\t\tDisplay status line.",NULL,statusbar,FXWindow::ID_TOGGLESHOWN);
  
  // Help Menu entries
  new FXMenuCommand(helpmenu,"&About FOX...",NULL,this,ID_ABOUT,0);
  
  // Make a tool tip
  new FXTooltip(getApp(),0);
  
  // Recent files
  mrufiles.setTarget(this);
  mrufiles.setSelector(ID_RECENTFILE);

  // Initialize file name
  filename="untitled";
  namechanged=FALSE;
  modified=TRUE;
  }
  


// About box
long TextWindow::onCmdAbout(FXObject*,FXSelector,void*){
  FXGIFIcon icon(getApp(),bigpenguin);
  FXMessageBox about(this,"About Text Edit","Text Editor Demonstrates the FOX Text Widget.\n\nUsing the FOX C++ GUI Library (http://www.cfdrc.com/FOX/fox.html)\n\nCopyright (C) 2000 Jeroen van der Zijp (jvz@cfdrc.com)",&icon,MBOX_OK|DECOR_TITLE|DECOR_BORDER);
  about.execute();  
  return 1;
  }


// Change font
long TextWindow::onCmdFont(FXObject*,FXSelector,void*){
  FXFontDialog fontdlg(this,"Change Font",DECOR_BORDER|DECOR_TITLE);
  FXFont *oldfont=editor->getFont();
  FXFontDesc fontdesc;
  oldfont->getFontDesc(fontdesc);
  fontdlg.setFontSelection(fontdesc);
  if(fontdlg.execute()){
    fontdlg.getFontSelection(fontdesc);
    FXFont *newfont=new FXFont(getApp(),fontdesc);
    newfont->create();
    editor->setFont(newfont);
    if(oldfont!=getApp()->getNormalFont()){
      delete oldfont;
      }
    }
  editor->setFocus();
  return 1;
  }


// Load file
FXbool TextWindow::loadfile(const FXString& file){
  struct stat info;
  char *text;
  FILE *fp;
  FXuint n;
  if(stat(file.text(),&info)!=0){
    FXMessageBox::error(this,MBOX_OK,"Error Loading File","Non-existing file: %s",file.text());
    return FALSE;
    }
  fp=fopen(file.text(),"r");
  if(!fp){
    FXMessageBox::error(this,MBOX_OK,"Error Loading File","Unable to open file: %s",file.text());
    return FALSE;
    }
  getApp()->beginWaitCursor();
  FXCALLOC(&text,FXchar,info.st_size+1);
  n=fread(text,1,info.st_size,fp);
  // Under WIN32, when the file is opened in text mode, fread() converts CR/LF pairs
  // to single LF characters; so the number of items returned by fread() will be less
  // than the file size reported by info.st_size.
#ifndef WIN32
  if(n!=info.st_size){
    FXMessageBox::error(this,MBOX_OK,"Error Reading File","Unable to load entire file: %s",file.text());
    }
  else{
#endif
    text[n]='\0';
    editor->setText(text);
    editor->setModified(FALSE);
#ifndef WIN32
    }
#endif
  FXFREE(&text);
  fclose(fp);
  getApp()->endWaitCursor();
  return TRUE;
  }


// Save file
FXbool TextWindow::savefile(const FXString& file){
  FILE *fp;
  const char *text;
  FXuint n,len;
  fp=fopen(file.text(),"w");
  if(!fp){
    FXMessageBox::error(this,MBOX_OK,"Error Saving File","Unable to open file: %s",file.text());
    return FALSE;
    }
  getApp()->beginWaitCursor();
  text=editor->getText();
  len=editor->getLength()-1;
  n=fwrite(text,1,len,fp);
  if(n!=len){
    FXMessageBox::error(this,MBOX_OK,"Error Saving File","Unable to save entire file: %s",file.text());
    }
  fclose(fp);
  editor->setModified(FALSE);
  getApp()->endWaitCursor();
  return TRUE;
  }


// Toggle wrap mode
long TextWindow::onCmdWrap(FXObject*,FXSelector,void*){
  editor->setTextStyle(editor->getTextStyle()^TEXT_WORDWRAP);
  return 1;
  }


// Update toggle wrap mode
long TextWindow::onUpdWrap(FXObject* sender,FXSelector,void*){
  if(editor->getTextStyle()&TEXT_WORDWRAP)
    sender->handle(this,MKUINT(ID_CHECK,SEL_COMMAND),NULL);
  else
    sender->handle(this,MKUINT(ID_UNCHECK,SEL_COMMAND),NULL);
  return 1;
  }


// Revert to saved
long TextWindow::onCmdRevert(FXObject*,FXSelector,void*){
  loadfile(filename);
  return 1;
  }


// Revert Update
long TextWindow::onUpdRevert(FXObject* sender,FXSelector,void* ptr){
  if(modified!=editor->isModified()){
    modified=editor->isModified();
    FXuint msg=modified ? FXWindow::ID_ENABLE : FXWindow::ID_DISABLE;
    sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
    }
  return 1;
  }


// Open
long TextWindow::onCmdOpen(FXObject*,FXSelector,void*){
  FXuint answer;
  if(editor->isModified()){
    answer=FXMessageBox::question(this,MBOX_YES_NO_CANCEL,"Unsaved Document","Save current document to file?");
    if(answer==MBOX_CLICKED_CANCEL) return 1;
    if(answer==MBOX_CLICKED_YES){
      if(!namechanged){
        FXFileDialog savedialog(this,"Save Document");
        savedialog.setFilename(filename);
        if(!savedialog.execute()) return 1;
        if(fxexists(savedialog.getFilename().text())){
          if(MBOX_CLICKED_NO==FXMessageBox::question(this,MBOX_YES_NO,"Overwrite Document","Overwrite existing document?")) return 1;
          }
        filename=savedialog.getFilename();
        }
      savefile(filename);
      }
    }
  FXFileDialog open(this,"Open Document");
  open.setFilename(filename);
  open.setPatternList(patterns);
  if(open.execute()){
    filename=open.getFilename();
    dirlist->setCurrentFile(filename);
    mrufiles.appendFile(filename);
    loadfile(filename);
    namechanged=TRUE;
    }
  return 1;
  }


// Open recent file
long TextWindow::onCmdRecentFile(FXObject*,FXSelector,void* ptr){
  FXuint answer;
  if(editor->isModified()){
    answer=FXMessageBox::question(this,MBOX_YES_NO_CANCEL,"Unsaved Document","Save current document to file?");
    if(answer==MBOX_CLICKED_CANCEL) return 1;
    if(answer==MBOX_CLICKED_YES){
      if(!namechanged){
        FXFileDialog savedialog(this,"Save Document");
        savedialog.setFilename(filename);
        if(!savedialog.execute()) return 1;
        if(fxexists(savedialog.getFilename().text())){
          if(MBOX_CLICKED_NO==FXMessageBox::question(this,MBOX_YES_NO,"Overwrite Document","Overwrite existing document?")) return 1;
          }
        filename=savedialog.getFilename();
        }
      savefile(filename);
      }
    }
  filename=(FXchar*)ptr;
  dirlist->setCurrentFile(filename);
  loadfile(filename);
  namechanged=TRUE;
  return 1;
  }


// Save
long TextWindow::onCmdSave(FXObject* sender,FXSelector sel,void* ptr){
  if(!namechanged) return onCmdSaveAs(sender,sel,ptr);
  savefile(filename);
  return 1;
  }


// Save Update
long TextWindow::onUpdSave(FXObject* sender,FXSelector,void* ptr){
  FXuint msg=editor->isModified() ? FXWindow::ID_ENABLE : FXWindow::ID_DISABLE;
  sender->handle(this,MKUINT(msg,SEL_COMMAND),ptr);
  return 1;
  }


// Save As
long TextWindow::onCmdSaveAs(FXObject*,FXSelector,void*){
  FXFileDialog savedialog(this,"Save Document");
  savedialog.setFilename(filename);
  if(savedialog.execute()){
    if(fxexists(savedialog.getFilename().text())){
      if(MBOX_CLICKED_NO==FXMessageBox::question(this,MBOX_YES_NO,"Overwrite Document","Overwrite existing document?")) return 1;
      }
    filename=savedialog.getFilename();
    dirlist->setCurrentFile(filename);
    mrufiles.appendFile(filename);
    savefile(filename);
    namechanged=TRUE;
    }
  return 1;
  }


// New
long TextWindow::onCmdNew(FXObject*,FXSelector,void*){
  FXuint answer;
  if(editor->isModified()){
    answer=FXMessageBox::question(this,MBOX_YES_NO_CANCEL,"Unsaved Document","Save current document to file?");
    if(answer==MBOX_CLICKED_CANCEL) return 1;
    if(answer==MBOX_CLICKED_YES){
      if(!namechanged){
        FXFileDialog savedialog(this,"Save Document");
        savedialog.setFilename(filename);
        if(!savedialog.execute()) return 1;
        if(fxexists(savedialog.getFilename().text())){
          if(MBOX_CLICKED_NO==FXMessageBox::question(this,MBOX_YES_NO,"Overwrite Document","Overwrite existing document?")) return 1;
          }
        filename=savedialog.getFilename();
        }
      savefile(filename);
      }
    }
  filename="untitled";
  editor->setText("");
  namechanged=FALSE;
  editor->setModified(FALSE);
  return 1;
  }


// Quit
long TextWindow::onCmdQuit(FXObject*,FXSelector,void*){
  FXuint answer;
  char fontspec[200];
  FXFontDesc fontdesc;
  if(editor->isModified()){
    answer=FXMessageBox::question(this,MBOX_YES_NO_CANCEL,"Unsaved Document","Save current document to file?");
    if(answer==MBOX_CLICKED_CANCEL) return 1;
    if(answer==MBOX_CLICKED_YES){
      if(!namechanged){
        FXFileDialog savedialog(this,"Save Document");
        savedialog.setFilename(filename);
        if(!savedialog.execute()) return 1;
        if(fxexists(savedialog.getFilename().text())){
          if(MBOX_CLICKED_NO==FXMessageBox::question(this,MBOX_YES_NO,"Overwrite Document","Overwrite existing document?")) return 1;
          }
        filename=savedialog.getFilename();
        }
      savefile(filename);
      }
    }
  
  // Write changed colors back to registry
  getApp()->reg().writeColorEntry("SETTINGS","textbackground",editor->getBackColor());
  getApp()->reg().writeColorEntry("SETTINGS","textforeground",editor->getTextColor());
  getApp()->reg().writeColorEntry("SETTINGS","textselbackground",editor->getSelBackColor());
  getApp()->reg().writeColorEntry("SETTINGS","textselforeground",editor->getSelTextColor());
  
  // Write new window size back to registry
  getApp()->reg().writeIntEntry("SETTINGS","x",getX());
  getApp()->reg().writeIntEntry("SETTINGS","y",getY());
  getApp()->reg().writeIntEntry("SETTINGS","width",getWidth());
  getApp()->reg().writeIntEntry("SETTINGS","height",getHeight());
  
  // Were showing hidden files
  getApp()->reg().writeIntEntry("SETTINGS","showhiddenfiles",dirlist->showHiddenFiles());
  
  // Was tree shown?
  getApp()->reg().writeIntEntry("SETTINGS","hidetree",!treebox->shown());
  
  // Width of tree
  getApp()->reg().writeIntEntry("SETTINGS","treewidth",treebox->getWidth());
  
  // Font
  editor->getFont()->getFontDesc(fontdesc);
  fxunparsefontdesc(fontspec,fontdesc);
  getApp()->reg().writeStringEntry("SETTINGS","font",fontspec);

  // Quit
  getApp()->exit(0);
  return 1;
  }


// Update title
long TextWindow::onUpdTitle(FXObject* sender,FXSelector,void*){
  FXString title="FOX Text Editor:- " + filename;
  if(editor->isModified()) title+="*";
  sender->handle(this,MKUINT(FXWindow::ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&title);
  return 1;
  }


// Print the text
long TextWindow::onCmdPrint(FXObject*,FXSelector,void*){
  FXPrintDialog dlg(this,"Print File");
  FXPrinter printer;
  if(dlg.execute()){
    dlg.getPrinter(printer);
    FXTRACE((100,"Printer = %s\n",printer.name));
    }
  return 1;
  }


// Command from the tree list
long TextWindow::onCmdTreeList(FXObject*,FXSelector,void* ptr){
  FXTreeItem *item=(FXTreeItem*)ptr;
  FXuint answer;
  if(!item || !dirlist->isItemFile(item)) return 1;
  if(editor->isModified()){
    answer=FXMessageBox::question(this,MBOX_YES_NO_CANCEL,"Unsaved Document","Save current document to file?");
    if(answer==MBOX_CLICKED_CANCEL) return 1;
    if(answer==MBOX_CLICKED_YES){
      if(!namechanged){
        FXFileDialog savedialog(this,"Save Document");
        if(!savedialog.execute()) return 1;
        if(fxexists(savedialog.getFilename().text())){
          if(MBOX_CLICKED_NO==FXMessageBox::question(this,MBOX_YES_NO,"Overwrite Document","Overwrite existing document?")) return 1;
          }
        filename=savedialog.getFilename();
        }
      savefile(filename);
      }
    }
  filename=dirlist->getItemPathname(item);
  mrufiles.appendFile(filename);
  loadfile(filename);
  namechanged=TRUE;
  return 1;
  }


// Change both text and tree background color
long TextWindow::onCmdBackColor(FXObject*,FXSelector,void* ptr){
  editor->setBackColor((FXColor)(long)ptr);
  dirlist->setBackColor((FXColor)(long)ptr);
  return 1;
  }


// Update background color
long TextWindow::onUpdBackColor(FXObject* sender,FXSelector,void* ptr){
  editor->handle(sender,MKUINT(FXText::ID_BACK_COLOR,SEL_UPDATE),ptr);
  return 1;
  }

// Change both text and tree selected background color
long TextWindow::onCmdSelBackColor(FXObject*,FXSelector,void* ptr){
  editor->setSelBackColor((FXColor)(long)ptr);
  dirlist->setSelBackColor((FXColor)(long)ptr);
  return 1;
  }


// Update selected background color
long TextWindow::onUpdSelBackColor(FXObject* sender,FXSelector,void* ptr){
  editor->handle(sender,MKUINT(FXText::ID_SELBACK_COLOR,SEL_UPDATE),ptr);
  return 1;
  }


// Change both text and tree text color
long TextWindow::onCmdTextColor(FXObject*,FXSelector,void* ptr){
  editor->setTextColor((FXColor)(long)ptr);
  dirlist->setTextColor((FXColor)(long)ptr);
  return 1;
  }


// Forward GUI update to text widget
long TextWindow::onUpdTextColor(FXObject* sender,FXSelector,void* ptr){
  editor->handle(sender,MKUINT(FXText::ID_TEXT_COLOR,SEL_UPDATE),ptr);
  return 1;
  }


// Change both text and tree
long TextWindow::onCmdSelTextColor(FXObject*,FXSelector,void* ptr){
  editor->setSelTextColor((FXColor)(long)ptr);
  dirlist->setSelTextColor((FXColor)(long)ptr);
  return 1;
  }


// Forward GUI update to text widget
long TextWindow::onUpdSelTextColor(FXObject* sender,FXSelector,void* ptr){
  editor->handle(sender,MKUINT(FXText::ID_SELTEXT_COLOR,SEL_UPDATE),ptr);
  return 1;
  }


// Start the ball rolling
void TextWindow::start(int& argc,char** argv){
  if(argc>1){
    filename=argv[1];
    dirlist->setCurrentFile(filename);
    loadfile(filename);
    namechanged=TRUE;
    }
  }


// Create and show window
void TextWindow::create(){
  FXColor textback,textfore,textselback,textselfore;
  FXint ww,hh,xx,yy,treewidth,hidetree,hiddenfiles;
  FXFontDesc fontdesc;
  FXFont *font;
  
  // Get colors
  textback=getApp()->reg().readColorEntry("SETTINGS","textbackground",FXRGB(255,255,255));
  textfore=getApp()->reg().readColorEntry("SETTINGS","textforeground",FXRGB(0,0,0));
  textselback=getApp()->reg().readColorEntry("SETTINGS","textselbackground",FXRGB(0,0,128));
  textselfore=getApp()->reg().readColorEntry("SETTINGS","textselforeground",FXRGB(255,255,255));
  
  // Font
  if(fxparsefontdesc(fontdesc,getApp()->reg().readStringEntry("SETTINGS","font","helvetica 90 700 1 1 0 2"))){
    font=new FXFont(getApp(),fontdesc);
    editor->setFont(font);
    }
  
  // Get size
  xx=getApp()->reg().readIntEntry("SETTINGS","x",600);
  yy=getApp()->reg().readIntEntry("SETTINGS","y",400);
  ww=getApp()->reg().readIntEntry("SETTINGS","width",600);
  hh=getApp()->reg().readIntEntry("SETTINGS","height",400);

  // Hidden files shown
  hiddenfiles=getApp()->reg().readIntEntry("SETTINGS","showhiddenfiles",FALSE);
  dirlist->showHiddenFiles(hiddenfiles);
  
  // Showing the tree?
  hidetree=getApp()->reg().readIntEntry("SETTINGS","hidetree",TRUE);

  // Width of tree
  treewidth=getApp()->reg().readIntEntry("SETTINGS","treewidth",100);

  // Change the colors
  editor->setTextColor(textfore);
  dirlist->setTextColor(textfore);
  editor->setBackColor(textback);
  dirlist->setBackColor(textback);
  editor->setSelBackColor(textselback);
  dirlist->setSelBackColor(textselback);
  editor->setSelTextColor(textselfore);
  dirlist->setSelTextColor(textselfore);
  
  // Hide tree if asked for
  if(hidetree) treebox->hide();
  
  // Set tree width
  treebox->setWidth(treewidth);
  
  // Open toward file
  dirlist->setCurrentFile(filename);
  
  // Reposition window
  position(xx,yy,ww,hh);
  
  FXMainWindow::create();
  show();
  }


/*******************************************************************************/


// Start the whole thing
int main(int argc,char *argv[]){

  // Make application
  FXApp* application=new FXApp("TextEdit","Test");
  
  // Open display
  application->init(argc,argv);

  // Make window
  TextWindow* window=new TextWindow(application);
  
  // Handle interrupt to save stuff nicely
  application->addSignal(SIGINT,window,TextWindow::ID_QUIT);

  // Create it
  application->create();
  
  // Start
  window->start(argc,argv);

  // Run
  application->run();

  return 0;
  }


