/********************************************************************************
*                                                                               *
*                          U t i l i t y   F u n c t i o n s                    *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: fxutils.cpp,v 1.25 2000/05/01 15:15:22 gui Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "fxkeys.h"

#ifdef __CYGWIN__
#define _MAX_PATH MAX_PATH
#endif

/*
  To do:
  - Those functions manipulating strings should perhaps become FXString type
    functions?
  - Need to pass buffer-size argument to all those fxdirpart() etc. functions
    to protect against memory overruns (make it a default argument last one
    so as to not impact anyone).
  - Revive my old malloc() replacement library to detect memory block overwrites.
*/

/*******************************************************************************/


// Global flag which controls tracing level
unsigned int fxTraceLevel=0;


/*******************************************************************************/

// Allocate memory
FXint fxmalloc(void** ptr,unsigned long size){
  FXASSERT(ptr);
  *ptr=malloc(size);
  return *ptr!=NULL;
  }


// Allocate cleaned memory
FXint fxcalloc(void** ptr,unsigned long size){
  FXASSERT(ptr);
  *ptr=calloc(size,1);
  return *ptr!=NULL;
  }


// Resize memory
FXint fxresize(void** ptr,unsigned long size){
  FXASSERT(ptr);
  void *p=realloc(*ptr,size);
  if(p){ *ptr=p; return TRUE; }
  return FALSE;
  }


// Free memory, resets ptr to NULL afterward
void fxfree(void** ptr){
  FXASSERT(ptr);
  if(*ptr) free(*ptr);
  *ptr=NULL;
  }


// String duplicate
FXchar *fxstrdup(const FXchar* str){
  FXchar *copy;
  FXMALLOC(&copy,FXchar,strlen(str)+1);
  strcpy(copy,str);
  return copy;
  }


// Assert failed routine
void fxassert(const char* expression,const char* filename,unsigned int lineno){
#ifndef WIN32
  fprintf(stderr,"%s:%d: FXASSERT(%s) failed.\n",filename,lineno,expression);
#else
  char msg[256];
  sprintf(msg,"%s(%d): FXASSERT(%s) failed.\n",filename,lineno,expression);
#ifdef _WINDOWS
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
#else
  fprintf(stderr,"%s",msg);
#endif
#endif
  }


// Log message to [typically] stderr
void fxmessage(const char* format,...){
#ifndef WIN32
  va_list arguments;
  va_start(arguments,format);
  vfprintf(stderr,format,arguments);
  va_end(arguments);
#else
  char msg[256];
  va_list arguments;
  va_start(arguments,format);
  vsprintf(msg,format,arguments);
  va_end(arguments);
#ifdef _WINDOWS
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
#else
  fprintf(stderr,"%s",msg);
#endif
#endif
  }


// Error routine
void fxerror(const char* format,...){
#ifndef WIN32
  va_list arguments;
  va_start(arguments,format);
  vfprintf(stderr,format,arguments);
  va_end(arguments);
  abort();
#else
#ifdef _WINDOWS
  char msg[512];
  va_list arguments;
  va_start(arguments,format);
  vsprintf(msg,format,arguments);
  va_end(arguments);
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
  MessageBox(NULL,msg,NULL,MB_OK|MB_ICONEXCLAMATION|MB_APPLMODAL);
  DebugBreak();
#else
  va_list arguments;
  va_start(arguments,format);
  vfprintf(stderr,format,arguments);
  va_end(arguments);
  abort();
#endif
#endif
  }


// Warning routine
void fxwarning(const char* format,...){
#ifndef WIN32
  va_list arguments;
  va_start(arguments,format);
  vfprintf(stderr,format,arguments);
  va_end(arguments);
#else
  char msg[256];
  va_list arguments;
  va_start(arguments,format);
  vsprintf(msg,format,arguments);
  va_end(arguments);
#ifdef _WINDOWS
  OutputDebugString(msg);
  fprintf(stderr,"%s",msg); // if a console is available
  MessageBox(NULL,msg,NULL,MB_OK|MB_ICONINFORMATION|MB_APPLMODAL);
#else
  fprintf(stderr,"%s",msg);
#endif
#endif
  }


// Trace printout routine
void fxtrace(unsigned int level,const char* format,...){
  if(fxTraceLevel>level){
#ifndef WIN32
    va_list arguments;
    va_start(arguments,format);
    vfprintf(stderr,format,arguments);
    va_end(arguments);
#else
    char msg[256];
    va_list arguments;
    va_start(arguments,format);
    vsprintf(msg,format,arguments);
    va_end(arguments);
#ifdef _WINDOWS
    OutputDebugString(msg);
    fprintf(stderr,"%s",msg); // if a console is available
#else
    fprintf(stderr,"%s",msg);
#endif
#endif
    }
  }


// Sleep n microseconds
void fxsleep(unsigned int n){
#ifdef WIN32
  unsigned int zzz=n/1000;
  if(zzz==0) zzz=1;
  Sleep(zzz);
#else
#ifdef __USE_POSIX199309
  struct timespec value;
  value.tv_nsec = 1000 * (n%1000000);
  value.tv_sec = n/1000000;
  nanosleep(&value,NULL);
#else
#ifndef BROKEN_SELECT  
  struct timeval value;
  value.tv_usec = n % 1000000;
  value.tv_sec = n / 1000000;
  select(1,0,0,0,&value);
#else
  unsigned int zzz=n/1000000;
  if(zzz==0) zzz=1;
  if(zzz){
    while((zzz=sleep(zzz))>0) ;
    }
#endif
#endif
#endif
  }


// Character value of keysym+control state
FXchar fxkeyval(FXuint keysym,FXuint state){
  if((state&CONTROLMASK) && 0x60<=keysym && keysym<=0x7f) return keysym-0x60;
  if(state&0x2000){   // Patch from mohme@hgb-leipzig.de
    switch(keysym){ 
      case 'q': return '@';
      case '7': return '{';
      case '8': return '[';
      case '9': return ']';
      case '0': return '}';
      case 0xdf: return '\\';
      }
    }
  if(KEY_space<=keysym && keysym<=255) return keysym;
  if(KEY_BackSpace<=keysym && keysym<=KEY_Return) return keysym&255;
  if(KEY_KP_0<=keysym && keysym<=KEY_KP_9) return '0'-KEY_KP_0+keysym;
  if(KEY_Escape==keysym) return '\033';
  if(KEY_KP_Space==keysym) return ' ';
  if(KEY_KP_Tab==keysym) return '\t';
  if(KEY_KP_Enter==keysym) return '\015';
  if(KEY_KP_Add==keysym) return '+';
  if(KEY_KP_Subtract==keysym) return '-';
  if(KEY_KP_Multiply==keysym) return '*';
  if(KEY_KP_Divide==keysym) return '/';
  if(KEY_KP_Equal==keysym) return '=';
  if(KEY_KP_Decimal==keysym) return '.';
  return 0;
  }

#define ISSEP(ch) ((ch)=='+' || (ch)=='-' || (ch)==' ')


// Parse for accelerator key codes in a string
FXHotKey fxparseaccel(const FXchar* s){
  FXuint code=0,mods=0;
  if(s){
    while(s[0] && s[0]!='\t' && s[0]!='\n'){
      if(ISSEP(s[0])){
        s++;
        continue;
        }
      if((tolower(s[0])=='c')&&(tolower(s[1])=='t')&&(tolower(s[2])=='l')&&ISSEP(s[3])){
        mods|=CONTROLMASK;
        s+=4;
        continue;
        }
      if((tolower(s[0])=='c')&&(tolower(s[1])=='t')&&(tolower(s[2])=='r')&&(tolower(s[3])=='l')&&ISSEP(s[4])){
        mods|=CONTROLMASK;
        s+=5;
        continue;
        }
      if((tolower(s[0])=='a')&&(tolower(s[1])=='l')&&(tolower(s[2])=='t')&&ISSEP(s[3])){
        mods|=ALTMASK;
        s+=4;
        continue;
        }
      if((tolower(s[0])=='s')&&(tolower(s[1])=='h')&&(tolower(s[2])=='i')&&(tolower(s[3])=='f')&&(tolower(s[4])=='t')&&ISSEP(s[5])){
        mods|=SHIFTMASK;
        s+=6;
        continue;
        }
      
      // One-digit function key 
      if(tolower(s[0])=='f' && isdigit(s[1]) && (s[2]=='\0' || s[2]=='\t' || s[2]=='\n')){
        code=KEY_F1+s[1]-'1';
        }
      
      // Two digit function key
      else if(tolower(s[0])=='f' && isdigit(s[1]) && isdigit(s[2]) && (s[3]=='\0' || s[3]=='\t' || s[3]=='\n')){
        code=KEY_F1+10*(s[1]-'0')+(s[2]-'0')-1;
        }
      
      // Simple alphanum with preceding modifiers
      else if(isalnum(s[0]) && (s[1]=='\0' || s[1]=='\t' || s[1]=='\n')){
        if(mods&SHIFTMASK) 
          code=toupper(s[0])+KEY_space-' ';
        else
          code=tolower(s[0])+KEY_space-' ';
        }
      FXTRACE((150,"fxparseaccel: code = %04x mods=%04x\n",code,mods));
      return MKUINT(code,mods);
      }
    }
  return 0;
  }


// Parse for hot key in a string
FXHotKey fxparsehotkey(const FXchar* s){
  FXuint code,mods;
  if(s){
    while(s[0] && s[0]!='\t'){
      if(s[0]=='&'){
        if(s[1]!='&'){
          if(isalnum(s[1])){
            mods=ALTMASK;
            code=tolower(s[1])+KEY_space-' ';
            FXTRACE((150,"fxparsehotkey: code = %04x mods=%04x\n",code,mods));
            return MKUINT(code,mods);
            }
          break;
          }
        s++;
        }
      s++;
      }
    }
  return 0;
  }


// Locate hot key underline offset from begin of string
FXint fxfindhotkeyoffset(const FXchar* s){
  register FXint pos=0;
  if(s){
    while(s[pos] && s[pos]!='\t'){
      if(s[pos]=='&'){
        if(s[pos+1]!='&') return pos;
        pos++;
        }
      pos++;
      }
    }
  return -1;
  }


#define FOLD(c)	  ((flags&FILEMATCH_CASEFOLD)&&islower(c)?toupper(c):(c))


// Match a file name with a pattern; returning 1 if it matches, 0 if not.
FXint fxfilematch(const char *pattern,const char *string,FXuint flags){
  if(pattern && string){
    register const char *p=pattern,*n=string;
    register char c,c1,cstart,cend;
    register int neg;
    
    // Loop over characters of pattern
    while((c=*p++)!='\0'){
      c=FOLD(c);
      switch(c){

        // Single character wild card
        case '?':
	  if(*n=='\0') return 0;
	  else if((flags&FILEMATCH_FILE_NAME) && *n==PATHSEP) return 0;
	  else if((flags&FILEMATCH_PERIOD) && *n=='.' && (n==string || ((flags&FILEMATCH_FILE_NAME) && n[-1]==PATHSEP))) return 0;
	  break;

        // Many character wild card
        case '*':
	  if((flags&FILEMATCH_PERIOD) && *n=='.' && (n==string || ((flags&FILEMATCH_FILE_NAME) && n[-1]==PATHSEP))) return 0;

	  for(c=*p++; c=='?' || c=='*'; c=*p++,n++){
	    if(((flags&FILEMATCH_FILE_NAME) && *n==PATHSEP) || (c=='?' && *n=='\0')) return 0;
            }

	  if(c=='\0') return 1;

	  c1=(!(flags&FILEMATCH_NOESCAPE) && c=='\\') ? *p : c;
	  c1=FOLD(c1);
          for(--p; *n!='\0'; n++){
	    if((c=='[' || FOLD(*n)==c1) && fxfilematch(p,n,flags&~FILEMATCH_PERIOD)) return 1;
            }
          return 0;

        // Character range wild card
        case '[':
	  if(*n=='\0') return 0;

	  if((flags&FILEMATCH_PERIOD) && *n=='.' && (n==string || ((flags&FILEMATCH_FILE_NAME) && n[-1]==PATHSEP))) return 0;

	  // Nonzero if the sense of the character class is inverted.
	  neg=(*p=='!' || *p=='^');

	  if(neg) ++p;

	  c=*p++;
	  for(;;){
	    cstart=c;
            cend=c;

	    if(!(flags&FILEMATCH_NOESCAPE) && c=='\\') cstart=cend=*p++;

	    cstart=cend=FOLD(cstart);

	    // [ (unterminated) loses.
	    if(c=='\0') return 0;

	    c=*p++;
	    c=FOLD(c);

	    // [/] can never match.
	    if((flags&FILEMATCH_FILE_NAME) && c==PATHSEP) return 0;

	    if(c=='-' && *p!=']'){
	      cend = *p++;
	      if(!(flags&FILEMATCH_NOESCAPE) && cend=='\\') cend=*p++;
	      if(cend=='\0') return 0;
	      cend=FOLD(cend);
	      c=*p++;
	      }

	    if(FOLD(*n)>=cstart && FOLD(*n)<=cend) goto matched;

	    if(c==']') break;
	    }
	  if(!neg) return 0;
	  break;

matched:  // Skip the rest of the [...] that already matched.
	  while(c!=']'){

	    // [... (unterminated) loses.
	    if(c=='\0') return 0;

	    c=*p++;

	    // XXX 1003.2d11 is unclear if this is right.
	    if(!(flags&FILEMATCH_NOESCAPE) && c=='\\') ++p;
	    }
	  if(neg) return 0;
	  break;

        // Escaped character
        case '\\':
	  if(!(flags&FILEMATCH_NOESCAPE)){
	    c=*p++;
	    c=FOLD(c);
	    }
          // Fall through...

        // Normal character
        default:
	  if(FOLD(*n)!=c) return 0;
	  break;
        }

      ++n;
      }

    if(*n=='\0') return 1;

    // The FILEMATCH_LEADING_DIR flag says that "foo*" matches "foobar/frobozz".
    if((flags&FILEMATCH_LEADING_DIR) && *n==PATHSEP) return 1;
    }

  return 0;
  }



// Get highlight color
FXColor makeHiliteColor(FXColor clr){
  FXuint r,g,b;
  r=FXREDVAL(clr);
  g=FXGREENVAL(clr);
  b=FXBLUEVAL(clr);
  r=FXMAX(31,r);
  g=FXMAX(31,g);
  b=FXMAX(31,b);
  r=(133*r)/100;
  g=(133*g)/100;
  b=(133*b)/100;
  r=FXMIN(255,r);
  g=FXMIN(255,g);
  b=FXMIN(255,b);
  return FXRGB(r,g,b);
  }


// Get shadow color
FXColor makeShadowColor(FXColor clr){
  FXuint r,g,b;
  r=FXREDVAL(clr);
  g=FXGREENVAL(clr);
  b=FXBLUEVAL(clr);
  r=(66*r)/100;
  g=(66*g)/100;
  b=(66*b)/100;
  return FXRGB(r,g,b);
  }


// Get user name from uid
FXchar* fxgetusername(FXchar* result,FXuint uid){
  if(!result){fxerror("fxgetusername: NULL result argument.\n");}
#ifndef WIN32
  struct passwd *pwd;
  if((pwd=getpwuid(uid))!=NULL) 
    strcpy(result,pwd->pw_name); 
  else 
    sprintf(result,"%d",uid);
#else
  sprintf(result,"%d",uid);
#endif
  return result;
  }


// Get group name from gid
FXchar* fxgetgroupname(FXchar* result,FXuint gid){
  if(!result){fxerror("fxgetgroupname: NULL result argument.\n");}
#ifndef WIN32
  struct group *grp;
  if((grp=getgrgid(gid))!=NULL) 
    strcpy(result,grp->gr_name); 
  else 
    sprintf(result,"%d",gid);
#else
  sprintf(result,"%d",gid);
#endif
  return result;
  }


// Get permissions string from mode
FXchar* fxgetpermissions(FXchar* result,FXuint mode){
  if(!result){fxerror("fxgetpermissions: NULL result argument.\n");}
#ifndef WIN32
  result[0]=S_ISLNK(mode) ? 'l' : S_ISREG(mode) ? '-' : S_ISDIR(mode) ? 'd' : S_ISCHR(mode) ? 'c' : S_ISBLK(mode) ? 'b' : S_ISFIFO(mode) ? 'p' : S_ISSOCK(mode) ? 's' : '?';
  result[1]=(mode&S_IRUSR) ? 'r' : '-';
  result[2]=(mode&S_IWUSR) ? 'w' : '-';
  result[3]=(mode&S_ISUID) ? 's' : (mode&S_IXUSR) ? 'x' : '-';
  result[4]=(mode&S_IRGRP) ? 'r' : '-';
  result[5]=(mode&S_IWGRP) ? 'w' : '-';
  result[6]=(mode&S_ISGID) ? 's' : (mode&S_IXGRP) ? 'x' : '-';
  result[7]=(mode&S_IROTH) ? 'r' : '-';
  result[8]=(mode&S_IWOTH) ? 'w' : '-';
  result[9]=(mode&S_ISVTX) ? 't' : (mode&S_IXOTH) ? 'x' : '-';
  result[10]=0;
#else
  result[0]='-';
  result[1]='r';
  result[2]='w';
  result[3]='x';
  result[4]='r';
  result[5]='w';
  result[6]='x';
  result[7]='r';
  result[8]='w';
  result[9]='x';
  result[10]=0;
#endif
  return result;
  }


// Return TRUE iff s is a prefix of t
FXbool fxprefix(const FXchar* s,const FXchar* t){
  const FXchar *p,*q;
  for(p=s,q=t; *p && *q && *p==*q; p++,q++);
  return !*p;
  }


// Return TRUE iff s is a suffix of t
FXbool fxsuffix(const FXchar* s, const FXchar* t){
  const FXchar *p,*q;
  for(p=s+strlen(s),q=t+strlen(t); p>s && q>t && *p==*q; p--,q--);
  return p==s && *p==*q;
  }


// Expand ~ in filenames; using HOME environment variable
// Note "~name" expands to "~name", "~/name" expands to "$HOME/name", 
// and "~" expands to "$HOME"
FXchar* fxexpand(FXchar* result,const FXchar* name){
  if(!result){fxerror("fxexpand: NULL result argument.\n");}
  *result='\0';
  if(name){
    if(name[0]=='~'){
      if(name[1]=='\0' || name[1]==PATHSEP){
        fxgethomedir(result);
        name++;
        }
      }
    strcat(result,name);
    }
  return result;
  }
        

// Construct a full pathname from the given directory and file name
FXchar* fxpathname(FXchar* result,const FXchar* dirname,const FXchar* filename){
  int l;
  if(!result){fxerror("fxpathname: NULL result argument.\n");}
  fxexpand(result,dirname);
  if((l=strlen(result)) && result[l-1]!=PATHSEP) result[l++]=PATHSEP;
  strcpy(result+l,filename);
  return result;
  }


// Return the directory part of pathname (incl drive name)
FXchar* fxdirpart(FXchar* result,const FXchar* pathname){
  FXchar *p;
  if(!result){fxerror("fxdirpart: NULL result argument.\n");}
  strcpy(result,pathname);
  if((p=strrchr(result,PATHSEP))){      // Fix by sancelot@usa.net
    if(!fxisdir(result)){
      if((p==result) || (p==result+2 && isalpha(result[0]) && result[1]==':')) p++;
      *p='\0';
      }
    }
  else{
    fxgetcurrentdir(result);
    }
  return result;
  }


// Return the file title, i.e. filename less extension part of pathname
FXchar* fxfiletitle(FXchar* result,const FXchar* pathname){
  FXchar *p;
  if(!result){fxerror("fxfiletitle: NULL result argument.\n");}
  result[0]='\0';
  if(pathname){
    if(isalpha(pathname[0]) && pathname[1]==':') pathname+=2;   // Skip drive name part
    p=(FXchar*)strrchr(pathname,PATHSEP);                       // Borland C++ compiler says strrchr() returns a const char*
    if(p) pathname=p+1;
    strcpy(result,pathname);
    p=strrchr(result,'.');
    if(p) *p=0;
    }
  return result;
  }


// Return the filename part of pathname
FXchar* fxfilepart(FXchar* result,const FXchar* pathname){
  const FXchar *p; // Borland C++ compiler says strrchr() returns const char*
  if(!result){fxerror("fxfilepart: NULL result argument.\n");}
  if(isalpha(pathname[0]) && pathname[1]==':') pathname+=2;   // Skip drive name part
  p=strrchr(pathname,PATHSEP);
  if(p) pathname=p+1;
  strcpy(result,pathname);
  return result;
  }


// Return the extension part of pathname
FXchar* fxfileext(FXchar* result,const FXchar* pathname){
  const FXchar* p; // Borland C++ compiler says strrchr() returns const char*
  if(!result){fxerror("fxfileext: NULL result argument.\n");}
  *result='\0';
  p=strrchr(pathname,PATHSEP);
  if(p) pathname=p+1;
  if((p=strrchr(pathname,'.'))) return strcpy(result,p+1);
  return result;
  }


// Check whether a file exists
FXbool fxexists(const FXchar *name){
  if(!name){fxerror("fxexists: NULL name argument.\n");}
#ifndef WIN32
  struct stat info;
  return stat(name,&info)==0;
#else
  DWORD atts=GetFileAttributes(name);
  return (atts!=0xFFFFFFFF);
#endif
  }


// Check whether its a directory
FXbool fxisdir(const FXchar *name){
  if(!name){fxerror("fxisdir: NULL name argument.\n");}
#ifndef WIN32
  struct stat info;
  return (stat(name,&info)==0) && S_ISDIR(info.st_mode);
#else
  DWORD atts=GetFileAttributes(name);
  return (atts!=0xFFFFFFFF) && (atts&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Check whether its a file
FXbool fxisfile(const FXchar *name){
  if(!name){fxerror("fxisdir: NULL name argument.\n");}
#ifndef WIN32
  struct stat info;
  return (stat(name,&info)==0) && S_ISREG(info.st_mode);
#else
  DWORD atts=GetFileAttributes(name);
  return (atts!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Create a unique numbered backup file name for the given pathname 
FXchar* fxbakname(FXchar* result,const FXchar* pathname){
  FXchar* p;
  int n=0;
  if(!result){fxerror("fxbakname: NULL result argument.\n");}
  if(!pathname){fxerror("fxbakname: NULL pathname argument.\n");}
  p=result+strlen(pathname);
  strcpy(result,pathname);
  do{
    sprintf(p,".~%d~",++n);
    } 
  while(fxexists(result));
  return result;
  }



// Check whether two files are identical (refer to the same inode)
FXbool fxidentical(const FXchar *name1,const FXchar *name2){
  if(!name1){fxerror("fxidentical: NULL name1 argument.\n");}
  if(!name2){fxerror("fxidentical: NULL name2 argument.\n");}
#ifndef WIN32
  struct stat info1,info2;
  return !lstat(name1,&info1) && !lstat(name2,&info2) && info1.st_ino==info2.st_ino;
#else
  HANDLE hFile1=CreateFile(name1,GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
  if(hFile1==INVALID_HANDLE_VALUE) return FALSE;
  HANDLE hFile2=CreateFile(name2,GENERIC_READ,FILE_SHARE_READ,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
  if(hFile2==INVALID_HANDLE_VALUE){
    CloseHandle(hFile1);
    return FALSE;
    }
  BY_HANDLE_FILE_INFORMATION info1;
  BY_HANDLE_FILE_INFORMATION info2;
  FXbool same=FALSE;
  if(GetFileInformationByHandle(hFile1,&info1) && GetFileInformationByHandle(hFile2,&info2)){
    if(info1.nFileIndexLow==info2.nFileIndexLow &&
       info1.nFileIndexHigh==info2.nFileIndexHigh &&
       info1.dwVolumeSerialNumber==info2.dwVolumeSerialNumber) same=TRUE;
    }
  CloseHandle(hFile1);
  CloseHandle(hFile2);
  return same;
#endif
  }


// Split a string into substrings delimited by a given character
FXchar *fxsplit(FXchar*& s,FXchar c){
  FXchar *t=s;
  if(s && (s=strchr(s,c))) *s++='\0';
  return t;
  }


// Return the shortest path equivalent to pathname (remove . and ..)
FXchar *fxshortestpath(FXchar *result,const FXchar *pathname){
  FXchar path[MAXPATHLEN+1],*cur,*last,*part,*tmp;
  if(!result){fxerror("fxshortestpath: NULL result argument.\n");}
  if(!pathname){fxerror("fxshortestpath: NULL pathname argument.\n");}
  strcpy(path,pathname);
  cur=result;
  *cur='\0';
  tmp=path;
  last=NULL;
  if(*tmp==PATHSEP){ 
    *cur++=PATHSEP;
    *cur='\0';
    tmp++; 
    }
  while((part=fxsplit(tmp,PATHSEP))){
    if(strcmp(part,".")==0)
      ;
    else if(strcmp(part,"..")==0 && (last=strrchr(result,PATHSEP))){
      if(last>result)
        cur=last;
      else
        cur=last+1;
      *cur='\0';
      } 
    else{
      if(cur>result && *(cur-1)!=PATHSEP) *cur++=PATHSEP;
      strcpy(cur,part);
      cur+=strlen(part);
      }
    }
  
  // Remove trailing backslash(es)
  while(result<cur-1 && *(cur-1)==PATHSEP && *(cur-2)!=':'){
    *--cur='\0';
    }

#ifdef WIN32
  // Use GetFullPathName() to ensure that the correct drive letter is used
  DWORD dwSize=GetFullPathName(result,0,NULL,NULL);
  if(dwSize>0){
    FXchar *buffer;
    if(FXMALLOC(&buffer,FXchar,dwSize+1)){
      if(0!=GetFullPathName(result,dwSize,buffer,NULL))
	strcpy(result,buffer);
      FXFREE(&buffer);
      }
    }
#endif

  return result;
  }


// Return directory one level above given one
FXchar* fxupdir(FXchar* result,const FXchar *dirname){
  FXchar *ptr;
  if(!result){fxerror("fxupdir: NULL result argument.\n");}
  if(!dirname){fxerror("fxupdir: NULL dirname argument.\n");}
  strcpy(result,dirname);
  if(!fxistopdir(result)){
    ptr=strrchr(result,PATHSEP);
    if(ptr && *(ptr+1)=='\0'){ *ptr='\0'; ptr=strrchr(result,PATHSEP); }
    if(ptr){
      if((result[0]==PATHSEP && ptr==result) || (isalpha(result[0]) && result[1]==':' && result[2]==PATHSEP && ptr==result+2)){
        *(ptr+1)='\0';
        }
      else{
        *ptr='\0';
        }
      }
    }
  return result;
  }


// Test if pathname is the toplevel directory
FXbool fxistopdir(const FXchar* path){
  return (path[0]==PATHSEP && path[1]=='\0') || (isalpha(path[0]) && path[1]==':' && path[2]==PATHSEP && path[3]=='\0');
  }


// Get current working directory
FXchar* fxgetcurrentdir(FXchar *result){
  if(!result){fxerror("fxgetcurrentdir: NULL result argument.\n");}
#ifndef WIN32
  if(getcwd(result,MAXPATHLEN)==NULL){fxerror("fxgetcurrentdir: failed.\n");}
#else
  GetCurrentDirectory(MAXPATHLEN,result);
#endif
  return result;
  }


// Make a directory
FXint fxmkdir(const FXchar* pathname,FXuint mode){
#if defined(WIN32) && !defined(__CYGWIN__)
  return _mkdir(pathname);
#else
  return mkdir(pathname,mode);
#endif
  }


// Get home directory
FXchar* fxgethomedir(FXchar *result){
  static FXchar* home=NULL;
  if(!result){fxerror("fxgethomedir: NULL result argument.\n");}
  if(!home) {
#ifdef WIN32
    HKEY hKey;
    char    szEnv[_MAX_PATH+5];     // "HOME="
    DWORD   dwSize=_MAX_PATH;
    if(RegOpenKeyEx(HKEY_CURRENT_USER,"Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders",0,KEY_READ,&hKey)==ERROR_SUCCESS){
      strcpy(szEnv,"HOME=");
      if(RegQueryValueEx(hKey,"Personal",NULL,NULL,(LPBYTE)szEnv+5,&dwSize)==ERROR_SUCCESS){
#ifdef _MSC_VER
        _putenv(szEnv);
#else
        putenv(szEnv);
#endif
        }
      RegCloseKey(hKey);
      }
#endif
    home=getenv("HOME");
    if(!home) home="";
    }
  strcpy(result,home);
  return result;
  }


// Translate filename to an absolute pathname; ~ in filename is expanded,
// and if the resulting pathname is still relative, basename is prepended
FXchar *fxabspath(FXchar *result,const FXchar *basename,const FXchar *filename){
  char s[MAXPATHLEN+1],t[MAXPATHLEN+1],cwd[MAXPATHLEN+1];
  
  // Test
  if(!result){fxerror("fxabspath: NULL result argument.\n");}

  // Expand tilde's
  fxexpand(s,filename);
  
  // Absolute path may be preceded by a drive letter as in C:\blabla\blabla
  // or just \blabla\blabla or on unix just /blabla/blabla
  if(!(s[0]==PATHSEP || (isalpha(s[0]) && s[1]==':' && s[2]==PATHSEP))){
    if(basename==NULL || *basename=='\0'){
      fxgetcurrentdir(cwd);
      basename=cwd;
      }
    fxpathname(t,basename,s);
    return fxshortestpath(result,t);
    }
  return fxshortestpath(result,s);
  }


// Search path for file name. A ~ in path is expanded. If name is absolute
// it is returned unchanged. Otherwise the absolute name is returned in
// result. If name is not found on path, NULL is returned. 
FXchar *fxsearchpath(FXchar *result,const FXchar *path,const FXchar *name){
  FXchar fullname[MAXPATHLEN+1];
  FXchar basedir[MAXPATHLEN+1];
  const FXchar *s,*t;
  if(!result){fxerror("fxsearchpath: NULL result argument.\n");}
  if(!path){fxerror("fxsearchpath: NULL path argument.\n");}
  if(!name){fxerror("fxsearchpath: NULL name argument.\n");}
  if(name[0]==PATHSEP || (isalpha(name[0]) && name[1]==':' && name[2]==PATHSEP)){
    strcpy(result,name);
    if(fxexists(result)) return result;
    return NULL;
    }
  fxgetcurrentdir(basedir);
  for(s=path; *s; s=t){
    int l;
    t=strchr(s,PATHLISTSEP);
    if(!t) t=s+strlen(s);
    if(s!=t){
      if(s[0]=='.'){ 
        if(t==s+1) s=t; else if(s[1]==PATHSEP) s+=2; 
        }
      l=t-s;
      strncpy(fullname,s,l);
      if(l>0 && fullname[l-1]!=PATHSEP){ fullname[l]=PATHSEP; l++; }
      strcpy(fullname+l,name);
      fxabspath(result,basedir,fullname);
      if(fxexists(result)) return result;
      }
    if(*t) t++;
    }
  return NULL;
  }


// Get process id
FXint fxgetpid(){
#ifndef WIN32
  return getpid();
#else
  return (int)GetCurrentProcessId();
#endif
  }


// Convert RGB to HSV
void fxrgb_to_hsv(FXfloat& h,FXfloat& s,FXfloat& v,FXfloat r,FXfloat g,FXfloat b){
  FXfloat t,delta;
  v=r; if(g>v) v=g; if(b>v) v=b;
  t=r; if(g<t) t=g; if(b<t) t=b;
  delta=v-t;
  if(v!=0.0) 
    s=delta/v; 
  else 
    s=0.0;
  if(s==0.0){
    h=0.0;
    }
  else{
    if(r==v) 
      h=(g-b)/delta;
    else if(g==v) 
      h=2.0f+(b-r)/delta;
    else if(b==v) 
      h=4.0f+(r-g)/delta;
    h=h*60.0f;
    if(h<0.0) h=h+360;
    }
  }


// Convert to RGB
void fxhsv_to_rgb(FXfloat& r,FXfloat& g,FXfloat& b,FXfloat h,FXfloat s,FXfloat v){
  FXfloat f,w,q,t;
  FXint i;
  if(s==0.0){
    r=v;
    g=v;
    b=v;
    }
  else{
    if(h==360.0) h=0.0;
    h=h/60.0f;
    i=(FXint)h;
    f=h-i;
    w=v*(1.0f-s);
    q=v*(1.0f-(s*f));
    t=v*(1.0f-(s*(1.0f-f)));
    switch(i){
      case 0: r=v; g=t; b=w; break;
      case 1: r=q; g=v; b=w; break;
      case 2: r=w; g=v; b=t; break;
      case 3: r=w; g=q; b=v; break;
      case 4: r=t; g=w; b=v; break;
      case 5: r=v; g=w; b=q; break;
      }
    }
  }


// Calculate a hash value from a string; algorithm same as in perl
FXint fxstrhash(const FXchar* str){
  register FXint h=0;
  register FXint g;
  while(*str) {
    h=(h<<4)+*str++;
    g=h&0xF0000000;
    if(g) h^=g>>24;
    h&=0x0fffffff;
    }
  FXASSERT(h<=0x0fffffff);
  return h;
  }


#ifdef WIN32

// Return the current state of the modifier keys and mouse buttons
unsigned int fxmodifierkeys(){
  FXuint state=0;
  if(GetKeyState(VK_SHIFT)<0) state|=SHIFTMASK;
  if(GetKeyState(VK_CAPITAL)>0) state|=CAPSLOCKMASK;
  if(GetKeyState(VK_CONTROL)<0) state|=CONTROLMASK;
  if(GetKeyState(VK_MENU)<0) state|=ALTMASK;
  if(GetKeyState(VK_NUMLOCK)>0) state|=NUMLOCKMASK;
  if(GetKeyState(VK_SCROLL)>0) state|=SCROLLLOCKMASK;
  if(GetKeyState(VK_LBUTTON)<0) state|=LEFTBUTTONMASK;
  if(GetKeyState(VK_MBUTTON)<0) state|=MIDDLEBUTTONMASK;
  if(GetKeyState(VK_RBUTTON)<0) state|=RIGHTBUTTONMASK;
  return state;
  }


// Convert font size (in decipoints) to device logical units
int fxpointsize_to_height(HDC hdc,unsigned size){
  // The calls to SetGraphicsMode() and ModifyWorldTransform() have no effect on Win95/98
  // but that's OK.
  SetGraphicsMode(hdc,GM_ADVANCED);
  XFORM xform;
  ModifyWorldTransform(hdc,&xform,MWT_IDENTITY);
  SetViewportOrgEx(hdc,0,0,NULL);
  SetWindowOrgEx(hdc,0,0,NULL);
  FLOAT cyDpi=(FLOAT)(25.4*GetDeviceCaps(hdc,VERTRES)/GetDeviceCaps(hdc,VERTSIZE));
  POINT pt;
  pt.x=0;
  pt.y=(int)(size*cyDpi/72);
  DPtoLP(hdc,&pt,1);
  return -(int)(fabs(pt.y)/10.0+0.5);
  }


// Convert logical units to decipoints
unsigned fxheight_to_pointsize(HDC hdc,int height){
  // The calls to SetGraphicsMode() and ModifyWorldTransform() have no effect on Win95/98
  // but that's OK.
  SetGraphicsMode(hdc,GM_ADVANCED);
  XFORM xform;
  xform.eM11=0.0f;
  xform.eM12=0.0f;
  xform.eM21=0.0f;
  xform.eM22=0.0f;
  xform.eDx=0.0f;
  xform.eDy=0.0f;
  ModifyWorldTransform(hdc,&xform,MWT_IDENTITY);
  SetViewportOrgEx(hdc,0,0,NULL);
  SetWindowOrgEx(hdc,0,0,NULL);
  FLOAT cyDpi=(FLOAT)(25.4*GetDeviceCaps(hdc,VERTRES)/GetDeviceCaps(hdc,VERTSIZE));
  POINT pt;
  pt.x=0;
  pt.y=10*height;
  LPtoDP(hdc,&pt,1);
  return (FXuint)(72*pt.y/cyDpi);
  }

#endif
