/********************************************************************************
*                                                                               *
*                          T r e e L i s t   O b j e c t                        *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXTreeList.cpp,v 1.35 2000/03/09 07:52:44 jeroen Exp $                   *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXButton.h"
#include "FXComposite.h"
#include "FXScrollbar.h"
#include "FXScrollArea.h"
#include "FXTreeList.h"


/*
  To do:
  - Tooltip should pop up exactly on top of current item.
  - Clicking on + does not make it current.
  - Need translate right-clicks into message with item figured out...
  - In autoselect mode, all items are expanded.
  - Sortfunc's will be hard to serialize, and hard to write w/o secretly #including
    the FXTreeItem header!

*/


#define ICON_SPACING        4         // Spacing between parent and child in x direction
#define TEXT_SPACING        4         // Spacing between icon and text
#define LINE_SPACING        1         // Spacing between lines
#define SIDE_SPACING        4         // Spacing between side and item

#define DEFAULT_INDENT      20        // Indent between parent and child


#define DRAG_NONE   0                 // No action going on
#define DRAG_DND    1                 // Dragging icons
#define DRAG_TRYDND 2                 // Tentative dragging

#define SELECT_MASK     (TREELIST_SINGLESELECT|TREELIST_BROWSESELECT)
#define TREELIST_MASK   (SELECT_MASK|TREELIST_AUTOSELECT|TREELIST_SHOWS_LINES|TREELIST_SHOWS_BOXES|TREELIST_ROOT_BOXES)

/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXTreeItem,FXObject,NULL,0)



// Draw item
void FXTreeItem::draw(const FXTreeList* list,FXDC& dc,FXint x,FXint y,FXint,FXint h) const {
  register FXFont *font=list->getFont();
  register FXIcon *icon=(state&OPENED)?openIcon:closedIcon;
  register FXint th=0,ih=0,tw,len;
  if(icon) ih=icon->getHeight();
  if(!label.empty()) th=4+font->getFontHeight();
  x+=SIDE_SPACING/2;
  if(icon){
    dc.drawIcon(icon,x,y+(h-ih)/2);
    x+=ICON_SPACING+icon->getWidth();
    }
  if(!label.empty()){
    len=label.length();
    tw=4+font->getTextWidth(label.text(),len);
    y+=(h-th)/2;
    if(isSelected()){
      dc.setForeground(list->getSelBackColor());
      dc.fillRectangle(x,y,tw,th);
      if(!isEnabled())
        dc.setForeground(makeShadowColor(list->getBackColor()));
      else
        dc.setForeground(list->getSelTextColor());
      }
    else{
      if(!isEnabled())
        dc.setForeground(makeShadowColor(list->getBackColor()));
      else
        dc.setForeground(list->getTextColor());
      }
    dc.drawText(x+2,y+font->getFontAscent()+2,label.text(),len);
    if(hasFocus()){
      drawFocus(list,dc,x,y,tw,th);
      }
    }
  }


// Draw dotted rectangle for focus
void FXTreeItem::drawFocus(const FXTreeList* list,FXDC& dc,FXint x,FXint y,FXint w,FXint h) const {
  dc.setFillStyle(FILL_OPAQUESTIPPLED);
  dc.setStipple(STIPPLE_GRAY,x,y);
  dc.setForeground(list->getTextColor());
  dc.setBackground(list->getBackColor());
  dc.drawRectangle(x+1,y+1,w-3,h-3);
  dc.setFillStyle(FILL_SOLID);
  dc.setStipple(STIPPLE_NONE);
  }



// See if item got hit, and where:- 1 is icon, 2 is text
FXint FXTreeItem::hitItem(const FXTreeList* list,FXint x,FXint y) const {
  register FXint oiw=0,ciw=0,oih=0,cih=0,tw=0,th=0,iw,ih,ix,iy,tx,ty,h;
  register FXFont *font=list->getFont();
  if(openIcon){
    oiw=openIcon->getWidth();
    oih=openIcon->getHeight();
    }
  if(closedIcon){
    ciw=closedIcon->getWidth();
    cih=closedIcon->getHeight();
    }
  if(!label.empty()){
    tw=4+font->getTextWidth(label.text(),label.length());
    th=4+font->getFontHeight();
    }
  iw=FXMAX(oiw,ciw);
  ih=FXMAX(oih,cih);
  h=LINE_SPACING+FXMAX(th,ih);
  ix=SIDE_SPACING/2;
  tx=SIDE_SPACING/2;
  if(iw) tx+=iw+ICON_SPACING;
  iy=(h-ih)/2;
  ty=(h-th)/2;
  
  // In icon?
  if(ix<=x && iy<=y && x<ix+iw && y<iy+ih) return 1;
    
  // In text?
  if(tx<=x && ty<=y && x<tx+tw && y<ty+th) return 2;
  
  // Outside
  return 0;
  }


// Set or kill focus
void FXTreeItem::setFocus(FXbool focus){
  if(focus) state|=FOCUS; else state&=~FOCUS;
  }

// Select or deselect item
void FXTreeItem::setSelected(FXbool selected){
  if(selected) state|=SELECTED; else state&=~SELECTED;
  }

// Set item opened
void FXTreeItem::setOpened(FXbool opened){
  if(opened) state|=OPENED; else state&=~OPENED;
  }

// Set item expanded
void FXTreeItem::setExpanded(FXbool expanded){
  if(expanded) state|=EXPANDED; else state&=~EXPANDED;
  }

// Enable or disable the item
void FXTreeItem::setEnabled(FXbool enabled){
  if(enabled) state&=~DISABLED; else state|=DISABLED;
  }


// Icon is draggable
void FXTreeItem::setDraggable(FXbool draggable){
  if(draggable) state|=DRAGGABLE; else state&=~DRAGGABLE;
  }


// Mark item
void FXTreeItem::setMarked(FXbool marked){
  if(marked) state|=MARKED; else state&=~MARKED;
  }


// Create icon
void FXTreeItem::create(){ 
  if(openIcon) openIcon->create(); 
  if(closedIcon) closedIcon->create(); 
  }


// Destroy icon
void FXTreeItem::destroy(){ 
  if(openIcon) openIcon->destroy(); 
  if(closedIcon) closedIcon->destroy(); 
  }


// Detach from icon resource
void FXTreeItem::detach(){ 
  if(openIcon) openIcon->detach(); 
  if(closedIcon) closedIcon->detach(); 
  }


// Get number of child items
FXint FXTreeItem::getNumChildren() const {
  register FXTreeItem *item=first;
  register FXint n=0;
  while(item){item=item->next;n++;}
  return n;
  }



// Get item (logically) below this one
FXTreeItem* FXTreeItem::getBelow() const {
  register FXTreeItem* item=(FXTreeItem*)this;
  if(first) return first;
  while(!item->next && item->parent) item=item->parent;
  return item->next;
  }


// Get item (logically) above this one
FXTreeItem* FXTreeItem::getAbove() const {
  register FXTreeItem* item=prev;
  if(!item) return parent;
  while(item->last) item=item->last;
  return item;
  }


// Get item width
FXint FXTreeItem::getWidth(const FXTreeList* list) const {
  FXint w=0,oiw=0,ciw=0;
  if(openIcon)   oiw=openIcon->getWidth();
  if(closedIcon) ciw=closedIcon->getWidth();
  w=FXMAX(oiw,ciw);
  if(!label.empty()){
    if(w) w+=ICON_SPACING;
    w+=4+list->getFont()->getTextWidth(label.text(),label.length());
    }
  return SIDE_SPACING+w;
  }


// Get item height
FXint FXTreeItem::getHeight(const FXTreeList* list) const {
  FXint th=0,oih=0,cih=0;
  if(openIcon)   oih=openIcon->getHeight();
  if(closedIcon) cih=closedIcon->getHeight();
  if(!label.empty()) th=4+list->getFont()->getFontHeight();
  return LINE_SPACING+FXMAX3(th,oih,cih);
  }



// Save data
void FXTreeItem::save(FXStream& store) const {
  FXObject::save(store);
  store << prev;
  store << next;
  store << parent;
  store << first;
  store << last;
  store << label;
  store << openIcon;
  store << closedIcon;
  store << state;
  }


// Load data
void FXTreeItem::load(FXStream& store){ 
  FXObject::load(store);
  store >> prev;
  store >> next;
  store >> parent;
  store >> first;
  store >> last;
  store >> label;
  store >> openIcon;
  store >> closedIcon;
  store >> state;
  }


/*******************************************************************************/

// Map
FXDEFMAP(FXTreeList) FXTreeListMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXTreeList::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXTreeList::onMotion),
  FXMAPFUNC(SEL_TIMEOUT,FXWindow::ID_AUTOSCROLL,FXTreeList::onAutoScroll),
  FXMAPFUNC(SEL_TIMEOUT,FXTreeList::ID_TIPTIMER,FXTreeList::onTipTimer),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXTreeList::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXTreeList::onLeftBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXTreeList::onRightBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXTreeList::onRightBtnRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXTreeList::onUngrabbed),
  FXMAPFUNC(SEL_KEYPRESS,0,FXTreeList::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXTreeList::onKeyRelease),
  FXMAPFUNC(SEL_ENTER,0,FXTreeList::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXTreeList::onLeave),
  FXMAPFUNC(SEL_ACTIVATE,0,FXTreeList::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXTreeList::onDeactivate),
  FXMAPFUNC(SEL_FOCUSIN,0,FXTreeList::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXTreeList::onFocusOut),
  FXMAPFUNC(SEL_OPENED,0,FXTreeList::onItemOpened),
  FXMAPFUNC(SEL_CLOSED,0,FXTreeList::onItemClosed),
  FXMAPFUNC(SEL_EXPANDED,0,FXTreeList::onItemExpanded),
  FXMAPFUNC(SEL_COLLAPSED,0,FXTreeList::onItemCollapsed),
  FXMAPFUNC(SEL_CHANGED,0,FXTreeList::onChanged),
  FXMAPFUNC(SEL_CLICKED,0,FXTreeList::onClicked),
  FXMAPFUNC(SEL_DOUBLECLICKED,0,FXTreeList::onDoubleClicked),
  FXMAPFUNC(SEL_TRIPLECLICKED,0,FXTreeList::onTripleClicked),
  FXMAPFUNC(SEL_SELECTION_LOST,0,FXTreeList::onSelectionLost),
  FXMAPFUNC(SEL_SELECTION_GAINED,0,FXTreeList::onSelectionGained),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_TIP,FXTreeList::onQueryTip),
  FXMAPFUNC(SEL_UPDATE,FXWindow::ID_QUERY_HELP,FXTreeList::onQueryHelp),
  FXMAPFUNC(SEL_COMMAND,0,FXTreeList::onCommand),
  };


// Object implementation
FXIMPLEMENT(FXTreeList,FXScrollArea,FXTreeListMap,ARRAYNUMBER(FXTreeListMap))

  
/*******************************************************************************/

  
// Tree List
FXTreeList::FXTreeList(){
  flags|=FLAG_ENABLED;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  extentitem=NULL;
  cursoritem=NULL;
  font=(FXFont*)-1;
  sortfunc=NULL;
  textColor=0;
  selbackColor=0;
  seltextColor=0;
  lineColor=0;
  itemWidth=1;
  itemHeight=1;
  visible=0;
  nrows=0;
  indent=DEFAULT_INDENT;
  grabx=0;
  graby=0;
  timer=NULL;
  }


// Tree List
FXTreeList::FXTreeList(FXComposite *p,FXint nvis,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXScrollArea(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  firstitem=NULL;
  lastitem=NULL;
  anchoritem=NULL;
  currentitem=NULL;
  extentitem=NULL;
  cursoritem=NULL;
  font=getApp()->getNormalFont();
  sortfunc=NULL;
  textColor=getApp()->foreColor;
  selbackColor=getApp()->selbackColor;
  seltextColor=getApp()->selforeColor;
  lineColor=getApp()->shadowColor;
  itemWidth=1;
  itemHeight=1;
  visible=FXMAX(nvis,0);
  nrows=0;
  indent=DEFAULT_INDENT;
  grabx=0;
  graby=0;
  timer=NULL;
  }


// Create window
void FXTreeList::create(){
  register FXTreeItem *item=firstitem;
  FXScrollArea::create();
  while(item){
    item->create();
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  font->create();
  }


// Detach window
void FXTreeList::detach(){
  register FXTreeItem *item=firstitem;
  FXScrollArea::detach();
  while(item){
    item->detach();
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  font->detach();
  }


// Can have focus
FXbool FXTreeList::canFocus() const { return TRUE; }


// Get default width
FXint FXTreeList::getDefaultWidth(){
  return FXScrollArea::getDefaultWidth();
  }


// Get default height
FXint FXTreeList::getDefaultHeight(){
  if(visible){
    if(flags&FLAG_RECALC) recompute();
    return visible*itemHeight;
    }
  return FXScrollArea::getDefaultHeight();
  }


// Propagate size change
void FXTreeList::recalc(){ 
  FXScrollArea::recalc();
  flags|=FLAG_RECALC;
  cursoritem=NULL;
  }


// List is multiple of nitems
void FXTreeList::setNumVisible(FXint nvis){
  if(nvis<0) nvis=0;
  if(visible!=nvis){
    visible=nvis;
    recalc();
    }
  }


// Get number of toplevel items
FXint FXTreeList::getNumItems() const {
  register FXTreeItem *item=firstitem;
  register FXint n=0;
  while(item){
    item=item->next;
    n++;
    }
  return n;
  }


// Recompute interior 
void FXTreeList::recompute(){
  register FXTreeItem* item;
  register FXint x,y,w,h;

  // Measure the items
  nrows=0;
  itemWidth=1;
  itemHeight=1;
  x=0;
  if(options&TREELIST_ROOT_BOXES) x+=indent;
  item=firstitem;
  while(item){
    w=item->getWidth(this);
    h=item->getHeight(this);
    if(h>itemHeight) itemHeight=h;
    if(x+w>itemWidth) itemWidth=x+w;
    nrows++;
    if(item->first && ((options&TREELIST_AUTOSELECT) || item->isExpanded())){
      x+=indent;
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      x-=indent;
      item=item->parent;
      }
    item=item->next;
    }
  if(itemHeight&1) itemHeight++;    // Got to be even for the dotted lines!!

  // Place the items
  x=y=0;
  if(options&TREELIST_ROOT_BOXES) x+=indent;
  item=firstitem;
  while(item){
    item->x=x;
    item->y=y;
    y+=itemHeight;
    if(item->first && ((options&TREELIST_AUTOSELECT) || item->isExpanded())){
      x+=indent;
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      x-=indent;
      item=item->parent;
      }
    item=item->next;
    }

  // Done
  flags&=~FLAG_RECALC;
  }

    
// Determine content width of tree list
FXint FXTreeList::getContentWidth(){
  if(flags&FLAG_RECALC) recompute();
  return itemWidth;
  }


// Determine content height of tree list
FXint FXTreeList::getContentHeight(){
  if(flags&FLAG_RECALC) recompute();
  return nrows*itemHeight;
  }


// Recalculate layout
void FXTreeList::layout(){

  // Repaint when content size changed
  //if(flags&FLAG_RECALC) update();

  // Calculate contents
  FXScrollArea::layout();
  
  // Set line size based on font
  vertical->setLine(itemHeight);
  horizontal->setLine(1);
  
  // Force repaint
  update();
  
  // No more dirty
  flags&=~FLAG_DIRTY;
  }


// Set item text
void FXTreeList::setItemText(FXTreeItem* item,const FXString& text){
  if(item==NULL){ fxerror("%s::setItemText: item is NULL.\n",getClassName()); }
  item->setText(text);
  recalc();
  }


// Get item text
FXString FXTreeList::getItemText(const FXTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemText: item is NULL.\n",getClassName()); }
  return item->getText();
  }


// Set item open icon
void FXTreeList::setItemOpenIcon(FXTreeItem* item,FXIcon* icon){
  if(item==NULL){ fxerror("%s::setItemOpenIcon: item is NULL.\n",getClassName()); }
  item->setOpenIcon(icon);
  recalc();
  }


// Get item open icon
FXIcon* FXTreeList::getItemOpenIcon(const FXTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemOpenIcon: item is NULL.\n",getClassName()); }
  return item->getOpenIcon();
  }


// Set item closed icon
void FXTreeList::setItemClosedIcon(FXTreeItem* item,FXIcon* icon){
  if(item==NULL){ fxerror("%s::setItemClosedIcon: item is NULL.\n",getClassName()); }
  item->setClosedIcon(icon);
  recalc();
  }


// Get item closed icon
FXIcon* FXTreeList::getItemClosedIcon(const FXTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemClosedIcon: item is NULL.\n",getClassName()); }
  return item->getClosedIcon();
  }


// Set item data
void FXTreeList::setItemData(FXTreeItem* item,void* ptr) const {
  if(item==NULL){ fxerror("%s::setItemData: item is NULL.\n",getClassName()); }
  item->setData(ptr);
  }


// Get item data
void* FXTreeList::getItemData(const FXTreeItem* item) const {
  if(item==NULL){ fxerror("%s::getItemData: item is NULL.\n",getClassName()); }
  return item->getData();
  }


// True if item is selected
FXbool FXTreeList::isItemSelected(const FXTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemSelected: item is NULL.\n",getClassName()); } 
  return item->isSelected();
  }


// True if item is current
FXbool FXTreeList::isItemCurrent(const FXTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemCurrent: item is NULL.\n",getClassName()); } 
  return currentitem==item; 
  }


// Check if item is expanded
FXbool FXTreeList::isItemExpanded(const FXTreeItem* item) const {
  if(!item){ fxerror("%s::isItemExpanded: item is NULL.\n",getClassName()); } 
  return (options&TREELIST_AUTOSELECT) || item->isExpanded(); 
  }
  

// Is item a leaf item
FXbool FXTreeList::isItemLeaf(const FXTreeItem* item) const {
  if(!item){ fxerror("%s::isItemLeaf: item is NULL.\n",getClassName()); } 
  return item->first!=NULL; 
  }


// Check if item is enabled
FXbool FXTreeList::isItemEnabled(const FXTreeItem* item) const {
  if(!item){ fxerror("%s::isItemEnabled: item is NULL.\n",getClassName()); } 
  return item->isEnabled(); 
  }


// Check item is open
FXbool FXTreeList::isItemOpened(const FXTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemOpen: item is NULL.\n",getClassName()); } 
  return item->isOpened(); 
  }


// True if item (partially) visible
FXbool FXTreeList::isItemVisible(const FXTreeItem* item) const { 
  if(!item){ fxerror("%s::isItemVisible: item is NULL.\n",getClassName()); } 
  return 0<(pos_y+item->y+itemHeight) && (pos_y+item->y)<viewport_h; 
  }


// Make item fully visible
void FXTreeList::makeItemVisible(FXTreeItem* item){
  FXint x,y,w,h;
  if(item){
    
    // Expand parents of this node
    if(!(options&TREELIST_AUTOSELECT)){
      FXTreeItem *par=item->parent;
      FXbool expanded=FALSE;
      while(par){
        if(!par->isExpanded()){
          par->setExpanded(TRUE);
          expanded=TRUE;
          }
        par=par->parent;
        }
      
      // If any nodes have expanded that weren't previously, recompute list size
      if(expanded){
        recalc();
        if(xid) layout();
        }
      }
      
    // Now we adjust the scrolled position to fit everything
    if(xid){
      x=pos_x;
      y=pos_y;

      w=item->getWidth(this);
      h=item->getHeight(this);

      if(viewport_w<=x+item->x+w) x=viewport_w-item->x-w;
      if(x+item->x<=0) x=-item->x; 

      if(viewport_h<=y+item->y+h) y=viewport_h-item->y-h;
      if(y+item->y<=0) y=-item->y; 

      setPosition(x,y);
      }
    }
  }


// Get item at position x,y
FXTreeItem* FXTreeList::getItemAt(FXint,FXint y) const {
  register FXTreeItem* item=firstitem;
  register FXint ix,iy;
  ix=pos_x;
  iy=pos_y;
  if(options&TREELIST_ROOT_BOXES) ix+=indent;
  while(item && iy<=y){
    if(y<iy+itemHeight) return item;
    iy+=itemHeight;
    if(item->first && ((options&TREELIST_AUTOSELECT) || item->isExpanded())){
      ix+=indent;
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      ix-=indent;
      item=item->parent;
      }
    item=item->next;
    }
  return NULL;
  }


// Did we hit the item, and which part of it did we hit (0=outside, 1=icon, 2=text, 3=box)
FXint FXTreeList::hitItem(const FXTreeItem* item,FXint x,FXint y) const {
  FXint ix,iy,xh,hit=0;
  if(item){
    x-=pos_x;
    y-=pos_y;
    ix=item->x;
    iy=item->y;
    if(iy<=y && y<iy+itemHeight){
      if((options&TREELIST_SHOWS_BOXES) && ((item->state&FXTreeItem::HASITEMS) || item->first)){
        xh=ix-indent+(SIDE_SPACING/2)+itemHeight/2-2;
        if(xh-4<=x && x<=xh+4) return 3;
        }
      hit=item->hitItem(this,x-ix,y-iy);
      }
    }
  return hit;
  }


// Repaint
void FXTreeList::updateItem(FXTreeItem* item){
  if(xid && item){
    update(0,pos_y+item->y,content_w,itemHeight);
    }
  }


// Enable one item
FXbool FXTreeList::enableItem(FXTreeItem* item){
  if(!item){ fxerror("%s::enableItem: item is NULL.\n",getClassName()); } 
  if(!item->isEnabled()){
    item->setEnabled(TRUE);
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }


// Disable one item
FXbool FXTreeList::disableItem(FXTreeItem* item){
  if(!item){ fxerror("%s::disableItem: item is NULL.\n",getClassName()); } 
  if(item->isEnabled()){
    item->setEnabled(FALSE);
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }


// Select one item
FXbool FXTreeList::selectItem(FXTreeItem* item){
  if(!item){ fxerror("%s::selectItem: item is NULL.\n",getClassName()); } 
  if(!item->isSelected()){
    switch(options&SELECT_MASK){
      case TREELIST_SINGLESELECT:
      case TREELIST_BROWSESELECT:
        killSelection();
        item->setSelected(TRUE);
        updateItem(item);
        break;
      case TREELIST_EXTENDEDSELECT:
      case TREELIST_MULTIPLESELECT:
        item->setSelected(TRUE);
        updateItem(item);
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// Deselect one item
FXbool FXTreeList::deselectItem(FXTreeItem* item){
  if(!item){ fxerror("%s::deselectItem: item is NULL.\n",getClassName()); } 
  if(item->isSelected()){
    switch(options&SELECT_MASK){
      case TREELIST_EXTENDEDSELECT:
      case TREELIST_MULTIPLESELECT:
      case TREELIST_SINGLESELECT:
        item->setSelected(FALSE);
        updateItem(item);
      case TREELIST_BROWSESELECT:
        break;
      }
    return TRUE;
    }
  return FALSE;
  }


// toggle one item
FXbool FXTreeList::toggleItem(FXTreeItem* item){
  if(!item){ fxerror("%s::toggleItem: item is NULL.\n",getClassName()); } 
  switch(options&SELECT_MASK){
    case TREELIST_BROWSESELECT:
      if(!item->isSelected()){
        killSelection();
        item->setSelected(TRUE);
        updateItem(item);
        }
      break;
    case TREELIST_SINGLESELECT:
      if(!item->isSelected()){
        killSelection();
        item->setSelected(TRUE);
        }
      else{
        item->setSelected(FALSE);
        }
      updateItem(item);
      break;
    case TREELIST_EXTENDEDSELECT:
    case TREELIST_MULTIPLESELECT:
      item->setSelected(!item->isSelected());
      updateItem(item);
      break;
    }
  return TRUE;
  }


// Open item
FXbool FXTreeList::openItem(FXTreeItem* item){
  if(item==NULL){ fxerror("%s::openItem: item is NULL.\n",getClassName()); }
  if(!item->isOpened()){
    item->setOpened(TRUE);
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }  
  
 
// Close item
FXbool FXTreeList::closeItem(FXTreeItem* item){
  if(item==NULL){ fxerror("%s::closeItem: item is NULL.\n",getClassName()); }
  if(item->isOpened()){
    item->setOpened(FALSE);
    updateItem(item);
    return TRUE;
    }
  return FALSE;
  }  


// Collapse all subtrees under item
FXbool FXTreeList::collapseTree(FXTreeItem* tree){
  if(tree==NULL){ fxerror("%s::collapseTree: tree is NULL.\n",getClassName()); }
  if(tree->isExpanded()){
    tree->setExpanded(FALSE);
    if(!(options&TREELIST_AUTOSELECT)){     // In autoselect, already shown as expanded!
      if(tree->first){
        recalc();
        }
      else{
        updateItem(tree);
        }
      }
    return TRUE;
    }
  return FALSE;
  }

 
// Expand subtree under item
FXbool FXTreeList::expandTree(FXTreeItem* tree){
  if(tree==NULL){ fxerror("%s::expandTree: tree is NULL.\n",getClassName()); }
  if(!tree->isExpanded()){
    tree->setExpanded(TRUE);
    if(!(options&TREELIST_AUTOSELECT)){     // In autoselect, already shown as expanded!
      if(tree->first){
        recalc();
        }
      else{
        updateItem(tree);
        }
      }
    return TRUE;
    }
  return FALSE;
  }


// Reparent item under a new parent
void FXTreeList::reparentItem(FXTreeItem* item,FXTreeItem* p){
  if(!item){ fxerror("%s::reparentItem: item is NULL.\n",getClassName()); } 
  if(item->parent!=p){
    if(item->prev) item->prev->next=item->next; else if(item->parent) item->parent->first=item->next; else firstitem=item->next;
    if(item->next) item->next->prev=item->prev; else if(item->parent) item->parent->last=item->prev; else lastitem=item->prev;
    if(p){
      item->prev=p->last;
      item->next=NULL;
      if(item->prev) item->prev->next=item; else p->first=item;
      p->last=item;
      }
    else{
      item->prev=lastitem;
      item->next=NULL;
      if(item->prev) item->prev->next=item; else firstitem=item;
      lastitem=item;
      }
    item->parent=p;
    recalc();
    }
  }


// Start motion timer while in this window
long FXTreeList::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onEnter(sender,sel,ptr);
  if(!timer){timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);}
  cursoritem=NULL;
  return 1;
  }


// Stop motion timer when leaving window
long FXTreeList::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onLeave(sender,sel,ptr);
  if(timer){getApp()->removeTimeout(timer);timer=NULL;}
  cursoritem=NULL;
  return 1;
  }


// Gained focus
long FXTreeList::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusIn(sender,sel,ptr);
  if(currentitem){ 
    currentitem->setFocus(TRUE);
    updateItem(currentitem);
    }
  return 1;
  }


// Lost focus
long FXTreeList::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onFocusOut(sender,sel,ptr);
  if(currentitem){
    currentitem->setFocus(FALSE);
    updateItem(currentitem);
    }
  return 1;
  }


// We have the selection
long FXTreeList::onSelectionGained(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionGained(sender,sel,ptr);
  return 1;
  }


// We lost the selection
long FXTreeList::onSelectionLost(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onSelectionLost(sender,sel,ptr);
  killSelection();
  return 1;
  }


// Draw item list
long FXTreeList::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTreeItem* item=firstitem;
  FXint yh,xh,x,y,xp,hh;
  FXDCWindow dc(this,event);
  dc.setTextFont(font);
  hh=(itemHeight/2)&~1;   // Got to be even for the dotted lines!
  x=pos_x;
  y=pos_y;
  if(options&TREELIST_ROOT_BOXES) x+=indent;
  while(item && y<event->rect.y+event->rect.h){
    if(event->rect.y<=y+itemHeight){
      
      // Draw item
      dc.setForeground(backColor);
      dc.fillRectangle(pos_x,y,content_w,itemHeight);
      item->draw(this,dc,x,y,content_w,itemHeight);
      
      // Show other paraphernalia such as dotted lines and expand-boxes
      if((options&(TREELIST_SHOWS_LINES|TREELIST_SHOWS_BOXES)) && (item->parent || (options&TREELIST_ROOT_BOXES))){
        yh=y+hh;
        xh=x-indent+(SIDE_SPACING/2)+itemHeight/2-2;
        dc.setForeground(lineColor);
        dc.setFillStyle(FILL_STIPPLED);
        dc.setStipple(STIPPLE_GRAY,x,y);
        if(options&TREELIST_SHOWS_LINES){                   // Connect items with lines
          FXTreeItem* p=item->parent;
          xp=xh;
          while(p){
            xp-=indent;
            if(p->next) dc.fillRectangle(xp,y,1,itemHeight);
            p=p->parent;
            }
          if((options&TREELIST_SHOWS_BOXES) && ((item->state&FXTreeItem::HASITEMS) || item->first)){
            if(item->prev || item->parent) dc.fillRectangle(xh,y,1,yh-y-4);
            if(item->next) dc.fillRectangle(xh,yh+4,1,y+itemHeight-yh-4);
           }
          else{
            if(item->prev || item->parent) dc.fillRectangle(xh,y,1,hh);
            if(item->next) dc.fillRectangle(xh,yh,1,itemHeight);
            dc.fillRectangle(xh,yh,x+(SIDE_SPACING/2)-2-xh,1);
            }
          //dc.setLineStyle(LINE_SOLID);
          dc.setFillStyle(FILL_SOLID);
          }
        
        // Boxes before items for expand/collapse of item
        if((options&TREELIST_SHOWS_BOXES) && ((item->state&FXTreeItem::HASITEMS) || item->first)){
          dc.setFillStyle(FILL_STIPPLED);
          dc.drawLine(xh+4,yh,x+(SIDE_SPACING/2)-2,yh);
          dc.setFillStyle(FILL_SOLID);
          dc.drawLine(xh-4,yh-4,xh+4,yh-4);
          dc.drawLine(xh-4,yh+4,xh+4,yh+4);
          dc.drawLine(xh+4,yh-4,xh+4,yh+4);
          dc.drawLine(xh-4,yh-4,xh-4,yh+4);
          dc.setForeground(textColor);
          dc.drawLine(xh-2,yh,xh+2,yh);
          if(!(options&TREELIST_AUTOSELECT) && !item->isExpanded()){
            dc.drawLine(xh,yh-2,xh,yh+2);
            }
          }
        }
      }
    
    y+=itemHeight;
    
    // Move on to the next item
    if(item->first && ((options&TREELIST_AUTOSELECT) || item->isExpanded())){
      x+=indent;
      item=item->first;
      continue;
      }
    while(!item->next && item->parent){
      x-=indent;
      item=item->parent;
      }
    item=item->next;
    }
  if(y<event->rect.y+event->rect.h){
    dc.setForeground(backColor);
    dc.fillRectangle(event->rect.x,y,event->rect.w,event->rect.y+event->rect.h-y);
    }
  return 1;
  }


// We were asked about tip text
long FXTreeList::onQueryTip(FXObject* sender,FXSelector,void*){
  FXint x,y; FXuint state;
  if((flags&FLAG_TIP) && !(options&TREELIST_AUTOSELECT)){   // No tip when autoselect!
    getCursorPosition(x,y,state);
    FXTRACE((250,"%s::onQueryTip %08x (%d,%d)\n",getClassName(),this,x,y));
    FXTreeItem *item=getItemAt(x,y);
    if(item){
      FXString string=item->getText();
      sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&string);
      return 1;
      }
    }
  return 0;
  }


// We were asked about status text
long FXTreeList::onQueryHelp(FXObject* sender,FXSelector,void*){
  if(!help.empty() && (flags&FLAG_HELP)){
    FXTRACE((200,"%s::onQueryHelp %08x\n",getClassName(),this));
    sender->handle(this,MKUINT(ID_SETSTRINGVALUE,SEL_COMMAND),(void*)&help);
    return 1;
    }
  return 0;
  }


// Key Press
long FXTreeList::onKeyPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTreeItem *item,*tmp;
  flags&=~FLAG_TIP;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr)) return 1;
  switch(event->code){
    case KEY_Up:
    case KEY_Down:
    case KEY_Right:
    case KEY_Left:
    case KEY_Home:
    case KEY_End:
      item=currentitem;
      switch(event->code){
        case KEY_Up:                      // Move up
          if(!item){ 
            item=lastitem;
            }
          else if(item->prev){
            item=item->prev; 
            while(item->first && item->isExpanded()) item=item->last;
            }
          else if(item->parent){ 
            item=item->parent;
            }
          break;
        case KEY_Down:                    // Move down
          if(!item){
            item=firstitem;
            }
          else if(item->first && item->isExpanded()){
            item=item->first;
            }
          else if(item->next){
            item=item->next;
            }
          else{
            tmp=item;
            while(item->parent && !item->parent->next) item=item->parent;
            if(item->parent && item->parent->next)
              item=item->parent->next;
            else
              item=tmp;
            }
          break;
        case KEY_Right:                   // Move right/down and open subtree
          if(!item){
            item=firstitem;
            }
          else if(!isItemExpanded(item)){
            handle(this,MKUINT(0,SEL_EXPANDED),item);
            }
          else if(item->first){
            item=item->first;
            }
          else if(item->next){
            item=item->next;
            }
          else{
            tmp=item;
            while(item->parent && !item->parent->next) item=item->parent;
            if(item->parent && item->parent->next)
              item=item->parent->next;
            else
              item=tmp;
            }
          break;
        case KEY_Left:                    // Move left/up and close subtree
          if(!item){
            item=firstitem;
            }
          else if(isItemExpanded(item)){
            handle(this,MKUINT(0,SEL_COLLAPSED),item);
            }
          else if(item->parent){
            item=item->parent;
            }
          else if(item->prev){
            item=item->prev;
            }
          break;
        case KEY_Home: 
          item=firstitem; 
          break;
        case KEY_End:  
          item=lastitem; 
          break;
        }
      handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
      if(item && item!=currentitem){
        if((options&SELECT_MASK)==TREELIST_EXTENDEDSELECT){
          extendSelection(item);
          handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
          }
        }
      if((options&SELECT_MASK)==TREELIST_BROWSESELECT){
        handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
        }
      flags&=~FLAG_UPDATE;
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      flags&=~FLAG_UPDATE;
      return 1;
    case KEY_Page_Up:
      setPosition(pos_x,pos_y+verticalScrollbar()->getPage());
      return 1;
    case KEY_Page_Down:
      setPosition(pos_x,pos_y-verticalScrollbar()->getPage());
      return 1;
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(flags&FLAG_DODRAG){
        handle(this,MKUINT(0,SEL_DRAGGED),ptr);
        }
      return 1;
    }
  return 0;
  }


// Key Release 
long FXTreeList::onKeyRelease(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(!isEnabled()) return 0;
  if(target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr)) return 1;
  switch(event->code){
    case KEY_Up:
    case KEY_Down:
    case KEY_Right:
    case KEY_Left:
    case KEY_Home:
    case KEY_End:
      flags|=FLAG_UPDATE;
      if((options&SELECT_MASK)==TREELIST_BROWSESELECT){
        handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
        }
      return 1;
    case KEY_space:
    case KEY_KP_Space:
      flags|=FLAG_UPDATE;
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      return 1;
    case KEY_Page_Up:
    case KEY_Page_Down:
      return 1;
    case KEY_Shift_L:
    case KEY_Shift_R:
    case KEY_Control_L:
    case KEY_Control_R:
    case KEY_Alt_L:
    case KEY_Alt_R:
      if(flags&FLAG_DODRAG){
        handle(this,MKUINT(0,SEL_DRAGGED),ptr);
        }
      return 1;
    }
  return 0;
  }


// We timed out, i.e. the user didn't move for a while
long FXTreeList::onTipTimer(FXObject*,FXSelector,void*){
  FXTRACE((200,"%s::onTipTimer %08x\n",getClassName(),this));
  timer=NULL;
  flags|=FLAG_TIP;
  return 1;
  }


// Scroll timer
long FXTreeList::onAutoScroll(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTreeItem *item;
  FXint xx,yy;
  
  // Scroll the content
  FXScrollArea::onAutoScroll(sender,sel,ptr);
 
  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    
    // Content scrolled, so perhaps something else under cursor
    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
  
    return 1;
    }
  
  // In autoselect mode, stop scrolling when mouse outside window
  if((flags&FLAG_PRESSED) || (options&TREELIST_AUTOSELECT)){

    // Validated position
    xx=event->win_x; if(xx<0) xx=0; else if(xx>=viewport_w) xx=viewport_w-1;
    yy=event->win_y; if(yy<0) yy=0; else if(yy>=viewport_h) yy=viewport_h-1;
    
    // Find item
    item=getItemAt(xx,yy);

    // Got item and different from last time
    if(item && item!=currentitem){

      // Make it the current item
      handle(this,MKUINT(0,SEL_CHANGED),(void*)item);

      // Extend the selection
      if((options&SELECT_MASK)==TREELIST_EXTENDEDSELECT){
        extendSelection(item);
        }
      }
    return 1;
    }
  return 0;
  }


// Mouse motion
long FXTreeList::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  FXTreeItem *oldcursoritem=cursoritem;
  FXuint flg=flags;
  FXTreeItem *item;
  
  // Kill the tip
  flags&=~FLAG_TIP;

  // Kill the tip timer
  if(timer) timer=getApp()->removeTimeout(timer);

  // Right mouse scrolling
  if(flags&FLAG_SCROLLING){
    setPosition(event->win_x-grabx,event->win_y-graby);
    return 1;
    }
  
  // Drag and drop mode
  if(flags&FLAG_DODRAG){
    if(startAutoScroll(event->win_x,event->win_y,TRUE)) return 1;
    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
    return 1;
    }

  // Tentative drag and drop
  if(flags&FLAG_TRYDRAG){
    if(event->moved){
      flags&=~FLAG_TRYDRAG;
      if(handle(this,MKUINT(0,SEL_BEGINDRAG),ptr)){
        flags|=FLAG_DODRAG;
        }
      }
    return 1;
    }

  // In autoselect, change to item under cursor
  if((flags&FLAG_PRESSED) || (options&TREELIST_AUTOSELECT)){
    if(startAutoScroll(event->win_x,event->win_y,FALSE)) return 1;
    item=getItemAt(event->win_x,event->win_y);
    if(item && item!=currentitem){
      handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
      if((options&SELECT_MASK)==TREELIST_EXTENDEDSELECT){
        extendSelection(item);
        }
      }
    return 1;
    }

  // Reset tip timer if nothing's going on
  timer=getApp()->addTimeout(getApp()->menuPause,this,ID_TIPTIMER);

  // Get item we're over
  cursoritem=getItemAt(event->win_x,event->win_y);
  
  // Force GUI update only when needed
  return (cursoritem!=oldcursoritem)||(flg&FLAG_TIP);
  }


// Pressed a button
long FXTreeList::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    if(options&TREELIST_AUTOSELECT) return 1;
    FXTreeItem *item=getItemAt(event->win_x,event->win_y);
    FXint code=hitItem(item,event->win_x,event->win_y);
    if(code==3 && !(options&TREELIST_AUTOSELECT)){
      if(isItemExpanded(item))
        handle(this,MKUINT(0,SEL_COLLAPSED),(void*)item);
      else
        handle(this,MKUINT(0,SEL_EXPANDED),(void*)item);
      return 1;
      }
    if(item || ((options&SELECT_MASK)!=TREELIST_BROWSESELECT)){
      handle(this,MKUINT(0,SEL_CHANGED),(void*)item);
      handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
      if(code==1 && item->isDraggable()) flags|=FLAG_TRYDRAG;
      flags|=FLAG_PRESSED;
      flags&=~FLAG_UPDATE;
      }
    return 1;
    }
  return 0;
  }


// Released button
long FXTreeList::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXuint flg=flags;
  if(isEnabled()){
    ungrab();
    stopAutoScroll();
    flags|=FLAG_UPDATE;
    flags&=~(FLAG_PRESSED|FLAG_TRYDRAG|FLAG_DODRAG);
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(flg&FLAG_DODRAG){
      handle(this,MKUINT(0,SEL_ENDDRAG),ptr);
      }
    if((flg&FLAG_PRESSED) || (options&TREELIST_AUTOSELECT)){
      handle(this,MKUINT(0,SEL_DEACTIVATE),ptr);
      }
    return 1;
    }
  return 0;
  }


// Pressed right button
long FXTreeList::onRightBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_RIGHTBUTTONPRESS),ptr)) return 1;
    flags&=~FLAG_UPDATE;
    flags|=FLAG_PRESSED|FLAG_SCROLLING;
    grabx=event->win_x-pos_x;
    graby=event->win_y-pos_y;
    return 1;
    }
  return 0;
  }


// Released right button
long FXTreeList::onRightBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    flags&=~(FLAG_PRESSED|FLAG_SCROLLING);
    flags|=FLAG_UPDATE;
    if(target && target->handle(this,MKUINT(message,SEL_RIGHTBUTTONRELEASE),ptr)) return 1;
    return 1;
    }
  return 0;
  }



// The widget lost the grab for some reason
long FXTreeList::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXScrollArea::onUngrabbed(sender,sel,ptr);
  flags&=~(FLAG_DODRAG|FLAG_LASSO|FLAG_TRYDRAG|FLAG_PRESSED|FLAG_CHANGED|FLAG_SCROLLING);
  flags|=FLAG_UPDATE;
  stopAutoScroll();
  return 1;
  }


// Current item changed
long FXTreeList::onChanged(FXObject*,FXSelector,void* ptr){
  FXTreeItem *olditem=currentitem;
  setCurrentItem((FXTreeItem*)ptr);
  makeItemVisible((FXTreeItem*)ptr);
  if(target && target->handle(this,MKUINT(message,SEL_CHANGED),ptr)) return 1;
  if(olditem){
    handle(this,MKUINT(0,SEL_CLOSED),olditem);
    }
  if(currentitem){
    handle(this,MKUINT(0,SEL_OPENED),currentitem);
    }
  return 1;
  }


// Command message
long FXTreeList::onCommand(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_COMMAND),ptr);
  }


// Clicked in list
long FXTreeList::onClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_CLICKED),ptr);
  }


// Double clicked in list; ptr may or may not point to an item
long FXTreeList::onDoubleClicked(FXObject*,FXSelector,void* ptr){
  
  // Double click anywhere in the widget
  if(target && target->handle(this,MKUINT(message,SEL_DOUBLECLICKED),ptr)) return 1;
  
  // Double click on an item
  if(ptr){
    
    // Expand the item 
    if(isItemExpanded((FXTreeItem*)ptr)){
      if(handle(this,MKUINT(0,SEL_COLLAPSED),ptr)) return 1;
      }
    else{
      if(handle(this,MKUINT(0,SEL_EXPANDED),ptr)) return 1;
      }
    }
  return 0;
  }


// Triple clicked in list; ptr may or may not point to an item
long FXTreeList::onTripleClicked(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_TRIPLECLICKED),ptr);
  }


// Item opened
long FXTreeList::onItemOpened(FXObject*,FXSelector,void* ptr){
  openItem((FXTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_OPENED),ptr);
  }


// Item closed
long FXTreeList::onItemClosed(FXObject*,FXSelector,void* ptr){
  closeItem((FXTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_CLOSED),ptr);
  }


// Item expanded
long FXTreeList::onItemExpanded(FXObject*,FXSelector,void* ptr){
  expandTree((FXTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_EXPANDED),ptr);
  }


// Item collapsed
long FXTreeList::onItemCollapsed(FXObject*,FXSelector,void* ptr){
  collapseTree((FXTreeItem*)ptr);
  return target && target->handle(this,MKUINT(message,SEL_COLLAPSED),ptr);
  }


// Button or Key activate
long FXTreeList::onActivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  if(currentitem && currentitem->isEnabled()){
    switch(options&SELECT_MASK){
      case TREELIST_EXTENDEDSELECT:
        if(event->state&SHIFTMASK){
          if(anchoritem){
            selectItem(anchoritem);
            extendSelection(currentitem);
            }
          else{
            selectItem(currentitem);
            setAnchorItem(currentitem);
            }
          }
        else if(event->state&CONTROLMASK){
          //if(!isItemSelected(current)) selectItem(current);
          //toggleItem(current);
          setAnchorItem(currentitem);
          }
        else{
          killSelection();
          selectItem(currentitem);
          setAnchorItem(currentitem);
          }
        break;
      case TREELIST_SINGLESELECT:
      case TREELIST_MULTIPLESELECT:
        toggleItem(currentitem);
        setAnchorItem(currentitem);
        break;
      case TREELIST_BROWSESELECT:
        break;
      }
    }
  else{
    switch(options&SELECT_MASK){
      case TREELIST_EXTENDEDSELECT:
      case TREELIST_SINGLESELECT:
        if(!(event->state&(SHIFTMASK|CONTROLMASK))) killSelection();
        break;
      case TREELIST_MULTIPLESELECT:
      case TREELIST_BROWSESELECT:
        break;
      }
    }
  return 1;
  }
  

// Button or Key deactivate
long FXTreeList::onDeactivate(FXObject*,FXSelector,void* ptr){
  FXEvent* event=(FXEvent*)ptr;
  setAnchorItem(currentitem);
  if(currentitem && currentitem->isEnabled()){
    switch(options&SELECT_MASK){
      case TREELIST_EXTENDEDSELECT:
        if(!event->moved && (event->state&CONTROLMASK)) toggleItem(currentitem);
        break;
      case TREELIST_MULTIPLESELECT:
      case TREELIST_SINGLESELECT:
      case TREELIST_BROWSESELECT:
        break;
      }
    }
  if(event->click_count==1){
    handle(this,MKUINT(0,SEL_CLICKED),(void*)currentitem);
    }
  else if(event->click_count==2){
    handle(this,MKUINT(0,SEL_DOUBLECLICKED),(void*)currentitem);
    }
  else if(event->click_count==3){
    handle(this,MKUINT(0,SEL_TRIPLECLICKED),(void*)currentitem);
    }
  if(currentitem && currentitem->isEnabled()){
    handle(this,MKUINT(0,SEL_COMMAND),(void*)currentitem);
    }
  return 1;
  }


// True if a before b
FXbool FXTreeList::before(const FXTreeItem* a,const FXTreeItem* b) const {
  if(a==b) return FALSE;
  while(b){
    if(a==b) return TRUE;
    b=b->getAbove();
    }
  return FALSE;
  }


// Mark items
FXbool FXTreeList::mark(FXTreeItem* beg,FXTreeItem* end,FXuint sel){
  register FXbool changes=FALSE;
  register FXTreeItem *item=beg;
  while(item){
    if(sel){
      if(!item->isSelected()){
        item->setMarked(FALSE);
        if(item->isEnabled()){
          item->setSelected(TRUE);
          updateItem(item);
          changes=TRUE;
          }
        }
      else{
        item->setMarked(TRUE);
        }
      }
    else{
      if(item->isSelected()){
        item->setMarked(TRUE);
        if(item->isEnabled()){
          item->setSelected(FALSE);
          updateItem(item);
          changes=TRUE;
          }
        }
      else{
        item->setMarked(FALSE);
        }
      }
    if(item==end) break;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  return changes;
  }


// Restore items to previous state
FXbool FXTreeList::restore(FXTreeItem* beg,FXTreeItem* end){
  register FXbool changes=FALSE;
  register FXTreeItem *item=beg;
  while(item){
    if(item->isMarked()){
      if(!item->isSelected()){
        item->setSelected(TRUE);
        updateItem(item);
        changes=TRUE;
        }
      }
    else{
      if(item->isSelected()){
        item->setSelected(FALSE);
        updateItem(item);
        changes=TRUE;
        }
      }
    if(item==end) break;
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  return changes;
  }


// Extend selection
FXbool FXTreeList::extendSelection(FXTreeItem* item){
  register FXbool changes=FALSE;
  register FXuint sel;
  if(item && anchoritem){
    sel=anchoritem->isSelected();
    if(before(extentitem,anchoritem)){
      if(before(item,extentitem)){                      // item < extent < anchor
        mark(item,extentitem->getAbove(),sel);
        }
      else if(before(anchoritem,item)){                 // extent < anchor < item
        restore(extentitem,anchoritem->getAbove());
        mark(anchoritem->getBelow(),item,sel);
        }
      else{                                             // extent <= item <= anchor
        restore(extentitem,item->getAbove());
        }
      }
    else{
      if(before(extentitem,item)){                      // anchor <= extent < item
        mark(extentitem->getBelow(),item,sel);
        }
      else if(before(item,anchoritem)){                 // item < anchor <= extent
        restore(anchoritem->getBelow(),extentitem);
        mark(item,anchoritem->getAbove(),sel);
        }
      else{                                             // anchor <= item <= extent
        restore(item->getBelow(),extentitem);
        }
      }
    extentitem=item;
    }
  return changes;
  }


// Kill selection
FXbool FXTreeList::killSelection(){
  register FXbool changes=FALSE;
  register FXTreeItem *item=firstitem;
  while(item){
    if(item->isSelected()){
      item->setSelected(FALSE);
      updateItem(item);
      changes=TRUE;
      }
    if(item->first){item=item->first;continue;}
    while(!item->next && item->parent){item=item->parent;}
    item=item->next;
    }
  extentitem=anchoritem;
  return changes;
  }


// Sort items
void FXTreeList::sort(FXTreeItem*& f1,FXTreeItem*& t1,FXTreeItem*& f2,FXTreeItem*& t2,int n){
  FXTreeItem *ff1,*tt1,*ff2,*tt2,*q;
  FXint m;
  if(f2==NULL){ 
    f1=NULL; 
    t1=NULL; 
    return; 
    }
  if(n>1){
    m=n/2;
    n=n-m;
    sort(ff1,tt1,f2,t2,n);  // 1 or more
    sort(ff2,tt2,f2,t2,m);  // 0 or more
    FXASSERT(ff1);
    if(ff2 && sortfunc(ff1,ff2)>0){
      f1=ff2;
      ff2->prev=NULL;
      ff2=ff2->next;
      }
    else{
      f1=ff1;
      ff1->prev=NULL;
      ff1=ff1->next;
      }
    t1=f1;
    t1->next=NULL;
    while(ff1 || ff2){
      if(ff1==NULL){ t1->next=ff2; ff2->prev=t1; t1=tt2; break; }
      if(ff2==NULL){ t1->next=ff1; ff1->prev=t1; t1=tt1; break; }
      if(sortfunc(ff1,ff2)>0){
        t1->next=ff2;
        ff2->prev=t1;
        t1=ff2;
        ff2=ff2->next;
        }
      else{
        t1->next=ff1;
        ff1->prev=t1;
        t1=ff1;
        ff1=ff1->next;
        }
      t1->next=NULL;
      }
    return;
    }
  FXASSERT(f2);
  f1=f2;
  t1=f2;
  f2=f2->next;
  while(f2){
    f2->prev=NULL;
    if(sortfunc(f2,t1)>0){
      t1->next=f2;
      f2->prev=t1;
      t1=f2;
      f2=f2->next;
      continue;
      }
    if(sortfunc(f1,f2)>0){
      q=f2;
      f2=f2->next;
      q->next=f1;
      f1->prev=q;
      f1=q;
      continue;
      }
    break;
    }
  FXASSERT(f1);
  FXASSERT(t1);
  f1->prev=NULL;
  t1->next=NULL;
  }


// Sort the items based on the sort function
void FXTreeList::sortItems(){
  if(sortfunc){
    FXTreeItem* f=firstitem;
    FXTreeItem* l=lastitem;
    sort(firstitem,lastitem,f,l,getNumItems());
    recalc();
    }
  }


// Sort child items
void FXTreeList::sortChildItems(FXTreeItem* item){
  if(sortfunc){
    FXTreeItem* f=item->first;
    FXTreeItem* l=item->last;
    sort(item->first,item->last,f,l,item->getNumChildren());
    if(item->isExpanded()) recalc();     // No need to recalc if it ain't visible!
    }
  }


// Set current item
void FXTreeList::setCurrentItem(FXTreeItem* item){
  if(item!=currentitem){
    
    // Deactivate old item
    if(currentitem){
      
      // No visible change if it doen't have the focus
      if(hasFocus()){
        currentitem->setFocus(FALSE);
        updateItem(currentitem);
        }
      }
    
    currentitem=item;
    
    // Activate new item
    if(currentitem){
      
      // No visible change if it doen't have the focus
      if(hasFocus()){
        currentitem->setFocus(TRUE);
        updateItem(currentitem);
        }
      }
    }
  
  // In browse select mode, select this item
  if((options&SELECT_MASK)==TREELIST_BROWSESELECT){
    if(currentitem && !currentitem->isSelected()){
      killSelection();
      selectItem(currentitem);
      }
    }
  }


// Set anchor item
void FXTreeList::setAnchorItem(FXTreeItem* item){
  anchoritem=item;
  extentitem=item;
  }



// Create item
FXTreeItem* FXTreeList::createItem(const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){ 
  return new FXTreeItem(text,oi,ci,ptr); 
  }


// Add item as first one under parent p
FXTreeItem* FXTreeList::addItemFirst(FXTreeItem* p,FXTreeItem* item){
  if(!item){ fxerror("%s::addItemFirst: item is NULL.\n",getClassName()); } 
  if(p){
    item->prev=NULL;
    item->next=p->first;
    if(item->next) item->next->prev=item; else p->last=item;
    p->first=item;
    }
  else{
    item->prev=NULL;
    item->next=firstitem;
    if(item->next) item->next->prev=item; else lastitem=item;
    firstitem=item;
    }
  item->parent=p;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Add item as last one under parent p
FXTreeItem* FXTreeList::addItemLast(FXTreeItem* p,FXTreeItem* item){
  if(!item){ fxerror("%s::addItemLast: item is NULL.\n",getClassName()); } 
  if(p){
    item->prev=p->last;
    item->next=NULL;
    if(item->prev) item->prev->next=item; else p->first=item;
    p->last=item;
    }
  else{
    item->prev=lastitem;
    item->next=NULL;
    if(item->prev) item->prev->next=item; else firstitem=item;
    lastitem=item;
    }
  item->parent=p;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Link item after other
FXTreeItem* FXTreeList::addItemAfter(FXTreeItem* other,FXTreeItem* item){
  if(!item){ fxerror("%s::addItemAfter: item is NULL.\n",getClassName()); } 
  if(!other){ fxerror("%s::addItemAfter: other item is NULL.\n",getClassName()); }
  item->prev=other;
  item->next=other->next;
  other->next=item;
  if(item->next) item->next->prev=item; else if(other->parent) other->parent->last=item; else lastitem=item; 
  item->parent=other->parent;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Link item before other 
FXTreeItem* FXTreeList::addItemBefore(FXTreeItem* other,FXTreeItem* item){
  if(!item){ fxerror("%s::addItemBefore: item is NULL.\n",getClassName()); } 
  if(!other){ fxerror("%s::addItemBefore: other item is NULL.\n",getClassName()); }
  item->next=other;
  item->prev=other->prev;
  other->prev=item;
  if(item->prev) item->prev->next=item; else if(other->parent) other->parent->first=item; else firstitem=item; 
  item->parent=other->parent;
  item->first=NULL;
  item->last=NULL;
  item->x=0;
  item->y=0;
  recalc();
  return item;
  }


// Add item as first one under parent p
FXTreeItem* FXTreeList::addItemFirst(FXTreeItem* p,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  return addItemFirst(p,createItem(text,oi,ci,ptr));
  }


// Add item as last one under parent p
FXTreeItem* FXTreeList::addItemLast(FXTreeItem* p,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  return addItemLast(p,createItem(text,oi,ci,ptr));
  }


// Link item after other
FXTreeItem* FXTreeList::addItemAfter(FXTreeItem* other,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  return addItemAfter(other,createItem(text,oi,ci,ptr));
  }


// Link item before other 
FXTreeItem* FXTreeList::addItemBefore(FXTreeItem* other,const FXString& text,FXIcon* oi,FXIcon* ci,void* ptr){
  return addItemBefore(other,createItem(text,oi,ci,ptr));
  }


// Remove node from list
void FXTreeList::removeItem(FXTreeItem* item){
  if(item){
    if(item->prev) item->prev->next=item->next; else if(item->parent) item->parent->first=item->next; else firstitem=item->next;
    if(item->next) item->next->prev=item->prev; else if(item->parent) item->parent->last=item->prev; else lastitem=item->prev;
    if(currentitem==item) currentitem=NULL;
    if(anchoritem==item) anchoritem=NULL;
    if(extentitem==item) extentitem=NULL;
    removeItems(item->first,item->last);
    delete item;
    recalc();
    }
  }


// Remove all siblings from [fm,to]
void FXTreeList::removeItems(FXTreeItem* fm,FXTreeItem* to){
  if(fm && to){
    FXTreeItem *item;
    if(fm->prev) fm->prev->next=to->next; else if(fm->parent) fm->parent->first=to->next; else firstitem=to->next;
    if(to->next) to->next->prev=fm->prev; else if(to->parent) to->parent->last=fm->prev; else lastitem=fm->prev;
    do{
      item=fm;
      if(currentitem==item) currentitem=NULL;
      if(anchoritem==item) anchoritem=NULL;
      if(extentitem==item) extentitem=NULL;
      removeItems(item->first,item->last);
      fm=fm->next;
      delete item;
      }
    while(item!=to);
    recalc();
    }
  }


// Remove all items
void FXTreeList::removeAllItems(){
  removeItems(firstitem,lastitem);
  }


// Find toplevel item by prefix of length len
FXTreeItem* FXTreeList::findItem(const FXString& text,FXuint len) const {
  register FXTreeItem* item;
  for(item=firstitem; item; item=item->next){
    if(compare(text,item->label,len)==0) return item;
    }
  return NULL;
  }


// Find child item by prefix of length len
FXTreeItem* FXTreeList::findChildItem(FXTreeItem* parentitem,const FXString& text,FXuint len) const {
  register FXTreeItem* item;
  if(parentitem==NULL){ fxerror("%s::findChildItem: parent is NULL.\n",getClassName()); }
  for(item=parentitem->first; item; item=item->next){
    if(compare(text,item->label,len)==0) return item;
    }
  return NULL;
  }


// Change the font
void FXTreeList::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  if(font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }


// Change help text
void FXTreeList::setHelpText(const FXString& text){
  help=text;
  }


// Set text color
void FXTreeList::setTextColor(FXColor clr){
  if(clr!=textColor){
    textColor=clr;
    update();
    }
  }


// Set select background color
void FXTreeList::setSelBackColor(FXColor clr){
  if(clr!=selbackColor){
    selbackColor=clr;
    update();
    }
  }


// Set selected text color
void FXTreeList::setSelTextColor(FXColor clr){
  if(clr!=seltextColor){
    seltextColor=clr;
    update();
    }
  }


// Set line color
void FXTreeList::setLineColor(FXColor clr){
  if(clr!=lineColor){
    lineColor=clr;
    update();
    }
  }


// Set parent to child indent amount
void FXTreeList::setIndent(FXint in){
  if(indent!=in){
    indent=in;
    recalc();
    }
  }


// Change list style
void FXTreeList::setListStyle(FXuint style){
  FXuint opts=(options&~TREELIST_MASK) | (style&TREELIST_MASK);
  if(options!=opts){
    options=opts;
    recalc();
    }
  }


// Get list style
FXuint FXTreeList::getListStyle() const { 
  return (options&TREELIST_MASK); 
  }


// Save data
void FXTreeList::save(FXStream& store) const {
  FXScrollArea::save(store);
  store << firstitem;
  store << lastitem;
  store << anchoritem;
  store << currentitem;
  store << extentitem;
  store << font;
  store << textColor;
  store << selbackColor;
  store << seltextColor;
  store << lineColor;
  store << itemWidth;
  store << itemHeight;
  store << nrows;
  store << indent;
  store << help;
  }


// Load data
void FXTreeList::load(FXStream& store){ 
  FXScrollArea::load(store);
  store >> firstitem;
  store >> lastitem;
  store >> anchoritem;
  store >> currentitem;
  store >> extentitem;
  store >> font;
  store >> textColor;
  store >> selbackColor;
  store >> seltextColor;
  store >> lineColor;
  store >> itemWidth;
  store >> itemHeight;
  store >> nrows;
  store >> indent;
  store >> help;
  }


// Cleanup
FXTreeList::~FXTreeList(){
  removeAllItems();
  if(timer) getApp()->removeTimeout(timer);
  firstitem=(FXTreeItem*)-1;
  lastitem=(FXTreeItem*)-1;
  anchoritem=(FXTreeItem*)-1;
  currentitem=(FXTreeItem*)-1;
  extentitem=(FXTreeItem*)-1;
  font=(FXFont*)-1;
  timer=(FXTimer*)-1;
  }


