/********************************************************************************
*                                                                               *
*                        T o o l b a r   W i d g e t                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 2000 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXToolbar.cpp,v 1.18 2000/04/27 19:46:21 jeroen Exp $                    *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXDrawable.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXPacker.h"
#include "FXToolbar.h"
#include "FXShell.h"
#include "FXTopWindow.h"

  
/*
  Notes:
  - We still need some seperator widgets which automatically flip
    when the bar wraps; also, perhaps the bar's orientation should
    automatically flip also?
  - Need to be able to fill up from left or right, so we can use
    it for menus also.
  - Need to allow for LAYOUT_FILL_X and such.
  - Orientation:- should it change when undocked?
  - Solid drag mode didn't work too well:- better way might be
    to undock as soon as dragging starts, then will probably keep
    tracking properly.
  - Need three types of toolbars:
    1) No grip at all.
    2) Single line grip to shuffle w/o undocking capability.
    3) Double line grip to allow undocking as well as shuffling.
  - Should a docked toolbar try to delete the FXToolbarShell if the
    toolbar is deleted?
  - Docking location should be determined by grip-point, i.e. where
    mouse is, rather than top-left corner.
*/

#define GRIP_SIZE    9      // Grip size
#define PROXIMITY    10     // How close to edge before considered docked

#define LAYOUT_SIDE_MASK (LAYOUT_SIDE_LEFT|LAYOUT_SIDE_RIGHT|LAYOUT_SIDE_TOP|LAYOUT_SIDE_BOTTOM)


#define GRIP_SIDE_MASK (DOCKBAR_GRIP_LEFT|DOCKBAR_GRIP_RIGHT|DOCKBAR_GRIP_TOP|DOCKBAR_GRIP_BOTTOM)


/*******************************************************************************/

// Map
FXDEFMAP(FXToolbar) FXToolbarMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXToolbar::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXToolbar::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXToolbar::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXToolbar::onLeftBtnRelease),
  FXMAPFUNC(SEL_DRAGGED,0,FXToolbar::onDragged),
  FXMAPFUNC(SEL_BEGINDRAG,0,FXToolbar::onBeginDrag),
  FXMAPFUNC(SEL_ENDDRAG,0,FXToolbar::onEndDrag),
  FXMAPFUNC(SEL_FOCUS_PREV,0,FXToolbar::onFocusLeft),
  FXMAPFUNC(SEL_FOCUS_NEXT,0,FXToolbar::onFocusRight),
  };


// Object implementation
FXIMPLEMENT(FXToolbar,FXPacker,FXToolbarMap,ARRAYNUMBER(FXToolbarMap))


// Deserialization
FXToolbar::FXToolbar(){
  flags|=FLAG_ENABLED;
  drydock=NULL;
  wetdock=NULL;
  location.x=0;
  location.y=0;
  location.w=0;
  location.h=0;
  docking=DOCK_TOP;
  }


// Make a horizontal one
FXToolbar::FXToolbar(FXTopWindow* p,FXTopWindow* q,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb,FXint hs,FXint vs):
  FXPacker(((opts&TOOLBAR_FLOATING)?q:p),opts&~(LAYOUT_LEFT|LAYOUT_RIGHT|LAYOUT_CENTER_X|LAYOUT_TOP|LAYOUT_BOTTOM|LAYOUT_CENTER_Y|LAYOUT_FIX_X|LAYOUT_FIX_Y),x,y,w,h,pl,pr,pt,pb,hs,vs){
  flags|=FLAG_ENABLED;
  defaultCursor=getApp()->moveCursor;
  dragCursor=getApp()->moveCursor;
  drydock=p;
  wetdock=q;
  location.x=x;
  location.y=y;
  location.w=w;
  location.h=h;
  if(options&TOOLBAR_FLOATING){ 
    docking=DOCK_FLOAT;
    }
  else if((options&LAYOUT_SIDE_MASK)==LAYOUT_SIDE_TOP){ 
    docking=DOCK_TOP;
    }
  else if((options&LAYOUT_SIDE_MASK)==LAYOUT_SIDE_BOTTOM){
    docking=DOCK_BOTTOM;
    }
  else if((options&LAYOUT_SIDE_MASK)==LAYOUT_SIDE_LEFT){
    docking=DOCK_LEFT;
    }
  else if((options&LAYOUT_SIDE_MASK)==LAYOUT_SIDE_RIGHT){
    docking=DOCK_RIGHT;
    }
  }


// Compute minimum width based on child layout hints
int FXToolbar::getDefaultWidth(){
  register FXint w,wcum,wmax,mw,n;
  register FXWindow* child;
  register FXuint hints;
  wcum=wmax=n=0;
  if(options&PACK_UNIFORM_WIDTH) mw=maxChildWidth();
  for(child=getFirst(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) w=child->getWidth(); 
      else if(options&PACK_UNIFORM_WIDTH) w=mw;
      else w=child->getDefaultWidth();
      if(wmax<w) wmax=w;
      wcum+=w; 
      n++;
      }
    }
  if(!(options&LAYOUT_SIDE_LEFT)){
    if(n>1) wcum+=(n-1)*hspacing;
    wcum+=GRIP_SIZE+hspacing;
    wmax=wcum;
    }
  return padleft+padright+wmax+(border<<1);
  }
  

// Compute minimum height based on child layout hints
int FXToolbar::getDefaultHeight(){
  register FXint h,hcum,hmax,mh,n;
  register FXWindow* child;
  register FXuint hints;
  hcum=hmax=n=0;
  if(options&PACK_UNIFORM_HEIGHT) mh=maxChildHeight();
  for(child=getFirst(); child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_HEIGHT) h=child->getHeight(); 
      else if(options&PACK_UNIFORM_HEIGHT) h=mh;
      else h=child->getDefaultHeight();
      if(hmax<h) hmax=h;
      hcum+=h; 
      n++;
      }
    }
  if(options&LAYOUT_SIDE_LEFT){
    if(n>1) hcum+=(n-1)*vspacing;
    hcum+=GRIP_SIZE+vspacing;
    hmax=hcum;
    }
  return padtop+padbottom+hmax+(border<<1);
  }


// Return width for given height
FXint FXToolbar::getWidthForHeight(FXint givenheight){
  register FXint wtot,w,h,mw,mh,wc,hc,space;
  register FXWindow* child;
  register FXuint hints;
  space=givenheight-padtop-padbottom-(border<<1);
  if(options&LAYOUT_SIDE_LEFT) space-=GRIP_SIZE+vspacing;
  if(space<=0) space=1;
  if(options&PACK_UNIFORM_WIDTH) mw=maxChildWidth();
  if(options&PACK_UNIFORM_HEIGHT) mh=maxChildHeight();
  for(child=getFirst(),wtot=wc=hc=0; child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) w=child->getWidth();
      else if(options&PACK_UNIFORM_WIDTH) w=mw;
      else w=child->getDefaultWidth();
      if(hints&LAYOUT_FIX_HEIGHT) h=child->getHeight(); 
      else if(options&PACK_UNIFORM_HEIGHT) h=mh;
      else h=child->getDefaultHeight();
      if(hc+h>space){
        wtot+=wc;
        hc=h;
        wc=w;
        }
      else{
        if(w>wc) wc=w;
        hc+=h+vspacing;
        }
      }
    }
  wtot+=wc;
  if(!(options&LAYOUT_SIDE_LEFT)) wtot+=GRIP_SIZE+hspacing;
  return padleft+padright+wtot+(border<<1);
  }


// Return height for given width
FXint FXToolbar::getHeightForWidth(FXint givenwidth){
  register FXint htot,w,h,mw,mh,wc,hc,space;
  register FXWindow* child;
  register FXuint hints;
  space=givenwidth-padleft-padright-(border<<1);
  if(!(options&LAYOUT_SIDE_LEFT)) space-=GRIP_SIZE+hspacing;
  if(space<=0) space=1;
  if(options&PACK_UNIFORM_WIDTH) mw=maxChildWidth();
  if(options&PACK_UNIFORM_HEIGHT) mh=maxChildHeight();
  for(child=getFirst(),htot=wc=hc=0; child; child=child->getNext()){
    if(child->shown()){
      hints=child->getLayoutHints();
      if(hints&LAYOUT_FIX_WIDTH) w=child->getWidth();
      else if(options&PACK_UNIFORM_WIDTH) w=mw;
      else w=child->getDefaultWidth();
      if(hints&LAYOUT_FIX_HEIGHT) h=child->getHeight(); 
      else if(options&PACK_UNIFORM_HEIGHT) h=mh;
      else h=child->getDefaultHeight();
      if(wc+w>space){
        htot+=hc;
        hc=h;
        wc=w;
        }
      else{
        if(h>hc) hc=h;
        wc+=w+hspacing;
        }
      }
    }
  htot+=hc;
  if(options&LAYOUT_SIDE_LEFT) htot+=GRIP_SIZE+vspacing;
  return padtop+padbottom+htot+(border<<1);
  }


// Recalculate layout
void FXToolbar::layout(){
  FXWindow *fc,*lc;
  FXint x,y,w,h,mw,mh,wc,hc,space;
  FXuint hints;

  FXTRACE((150,"%s::layout\n",getClassName()));

  // Get maximum child size
  if(options&PACK_UNIFORM_WIDTH) mw=maxChildWidth();
  if(options&PACK_UNIFORM_HEIGHT) mh=maxChildHeight();
  
  // Vertical toolbar
  if(options&LAYOUT_SIDE_LEFT){
    space=height-padtop-padbottom-(border<<1)-GRIP_SIZE-vspacing;
    x=border+padleft;
    y=border+padtop+GRIP_SIZE+vspacing;
    fc=getFirst();
    while(fc){
      while(fc && !fc->shown()) fc=fc->getNext();
      for(lc=fc,wc=hc=0; lc; lc=lc->getNext()){
        if(!lc->shown()) continue;
        hints=lc->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) h=lc->getHeight(); 
        else if(options&PACK_UNIFORM_HEIGHT) h=mh;
        else h=lc->getDefaultHeight();
        if(hints&LAYOUT_FIX_WIDTH) w=lc->getWidth();
        else if(options&PACK_UNIFORM_WIDTH) w=mw;
        else w=lc->getDefaultWidth();
        if(hc+h>space && fc!=lc) break;
        if(w>wc) wc=w;
        hc+=h+vspacing;
        }
      FXTRACE((150,"strip: fc=%08x lc=%08x wc=%d hc=%d\n",fc,lc,wc,hc));
      for(; fc!=lc; fc=fc->getNext()){
        if(!fc->shown()) continue;
        hints=fc->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) h=fc->getHeight(); 
        else if(options&PACK_UNIFORM_HEIGHT) h=mh;
        else h=fc->getDefaultHeight();
        if(hints&LAYOUT_FIX_WIDTH) w=fc->getWidth();
        else if(options&PACK_UNIFORM_WIDTH) w=mw;
        else w=fc->getDefaultWidth();
        if((hints&LAYOUT_FILL_X) && !(hints&LAYOUT_FIX_WIDTH)){
          fc->position(x,y,wc,h);
          }
        else if(hints&LAYOUT_CENTER_X){
          fc->position(x+(wc-w)/2,y,w,h);
          }
        else{
          if(hints&LAYOUT_RIGHT)
            fc->position(x+wc-w,y,w,h);
          else
            fc->position(x,y,w,h);
          }
        y+=h+vspacing;
        }
      x+=wc;
      y=border+padtop+GRIP_SIZE+vspacing;
      }
    }
  
  // Horizontal toolbar
  else{
    space=width-padleft-padright-(border<<1)-GRIP_SIZE-hspacing;
    x=border+padleft+GRIP_SIZE+hspacing;
    y=border+padtop;
    fc=getFirst();
    while(fc){
      while(fc && !fc->shown()) fc=fc->getNext();
      for(lc=fc,wc=hc=0; lc; lc=lc->getNext()){
        if(!lc->shown()) continue;
        hints=lc->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) h=lc->getHeight(); 
        else if(options&PACK_UNIFORM_HEIGHT) h=mh;
        else h=lc->getDefaultHeight();
        if(hints&LAYOUT_FIX_WIDTH) w=lc->getWidth();
        else if(options&PACK_UNIFORM_WIDTH) w=mw;
        else w=lc->getDefaultWidth();
        if(wc+w>space && fc!=lc) break;
        if(h>hc) hc=h;
        wc+=w+hspacing;
        }
      FXTRACE((150,"strip: fc=%08x lc=%08x wc=%d hc=%d\n",fc,lc,wc,hc));
      for(; fc!=lc; fc=fc->getNext()){
        if(!fc->shown()) continue;
        hints=fc->getLayoutHints();
        if(hints&LAYOUT_FIX_HEIGHT) h=fc->getHeight(); 
        else if(options&PACK_UNIFORM_HEIGHT) h=mh;
        else h=fc->getDefaultHeight();
        if(hints&LAYOUT_FIX_WIDTH) w=fc->getWidth();
        else if(options&PACK_UNIFORM_WIDTH) w=mw;
        else w=fc->getDefaultWidth();
        if((hints&LAYOUT_FILL_Y) && !(hints&LAYOUT_FIX_HEIGHT)){
          fc->position(x,y,w,hc);
          }
        else if(hints&LAYOUT_CENTER_Y){
          fc->position(x,y+(hc-h)/2,w,h);
          }
        else{
          if(hints&LAYOUT_BOTTOM)
            fc->position(x,y+hc-h,w,h);
          else
            fc->position(x,y,w,h);
          }
        x+=w+hspacing;
        }
      y+=hc;
      x=border+padleft+GRIP_SIZE+hspacing;
      }
    }
  flags&=~FLAG_DIRTY;
  }


// Find dock side
void FXToolbar::findside(FXint x,FXint y,FXRectangle& pos,FXDocking& side,FXWindow*& before){
  FXint left,right,top,bottom;
  FXWindow* child;
  side=DOCK_FLOAT;
  before=NULL;
  pos.x=x;
  pos.y=y;
  pos.w=location.w;
  pos.h=location.h;
  if(drydock){
    drydock->translateCoordinatesTo(left,top,getRoot(),0,0);
    right=left+drydock->getWidth();
    bottom=top+drydock->getHeight();
    child=drydock->getFirst();
    while(1){

      // Skip myself, invisible children and hard-positioned children
      while(child && ((child==this) || !child->shown() || (child->getLayoutHints()&(LAYOUT_FIX_X|LAYOUT_FIX_Y)))){
        child=child->getNext();
        }
      
      // Test if close to an edge
      if(top<=y && y<bottom){
        if(FXABS(x-left)<PROXIMITY){
          pos.h=bottom-top;
          pos.w=getWidthForHeight(pos.h);
          pos.y=top;
          pos.x=left;
          side=DOCK_LEFT;
          before=child;
          }
        if(FXABS(x-right)<PROXIMITY){
          pos.h=bottom-top;
          pos.w=getWidthForHeight(pos.h);
          pos.y=top;
          pos.x=right-pos.w;
          side=DOCK_RIGHT;
          before=child;
          }
        }
      if(left<=x && x<right){
        if(FXABS(y-top)<PROXIMITY){
          pos.w=right-left;
          pos.h=getHeightForWidth(pos.w);
          pos.y=top;
          pos.x=left;
          side=DOCK_TOP;
          before=child;
          }
        if(FXABS(y-bottom)<PROXIMITY){
          pos.w=right-left;
          pos.h=getHeightForWidth(pos.w);
          pos.y=bottom-pos.h;
          pos.x=left;
          side=DOCK_BOTTOM;
          before=child;
          }
        }

      // We have not found a docking position
      if(!child) break;
      
      // Some final fully stretched child also marks the end
      if((child->getLayoutHints()&LAYOUT_FILL_X)&&(child->getLayoutHints()&LAYOUT_FILL_Y)) break;

      // Grow edge inward
      switch(child->getLayoutHints()&LAYOUT_SIDE_MASK){
        case LAYOUT_SIDE_LEFT: left+=drydock->getHSpacing()+child->getWidth(); break;
        case LAYOUT_SIDE_RIGHT: right-=drydock->getHSpacing()+child->getWidth(); break;
        case LAYOUT_SIDE_BOTTOM: bottom-=drydock->getVSpacing()+child->getHeight(); break;
        case LAYOUT_SIDE_TOP: top+=drydock->getVSpacing()+child->getHeight(); break;
        }

      // Next one
      child=child->getNext();
      }
    }
  }


// Change bar docking
void FXToolbar::dockBar(FXint x,FXint y){
  FXuint hints=(LAYOUT_FIX_WIDTH|LAYOUT_FIX_HEIGHT)&getLayoutHints();
  FXWindow* before;
  FXDocking side; 
  findside(x,y,location,side,before);
  if(side!=DOCK_FLOAT){
    if(docking==DOCK_FLOAT){
      reparent(drydock);
      wetdock->hide();
      }
    switch(side){
      case DOCK_TOP: 
        FXTRACE((1,"DOCK_TOP\n"));
        setLayoutHints(hints|LAYOUT_SIDE_TOP|LAYOUT_FILL_X); 
        break;
      case DOCK_BOTTOM: 
        FXTRACE((1,"DOCK_BOTTOM\n"));
        setLayoutHints(hints|LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X); 
        break;
      case DOCK_LEFT: 
        FXTRACE((1,"DOCK_LEFT\n"));
        setLayoutHints(hints|LAYOUT_SIDE_LEFT|LAYOUT_FILL_Y); 
        break;
      case DOCK_RIGHT: 
        FXTRACE((1,"DOCK_RIGHT\n"));
        setLayoutHints(hints|LAYOUT_SIDE_RIGHT|LAYOUT_FILL_Y); 
        break;
      }
    if(before) linkBefore(before);
    }
  else{
    if(docking!=DOCK_FLOAT){
      reparent(wetdock);
      wetdock->show();
      }
    setLayoutHints(hints|LAYOUT_SIDE_TOP|LAYOUT_FILL_X); 
    //resize(pos.w,pos.h);
    wetdock->position(location.x,location.y,wetdock->getDefaultWidth(),wetdock->getDefaultHeight());
    }
  docking=side;
  }


// Move the bar
void FXToolbar::moveBar(FXint x,FXint y){
  FXWindow* before;
  FXDocking side; 
  findside(x,y,location,side,before);
  }


// Draw horizontal bar grip
void FXToolbar::drawhgrip(FXDCWindow& dc,FXint l,FXint r,FXint t,FXint b){
  dc.setForeground(hiliteColor);
  dc.fillRectangle(l,t,1,2);
  dc.fillRectangle(l,t+4,1,2);
  dc.fillRectangle(l,t,r-l,1);
  dc.fillRectangle(l,t+4,r-l,1);
  dc.setForeground(shadowColor);
  dc.fillRectangle(r,t,1,3);
  dc.fillRectangle(r,t+4,1,3);
  dc.fillRectangle(l,t+2,r-l,1);
  dc.fillRectangle(l,t+6,r-l,1);
  }


// Draw vertical bar grip
void FXToolbar::drawvgrip(FXDCWindow& dc,FXint l,FXint r,FXint t,FXint b){
  dc.setForeground(hiliteColor);  
  dc.fillRectangle(l,t,2,1);
  dc.fillRectangle(l+4,t,2,1);
  dc.fillRectangle(l,t,1,b-t);
  dc.fillRectangle(l+4,t,1,b-t);
  dc.setForeground(shadowColor);
  dc.fillRectangle(l,b,3,1);
  dc.fillRectangle(l+4,b,3,1);
  dc.fillRectangle(l+2,t,1,b-t);
  dc.fillRectangle(l+6,t,1,b-t);
  }


// Slightly different from Frame border
long FXToolbar::onPaint(FXObject* sender,FXSelector sel,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  register FXint t,b,l,r;
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  drawFrame(dc,0,0,width,height);
  if(options&LAYOUT_SIDE_LEFT){
    l=padleft+border+2;
    r=width-padright-border-3;
    t=padtop+border;
    b=t+GRIP_SIZE;
    drawhgrip(dc,l,r,t,b);
    }
  else{
    t=padtop+border+2;
    b=height-padbottom-border-3;
    l=padleft+border;
    r=l+GRIP_SIZE;
    drawvgrip(dc,l,r,t,b);
    }
  return 1;
  }



// Start a drag operation
long FXToolbar::onBeginDrag(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  FXDCWindow dc(getRoot());
  FXTRACE((1,"FXToolbar::onBeginDrag\n"));
  location.x=event->root_x-event->click_x;
  location.y=event->root_y-event->click_y;
  location.w=width;
  location.h=height;
  dc.clipChildren(FALSE);
  dc.setFunction(BLT_SRC_XOR_DST);
  dc.setForeground(FXRGB(255,255,255));
  dc.setLineWidth(3);
  dc.drawRectangles(&location,1);
  return 1;
  }


// End drag operation
long FXToolbar::onEndDrag(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  FXDCWindow dc(getRoot());
  FXTRACE((1,"FXToolbar::onEndDrag\n"));
  dc.clipChildren(FALSE);
  dc.setFunction(BLT_SRC_XOR_DST);
  dc.setForeground(FXRGB(255,255,255));
  dc.setLineWidth(3);
  dc.drawRectangles(&location,1);
  dockBar(event->root_x-event->click_x,event->root_y-event->click_y);
  return 1;
  }


// Dragged stuff around
long FXToolbar::onDragged(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  FXRectangle pos=location;
  FXTRACE((1,"FXToolbar::onDragged\n"));
  moveBar(event->root_x-event->click_x,event->root_y-event->click_y);
  if(pos.x!=location.x || pos.y!=location.y || pos.w!=location.w || pos.h!=location.h){
    FXDCWindow dc(getRoot());
    dc.clipChildren(FALSE);
    dc.setFunction(BLT_SRC_XOR_DST);
    dc.setLineWidth(3);
    dc.setForeground(FXRGB(255,255,255));
    dc.drawRectangles(&pos,1);
    dc.drawRectangles(&location,1);
    }
  return 1;
  }


// Moved
long FXToolbar::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent *event=(FXEvent*)ptr;
  if(flags&FLAG_DODRAG){
    handle(this,MKUINT(0,SEL_DRAGGED),ptr);
    return 1;
    }
  if((flags&FLAG_TRYDRAG) && event->moved){
    flags&=~FLAG_TRYDRAG;
    if(handle(this,MKUINT(0,SEL_BEGINDRAG),ptr)){
      flags|=FLAG_DODRAG;
      return 1;
      }
    }
  return 0;
  }


// Pressed LEFT button 
long FXToolbar::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  if(isEnabled()){
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    flags&=~FLAG_UPDATE;
    flags|=FLAG_TRYDRAG;
    }
  return 1;
  }


// Released LEFT button 
long FXToolbar::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  FXuint flg=flags;
  if(isEnabled()){
    ungrab();
    flags|=FLAG_UPDATE;
    flags&=~(FLAG_TRYDRAG|FLAG_DODRAG);
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(flg&FLAG_DODRAG){
      handle(this,MKUINT(0,SEL_ENDDRAG),ptr);
      }
    }
  return 1;
  }


// Save data
void FXToolbar::save(FXStream& store) const {
  FXPacker::save(store);
  }


// Load data
void FXToolbar::load(FXStream& store){ 
  FXPacker::load(store);
  }


// Destroy
FXToolbar::~FXToolbar(){
  drydock=(FXTopWindow*)-1;
  wetdock=(FXTopWindow*)-1;
  }
