/********************************************************************************
*                                                                               *
*                           R e g i s t r y   C l a s s                         *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXRegistry.cpp,v 1.8 2000/03/20 14:29:22 jeroen Exp $                    *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"

/*
  Notes:
  
  - Default directories for registry files where search for the FOX settings
    database is tried:
    
     1 $FOXDIR
     
     2 /etc/foxrc /usr/lib/foxrc /usr/local/lib/foxrc
     
     3 $PATH/foxrc
     
     4 ~/.foxrc
       
  - The latter one is writable, and also the one that overrides all other
    values.
    
  - Search $PATH for registry directory [$PATH/VendorKey/AppKey] or
    [$PATH/AppKey]
    
  - Compile-time #define for registry directories [for global settings].

  - Registry directory is organized as follows:
  
    Desktop             Registry file common to all fox applications, from all vendors
    Vendor/Vendor       Registry file common to all applications from Vendor
    Vendor/Application  Registry file for Application from Vendor
    
    Rationale:  
    
      1)    When installing an application, simply copy ``seed'' registry files
            in the appropriate places; having a subdirectory Vendor prevents clobbering
            other people's registry files, even if their application has the same name.
            
      2)    System-wide registry files are, as a matter of principle, read-only.
      
      3)    System registry files are loaded first, then per-user registry files are
            loaded on top of that.
            
      4)    Registry files loaded later will take precedence over those loaded earlier;
            i.e. key/value pairs in a later file will override a key/value pair with the
            same key loaded earlier.
              
      5)    The exception to the rule is that a key/value pair will not be overridden if
            the value of the key has changed since loading it. In other words, changes 
            will persist.
              
      6)    An application reads files in the order of system: (Desktop, Vendor/Vendor,
            Vendor/App), then user: (Desktop, Vendor/Vendor, Vendor/App).
              
      7)    All changed entries are written to the per-user directory.  An application
            starts by reading the registry files, runs for a while, then at the end writes
            the settings back out to the per-user registry files.
            It will ONLY write those entries which (a) have been changed, or (b) were
            previously read from the same per-user registry files.

  - Format for Registry file:

    [Section Key]
    EntryKey = string-with-no-spaces
    EntryKey = "string\nwith a\nnewline in it\n"
    EntryKey = "string with spaces and \"embedded\" in it"
  
  - EntryKey may is of the form "ali baba", "ali-baba", "ali_baba", or "ali.baba".

  - Leading/trailing spaces are NOT part of the EntryKey.
  
*/

#define MAXBUFFER 2000
#define MAXNAME   200
#define MAXVALUE  2000

#ifndef REGISTRYPATH
#ifndef WIN32
#define REGISTRYPATH   "/etc:/usr/lib:/usr/local/lib"
#else
#define REGISTRYPATH   "\\WINDOWS\\foxrc"
#endif
#endif


#ifndef WIN32
#define FOXRC          "~/.foxrc"
#else
#define FOXRC          "~\\foxrc"
#endif


/*******************************************************************************/

// Make registry object
FXRegistry::FXRegistry(const FXchar* akey,const FXchar* vkey){
  applicationkey=akey;
  vendorkey=vkey;
  modified=FALSE;
#ifndef WIN32
  ascii=TRUE;
#else
  ascii=FALSE;
#endif
  }


// Read registry
FXbool FXRegistry::read(){
  FXchar dirname[MAXPATHLEN+1];
  FXchar *foxregistry,*searchpath;
  register FXbool ok=FALSE;
  
  // Either file based or system registry for WIN32
#ifdef WIN32
  
  if(ascii){

    // File based registry
    FXTRACE((100,"Reading from file based settings database.\n"));

    // Try get location of system-wide registry from environment variable
    foxregistry=getenv("FOXDIR");
    if(foxregistry){
      FXTRACE((100,"Found registry %s in $FOXDIR.\n",foxregistry));
      ok=readFromDir(foxregistry,FALSE);
      }

    // Try the standard places always; this where installation would have put stuff
    if(!ok && fxsearchpath(dirname,REGISTRYPATH,"foxrc")){
      FXTRACE((100,"Found registry %s in default location.\n",dirname));
      ok=readFromDir(dirname,FALSE);
      }

    // Try search along path if still not found
    if(!ok){
      searchpath=getenv("PATH");
      if(searchpath){
        if(fxsearchpath(dirname,searchpath,"foxrc")){
          FXTRACE((100,"Found registry %s in $PATH.\n",dirname));
          ok=readFromDir(dirname,FALSE);
          }
        }
      }
  
    // Get path to per-user settings directory
    fxexpand(dirname,FOXRC);
    
    // Then read per-user settings; overriding system-wide ones
    if(readFromDir(dirname,TRUE)) ok=TRUE;
    }
  
  else{
    
    // System registry
    FXTRACE((100,"Reading from system registry database.\n"));

    // Load system-wide resources first
    if(readFromRegistry(HKEY_LOCAL_MACHINE,FALSE)) ok=TRUE;

    // Now any modified resources for current user
    if(readFromRegistry(HKEY_CURRENT_USER,TRUE)) ok=TRUE;
    }
  
  // File based registry for UNIX
#else

  // Try get location of system-wide registry from environment variable
  foxregistry=getenv("FOXDIR");
  if(foxregistry){
    FXTRACE((100,"Found registry %s in $FOXDIR.\n",foxregistry));
    ok=readFromDir(foxregistry,FALSE);
    }
  
  // Try the standard places always; this where installation would have put stuff
  if(!ok && fxsearchpath(dirname,REGISTRYPATH,"foxrc")){
    FXTRACE((100,"Found registry %s in default location.\n",dirname));
    ok=readFromDir(dirname,FALSE);
    }

  // Try search along path if still not found
  if(!ok){
    searchpath=getenv("PATH");
    if(searchpath){
      if(fxsearchpath(dirname,searchpath,"foxrc")){
        FXTRACE((100,"Found registry %s in $PATH.\n",dirname));
        ok=readFromDir(dirname,FALSE);
        }
      }
    }   
    
  // Get path to per-user settings directory
  fxexpand(dirname,FOXRC);

  // Then read per-user settings; overriding system-wide ones
  if(readFromDir(dirname,TRUE)) ok=TRUE;
  
#endif
  
  return ok;
  }


// Try read registry from directory
FXbool FXRegistry::readFromDir(const FXchar* dirname,FXbool mark){
  FXchar pathname[MAXPATHLEN+1];
  FXbool ok=FALSE;
  
  // First try to load desktop registry
  strcpy(pathname,dirname);
  strcat(pathname,PATHSEPSTRING);
  strcat(pathname,"Desktop");
#ifdef WIN32
  strcat(pathname,".ini");
#endif
  if(parseFile(pathname,FALSE)) ok=TRUE;
  
  // Then try to load vendor-wide registry
  if(vendorkey){
    strcpy(pathname,dirname);
    strcat(pathname,PATHSEPSTRING);
    strcat(pathname,vendorkey);
    strcat(pathname,PATHSEPSTRING);
    strcat(pathname,vendorkey);
#ifdef WIN32
    strcat(pathname,".ini");
#endif
    if(parseFile(pathname,FALSE)) ok=TRUE;
    }
  
  // Then try to load application registry
  if(applicationkey){
    strcpy(pathname,dirname);
    if(vendorkey){
      strcat(pathname,PATHSEPSTRING);
      strcat(pathname,vendorkey);
      }
    strcat(pathname,PATHSEPSTRING);
    strcat(pathname,applicationkey);
#ifdef WIN32
    strcat(pathname,".ini");
#endif
    if(parseFile(pathname,mark)) ok=TRUE;
    }
  return ok; 
  }
  

#ifdef WIN32

// Read from Windows Registry
FXbool FXRegistry::readFromRegistry(void* hRootKey,FXbool mark){
  FXbool ok=FALSE;
  HKEY hSoftKey,hDesktopKey,hOrgKey,hAppKey;
  DWORD dwDisposition;
  FXchar name[MAXNAME];
  FILETIME ftLastWriteTime;
  FXStringDict *group=NULL;
  
  if(RegOpenKeyEx((HKEY)hRootKey,"Software",0,KEY_READ,&hSoftKey)==ERROR_SUCCESS){
    
    // Load FOX-wide settings from Software\Desktop
    if(RegCreateKeyEx(hSoftKey,"Desktop",0,REG_NONE,REG_OPTION_NON_VOLATILE,KEY_WRITE|KEY_READ,NULL,&hDesktopKey,&dwDisposition)==ERROR_SUCCESS){
      DWORD dwSectIndex=0;
      DWORD cbNameSize=sizeof(name);
      while(RegEnumKeyEx(hDesktopKey,dwSectIndex,name,&cbNameSize,NULL,NULL,NULL,&ftLastWriteTime)==ERROR_SUCCESS){
        group=sections.insert(name);
        readRegistrySection(hDesktopKey,group,name,mark);
        cbNameSize=sizeof(name);
        dwSectIndex++;
        }
      RegCloseKey(hDesktopKey);
      ok=TRUE;
      }

    // Load vendor-wide settings from Software\VendorKey\VendorKey
    if(RegCreateKeyEx(hSoftKey,vendorkey,0,REG_NONE,REG_OPTION_NON_VOLATILE,KEY_WRITE|KEY_READ,NULL,&hOrgKey,&dwDisposition)==ERROR_SUCCESS){
      if(RegCreateKeyEx(hOrgKey,vendorkey,0,REG_NONE,REG_OPTION_NON_VOLATILE,KEY_WRITE|KEY_READ,NULL,&hAppKey,&dwDisposition)==ERROR_SUCCESS){
	DWORD dwSectIndex=0;
	DWORD cbNameSize=sizeof(name);
	while(RegEnumKeyEx(hAppKey,dwSectIndex,name,&cbNameSize,NULL,NULL,NULL,&ftLastWriteTime)==ERROR_SUCCESS){
	  group=sections.insert(name);
	  readRegistrySection(hAppKey,group,name,mark);
	  cbNameSize=sizeof(name);
	  dwSectIndex++;
	  }
	RegCloseKey(hAppKey);
	ok=TRUE;
	}

      // Now load app-specific settings from Software\VendorKey\AppKey
      if(RegCreateKeyEx(hOrgKey,applicationkey,0,REG_NONE,REG_OPTION_NON_VOLATILE,KEY_WRITE|KEY_READ,NULL,&hAppKey,&dwDisposition)==ERROR_SUCCESS){
	DWORD dwSectIndex=0;
	DWORD cbNameSize=sizeof(name);
	while(RegEnumKeyEx(hAppKey,dwSectIndex,name,&cbNameSize,NULL,NULL,NULL,&ftLastWriteTime)==ERROR_SUCCESS){
	  group=sections.insert(name);
	  readRegistrySection(hAppKey,group,name,mark);
	  cbNameSize=sizeof(name);
	  dwSectIndex++;
	  }
	RegCloseKey(hAppKey);
	ok=TRUE;
	}
      RegCloseKey(hOrgKey);
      ok=TRUE;
      }
    RegCloseKey(hSoftKey);
    }
  return ok;
  }


// Read all values for a particular section
FXbool FXRegistry::readRegistrySection(void* hAppKey,FXStringDict*& group,const char *groupname,FXbool mark){
  FXASSERT(group);
  FXASSERT(groupname);
  HKEY hSectKey=NULL;
  FXbool ok=FALSE;
  if(RegOpenKeyEx((HKEY)hAppKey,groupname,0,KEY_READ,&hSectKey)==ERROR_SUCCESS){
    DWORD dwValueIndex=0;
    DWORD dwType;
    FXchar name[MAXNAME];
    DWORD cbNameSize=sizeof(name);
    FXchar value[MAXVALUE];
    BYTE valuebytes[MAXVALUE];
    DWORD cbDataSize=sizeof(valuebytes);
    while(RegEnumValue(hSectKey,dwValueIndex,name,&cbNameSize,NULL,&dwType,valuebytes,&cbDataSize)!=ERROR_NO_MORE_ITEMS){
      FXASSERT(dwType==REG_SZ);
      for(FXuint i=0; i<cbDataSize; i++) value[i]=valuebytes[i];
      group->replace(name,value,mark);
      cbNameSize=sizeof(name);
      cbDataSize=sizeof(value);
      dwValueIndex++;
      }
    RegCloseKey(hSectKey);
    ok=TRUE;
    }
  return ok;
  }


#endif


// Parse filename
FXbool FXRegistry::parseFile(const FXchar* filename,FXbool mark){
  FXchar buffer[MAXBUFFER];
  FXStringDict *group=NULL;
  FXint lineno=1;
  FXint numerrors=0;
  FILE *file;
  file=fopen(filename,"r");
  if(!file) return FALSE; 
  FXTRACE((100,"Reading registry file: %s\n",filename));
  while(fgets(buffer,MAXBUFFER,file)!=NULL){
    if(!parseBuffer(group,buffer,filename,lineno,mark)) numerrors++;
    if(numerrors>10){
      fxwarning("%s:%d: too many errors; parsing terminated.\n",filename,lineno);
      fclose(file);
      return FALSE;
      }
    lineno++;
    }
  fclose(file);
  return TRUE;
  }


// Parse buffer
FXbool FXRegistry::parseBuffer(FXStringDict*& group,const FXchar* buffer,const FXchar *filename,FXint lineno,FXbool mark){
  register const FXchar *ptr=buffer;
  register FXchar c;
  register FXint len;
  FXchar name[MAXNAME];
  FXchar value[MAXVALUE];
    
  // Skip leading spaces
  while(*ptr && isspace(*ptr)) ptr++;
  
  // Test for comments
  if(*ptr=='#' || *ptr==';' || *ptr=='\0') return TRUE;
    
  // Parse section name
  if(*ptr=='['){
    ptr++;
    len=0;
    while((c=*ptr)!='\0' && c!=']'){
      if(!isalnum(c) && c!='_' && c!='-' && c!='.' && c!=' '){
        fxwarning("%s:%d: illegal section name.\n",filename,lineno);
        return FALSE;
        }
      if(len>=MAXNAME){
        fxwarning("%s:%d: section name to long.\n",filename,lineno);
        return FALSE;
        }
      name[len]=c;
      len++;
      ptr++;
      }
    name[len]='\0';
    
    // Add new section dict
    group=sections.insert(name);
    }
  
  // Parse key name
  else{
    
    // Should have a group
    if(!group){
      fxwarning("%s:%d: registry entry should follow a section.\n",filename,lineno);
      return FALSE;
      }
    
    // Transfer key, checking validity
    len=0;
    while((c=*ptr)!='\0' && c!='='){
      if(!isalnum(c) && c!='_' && c!='-' && c!='.' && c!=' '){
        fxwarning("%s:%d: illegal key name.\n",filename,lineno);
        }
      if(len>=MAXNAME){
        fxwarning("%s:%d: key name to long.\n",filename,lineno);
        return FALSE;
        }
      name[len]=c;
      len++;
      ptr++;
      }
    
    // Remove trailing spaces from key
    while(len && name[len-1]==' ') len--;
    name[len]='\0';

    // Should be a '='
    if(*ptr++!='='){
      fxwarning("%s:%d: expected '=' to follow key.\n",filename,lineno);
      return FALSE;
      }
    
    // Skip more spaces
    while(*ptr && isspace(*ptr)) ptr++;
    
    // Parse value
    if(!parseValue(value,ptr,filename,lineno)) return FALSE;
    
    // Add entry to current section
    group->replace(name,value,mark);
    }
  return TRUE;
  }


// Parse value
FXbool FXRegistry::parseValue(FXchar* value,const FXchar* buffer,const FXchar *filename,FXint lineno){
  register const FXchar *ptr=buffer;
  register FXchar *out=value;
  unsigned int v1,v2,h,l;
  
  // Was quoted string; copy verbatim
  if(*ptr=='"'){
    ptr++;
    while(*ptr){
      switch(*ptr){
        case '\\':
          ptr++;
          switch(*ptr){
            case 'n': 
              *out++='\n'; 
              break;
            case 'r': 
              *out++='\r'; 
              break;
            case 'b': 
              *out++='\b'; 
              break;
            case 'v': 
              *out++='\v'; 
              break;
            case 'a': 
              *out++='\a'; 
              break;
            case 'f': 
              *out++='\f'; 
              break;
            case 't': 
              *out++='\t'; 
              break;
            case '\\': 
              *out++='\\'; 
              break;
            case '"': 
              *out++='"'; 
              break;
            case 'x': 
              ptr++; 
              v1=*ptr++; 
              v2=*ptr;
              h=v1<='9'?v1-'0':toupper(v1)-'A'+10; 
              l=v2<='9'?v2-'0':toupper(v2)-'A'+10; 
              *out++=(h<<4)+l; 
              break;
            default: 
              *out++=*ptr; 
              break;
            }
          break;
        case '"':
          *out='\0';
          return TRUE;
        default:
          *out++=*ptr;
          break;
        }
      ptr++;
      }
    *value='\0';
    fxwarning("%s:%d: unterminated string constant.\n",filename,lineno);
    return FALSE;
    }
  
  // Non-quoted string copy sequence of non-white space
  else{
    while(*ptr && !isspace(*ptr) && isprint(*ptr)){
      *out++=*ptr++;
      }
    *out='\0';
    }
  return TRUE;
  }


// Write registry
FXbool FXRegistry::write(){
  FXchar pathname[MAXPATHLEN+1];
  FXchar tempname[MAXPATHLEN+1];
  
  // Settings have not changed
  if(!modified) return TRUE;
  
  // We can not save if no application key given
  if(applicationkey){
    
  // Either file based or system registry for WIN32
#ifdef WIN32

    if(ascii){

      // Changes written only in the per-user registry
      fxexpand(pathname,FOXRC);

      // If this directory does not exist, make it
      if(!fxexists(pathname)){
	if(fxmkdir(pathname,0777)!=0){
	  fxwarning("%s: unable to create directory.\n",pathname);
	  return FALSE;
	  }
	}
      else{
	if(!fxisdir(pathname)){
	  fxwarning("%s: is not a directory.\n",pathname);
	  return FALSE;
	  }
	}

      // Add vendor subdirectory
      if(vendorkey){
	strcat(pathname,PATHSEPSTRING);
	strcat(pathname,vendorkey);
	if(!fxexists(pathname)){
	  if(fxmkdir(pathname,0777)!=0){
	    fxwarning("%s: unable to create directory.\n",pathname);
	    return FALSE;
	    }
	  }
	else{
	  if(!fxisdir(pathname)){
            fxwarning("%s: is not a directory.\n",pathname);
	    return FALSE;
	    }
	  }
	}

      // Add application key
      strcat(pathname,PATHSEPSTRING);
      strcat(pathname,applicationkey);

      // Construct temp name
      sprintf(tempname,"%s_%d",pathname,fxgetpid());

      // Unparse settings into temp file first
      if(unparseFile(tempname)){

	// Rename ATOMICALLY to proper name
	if(rename(tempname,pathname)!=0){
	  fxwarning("Unable to save registry.\n");
	  return FALSE;
	  }

	modified=FALSE;
	return TRUE;
	}
      }

    else{
      if(writeToRegistry(HKEY_CURRENT_USER)) return TRUE;
      }
    
    // File based registry for X11
#else

    // Changes written only in the per-user registry
    fxexpand(pathname,FOXRC);

    // If this directory does not exist, make it
    if(!fxexists(pathname)){
      if(fxmkdir(pathname,0777)!=0){
	fxwarning("%s: unable to create directory.\n",pathname);
	return FALSE;
	}
      }
    else{
      if(!fxisdir(pathname)){
	fxwarning("%s: is not a directory.\n",pathname);
	return FALSE;
	}
      }

    // Add vendor subdirectory
    if(vendorkey){
      strcat(pathname,PATHSEPSTRING);
      strcat(pathname,vendorkey);
      if(!fxexists(pathname)){
	if(fxmkdir(pathname,0777)!=0){
	  fxwarning("%s: unable to create directory.\n",pathname);
	  return FALSE;
	  }
	}
      else{
	if(!fxisdir(pathname)){
          fxwarning("%s: is not a directory.\n",pathname);
	  return FALSE;
	  }
	}
      }

    // Add application key
    strcat(pathname,PATHSEPSTRING);
    strcat(pathname,applicationkey);

    // Construct temp name
    sprintf(tempname,"%s_%d",pathname,fxgetpid());

    // Unparse settings into temp file first
    if(unparseFile(tempname)){

      // Rename ATOMICALLY to proper name
      if(rename(tempname,pathname)!=0){
	fxwarning("Unable to save registry.\n");
	return FALSE;
	}
      modified=FALSE;
      return TRUE;
      }

#endif

    }
  return FALSE;
  }



#ifdef WIN32

// Update current user's settings
FXbool FXRegistry::writeToRegistry(void* hRootKey){
  FXbool ok=FALSE;
  HKEY hSoftKey;
  if(RegOpenKeyEx((HKEY)hRootKey,"Software",0,KEY_WRITE,&hSoftKey)==ERROR_SUCCESS){
    
    // Add vendor key
    HKEY hVendorKey;
    DWORD dwDisposition;
    if(RegCreateKeyEx(hSoftKey,vendorkey,0,REG_NONE,REG_OPTION_NON_VOLATILE,KEY_WRITE|KEY_READ,NULL,&hVendorKey,&dwDisposition)==ERROR_SUCCESS){

      // Add application key
      HKEY hAppKey;
      if(RegCreateKeyEx(hVendorKey,applicationkey,0,REG_NONE,REG_OPTION_NON_VOLATILE,KEY_WRITE|KEY_READ,NULL,&hAppKey,&dwDisposition)==ERROR_SUCCESS){

	// First, purge all existing sections (keys)
	DWORD dwSectIndex=0;
	FXchar name[MAXNAME];
	DWORD cbNameSize=sizeof(name);
	FILETIME ftLastWriteTime;
	while(RegEnumKeyEx(hAppKey,dwSectIndex,name,&cbNameSize,NULL,NULL,NULL,&ftLastWriteTime)==ERROR_SUCCESS){
	  LONG lResult=RegDeleteKey(hAppKey,name);
	  FXASSERT(lResult==ERROR_SUCCESS);
	  cbNameSize=sizeof(name);
	  dwSectIndex++;
	  }

	// Dump the registry
	FXint s=sections.first();
	while(s<sections.size()){
	  HKEY hSectKey=NULL;
	  FXbool sec=FALSE;
	  FXStringDict *group=sections.data(s);
	  FXASSERT(group);
	  FXint e=group->first();
	  while(e<group->size()){
	    FXbool mrk=group->mark(e);
	    if(mrk && !sec){
	      FXASSERT(sections.key(s));
	      if(RegCreateKeyEx(hAppKey,sections.key(s),0,REG_NONE,REG_OPTION_NON_VOLATILE,KEY_WRITE|KEY_READ,NULL,&hSectKey,&dwDisposition)==ERROR_SUCCESS){
		sec=TRUE;
		}
	      }
	    if(mrk){
	      FXASSERT(group->key(e));
	      FXASSERT(group->data(e));
	      BYTE bytes[MAXVALUE];
	      for(FXuint i=0; i<strlen(group->data(e)); i++) bytes[i]=group->data(e)[i];
	      bytes[strlen(group->data(e))]=0;
	      if(RegSetValueEx(hSectKey,group->key(e),0,REG_SZ,bytes,strlen(group->data(e))+1)==ERROR_SUCCESS){
		}
	      }
	    e=group->next(e);
	    }

	  // Close this section's key (if it exists)
	  if(hSectKey)
	    RegCloseKey(hSectKey);

	  // Process next registry section
	  s=sections.next(s);
	  }

	// Done with application key
	ok=TRUE;
	RegCloseKey(hAppKey);
	}
      
      // Done with vendor key
      RegCloseKey(hVendorKey);
      }

    // Done with Software key
    RegCloseKey(hSoftKey);
    }
  return ok;
  }

#endif


// Unparse registry file
FXbool FXRegistry::unparseFile(const FXchar* filename){
  FXchar buffer[MAXVALUE];
  FXStringDict *group;
  FILE *file;
  FXint s,e;
  FXbool sec,mrk;
  file=fopen(filename,"w");
  if(!file){ fxwarning("%s: unable to write file.\n",filename); return FALSE; }
  FXTRACE((100,"Writing registry file: %s\n",filename));
  s=sections.first();
  while(s<sections.size()){
    sec=FALSE;
    group=sections.data(s);
    FXASSERT(group);
    e=group->first();
    while(e<group->size()){
      mrk=group->mark(e);
      if(mrk && !sec){
        FXASSERT(sections.key(s));
        fputc('[',file);
        fputs(sections.key(s),file);
        fputc(']',file);
        fputc('\n',file);
        sec=TRUE;
        }
      if(mrk){
        FXASSERT(group->key(e));
        FXASSERT(group->data(e));
        fputs(group->key(e),file);
        fputc(' ',file);
        fputc('=',file);
        fputc(' ',file);
        if(unparseValue(buffer,group->data(e))){
          fputc('"',file);
          fputs(buffer,file);
          fputc('"',file);
          }
        else{
          fputs(buffer,file);
          }
        fputc('\n',file);
        }
      e=group->next(e);
      }
    if(sec){
      fputc('\n',file);
      }
    s=sections.next(s);
    }
  fclose(file);
  return TRUE;
  }


// Unparse value by quoting strings; return TRUE if quote needed
FXbool FXRegistry::unparseValue(FXchar* buffer,const FXchar* value){
  const FXchar hex[]="0123456789ABCDEF";
  FXbool mustquote=FALSE;
  FXuint v;
  FXASSERT(value);
  while(*value){
    switch(*value){
      case '\n': 
        *buffer++='\\'; 
        *buffer++='n'; 
        mustquote=TRUE; 
        break;
      case '\r': 
        *buffer++='\\'; 
        *buffer++='r'; 
        mustquote=TRUE; 
        break;
      case '\b': 
        *buffer++='\\'; 
        *buffer++='b'; 
        mustquote=TRUE; 
        break;
      case '\v': 
        *buffer++='\\'; 
        *buffer++='v'; 
        mustquote=TRUE; 
        break;
      case '\a': 
        *buffer++='\\'; 
        *buffer++='a'; 
        mustquote=TRUE; 
        break;
      case '\f': 
        *buffer++='\\'; 
        *buffer++='f'; 
        mustquote=TRUE; 
        break;
      case '\t': 
        *buffer++='\\'; 
        *buffer++='t'; 
        mustquote=TRUE; 
        break;
      case '\\': 
        *buffer++='\\'; 
        *buffer++='\\';
        mustquote=TRUE; 
        break;
      case '"':  
        *buffer++='\\'; 
        *buffer++='"'; 
        mustquote=TRUE; 
        break;
      case ' ':  
        *buffer++=' '; 
        mustquote=TRUE; 
        break;
      default: 
        v=*value;
        if(v<0x20 || 0x7f<v){ 
          *buffer++='\\'; 
          *buffer++='x';
          *buffer++=hex[((v>>4)&15)];
          *buffer++=hex[v&15];
          mustquote=TRUE;
          }
        else{
          *buffer++=v;
          }
        break;
      }
    value++;
    }
  *buffer='\0';
  return mustquote;
  }


// Read a string-valued registry entry
const FXchar *FXRegistry::readStringEntry(const FXchar *section,const FXchar *key,const FXchar *def){
  if(!section){ fxerror("FXRegistry::readStringEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::readStringEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.find(section);
  if(group){
    const char *value=group->find(key);
    if(value) return value;
    }
  return def;
  }


// Read a int-valued registry entry
FXint FXRegistry::readIntEntry(const FXchar *section,const FXchar *key,FXint def){
  if(!section){ fxerror("FXRegistry::readIntEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::readIntEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.find(section);
  if(group){
    const char *value=group->find(key);
    if(value){
      FXint ivalue;
      if(value[0]=='0' && (value[1]=='x' || value[1]=='X')){
        if(sscanf(value+2,"%x",&ivalue)) return ivalue;
        }
      else{
        if(sscanf(value,"%d",&ivalue)==1) return ivalue;
        }
      }
    }
  return def;
  }


// Read a unsigned int-valued registry entry
FXuint FXRegistry::readUnsignedEntry(const FXchar *section,const FXchar *key,FXuint def){
  if(!section){ fxerror("FXRegistry::readUnsignedEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::readUnsignedEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.find(section);
  if(group){
    const char *value=group->find(key);
    if(value){
      FXuint ivalue;
      if(value[0]=='0' && (value[1]=='x' || value[1]=='X')){
        if(sscanf(value+2,"%x",&ivalue)) return ivalue;
        }
      else{
        if(sscanf(value,"%u",&ivalue)==1) return ivalue;
        }
      }
    }
  return def;
  }


// Read a double-valued registry entry
FXdouble FXRegistry::readRealEntry(const FXchar *section,const FXchar *key,FXdouble def){
  if(!section){ fxerror("FXRegistry::readRealEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::readRealEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.find(section);
  if(group){
    const char *value=group->find(key);
    if(value){
      FXdouble dvalue;
      if(sscanf(value,"%lf",&dvalue)==1) return dvalue;
      }
    }
  return def;
  }


// Read a color registry entry
FXColor FXRegistry::readColorEntry(const FXchar *section,const FXchar *key,FXColor def){
  if(!section){ fxerror("FXRegistry::readColorEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::readColorEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.find(section);
  if(group){
    const char *value=group->find(key);
    if(value){
      return fxcolorfromname(value);
      }
    }
  return def;
  }


// Write a string-valued registry entry
FXbool FXRegistry::writeStringEntry(const FXchar *section,const FXchar *key,const FXchar *val){
  if(!section){ fxerror("FXRegistry::writeStringEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::writeStringEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.insert(section);
  if(group){
    group->replace(key,val,TRUE);
    modified=TRUE;
    return TRUE;
    }
  return FALSE;
  }


// Write a int-valued registry entry
FXbool FXRegistry::writeIntEntry(const FXchar *section,const FXchar *key,FXint val){
  if(!section){ fxerror("FXRegistry::writeIntEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::writeIntEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.insert(section);
  if(group){
    FXchar buffer[10];
    sprintf(buffer,"%d",val);
    group->replace(key,buffer,TRUE);
    modified=TRUE;
    return TRUE;
    }
  return FALSE;
  }


// Write a unsigned int-valued registry entry
FXbool FXRegistry::writeUnsignedEntry(const FXchar *section,const FXchar *key,FXuint val){
  if(!section){ fxerror("FXRegistry::writeUnsignedEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::writeUnsignedEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.insert(section);
  if(group){
    FXchar buffer[10];
    sprintf(buffer,"%u",val);
    group->replace(key,buffer,TRUE);
    modified=TRUE;
    return TRUE;
    }
  return FALSE;
  }


// Write a double-valued registry entry
FXbool FXRegistry::writeRealEntry(const FXchar *section,const FXchar *key,FXdouble val){
  if(!section){ fxerror("FXRegistry::writeRealEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::writeRealEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.insert(section);
  if(group){
    FXchar buffer[60];
    sprintf(buffer,"%.16g",val);
    group->replace(key,buffer,TRUE);
    modified=TRUE;
    return TRUE;
    }
  return FALSE;
  }


// Write a color registry entry
FXbool FXRegistry::writeColorEntry(const FXchar *section,const FXchar *key,FXColor val){
  if(!section){ fxerror("FXRegistry::writeColorEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::writeColorEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.insert(section);
  if(group){
    FXchar buffer[60];
    group->replace(key,fxnamefromcolor(buffer,val),TRUE);
    modified=TRUE;
    return TRUE;
    }
  return FALSE;
  }


// Delete a registry entry
FXbool FXRegistry::deleteEntry(const FXchar *section,const FXchar *key){
  if(!section){ fxerror("FXRegistry::deleteEntry: NULL section argument.\n"); }
  if(!key){ fxerror("FXRegistry::deleteEntry: NULL key argument.\n"); }
  FXStringDict *group=sections.insert(section);
  if(group){
    group->remove(key);
    modified=TRUE;
    return TRUE;
    }
  return FALSE;
  }


// Delete section
FXbool FXRegistry::deleteSection(const FXchar *section){
  if(!section){ fxerror("FXRegistry::deleteSection: NULL section argument.\n"); }
  sections.remove(section);
  modified=TRUE;
  return TRUE;
  }


// Delete registry object
FXRegistry::~FXRegistry(){
  }
