/********************************************************************************
*                                                                               *
*                  R a d i o   B u t t o n    O b j e c t                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXRadioButton.cpp,v 1.13 2000/03/02 00:14:19 jeroen Exp $                *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXRadioButton.h"

/*
  To do:
  - Need check-style also (stay in when pressed, pop out when unpressed).
  - Who owns the icon(s)?
  - Arrow buttons should auto-repeat with a timer of some kind
  - "&Label\tTooltip\tHelptext\thttp://server/application/helponitem.html"
  - CheckButton should send SEL_COMMAND.
  - Default button mode:- should somehow get focus.
  - Add button multiple-click translations elsewhere
*/


#define RADIOBUTTON_MASK  (RADIOBUTTON_AUTOGRAY|RADIOBUTTON_AUTOHIDE)

/*******************************************************************************/

// Map
FXDEFMAP(FXRadioButton) FXRadioButtonMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXRadioButton::onPaint),
  FXMAPFUNC(SEL_UPDATE,0,FXRadioButton::onUpdate),
  FXMAPFUNC(SEL_ENTER,0,FXRadioButton::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXRadioButton::onLeave),
  FXMAPFUNC(SEL_FOCUSIN,0,FXRadioButton::onFocusIn),
  FXMAPFUNC(SEL_FOCUSOUT,0,FXRadioButton::onFocusOut),
  FXMAPFUNC(SEL_ACTIVATE,0,FXRadioButton::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXRadioButton::onDeactivate),
  FXMAPFUNC(SEL_KEYPRESS,FXWindow::ID_HOTKEY,FXRadioButton::onHotKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,FXWindow::ID_HOTKEY,FXRadioButton::onHotKeyRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXRadioButton::onUngrabbed),
  FXMAPFUNC(SEL_UNCHECK_RADIO,0,FXRadioButton::onUncheckRadio),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_CHECK,FXRadioButton::onCheck),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_UNCHECK,FXRadioButton::onUncheck),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_UNKNOWN,FXRadioButton::onUnknown),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETVALUE,FXRadioButton::onCmdSetValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_SETINTVALUE,FXRadioButton::onCmdSetIntValue),
  FXMAPFUNC(SEL_COMMAND,FXWindow::ID_GETINTVALUE,FXRadioButton::onCmdGetIntValue),
  };


// Object implementation
FXIMPLEMENT(FXRadioButton,FXLabel,FXRadioButtonMap,ARRAYNUMBER(FXRadioButtonMap))


// Deserialization
FXRadioButton::FXRadioButton(){
  check=FALSE;
  oldcheck=FALSE;
  radioColor=0;
  }


// Make a check button
FXRadioButton::FXRadioButton(FXComposite* p,const FXString& text,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXLabel(p,text,NULL,opts,x,y,w,h,pl,pr,pt,pb){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  check=FALSE;
  oldcheck=FALSE;
  radioColor=getApp()->backColor;
  }


// If window can have focus
FXbool FXRadioButton::canFocus() const { return 1; }


// Enable the window
void FXRadioButton::enable(){
  if(!(flags&FLAG_ENABLED)){
    FXWindow::enable();
    update();
    }
  }


// Disable the window
void FXRadioButton::disable(){
  if(flags&FLAG_ENABLED){
    FXWindow::disable();
    update();
    }
  }


// Get default width
FXint FXRadioButton::getDefaultWidth(){
  FXint tw=0,s=0,w;
  if(!label.empty()){ 
    tw=labelWidth(label);
    s=4;
    }
  if(!(options&(ICON_AFTER_TEXT|ICON_BEFORE_TEXT))) w=FXMAX(tw,13); else w=tw+13+s;
  return padleft+padright+w+(border<<1);
  }


// Get default height
FXint FXRadioButton::getDefaultHeight(){
  FXint th=0,h;
  if(!label.empty()){ 
    th=labelHeight(label);
    }
  if(!(options&(ICON_ABOVE_TEXT|ICON_BELOW_TEXT))) h=FXMAX(th,13); else h=th+13;
  return padtop+padbottom+h+(border<<1);
  }


// Check button
void FXRadioButton::setCheck(FXuint s){
  if(check!=s){
    check=s;
    update();
    }
  }


// Change state to checked
long FXRadioButton::onCheck(FXObject*,FXSelector,void*){ 
  setCheck(TRUE); 
  return 1; 
  }


// Change state to unchecked
long FXRadioButton::onUncheck(FXObject*,FXSelector,void*){ 
  setCheck(FALSE); 
  return 1; 
  }


// Change state to indeterminate
long FXRadioButton::onUnknown(FXObject*,FXSelector,void*){ 
  setCheck(MAYBE); 
  return 1; 
  }


// Update value from a message
long FXRadioButton::onCmdSetValue(FXObject*,FXSelector,void* ptr){
  setCheck((FXint)(long)ptr);
  return 1;
  }


// Update value from a message
long FXRadioButton::onCmdSetIntValue(FXObject*,FXSelector,void* ptr){
  if(ptr==NULL){ fxerror("%s::onCmdSetIntValue: NULL pointer.\n",getClassName()); }
  setCheck(*((FXint*)ptr));
  return 1;
  }


// Obtain value from text field
long FXRadioButton::onCmdGetIntValue(FXObject*,FXSelector,void* ptr){
  if(ptr==NULL){ fxerror("%s::onCmdGetIntValue: NULL pointer.\n",getClassName()); }
  *((FXint*)ptr)=getCheck();
  return 1;
  }


// Uncheck radio button, sent from parent
long FXRadioButton::onUncheckRadio(FXObject*,FXSelector,void*){
  if(getCheck()){
    setCheck(FALSE);
    if(target && target->handle(this,MKUINT(message,SEL_COMMAND),(void*)check)) return 1;
    }
  return 0;
  }


// Gained focus
long FXRadioButton::onFocusIn(FXObject* sender,FXSelector sel,void* ptr){
  FXLabel::onFocusIn(sender,sel,ptr);
  update(border,border,width-(border<<1),height-(border<<1));
  return 1;
  }

  
// Lost focus
long FXRadioButton::onFocusOut(FXObject* sender,FXSelector sel,void* ptr){
  FXLabel::onFocusOut(sender,sel,ptr);
  update(border,border,width-(border<<1),height-(border<<1));
  return 1;
  }


// Hot key combination pressed
long FXRadioButton::onHotKeyPress(FXObject*,FXSelector,void* ptr){
  FXTRACE((200,"%s::onHotKeyPress %08x\n",getClassName(),this));
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    handle(this,MKUINT(0,SEL_ACTIVATE),ptr);
    }
  return 1;
  }


// Hot key combination released
long FXRadioButton::onHotKeyRelease(FXObject*,FXSelector,void* ptr){
  FXTRACE((200,"%s::onHotKeyRelease %08x\n",getClassName(),this));
  flags&=~FLAG_TIP;
  if(isEnabled()){ 
    handle(this,MKUINT(0,SEL_DEACTIVATE),ptr); 
    }
  return 1;
  }


// Handle repaint 
long FXRadioButton::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXint tw=0,th=0,tx,ty,ix,iy;

  FXDCWindow dc(this,ev);
  
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  
  if(!label.empty()){
    tw=labelWidth(label);
    th=labelHeight(label);
    }

  just_x(tx,ix,tw,13);
  just_y(ty,iy,th,13);

  if(!isEnabled())                   // fix by Daniel Gehriger (gehriger@linkcad.com)
    dc.setForeground(baseColor); 
  else
    dc.setForeground(radioColor);

  dc.fillArc(ix+1,iy+1,11,11,0,23040);

  dc.setForeground(shadowColor);
  dc.drawArc(ix,iy,13,13,45*64,180*64);

  dc.setForeground(hiliteColor);
  dc.drawArc(ix,iy,13,13,225*64,180*64);

  dc.setForeground(borderColor);
  dc.drawArc(ix+1,iy+1,11,11,45*64,180*64);

  dc.setForeground(baseColor);
  dc.drawArc(ix+1,iy+1,11,11,225*64,180*64);

  if(check!=FALSE){
    if(isEnabled())
      dc.setForeground(textColor);
    else
      dc.setForeground(shadowColor);
    dc.fillArc(ix+3,iy+3,7,7,0,23040);
    }

  if(!label.empty()){
    dc.setTextFont(font);
    if(isEnabled()){
      dc.setForeground(textColor);
      drawLabel(dc,label,hotoff,tx,ty,tw,th);
      if(hasFocus()){
        drawFocusRectangle(dc,tx-1,ty-1,tw+2,th+2);
        }
      }
    else{
      dc.setForeground(hiliteColor);
      drawLabel(dc,label,hotoff,tx+1,ty+1,tw,th);
      dc.setForeground(shadowColor);
      drawLabel(dc,label,hotoff,tx,ty,tw,th);
      }
    }
  
  drawFrame(dc,0,0,width,height);

  return 1;
  }


// Implement auto-hide or auto-gray modes
long FXRadioButton::onUpdate(FXObject* sender,FXSelector sel,void* ptr){
  if(!FXLabel::onUpdate(sender,sel,ptr)){
    if(options&RADIOBUTTON_AUTOHIDE){if(shown()){hide();recalc();}}
    if(options&RADIOBUTTON_AUTOGRAY){disable();}
    }
  return 1;
  }


// The widget lost the grab for some reason
long FXRadioButton::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXLabel::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags|=FLAG_UPDATE;
  setCheck(oldcheck);
  return 1;
  }


// Button being pressed
long FXRadioButton::onActivate(FXObject*,FXSelector,void*){
  flags|=FLAG_PRESSED;
  flags&=~FLAG_UPDATE;
  oldcheck=check;
  setCheck(TRUE);
  return 1;
  }


// Button being released
long FXRadioButton::onDeactivate(FXObject*,FXSelector,void*){
  flags&=~FLAG_PRESSED;
  flags|=FLAG_UPDATE;
  if(oldcheck!=check){ 
    getParent()->handle(this,MKUINT(0,SEL_UNCHECK_OTHER),NULL); 
    if(target && target->handle(this,MKUINT(message,SEL_COMMAND),(void*)check)) return 1;
    }
  return 1;
  }


// Entered button
long FXRadioButton::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXLabel::onEnter(sender,sel,ptr);
  if(isEnabled()){
    if(flags&FLAG_PRESSED) setCheck(TRUE);
    }
  return 1;
  }


// Left button
long FXRadioButton::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXLabel::onLeave(sender,sel,ptr);
  if(isEnabled()){
    if(flags&FLAG_PRESSED) setCheck(oldcheck);
    }
  return 1;
  }


// Set box color
void FXRadioButton::setRadioColor(FXColor clr){
  if(radioColor!=clr){
    radioColor=clr;
    update();
    }
  }


// Change radio button style
void FXRadioButton::setRadioButtonStyle(FXuint style){
  FXuint opts=(options&~RADIOBUTTON_MASK) | (style&RADIOBUTTON_MASK);
  if(options!=opts){
    options=opts;
    update();
    }
  }


// Return current radio button style
FXuint FXRadioButton::getRadioButtonStyle() const {
  return (options&RADIOBUTTON_MASK); 
  }


// Save object to stream
void FXRadioButton::save(FXStream& store) const {
  FXLabel::save(store);
  store << radioColor;
  }


// Load object from stream
void FXRadioButton::load(FXStream& store){
  FXLabel::load(store);
  store >> radioColor;
  }  

