/********************************************************************************
*                                                                               *
*                             I m a g e    O b j e c t                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXImage.cpp,v 1.35 2000/03/08 03:43:29 jeroen Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXDC.h"
#include "FXDCWindow.h"


/*
  Notes:
  - FXImage::create() renders rgb[a] data into X/GDI resident, device
    dependent pixmap.
  - Need to be able to re-render subpart of image.
  - We should implement shared pixmaps.
  - If IMAGE_KEEP, repeated rendering is usually desired; should we
    hang on to XImage, and the shared memory segment in that case?
    How about shared pixmaps...
  - Slight change in interpretation of IMAGE_OWNED flag:- if passed, the
    FXImage will own client-side pixel buffer, otherwise it will not; if
    no pixel-buffer has been passed and IMAGE_OWNED *is* passed, a pixel
    buffer will be allocated [and cleared to zero].
    No pixel buffer will be allocated if neither IMAGE_OWNED nor pixels
    are passed.
  - When using shared image/pixmaps, if IMAGE_KEEP is set, hang on to pixel buffer.
  - Need resize, rotate member functions.
  - We need to speed up 16/15 bpp true color.
  - We need dither tables with 3bit and 2bit rounding for 5,6,5/5,5,5 modes
  - We need dither tables with 5bit, 6bit rounding for 3,3,2 mode.
  - We need to split true color from direct color, because direct color
    has random mapping, true has not.
  - Just because I always forget:
  
      StaticGray   0
      GrayScale    1
      StaticColor  2
      PseudoColor  3
      TrueColor    4
      DirectColor  5
*/


// RGB Ordering code
enum {
  RGB = 7,   // RGB 111      > | R  G  B
  BGR = 0,   // BGR 000      --+--------
  RBG = 6,   // RBG 110      R | x  4  2
  GBR = 1,   // GBR 001      G |    x  1
  BRG = 4,   // BRG 100      B |       x
  GRB = 3    // GRB 011
  };

#define DISPLAY(app) ((Display*)((app)->display))

  
// Maximum size of the colormap; for high-end graphics systems
// you may want to define HIGHENDGRAPHICS to allow for large colormaps
#ifdef HIGHENDGRAPHICS
#define MAX_MAPSIZE 4096
#else 
#define MAX_MAPSIZE 256
#endif


/*******************************************************************************/

// Object implementation
FXIMPLEMENT(FXImage,FXDrawable,NULL,0)


// For deserialization
FXImage::FXImage(){
  data=NULL;
  options=0;
  channels=4;
  }


// Initialize
FXImage::FXImage(FXApp* a,const void *pix,FXuint opts,FXint w,FXint h):FXDrawable(a,w,h){
  FXTRACE((100,"FXImage::FXImage %08x\n",this));
  visual=getApp()->getDefaultVisual();
  if(opts&IMAGE_ALPHA) channels=4; else channels=3;
  if(!pix && (opts&IMAGE_OWNED)){
    FXuint size=width*height*channels;
    FXCALLOC(&pix,FXuchar,size);
    }
  data=(FXuchar*)pix;
  options=opts;
  }


// Create image
void FXImage::create(){
  if(!xid){
    FXTRACE((100,"%s::create %08x\n",getClassName(),this));
    
#ifndef WIN32
    
    // App should exist
    if(!DISPLAY(getApp())){ fxerror("%s::create: trying to create image before opening display.\n",getClassName()); }
  
    // Initialize visual
    visual->init();

    // Get depth (should use visual!!)
    int dd=visual->getDepth();
    
    // Make pixmap
    xid=XCreatePixmap(DISPLAY(getApp()),XDefaultRootWindow(DISPLAY(getApp())),width,height,dd);
    if(!xid){ fxerror("%s::create: unable to create image.\n",getClassName()); }
  
#else

    // Initialize visual
    visual->init();

    // Create a bitmap compatible with current display
    HDC hdc=::GetDC(GetDesktopWindow());
    xid=CreateCompatibleBitmap(hdc,width,height);
    ::ReleaseDC(GetDesktopWindow(),hdc);
    if(!xid){ fxerror("%s::create: unable to create image.\n",getClassName()); }

#endif

    // Render pixels
    render();

    // Zap data
    if(!(options&IMAGE_KEEP) && (options&IMAGE_OWNED)){
      options&=~IMAGE_OWNED;
      FXFREE(&data);
      }
    }
  }


// Detach image
void FXImage::detach(){
  if(xid){
    FXTRACE((100,"%s::detach %08x\n",getClassName(),this));
    xid=0;
    }
  }


// Destroy image
void FXImage::destroy(){
  if(xid){
    FXTRACE((100,"%s::destroy %08x\n",getClassName(),this));
#ifndef WIN32
    XFreePixmap(DISPLAY(getApp()),xid);
#else
    DeleteObject(xid);
#endif
    xid=0;
    }
  }


#ifndef WIN32

// Find shift amount
static inline FXuint findshift(unsigned long mask){
  register FXuint sh=0;
  while(!(mask&(1<<sh))) sh++;
  return sh;
  }


// Find low bit in mask
static inline FXPixel lowbit(FXPixel mask){
  return (~mask+1)&mask;
  }


// Restore client-side pixel buffer from image 
void FXImage::restore(){
#ifdef HAVE_XSHM
  XShmSegmentInfo shminfo;
#endif
  register FXPixel red,green,blue;
  register FXPixel red1,green1,blue1;
  register FXPixel pixel;
  register FXuint  redshift,greenshift,blueshift; 
  register FXPixel redmask,greenmask,bluemask;  
  register int size,dd,i;
  register FXbool shmi=FALSE;
  register XImage *xim=NULL;
  register Visual *vis;
  register FXint x,y;
  register FXuchar *img;
  register FXuint r,g,b;
  FXuchar rtab[MAX_MAPSIZE];
  FXuchar gtab[MAX_MAPSIZE];
  FXuchar btab[MAX_MAPSIZE];

  FXTRACE((100,"%s::restore image %0x8\n",getClassName(),this));

  // Can not restore before creation
  if(!xid){ fxerror("%s::restore: trying to restore image before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<1 || height<1){ fxerror("%s::restore: illegal image size %dx%d.\n",getClassName(),width,height); }

  // Get Visual
  vis=(Visual*)visual->visual;
  dd=visual->getDepth();
  redmask=vis->red_mask;
  greenmask=vis->green_mask;
  bluemask=vis->blue_mask;
  redshift=findshift(redmask);
  greenshift=findshift(greenmask);
  blueshift=findshift(bluemask);

  // Just in case you're on a high-end system
  FXASSERT(vis->map_entries<=MAX_MAPSIZE);
  
  // Make array for data if needed
  if(!data){
    size=width*height*channels;
    FXCALLOC(&data,FXuchar,size);
    options|=IMAGE_OWNED;
    }
  
  // Got local buffer to receive into
  if(data){
    
    // Turn it on iff both supported and desired
#ifdef HAVE_XSHM
    if(options&IMAGE_SHMI) shmi=getApp()->shmi;
#endif

  // First try XShm
#ifdef HAVE_XSHM
    if(shmi){
      xim=XShmCreateImage(DISPLAY(getApp()),vis,dd,(dd==1)?XYPixmap:ZPixmap,NULL,&shminfo,width,height);
      if(!xim){ shmi=0; }
      if(shmi){
        shminfo.shmid=shmget(IPC_PRIVATE,xim->bytes_per_line*xim->height,IPC_CREAT|0777);
        if(shminfo.shmid==-1){ XDestroyImage(xim); xim=NULL; shmi=0; }
        if(shmi){
          shminfo.shmaddr=xim->data=(char*)shmat(shminfo.shmid,0,0);
          shminfo.readOnly=FALSE;
          XShmAttach(DISPLAY(getApp()),&shminfo);
          FXTRACE((150,"RGBPixmap XSHM attached at memory=0x%08x (%d bytes)\n",xim->data,xim->bytes_per_line*xim->height));
          XShmGetImage(DISPLAY(getApp()),xid,xim,0,0,AllPlanes);
          XSync(DISPLAY(getApp()),False);
          }
        }
      }
#endif

    // Try the old fashioned way
    if(!shmi){
      xim=XGetImage(DISPLAY(getApp()),xid,0,0,width,height,AllPlanes,ZPixmap);
      if(!xim){ fxerror("%s::restore: unable to restore image.\n",getClassName()); }
      }

    // Should have succeeded
    FXASSERT(xim);

    FXTRACE((150,"im width = %d\n",xim->width));
    FXTRACE((150,"im height = %d\n",xim->height));
    FXTRACE((150,"im format = %s\n",xim->format==XYBitmap?"XYBitmap":xim->format==XYPixmap?"XYPixmap":"ZPixmap"));
    FXTRACE((150,"im byte_order = %s\n",(xim->byte_order==MSBFirst)?"MSBFirst":"LSBFirst"));
    FXTRACE((150,"im bitmap_unit = %d\n",xim->bitmap_unit));
    FXTRACE((150,"im bitmap_bit_order = %s\n",(xim->bitmap_bit_order==MSBFirst)?"MSBFirst":"LSBFirst"));
    FXTRACE((150,"im bitmap_pad = %d\n",xim->bitmap_pad));
    FXTRACE((150,"im bitmap_unit = %d\n",xim->bitmap_unit));
    FXTRACE((150,"im depth = %d\n",xim->depth));
    FXTRACE((150,"im bytes_per_line = %d\n",xim->bytes_per_line));
    FXTRACE((150,"im bits_per_pixel = %d\n",xim->bits_per_pixel));


    {
    XColor colors[MAX_MAPSIZE];

    // Read back the colormap and convert to more usable form
    if(vis->c_class!=TrueColor && vis->c_class!=DirectColor){
      for(i=0 ; i<vis->map_entries; i++){
        colors[i].pixel=i;
        colors[i].flags=DoRed|DoGreen|DoBlue;
        }
      }
    else{
      red=green=blue=0;
      red1=lowbit(vis->red_mask);
      green1=lowbit(vis->green_mask);
      blue1=lowbit(vis->blue_mask);
      for(i=0; i<vis->map_entries; i++){
        colors[i].pixel=red|green|blue;
        colors[i].flags=DoRed|DoGreen|DoBlue;
        if(red<vis->red_mask) red+=red1;
        if(green<vis->green_mask) green+=green1;
        if(blue<vis->blue_mask) blue+=blue1;
        }
      }
    XQueryColors(DISPLAY(getApp()),visual->colormap,colors,vis->map_entries);
    for(i=0; i<vis->map_entries; i++){
      rtab[i]=colors[i].red >> 8;
      gtab[i]=colors[i].green >> 8;
      btab[i]=colors[i].blue >> 8;
      }
    }
  
    // Now we convert the pixels back to color
    img=data;
    switch(xim->bits_per_pixel){
      case 0:
      case 1:
      case 2:
      case 3:
      case 4:
      case 5:
      case 6:
      case 7:
      case 8:
        for(y=0; y<height; y++){
          for(x=0; x<width; x++){
            pixel=XGetPixel(xim,x,y);
            img[0]=rtab[pixel];
            img[1]=gtab[pixel];
            img[2]=btab[pixel];
            img+=channels;
            }
          }
        break;
      case 15:
      case 16:
      case 24:
      case 32:
      default:
        FXASSERT(vis->c_class==TrueColor || vis->c_class==DirectColor);
        for(y=0; y<height; y++){
          for(x=0; x<width; x++){
            pixel=XGetPixel(xim,x,y);
            r=(pixel&redmask)>>redshift;
            g=(pixel&greenmask)>>greenshift;
            b=(pixel&bluemask)>>blueshift;
            img[0]=rtab[r];
            img[1]=gtab[g];
            img[2]=btab[b];
            img+=channels;
            }
          }
        break;
      }
  
    // Destroy image
#ifdef HAVE_XSHM
    if(shmi){
      FXTRACE((150,"RGBPixmap XSHM detached at memory=0x%08x (%d bytes)\n",xim->data,xim->bytes_per_line*xim->height));
      XShmDetach(DISPLAY(getApp()),&shminfo);
      XDestroyImage(xim);
      shmdt(shminfo.shmaddr);
      shmctl(shminfo.shmid,IPC_RMID,0);
      }
#endif

    // Destroy image
    if(!shmi){
      XDestroyImage(xim);
      }
    }
  }


#else


// Restore client-side pixel buffer from image 
void FXImage::restore(){
  register FXint size,bytes_per_line,skip;
  register FXint x,y;
  register FXuchar *pixels,*pix,*img;
  BITMAPINFO bmi;
  HDC hdcmem;

  FXTRACE((100,"%s::restore image %0x8\n",getClassName(),this));

  // Can not restore before creation
  if(!xid){ fxerror("%s::restore: trying to restore image before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<1 || height<1){ fxerror("%s::restore: illegal image size %dx%d.\n",getClassName(),width,height); }

  // Make array for data if needed
  if(!data){
    size=width*height*channels;
    FXCALLOC(&data,FXuchar,size);
    options|=IMAGE_OWNED;
    }
  
  // Got local buffer to receive into
  if(data){

    // Set up the bitmap info
    bytes_per_line=(width*3+3)/4*4;
    skip=bytes_per_line-width*3;

    bmi.bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
    bmi.bmiHeader.biWidth=width;
    bmi.bmiHeader.biHeight=-height; // Negative heights means upside down!
    bmi.bmiHeader.biPlanes=1;
    bmi.bmiHeader.biBitCount=24;
    bmi.bmiHeader.biCompression=BI_RGB;
    bmi.bmiHeader.biSizeImage=0;
    bmi.bmiHeader.biXPelsPerMeter=0;
    bmi.bmiHeader.biYPelsPerMeter=0;
    bmi.bmiHeader.biClrUsed=0;
    bmi.bmiHeader.biClrImportant=0;

    // DIB format pads to multiples of 4 bytes...
    FXMALLOC(&pixels,FXuchar,bytes_per_line*height);

    // Make device context
    hdcmem=::CreateCompatibleDC(NULL);
    if(!GetDIBits(hdcmem,(HBITMAP)xid,0,height,pixels,&bmi,DIB_RGB_COLORS)){
      fxerror("%s::render: unable to restore pixels\n",getClassName());
      }

    // Stuff it into our own data structure
    img=data;
    pix=pixels;
    for(y=0; y<height; y++){
      for(x=0; x<width; x++){
        img[0]=pix[0];
        img[1]=pix[1];
        img[2]=pix[2];
        img+=channels;
        pix+=3;
        }
      pix+=skip;
      }
    FXFREE(&pixels);
    ::DeleteDC(hdcmem);
    }
  }


#endif

#if 0
/************************************************************************* 
 * 
 * BitmapToDIB() 
 * 
 * Parameters: 
 * 
 * HBITMAP hBitmap  - specifies the bitmap to convert 
 * 
 * HPALETTE hPal    - specifies the palette to use with the bitmap 
 * 
 * Return Value: 
 * 
 * HDIB             - identifies the device-dependent bitmap 
 * 
 * Description: 
 * 
 * This function creates a DIB from a bitmap using the specified palette. 
 * 
 ************************************************************************/ 
 
HDIB BitmapToDIB(HBITMAP hBitmap, HPALETTE hPal) 
{ 
    BITMAP              bm;         // bitmap structure 
    BITMAPINFOHEADER    bi;         // bitmap header 
    LPBITMAPINFOHEADER  lpbi;       // pointer to BITMAPINFOHEADER 
    DWORD               dwLen;      // size of memory block 
    HANDLE              hDIB, h;    // handle to DIB, temp handle 
    HDC                 hDC;        // handle to DC 
    WORD                biBits;     // bits per pixel 
 
    // check if bitmap handle is valid 
 
    if (!hBitmap) 
        return NULL; 
 
    // fill in BITMAP structure, return NULL if it didn't work 
 
    if (!GetObject(hBitmap, sizeof(bm), (LPSTR)&bm)) 
        return NULL; 
 
    // if no palette is specified, use default palette 
 
    if (hPal == NULL) 
        hPal = GetStockObject(DEFAULT_PALETTE); 
 
    // calculate bits per pixel 
 
    biBits = bm.bmPlanes * bm.bmBitsPixel; 
 
    // make sure bits per pixel is valid 
 
    if (biBits <= 1) 
        biBits = 1; 
    else if (biBits <= 4) 
        biBits = 4; 
    else if (biBits <= 8) 
        biBits = 8; 
    else // if greater than 8-bit, force to 24-bit 
        biBits = 24; 
 
    // initialize BITMAPINFOHEADER 
 
    bi.biSize = sizeof(BITMAPINFOHEADER); 
    bi.biWidth = bm.bmWidth; 
    bi.biHeight = bm.bmHeight; 
    bi.biPlanes = 1; 
    bi.biBitCount = biBits; 
    bi.biCompression = BI_RGB; 
    bi.biSizeImage = 0; 
    bi.biXPelsPerMeter = 0; 
    bi.biYPelsPerMeter = 0; 
    bi.biClrUsed = 0; 
    bi.biClrImportant = 0; 
 
    // calculate size of memory block required to store BITMAPINFO 
 
    dwLen = bi.biSize + PaletteSize((LPSTR)&bi); 
 
    // get a DC 
 
    hDC = GetDC(NULL); 
 
    // select and realize our palette 
 
    hPal = SelectPalette(hDC, hPal, FALSE); 
    RealizePalette(hDC); 
 
    // alloc memory block to store our bitmap 
 
    hDIB = GlobalAlloc(GHND, dwLen); 
 
    // if we couldn't get memory block 
 
    if (!hDIB) 
    { 
      // clean up and return NULL 
 
      SelectPalette(hDC, hPal, TRUE); 
      RealizePalette(hDC); 
      ReleaseDC(NULL, hDC); 
      return NULL; 
    } 
 
    // lock memory and get pointer to it 
 
    lpbi = (LPBITMAPINFOHEADER)GlobalLock(hDIB); 
 
    /// use our bitmap info. to fill BITMAPINFOHEADER 
 
    *lpbi = bi; 
 
    // call GetDIBits with a NULL lpBits param, so it will calculate the 
    // biSizeImage field for us     
 
    GetDIBits(hDC, hBitmap, 0, (UINT)bi.biHeight, NULL, (LPBITMAPINFO)lpbi, 
        DIB_RGB_COLORS); 
 
    // get the info. returned by GetDIBits and unlock memory block 
 
    bi = *lpbi; 
    GlobalUnlock(hDIB); 
 
    // if the driver did not fill in the biSizeImage field, make one up  
    if (bi.biSizeImage == 0) 
        bi.biSizeImage = WIDTHBYTES((DWORD)bm.bmWidth * biBits) * bm.bmHeight; 
 
    // realloc the buffer big enough to hold all the bits 
 
    dwLen = bi.biSize + PaletteSize((LPSTR)&bi) + bi.biSizeImage; 
 
    if (h = GlobalReAlloc(hDIB, dwLen, 0)) 
        hDIB = h; 
    else 
    { 
        // clean up and return NULL 
 
        GlobalFree(hDIB); 
        hDIB = NULL; 
        SelectPalette(hDC, hPal, TRUE); 
        RealizePalette(hDC); 
        ReleaseDC(NULL, hDC); 
        return NULL; 
    } 
 
    // lock memory block and get pointer to it */ 
 
    lpbi = (LPBITMAPINFOHEADER)GlobalLock(hDIB); 
 
    // call GetDIBits with a NON-NULL lpBits param, and actualy get the 
    // bits this time 
 
    if (GetDIBits(hDC, hBitmap, 0, (UINT)bi.biHeight, (LPSTR)lpbi + 
            (WORD)lpbi->biSize + PaletteSize((LPSTR)lpbi), (LPBITMAPINFO)lpbi, 
            DIB_RGB_COLORS) == 0) 
    { 
        // clean up and return NULL 
 
        GlobalUnlock(hDIB); 
        hDIB = NULL; 
        SelectPalette(hDC, hPal, TRUE); 
        RealizePalette(hDC); 
        ReleaseDC(NULL, hDC); 
        return NULL; 
    } 
 
    bi = *lpbi; 
 
    // clean up  
    GlobalUnlock(hDIB); 
    SelectPalette(hDC, hPal, TRUE); 
    RealizePalette(hDC); 
    ReleaseDC(NULL, hDC); 
 
    // return handle to the DIB 
    return hDIB; 
} 
#endif




#ifndef WIN32



// True generic mode
void FXImage::render_true_N_fast(void *xim,FXuchar *img){
  register FXint x,y;
  FXTRACE((150,"True MSB/LSB N bpp render nearest\n"));
  y=0;
  do{
    x=0;
    do{
      XPutPixel(((XImage*)xim),x,y,visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }



// True generic mode
void FXImage::render_true_N_dither(void *xim,FXuchar *img){
  register FXint x,y,d;
  FXTRACE((150,"True MSB/LSB N bpp render dither\n"));
  y=0;
  do{
    x=0;
    do{
      d=((y&3)<<2)|(x&3);
      XPutPixel(((XImage*)xim),x,y,visual->rpix[d][img[0]] | visual->gpix[d][img[1]] | visual->bpix[d][img[2]]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }



// True 24 bit color
void FXImage::render_true_24(void *xim,FXuchar *img){
  register FXuint jmp=((XImage*)xim)->bytes_per_line-(width*3);
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXPixel val;
  register FXint w,h;
  if(((XImage*)xim)->byte_order==MSBFirst){    // MSB
    FXTRACE((150,"True MSB 24bpp render\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        val=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        pix[0]=(FXuchar)(val>>16);
        pix[1]=(FXuchar)(val>>8);
        pix[2]=(FXuchar)val;
        img+=channels;
        pix+=3;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  else{                             // LSB
    FXTRACE((150,"True LSB 24bpp render\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        val=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        pix[0]=(FXuchar)val;
        pix[1]=(FXuchar)(val>>8);
        pix[2]=(FXuchar)(val>>16);
        img+=channels;
        pix+=3;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  }



// True 32 bit color
void FXImage::render_true_32(void *xim,FXuchar *img){
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXuint jmp=((XImage*)xim)->bytes_per_line-(width<<2);
  register FXPixel val;
  register FXint w,h;
  
  // Byte order matches
  if(((XImage*)xim)->byte_order == FOX_BIGENDIAN){
    FXTRACE((150,"True MSB/LSB 32bpp render\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        *((FXuint*)pix)=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        img+=channels;
        pix+=4;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  
  // MSB Byte order
  else if(((XImage*)xim)->byte_order==MSBFirst){
    FXTRACE((150,"True MSB 32bpp render\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        val=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        pix[0]=(FXuchar)(val>>24);
        pix[1]=(FXuchar)(val>>16);
        pix[2]=(FXuchar)(val>>8);
        pix[3]=(FXuchar)val;
        img+=channels;
        pix+=4;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  
  // LSB Byte order
  else{
    FXTRACE((150,"True LSB 32bpp render\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        val=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        pix[0]=(FXuchar)val;
        pix[1]=(FXuchar)(val>>8);
        pix[2]=(FXuchar)(val>>16);
        pix[3]=(FXuchar)(val>>24);
        img+=channels;
        pix+=4;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  }



// True 16 bit color
void FXImage::render_true_16_fast(void *xim,FXuchar *img){
  register FXuint jmp=((XImage*)xim)->bytes_per_line-(width<<1);
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXPixel val;
  register FXint w,h;
  
  // Byte order matches
  if(((XImage*)xim)->byte_order == FOX_BIGENDIAN){
    FXTRACE((150,"True MSB/LSB 16bpp 5,6,5/5,5,5 render nearest\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        *((FXushort*)pix)=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        img+=channels;
        pix+=2;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }

  // MSB Byte order
  else if(((XImage*)xim)->byte_order==MSBFirst){
    FXTRACE((150,"True MSB 16bpp 5,6,5/5,5,5 render nearest\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        val=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        pix[0]=(FXuchar)(val>>8);
        pix[1]=(FXuchar)val;
        img+=channels;
        pix+=2;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }

  // LSB Byte order
  else{
    FXTRACE((150,"True LSB 16bpp 5,6,5/5,5,5 render nearest\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        val=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
        pix[0]=(FXuchar)val;
        pix[1]=(FXuchar)(val>>8);
        img+=channels;
        pix+=2;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  }


// True 16 bit color, dithered
void FXImage::render_true_16_dither(void *xim,FXuchar *img){
  register FXuint jmp=((XImage*)xim)->bytes_per_line-(width<<1);
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXPixel val;
  register FXint w,h,d;
  
  // Byte order matches
  if(((XImage*)xim)->byte_order == FOX_BIGENDIAN){
    FXTRACE((150,"True MSB/LSB 16bpp 5,6,5/5,5,5 render dither\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        d=((h&3)<<2)|(w&3);
        *((FXushort*)pix)=visual->rpix[d][img[0]] | visual->gpix[d][img[1]] | visual->bpix[d][img[2]];
        img+=channels;
        pix+=2;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }

  // MSB Byte order
  else if(((XImage*)xim)->byte_order==MSBFirst){
    FXTRACE((150,"True MSB 16bpp 5,6,5/5,5,5 render dither\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        d=((h&3)<<2)|(w&3);
        val=visual->rpix[d][img[0]] | visual->gpix[d][img[1]] | visual->bpix[d][img[2]];
        pix[0]=(FXuchar)(val>>8);
        pix[1]=(FXuchar)val;
        img+=channels;
        pix+=2;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }

  // LSB Byte order
  else{
    FXTRACE((150,"True LSB 16bpp 5,6,5/5,5,5 render dither\n"));
    h=height-1;
    do{
      w=width-1;
      do{
        d=((h&3)<<2)|(w&3);
        val=visual->rpix[d][img[0]] | visual->gpix[d][img[1]] | visual->bpix[d][img[2]];
        pix[0]=(FXuchar)val;
        pix[1]=(FXuchar)(val>>8);
        img+=channels;
        pix+=2;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  }



// True 8 bit color
void FXImage::render_true_8_fast(void *xim,FXuchar *img){
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXint w,h;
  FXTRACE((150,"True MSB/LSB 8bpp render nearest\n"));
  h=height-1;
  do{
    w=width-1;
    do{
      *pix=visual->rpix[1][img[0]] | visual->gpix[1][img[1]] | visual->bpix[1][img[2]];
      img+=channels;
      pix++;
      }
    while(--w>=0);
    pix+=jmp;
    }
  while(--h>=0);
  }



// True 8 bit color, dithered
void FXImage::render_true_8_dither(void *xim,FXuchar *img){
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXint w,h,d;
  FXTRACE((150,"True MSB/LSB 8bpp render dither\n"));
  h=height-1;
  do{
    w=width-1;
    do{
      d=((h&3)<<2)|(w&3);
      *pix=visual->rpix[d][img[0]] | visual->gpix[d][img[1]] | visual->bpix[d][img[2]];
      img+=channels;
      pix++;
      }
    while(--w>=0);
    pix+=jmp;
    }
  while(--h>=0);
  }



// Render 4 bit index color mode
void FXImage::render_index_4_fast(void *xim,FXuchar *img){
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXuint val,half;
  register FXint w,h;
  if(((XImage*)xim)->byte_order==MSBFirst){    // MSB
    FXTRACE((150,"Index MSB 4bpp render nearest\n"));
    h=height-1;
    do{
      w=width-1;
      half=0;
      do{
        val=visual->lut[visual->rpix[1][img[0]]+visual->gpix[1][img[1]]+visual->bpix[1][img[2]]];
        if(half) *pix++|=val; 
        else *pix=val<<4; 
        half^=1;
        img+=channels;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  else{                               // LSB
    FXTRACE((150,"Index LSB 4bpp render nearest\n"));
    h=height-1;
    do{
      w=width-1;
      half=0;
      do{
        val=visual->lut[visual->rpix[1][img[0]]+visual->gpix[1][img[1]]+visual->bpix[1][img[2]]];
        if(half) *pix++|=val<<4;
        else *pix=val;
        half^=1;
        img+=channels;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  }



// Render 4 bit index color mode
void FXImage::render_index_4_dither(void *xim,FXuchar *img){
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXuint val,half,d;
  register FXint w,h;
  if(((XImage*)xim)->byte_order==MSBFirst){    // MSB
    FXTRACE((150,"Index MSB 4bpp render dither\n"));
    h=height-1;
    do{
      w=width-1;
      half=0;
      do{
        d=((h&3)<<2)|(w&3);
        val=visual->lut[visual->rpix[d][img[0]]+visual->gpix[d][img[1]]+visual->bpix[d][img[2]]];
        if(half) *pix++|=val; 
        else *pix=val<<4; 
        half^=1;
        img+=channels;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  else{                               // LSB
    FXTRACE((150,"Index LSB 4bpp render dither\n"));
    h=height-1;
    do{
      w=width-1;
      half=0;
      do{
        d=((h&3)<<2)|(w&3);
        val=visual->lut[visual->rpix[d][img[0]]+visual->gpix[d][img[1]]+visual->bpix[d][img[2]]];
        if(half) *pix++|=val<<4;
        else *pix=val;
        half^=1;
        img+=channels;
        }
      while(--w>=0);
      pix+=jmp;
      }
    while(--h>=0);
    }
  }



// Render 8 bit index color mode
void FXImage::render_index_8_fast(void *xim,FXuchar *img){
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXint w,h;
  FXTRACE((150,"Index MSB/LSB 8bpp render nearest\n"));
  h=height-1;
  do{
    w=width-1;
    do{
      *pix=visual->lut[visual->rpix[1][img[0]]+visual->gpix[1][img[1]]+visual->bpix[1][img[2]]];
      img+=channels;
      pix++;
      }
    while(--w>=0);
    pix+=jmp;
    }
  while(--h>=0);
  }



// Render 8 bit index color mode
void FXImage::render_index_8_dither(void *xim,FXuchar *img){
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXint w,h,d;
  FXTRACE((150,"Index MSB/LSB 8bpp render dither\n"));
  h=height-1;
  do{
    w=width-1;
    do{
      d=((h&3)<<2)|(w&3);
      *pix=visual->lut[visual->rpix[d][img[0]]+visual->gpix[d][img[1]]+visual->bpix[d][img[2]]];
      img+=channels;
      pix++;
      }
    while(--w>=0);
    pix+=jmp;
    }
  while(--h>=0);
  }



// Render generic N bit index color mode
void FXImage::render_index_N_fast(void *xim,FXuchar *img){
  register FXint x,y;
  FXTRACE((150,"Index MSB/LSB N bpp render nearest\n"));
  y=0;
  do{
    x=0;
    do{
      XPutPixel(((XImage*)xim),x,y,visual->lut[visual->rpix[1][img[0]]+visual->gpix[1][img[1]]+visual->bpix[1][img[2]]]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }



// Render generic N bit index color mode
void FXImage::render_index_N_dither(void *xim,FXuchar *img){
  register FXint x,y,d;
  FXTRACE((150,"Index MSB/LSB N bpp render dither\n"));
  y=0;
  do{
    x=0;
    do{
      d=((y&3)<<2)|(x&3);
      XPutPixel(((XImage*)xim),x,y,visual->lut[visual->rpix[d][img[0]]+visual->gpix[d][img[1]]+visual->bpix[d][img[2]]]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }



// Render 8 bit gray mode
void FXImage::render_gray_8_fast(void *xim,FXuchar *img){
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXint w,h;
  FXTRACE((150,"Gray MSB/LSB 8bpp render nearest\n"));
  h=height-1;
  do{
    w=width-1;
    do{
      *pix=visual->gpix[1][(77*img[0]+151*img[1]+29*img[2])>>8];
      img+=channels;
      pix++;
      }
    while(--w>=0);
    pix+=jmp;
    }
  while(--h>=0);
  }



// Render 8 bit gray mode
void FXImage::render_gray_8_dither(void *xim,FXuchar *img){
  register FXuchar *pix=(FXuchar*)((XImage*)xim)->data;
  register FXuint jmp=((XImage*)xim)->bytes_per_line-width;
  register FXint w,h;
  FXTRACE((150,"Gray MSB/LSB 8bpp render dither\n"));
  h=height-1;
  do{
    w=width-1;
    do{
      *pix=visual->gpix[((h&3)<<2)|(w&3)][(77*img[0]+151*img[1]+29*img[2])>>8];
      img+=channels;
      pix++;
      }
    while(--w>=0);
    pix+=jmp;
    }
  while(--h>=0);
  }



// Render generic N bit gray mode
void FXImage::render_gray_N_fast(void *xim,FXuchar *img){
  register FXint x,y;
  FXTRACE((150,"Gray MSB/LSB N bpp render nearest\n"));
  y=0;
  do{
    x=0;
    do{
      XPutPixel(((XImage*)xim),x,y,visual->gpix[1][(77*img[0]+151*img[1]+29*img[2])>>8]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }



// Render generic N bit gray mode
void FXImage::render_gray_N_dither(void *xim,FXuchar *img){
  register FXint x,y;
  FXTRACE((150,"Gray MSB/LSB N bpp render dither\n"));
  y=0;
  do{
    x=0;
    do{
      XPutPixel(((XImage*)xim),x,y,visual->gpix[((y&3)<<2)|(x&3)][(77*img[0]+151*img[1]+29*img[2])>>8]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }




// Render monochrome mode
void FXImage::render_mono_1_fast(void *xim,FXuchar *img){
  register FXint x,y;
  FXTRACE((150,"Monochrome MSB/LSB 1bpp render nearest\n"));
  y=0;
  do{
    x=0;
    do{
      XPutPixel(((XImage*)xim),x,y,visual->gpix[1][(77*img[0]+151*img[1]+29*img[2])>>8]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }



// Render monochrome mode
void FXImage::render_mono_1_dither(void *xim,FXuchar *img){
  register FXint x,y;
  FXTRACE((150,"Monochrome MSB/LSB 1bpp render dither\n"));
  y=0;
  do{
    x=0;
    do{
      XPutPixel(((XImage*)xim),x,y,visual->gpix[((y&3)<<2)|(x&3)][(77*img[0]+151*img[1]+29*img[2])>>8]);
      img+=channels;
      }
    while(++x<width);
    }
  while(++y<height);
  }



#endif



#ifndef WIN32

// Render into pixmap
void FXImage::render(){
#ifdef HAVE_XSHM
  XShmSegmentInfo shminfo;
#endif
  register FXbool shmi=FALSE;
  register XImage *xim=NULL;
  register Visual *vis;
  register int dd;
  XGCValues values;
  GC gc;
  
  FXTRACE((100,"%s::render image %0x8\n",getClassName(),this));

  // Can not render before creation
  if(!xid){ fxerror("%s::render: trying to render image before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<1 || height<1){ fxerror("%s::render: illegal image size %dx%d.\n",getClassName(),width,height); }

  // Just leave if black if no data
  if(data){

    // Make GC
    values.foreground=BlackPixel(DISPLAY(getApp()),DefaultScreen(DISPLAY(getApp())));
    values.background=WhitePixel(DISPLAY(getApp()),DefaultScreen(DISPLAY(getApp())));
    gc=XCreateGC(DISPLAY(getApp()),xid,GCForeground|GCBackground,&values);

    // Get Visual
    vis=(Visual*)visual->visual;

    dd=visual->getDepth();

    // Turn it on iff both supported and desired
#ifdef HAVE_XSHM
    if(options&IMAGE_SHMI) shmi=getApp()->shmi;
#endif

    // First try XShm
#ifdef HAVE_XSHM
    if(shmi){
      xim=XShmCreateImage(DISPLAY(getApp()),vis,dd,(dd==1)?XYPixmap:ZPixmap,NULL,&shminfo,width,height);
      if(!xim){ shmi=0; }
      if(shmi){
        shminfo.shmid=shmget(IPC_PRIVATE,xim->bytes_per_line*xim->height,IPC_CREAT|0777);
        if(shminfo.shmid==-1){ XDestroyImage(xim); xim=NULL; shmi=0; }
        if(shmi){
          shminfo.shmaddr=xim->data=(char*)shmat(shminfo.shmid,0,0);
          shminfo.readOnly=FALSE;
          XShmAttach(DISPLAY(getApp()),&shminfo);
          FXTRACE((150,"RGBPixmap XSHM attached at memory=0x%08x (%d bytes)\n",xim->data,xim->bytes_per_line*xim->height));
          }
        }
      }
#endif

    // Try the old fashioned way
    if(!shmi){
      xim=XCreateImage(DISPLAY(getApp()),vis,dd,(dd==1)?XYPixmap:ZPixmap,0,NULL,width,height,32,0);
      if(!xim){ fxerror("%s::render: unable to render image.\n",getClassName()); }

      // Try create temp pixel store
      xim->data=(char*)malloc(xim->bytes_per_line*height);

      // Failed completely
      if(!xim->data){ fxerror("%s::render: unable to allocate memory.\n",getClassName()); }
      }

    // Should have succeeded
    FXASSERT(xim);

    FXTRACE((150,"im width = %d\n",xim->width));
    FXTRACE((150,"im height = %d\n",xim->height));
    FXTRACE((150,"im format = %s\n",xim->format==XYBitmap?"XYBitmap":xim->format==XYPixmap?"XYPixmap":"ZPixmap"));
    FXTRACE((150,"im byte_order = %s\n",(xim->byte_order==MSBFirst)?"MSBFirst":"LSBFirst"));
    FXTRACE((150,"im bitmap_unit = %d\n",xim->bitmap_unit));
    FXTRACE((150,"im bitmap_bit_order = %s\n",(xim->bitmap_bit_order==MSBFirst)?"MSBFirst":"LSBFirst"));
    FXTRACE((150,"im bitmap_pad = %d\n",xim->bitmap_pad));
    FXTRACE((150,"im bitmap_unit = %d\n",xim->bitmap_unit));
    FXTRACE((150,"im depth = %d\n",xim->depth));
    FXTRACE((150,"im bytes_per_line = %d\n",xim->bytes_per_line));
    FXTRACE((150,"im bits_per_pixel = %d\n",xim->bits_per_pixel));
    
    // Determine what to do
    switch(visual->getType()){
      case VISUALTYPE_TRUE:
        switch(xim->bits_per_pixel){
          case 32: 
            render_true_32(xim,data); 
            break;
          case 24: 
            render_true_24(xim,data); 
            break;
          case 15:
          case 16: 
            if(options&IMAGE_NEAREST) 
              render_true_16_fast(xim,data); 
            else 
              render_true_16_dither(xim,data); 
            break;
          case 8:  
            if(options&IMAGE_NEAREST) 
              render_true_8_fast(xim,data); 
            else 
              render_true_8_dither(xim,data); 
            break;
          default: 
            if(options&IMAGE_NEAREST) 
              render_true_N_fast(xim,data); 
            else 
              render_true_N_dither(xim,data); 
            break;
          }
        break;
      case VISUALTYPE_GRAY:
        switch(xim->bits_per_pixel){
          case 1:
            if(options&IMAGE_NEAREST) 
              render_mono_1_fast(xim,data); 
            else 
              render_mono_1_dither(xim,data);
            break;
          case 8:
            if(options&IMAGE_NEAREST) 
              render_gray_8_fast(xim,data); 
            else 
              render_gray_8_dither(xim,data);
            break;
          default:
            if(options&IMAGE_NEAREST) 
              render_gray_N_fast(xim,data); 
            else 
              render_gray_N_dither(xim,data);
            break;
          }
        break;
      case VISUALTYPE_INDEX:
        switch(xim->bits_per_pixel){
          case 4:
            if(options&IMAGE_NEAREST) 
              render_index_4_fast(xim,data); 
            else 
              render_index_4_dither(xim,data);
            break;
          case 8:
            if(options&IMAGE_NEAREST) 
              render_index_8_fast(xim,data); 
            else 
              render_index_8_dither(xim,data);
            break;
          default:
            if(options&IMAGE_NEAREST) 
              render_index_N_fast(xim,data); 
            else 
              render_index_N_dither(xim,data);
            break;
          }
        break;
      case VISUALTYPE_MONO:
        if(options&IMAGE_NEAREST) 
          render_mono_1_fast(xim,data); 
        else 
          render_mono_1_dither(xim,data);
      case VISUALTYPE_UNKNOWN:
        break;
      }

    // Transfer image with shared memory
#ifdef HAVE_XSHM
    if(shmi){
      XShmPutImage(DISPLAY(getApp()),xid,gc,xim,0,0,0,0,width,height,False);
      XSync(DISPLAY(getApp()),False);
      FXTRACE((150,"RGBPixmap XSHM detached at memory=0x%08x (%d bytes)\n",xim->data,xim->bytes_per_line*xim->height));
      XShmDetach(DISPLAY(getApp()),&shminfo);
      XDestroyImage(xim);
      shmdt(shminfo.shmaddr);
      shmctl(shminfo.shmid,IPC_RMID,0);
      }
#endif
   
    // Transfer the image old way
    if(!shmi){
      XPutImage(DISPLAY(getApp()),xid,gc,xim,0,0,0,0,width,height);
      XDestroyImage(xim);
      }
    XFreeGC(DISPLAY(getApp()),gc);
    }
  }


#else


void FXImage::render(){
  register FXuint bytes_per_line,skip;
  register FXuchar *src,*dst;
  register FXint h,w;
  BITMAPINFO bmi;
  FXuchar *pixels;
  HDC hdcmem;

  FXTRACE((100,"%s::render %0x8\n",getClassName(),this));
  
  // Can not render before creation
  if(!xid){ fxerror("%s::render: trying to render image before it has been created.\n",getClassName()); }

  // Check for legal size
  if(width<1 || height<1){ fxerror("%s::render: illegal image size %dx%d.\n",getClassName(),width,height); }


  // Just leave if black if no data
  if(data){
    
    // Set up the bitmap info
    bmi.bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
    bmi.bmiHeader.biWidth=width;
    bmi.bmiHeader.biHeight=height;
    bmi.bmiHeader.biPlanes=1;
    bmi.bmiHeader.biBitCount=24;
    bmi.bmiHeader.biCompression=BI_RGB;
    bmi.bmiHeader.biSizeImage=0;
    bmi.bmiHeader.biXPelsPerMeter=0;
    bmi.bmiHeader.biYPelsPerMeter=0;
    bmi.bmiHeader.biClrUsed=0;
    bmi.bmiHeader.biClrImportant=0;

    // DIB format pads to multiples of 4 bytes...
    bytes_per_line=(width*3+3)/4*4;
    skip=bytes_per_line+width*3;
    FXMALLOC(&pixels,FXuchar,bytes_per_line*height);
    h=height-1;
    src=data;
    dst=pixels+h*bytes_per_line;
    do{
      w=width-1;
      do{
        dst[0]=src[2];
        dst[1]=src[1];
        dst[2]=src[0];
        src+=channels;
        dst+=3;
        }
      while(--w>=0);
      dst-=skip;
      } 
    while(--h>=0);
    
    // The MSDN documentation for SetDIBits() states that "the device context
    // identified by the (first) parameter is used only if the DIB_PAL_COLORS
    // constant is set for the (last) parameter". This may be true, but under
    // Win95 you must pass in a non-NULL hdc for the first parameter; otherwise
    // this call to SetDIBits() will fail (in contrast, it works fine under
    // Windows NT if you pass in a NULL hdc).
    hdcmem=::CreateCompatibleDC(NULL);
    if(!SetDIBits(hdcmem,(HBITMAP)xid,0,height,pixels,&bmi,DIB_RGB_COLORS)){
      fxerror("%s::render: unable to render pixels\n",getClassName());
      }
    GdiFlush();
    FXFREE(&pixels);
    ::DeleteDC(hdcmem);
    }
  }

#endif



// Resize pixmap to the specified width and height; the contents become undefined 
void FXImage::resize(FXint w,FXint h){
  FXTRACE((100,"%s::resize(%d,%d) %08x\n",getClassName(),w,h,this));
  if(w<1) w=1;
  if(h<1) h=1;
  if(width!=w || height!=h){
    
    // Resize device dependent pixmap
    if(xid){
#ifndef WIN32

      // Get depth (should use visual!!)
      int dd=visual->getDepth();

      // Free old pixmap
      XFreePixmap(DISPLAY(getApp()),xid);

      // Make new pixmap
      xid=XCreatePixmap(DISPLAY(getApp()),XDefaultRootWindow(DISPLAY(getApp())),w,h,dd);
      if(!xid){ fxerror("%s::resize: unable to resize image.\n",getClassName()); }
#else

      // Delete old bitmap
      DeleteObject(xid);

      // Create a bitmap compatible with current display
      HDC hdc=::GetDC(GetDesktopWindow());
      xid=CreateCompatibleBitmap(hdc,w,h);
      ::ReleaseDC(GetDesktopWindow(),hdc);
      if(!xid){ fxerror("%s::resize: unable to resize image.\n",getClassName()); }
#endif
      }
    
    // Resize data array iff total size changed
    if(data && (w*h)!=(width*height)){
      if(options&IMAGE_OWNED){
        FXRESIZE(&data,FXuchar,w*h*channels);
        }
      else{
        FXCALLOC(&data,FXuchar,w*h*channels);
        options|=IMAGE_OWNED;
        }
      }
    
    // Remember new size
    width=w;
    height=h;
    }
  }



// Resize drawable to the specified width and height 
void FXImage::scale(FXint sx,FXint sy){
  FXTRACE((100,"%s::scale %08x\n",getClassName(),this));
  }


// Mirror image horizontally and/or vertically
void FXImage::mirror(FXbool horizontal,FXbool vertical){
  register FXuchar *paa,*pbb,*pa,*pb;
  register FXint nbytes=channels*width;
  register FXColor t1;
  register FXuchar t2;
  FXTRACE((100,"%s::mirror(%d,%d) %08x\n",getClassName(),horizontal,vertical,this));
  if(data && width>0 && height>0){

    // Mirror vertically
    if(vertical && height>1){
      paa=data;
      pbb=data+nbytes*(height-1);
      if(channels==4){
        FXASSERT((((long)data)&3)==0);
        do{
          pa=paa; paa+=nbytes;
          pb=pbb; pbb-=nbytes;
          do{
            t1=*((FXColor*)pa); *((FXColor*)pa)=*((FXColor*)pb); *((FXColor*)pb)=t1;
            pa+=4;
            pb+=4;
            }
          while(pa<paa);
          }
        while(paa<pbb);
        }
      else{
        do{
          pa=paa; paa+=nbytes;
          pb=pbb; pbb-=nbytes;
          do{
            t2=pa[0]; pa[0]=pb[0]; pb[0]=t2;
            t2=pa[1]; pa[1]=pb[1]; pb[1]=t2;
            t2=pa[2]; pa[2]=pb[2]; pb[2]=t2;
            pa+=3;
            pb+=3;
            }
          while(pa<paa);
          }
        while(paa<pbb);
        }
      }

    // Mirror horizontally
    if(horizontal && width>1){
      paa=data;
      pbb=data+nbytes*height;
      if(channels==4){
        FXASSERT((((long)data)&3)==0);
        do{
          pa=paa; paa+=nbytes; pb=paa-4;
          do{
            t1=*((FXColor*)pa); *((FXColor*)pa)=*((FXColor*)pb); *((FXColor*)pb)=t1;
            pa+=4;
            pb-=4;
            }
          while(pa<pb);
          }
        while(paa<pbb);
        }
      else{
        do{
          pa=paa; paa+=nbytes; pb=paa-3;
          do{
            t2=pa[0]; pa[0]=pb[0]; pb[0]=t2;
            t2=pa[1]; pa[1]=pb[1]; pb[1]=t2;
            t2=pa[2]; pa[2]=pb[2]; pb[2]=t2;
            pa+=3;
            pb-=3;
            }
          while(pa<pb);
          }
        while(paa<pbb);
        }
      }

    // Render pixmap 
    if(xid) render();
    }
  }



// Rotate image by degrees ccw
void FXImage::rotate(FXint degrees){
  register FXuchar *paa,*pbb,*end,*pa,*pb;
  register FXint nbytesa;
  register FXint nbytesb;
  register FXint size;
  FXuchar *olddata;
  FXTRACE((100,"%s::rotate(%d) %08x\n",getClassName(),degrees,this));
  if(width>1 && height>1){
    switch((degrees+360)%360){
      case 90:
        resize(height,width);
        if(data){
          nbytesa=channels*width;
          nbytesb=channels*height;
          size=channels*width*height;
          FXMALLOC(&olddata,FXuchar,size);
          memcpy(olddata,data,size);
          paa=data;
          pbb=olddata+channels*(height-1);
          end=data+size;
          if(channels==4){
            do{
              pa=paa; paa+=nbytesa;
              pb=pbb; pbb-=4;
              do{
                *((FXColor*)pa)=*((FXColor*)pb);
                pa+=4;
                pb+=nbytesb;
                }
              while(pa<paa);
              }
            while(paa<end);
            }
          else{
            do{
              pa=paa; paa+=nbytesa;
              pb=pbb; pbb-=3;
              do{
                pa[0]=pb[0];
                pa[1]=pb[1];
                pa[2]=pb[2];
                pa+=3;
                pb+=nbytesb;
                }
              while(pa<paa);
              }
            while(paa<end);
            }
          FXFREE(&olddata);
          if(xid) render();
          }
        break;
      case 180:
        if(data){
          nbytesa=channels*width;
          nbytesb=channels*width;
          size=channels*width*height;
          FXMALLOC(&olddata,FXuchar,size);
          memcpy(olddata,data,size);
          paa=data;
          pbb=olddata+channels*(width*height-1);
          end=data+size;
          if(channels==4){
            do{
              pa=paa; paa+=nbytesa;
              pb=pbb; pbb-=nbytesb;
              do{
                *((FXColor*)pa)=*((FXColor*)pb);
                pa+=4;
                pb-=4;
                }
              while(pa<paa);
              }
            while(paa<end);
            }
          else{
            do{
              pa=paa; paa+=nbytesa;
              pb=pbb; pbb-=nbytesb;
              do{
                pa[0]=pb[0];
                pa[1]=pb[1];
                pa[2]=pb[2];
                pa+=3;
                pb-=3;
                }
              while(pa<paa);
              }
            while(paa<end);
            }
          FXFREE(&olddata);
          if(xid) render();
          }
        break;
      case 270:
        resize(height,width);
        if(data){
          nbytesa=channels*width;
          nbytesb=channels*height;
          size=channels*width*height;
          FXMALLOC(&olddata,FXuchar,size);
          memcpy(olddata,data,size);
          paa=data;
          pbb=olddata+nbytesb*(width-1);
          end=data+size;
          if(channels==4){
            do{
              pa=paa; paa+=nbytesa;
              pb=pbb; pbb+=4;
              do{
                *((FXColor*)pa)=*((FXColor*)pb);
                pa+=4;
                pb-=nbytesb;
                }
              while(pa<paa);
              }
            while(paa<end);
            }
          else{
            do{
              pa=paa; paa+=nbytesa;
              pb=pbb; pbb+=3;
              do{
                pa[0]=pb[0];
                pa[1]=pb[1];
                pa[2]=pb[2];
                pa+=3;
                pb-=nbytesb;
                }
              while(pa<paa);
              }
            while(paa<end);
            }
          FXFREE(&olddata);
          if(xid) render();
          }
        break;
      default:
        fxwarning("%s::rotate: rotation by %d degrees not implemented\n",getClassName(),degrees);
      case 0:
        break;
      }
    }
  }



#ifdef WIN32

// Return the device context; the image already selected into it
FXID FXImage::GetDC() const { 
  HDC hdc=::CreateCompatibleDC(NULL);
  SelectObject(hdc,(HBITMAP)xid);
  return hdc;
  }


// Release it (no-op)
int FXImage::ReleaseDC(FXID hdc) const { 
  return ::DeleteDC((HDC)hdc);
  }

#endif



// Save pixel data only
void FXImage::savePixels(FXStream& store) const {
  FXuint size=width*height*channels;
  store.save(data,size);
  }


// Load pixel data only
void FXImage::loadPixels(FXStream& store){
  FXuint size=width*height*channels;
  if(options&IMAGE_OWNED){FXFREE(&data);}
  FXMALLOC(&data,FXuchar,size);
  store.load(data,size);
  options|=IMAGE_OWNED;
  }


// Save data
void FXImage::save(FXStream& store) const {
  FXuchar haspixels=(data!=NULL);
  FXDrawable::save(store);
  store << options;
  store << channels;
  store << haspixels;
  if(haspixels) savePixels(store);
  }


// Load data
void FXImage::load(FXStream& store){
  FXuchar haspixels;
  FXDrawable::load(store);
  store >> options;
  store >> channels;
  store >> haspixels;
  if(haspixels) loadPixels(store);
  }


// Clean up
FXImage::~FXImage(){
  FXTRACE((100,"FXImage::~FXImage %08x\n",this));
  if(xid){
#ifndef WIN32
    XFreePixmap(DISPLAY(getApp()),xid);
#else
    DeleteObject(xid);
#endif
    }
  if(options&IMAGE_OWNED){FXFREE(&data);}
  data=(FXuchar*)-1;
  }
