/********************************************************************************
*                                                                               *
*                               H e a d e r   O b j e c t                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXHeader.cpp,v 1.8 2000/02/20 19:17:50 jeroen Exp $                      *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXHeader.h"


#define FUDGE   8


/*
  To do:
  - FXHeader is nigh perfect.
  - Perhaps only, maybe, some way to drive via keyboard?
*/




#define HEADER_MASK (HEADER_BUTTON|HEADER_TRACKING)


/*******************************************************************************/

// Object implementation
FXIMPLEMENT(FXHeaderItem,FXObject,NULL,0)


// Draw item
void FXHeaderItem::draw(const FXHeader* header,FXDC& dc,FXint x,FXint y,FXint w,FXint h){
  FXFont *font=header->getFont();
  FXint th=0,ih=0;
  
  // Placement
  if(icon) ih=icon->getHeight();
  if(!label.empty()) th=header->getFont()->getFontHeight();
  w=w-(header->getPadLeft()+header->getPadRight()+(header->getBorderWidth()<<1));
  x+=header->getBorderWidth()+header->getPadLeft();
  
  // Draw icon
  if(icon){
    if(icon->getWidth()<=w){
      dc.drawIcon(icon,x,y+(h-ih)/2);
      x+=icon->getWidth()+4;
      }
    w-=icon->getWidth()+4;
    }
  
  // Draw text
  if(!label.empty() && w>0){
    FXint dw=font->getTextWidth("...",3);
    FXint num=label.length();
    FXint tw;
    y=y+(h-th)/2+font->getFontAscent();
    dc.setTextFont(font);
    if(font->getTextWidth(label.text(),num)>w){
      while(num>0 && (tw=font->getTextWidth(label.text(),num))>(w-dw)) num--;
      if(num>0){
        dc.setForeground(header->getTextColor());
        dc.drawText(x,y,label.text(),num);
        dc.drawText(x+tw,y,"...",3);
        }
      else{
        if(font->getTextWidth(label.text(),1)<=w){
          dc.setForeground(header->getTextColor());
          dc.drawText(x,y,label.text(),1);
          }
        }
      }
    else{
      dc.setForeground(header->getTextColor());
      dc.drawText(x,y,label.text(),num);
      }
    }
  }


// Create icon
void FXHeaderItem::create(){ if(icon) icon->create(); }


// No op, we don't own icon
void FXHeaderItem::destroy(){ }


// Detach from icon resource
void FXHeaderItem::detach(){ if(icon) icon->detach(); }


// Get width of item
FXint FXHeaderItem::getWidth(const FXHeader*) const {
  return size;
  }


// Get height of item
FXint FXHeaderItem::getHeight(const FXHeader* header) const {
  FXint th,ih;
  th=ih=0;
  if(!label.empty()) th=header->getFont()->getFontHeight();
  if(icon) ih=icon->getHeight();
  return FXMAX(th,ih)+header->getPadTop()+header->getPadBottom()+(header->getBorderWidth()<<1);
  }



/*******************************************************************************/

// Map
FXDEFMAP(FXHeader) FXHeaderMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXHeader::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXHeader::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXHeader::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXHeader::onLeftBtnRelease),
  FXMAPFUNC(SEL_UNGRABBED,0,FXHeader::onUngrabbed),
  };


// Object implementation
FXIMPLEMENT(FXHeader,FXFrame,FXHeaderMap,ARRAYNUMBER(FXHeaderMap))

  
// Make a Header
FXHeader::FXHeader(){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  items=NULL;
  nitems=0;
  textColor=0;
  state=FALSE;
  font=(FXFont*)-1;
  active=-1;
  activex=0;
  activew=0;
  off=0;
  }


// Make a Header
FXHeader::FXHeader(FXComposite* p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXFrame(p,opts,x,y,w,h,pl,pr,pt,pb){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  target=tgt;
  message=sel;
  items=NULL;
  nitems=0;
  textColor=getApp()->foreColor;
  state=FALSE;
  font=getApp()->getNormalFont();
  active=-1;
  activex=0;
  activew=0;
  off=0;
  }


// Create window
void FXHeader::create(){
  register FXint i;
  FXFrame::create();
  font->create();
  for(i=0; i<nitems; i++){if(items[i]->icon){items[i]->icon->create();}}
  }


// Detach window
void FXHeader::detach(){
  register FXint i;
  FXFrame::detach();
  font->detach();
  for(i=0; i<nitems; i++){if(items[i]->icon){items[i]->icon->detach();}}
  }


// Get default width
FXint FXHeader::getDefaultWidth(){
  register FXint w,i;
  for(i=0,w=0; i<nitems; i++){w+=items[i]->getWidth(this);}
  return w; 
  }


// Get default height
FXint FXHeader::getDefaultHeight(){
  register FXint h,i,t;
  for(i=0,h=0; i<nitems; i++){if((t=items[i]->getHeight(this))>h) h=t;}
  return h; 
  }


// Create custom item
FXHeaderItem *FXHeader::createItem(const FXString& text,FXIcon* icon,FXint size,void* ptr){
  return new FXHeaderItem(text,icon,size,ptr);
  }


// Retrieve item
FXHeaderItem *FXHeader::retrieveItem(FXint index) const {
  if(index<0 || nitems<index){ fxerror("%s::retrieveItem: index out of range.\n",getClassName()); } 
  return items[index];
  }


// Append item at end
FXint FXHeader::appendItem(FXHeaderItem* item){
  if(!item){ fxerror("%s::appendItem: item is NULL.\n",getClassName()); }
  FXRESIZE(&items,FXHeaderItem*,nitems+1);
  items[nitems]=item;
  nitems++;
  recalc();
  return nitems-1;
  }


// Append item at end
FXint FXHeader::appendItem(const FXString& text,FXIcon *icon,FXint size,void* ptr){
  return appendItem(createItem(text,icon,FXMAX(size,0),ptr));
  }


// Remove node from list
void FXHeader::removeItem(FXint index){
  if(index<0 || nitems<=index){ fxerror("%s::removeItem: index out of range.\n",getClassName()); } 
  delete items[index];
  memmove(&items[index],&items[index+1],sizeof(FXHeaderItem*)*(nitems-index));
  nitems--;
  recalc();
  }


// Remove all items
void FXHeader::clearItems(){
  register FXint i;
  for(i=0; i<nitems; i++){delete items[i];}
  FXFREE(&items);
  nitems=0;
  recalc();
  }


// Get item at offset
FXint FXHeader::getItemAt(FXint offset) const {
  register FXint x,i,w;
  for(i=0,x=0; i<nitems; i++){
    w=items[i]->getWidth(this);
    if(x<=offset && offset<x+w) return i;
    x+=w;
    }
  return -1;
  }


// Change item's text
void FXHeader::setItemText(FXint index,const FXString& text){
  if(index<0 || nitems<=index){ fxerror("%s::setItemText: index out of range.\n",getClassName()); } 
  if(items[index]->label!=text){
    items[index]->label=text;
    update();
    }
  }


// Get item's text
FXString FXHeader::getItemText(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemText: index out of range.\n",getClassName()); } 
  return items[index]->label;
  }


// Change item's icon
void FXHeader::setItemIcon(FXint index,FXIcon* icon){
  if(index<0 || nitems<=index){ fxerror("%s::setItemIcon: index out of range.\n",getClassName()); } 
  if(items[index]->icon!=icon){
    items[index]->icon=icon;
    update();
    }
  }


// Get item's icon
FXIcon* FXHeader::getItemIcon(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemIcon: index out of range.\n",getClassName()); } 
  return items[index]->icon;
  }


// Change item's size
void FXHeader::setItemSize(FXint index,FXint size){
  if(index<0 || nitems<=index){ fxerror("%s::setItemSize: index out of range.\n",getClassName()); } 
  if(size<0) size=0;
  if(items[index]->size!=size){
    items[index]->size=size;
    recalc();
    }
  }


// Get item's size
FXint FXHeader::getItemSize(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemSize: index out of range.\n",getClassName()); } 
  return items[index]->size;
  }


// Get item's offset
FXint FXHeader::getItemOffset(FXint index) const {
  register FXint x,i;
  if(index<0 || nitems<=index){ fxerror("%s::getItemOffset: index out of range.\n",getClassName()); } 
  for(i=0,x=0; i<index; i++){x+=items[i]->getWidth(this);}
  return x;
  }


// Set item data
void FXHeader::setItemData(FXint index,void* ptr){
  if(index<0 || nitems<=index){ fxerror("%s::setItemData: index out of range.\n",getClassName()); } 
  items[index]->data=ptr;
  recalc();
  }


// Get item data
void* FXHeader::getItemData(FXint index) const {
  if(index<0 || nitems<=index){ fxerror("%s::getItemData: index out of range.\n",getClassName()); } 
  return items[index]->data;
  }


// Do layout
void FXHeader::layout(){
  
  // Force repaint
  update();
  
  // No more dirty
  flags&=~FLAG_DIRTY;
  }


// Handle repaint 
long FXHeader::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  FXint x,w,i;
  dc.setForeground(backColor);
  dc.fillRectangle(ev->rect.x,ev->rect.y,ev->rect.w,ev->rect.h);
  for(i=0,x=0; i<nitems; i++){
    w=items[i]->getWidth(this);
    if(ev->rect.x<x+w && x<ev->rect.x+ev->rect.w){
      items[i]->draw(this,dc,x,0,w,height);
      if(i==active && state){
        if(options&FRAME_THICK) drawDoubleSunkenRectangle(dc,x,0,w,height);
        else drawSunkenRectangle(dc,x,0,w,height);
        }
      else{
        if(options&FRAME_THICK) drawDoubleRaisedRectangle(dc,x,0,w,height);
        else drawRaisedRectangle(dc,x,0,w,height);
        }
      }
    x+=w;
    }
  if(x<width){
    if(options&FRAME_THICK) drawDoubleRaisedRectangle(dc,x,0,width-x,height);
    else drawRaisedRectangle(dc,x,0,width-x,height);
    }
  return 1;
  }



// Button being pressed
long FXHeader::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  FXEvent* ev=(FXEvent*)ptr;
  FXint i,x,w;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr)) return 1;
    for(i=0,x=0; i<nitems; i++){
      w=items[i]->getWidth(this);
      
      // Hit a header button?
      if((options&HEADER_BUTTON) && x+FUDGE<=ev->win_x && ev->win_x<x+w-FUDGE){
        active=i;
        activex=x;
        activew=w;
        state=TRUE;
        update(activex,0,activew,height);
        flags&=~FLAG_UPDATE;
        flags|=FLAG_PRESSED;
        break;
        }

      // Upper end of a button
      if(x+w-FUDGE<=ev->win_x && ev->win_x<x+w){
        setDragCursor(getApp()->hsplitCursor);
        active=i;
        activex=x;
        activew=w;
        flags|=FLAG_DODRAG;
        flags&=~FLAG_UPDATE;
        flags|=FLAG_PRESSED;
        break;
        }
      
      // Lower end of last button at this place
      if(x+w<=ev->win_x && ev->win_x<x+w+FUDGE){
        setDragCursor(getApp()->hsplitCursor);
        active=i;
        activex=x;
        activew=w;
        flags|=FLAG_DODRAG;
        flags&=~FLAG_UPDATE;
        flags|=FLAG_PRESSED;
        }
      
      
      // Hit nothing, next item
      x+=w;
      }
    
    // Dragging
    if(flags&FLAG_DODRAG){
      off=ev->win_x-activex-activew;
      if(!(options&HEADER_TRACKING)) drawSplit(activex+activew);
      }
    return 1;
    }
  return 0;
  }
  

// Button being released
long FXHeader::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_PRESSED;
  flags|=FLAG_UPDATE;
  if(isEnabled()){
    ungrab();
    if(target && target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr)) return 1;
    if(flags&FLAG_DODRAG){
      setDragCursor(getApp()->arrowCursor);
      if(!(options&HEADER_TRACKING)){
        drawSplit(activex+activew);
        setItemSize(active,activew);
        if(target) target->handle(this,MKUINT(message,SEL_CHANGED),(void*)active);
        }
      flags&=~FLAG_DODRAG;
      }
    else if(state){
      state=FALSE;
      update(activex,0,activew,height);
      if(target) target->handle(this,MKUINT(message,SEL_COMMAND),(void*)active);
      }
    return 1;
    }
  return 0;
  }


// Button being moved
long FXHeader::onMotion(FXObject*,FXSelector,void* ptr){
  FXEvent* ev=(FXEvent*)ptr;
  FXint i,x,w,oldsplit,newsplit;
  
  // Had the button pressed
  if(flags&FLAG_PRESSED){
    
    // We were dragging a split
    if(flags&FLAG_DODRAG){
      oldsplit=activex+activew;
      activew=ev->win_x-off-activex;
      if(activew<0) activew=0;
      newsplit=activex+activew;
      if(newsplit!=oldsplit){
        if(!(options&HEADER_TRACKING)){
          drawSplit(oldsplit);
          drawSplit(newsplit);
          }
        else{
          setItemSize(active,activew);
          if(target) target->handle(this,MKUINT(message,SEL_CHANGED),(void*)active);
          }
        }
      }
    
    // We pressed the button
    else{
      if(activex<=ev->win_x && ev->win_x<activex+activew && 0<=ev->win_y && ev->win_y<height){
        if(!state){
          state=TRUE;
          update(activex,0,activew,height);
          }
        }
      else{
        if(state){
          state=FALSE;
          update(activex,0,activew,height);
          }
        }
      }
    return 1;
    }
  
  // If over a split, show split cursor
  if(isEnabled()){
    for(i=0,x=0; i<nitems; i++){
      w=items[i]->getWidth(this);
      if(x+w-FUDGE<=ev->win_x && ev->win_x<x+w+FUDGE){
        setDefaultCursor(getApp()->hsplitCursor);
        return 1;
        }
      x+=w;
      }
    }
  
  // Not over a split, back to normal cursor
  setDefaultCursor(getApp()->arrowCursor);

  return 0;
  }


// The widget lost the grab for some reason
long FXHeader::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXFrame::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags&=~FLAG_CHANGED;
  flags|=FLAG_UPDATE;
  return 1;
  }


// Draw the split
void FXHeader::drawSplit(FXint pos){
  FXDCWindow dc(getParent());
  FXint px,py;
  translateCoordinatesTo(px,py,getParent(),pos,0);
  dc.clipChildren(FALSE);
  dc.setFunction(BLT_NOT_DST);
  dc.fillRectangle(px,0,2,getParent()->getHeight());
  }


// Change the font
void FXHeader::setFont(FXFont* fnt){
  if(!fnt){ fxerror("%s::setFont: NULL font specified.\n",getClassName()); }
  if(font!=fnt){
    font=fnt;
    recalc();
    update();
    }
  }


// Set text color
void FXHeader::setTextColor(FXColor clr){
  if(textColor!=clr){
    textColor=clr;
    update();
    }
  }


// Header style change
void FXHeader::setHeaderStyle(FXuint style){
  options=(options&~HEADER_MASK) | (style&HEADER_MASK);
  }


// Get header style
FXuint FXHeader::getHeaderStyle() const {
  return (options&HEADER_MASK); 
  }


// Save object to stream
void FXHeader::save(FXStream& store) const {
  register FXint i;
  FXFrame::save(store);
  store << nitems;
  for(i=0; i<nitems; i++){store<<items[i];}
  store << textColor;
  store << font;
  }

      

// Load object from stream
void FXHeader::load(FXStream& store){
  register FXint i;
  FXFrame::load(store);
  store >> nitems;
  FXRESIZE(&items,FXHeaderItem*,nitems);
  for(i=0; i<nitems; i++){store>>items[i];}
  store >> textColor;
  store >> font;
  }  


// Clean up
FXHeader::~FXHeader(){
  clearItems();
  items=(FXHeaderItem**)-1;
  font=(FXFont*)-1;
  }
