/********************************************************************************
*                                                                               *
*                    O p e n G L   C a n v a s   O b j e c t                    *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXGLCanvas.cpp,v 1.21 2000/02/23 07:29:23 jeroen Exp $                   *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXGLVisual.h"
#include "FXDrawable.h"
#include "FXWindow.h"
#include "FXCursor.h"
#include "FXGLCanvas.h"


/*
  Notes:
  - GL Context is part of the window, as it contains much stuff we don't share it with 
    other windows.
  - All GL canvas widgets which share a display lists are in a ring-like linked list.
    We do not know which of the widgets will get create()-ed first.  Each time we
    create a GL canvas, we just need to go through the list and find one which has
    already been created.  If none are found, then this window is the first one,
    and we do the appropriate thing.
  - When serialized, we remember which widgets were members of the list, so we
    can reconstruct everything.
  - OpenGL documentation says that in order for display lists to be shared, the
    visuals/pixelformats have to be identical; we test for this on the window whose
    display list we're trying to share.
*/

#define DISPLAY(app) ((Display*)((app)->display))

/*******************************************************************************/


// Map
FXDEFMAP(FXGLCanvas) FXGLCanvasMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXGLCanvas::onPaint),
  FXMAPFUNC(SEL_MOTION,0,FXGLCanvas::onMotion),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXGLCanvas::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXGLCanvas::onLeftBtnRelease),
  FXMAPFUNC(SEL_MIDDLEBUTTONPRESS,0,FXGLCanvas::onMiddleBtnPress),
  FXMAPFUNC(SEL_MIDDLEBUTTONRELEASE,0,FXGLCanvas::onMiddleBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXGLCanvas::onRightBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXGLCanvas::onRightBtnRelease),
  FXMAPFUNC(SEL_KEYPRESS,0,FXGLCanvas::onKeyPress),
  FXMAPFUNC(SEL_KEYRELEASE,0,FXGLCanvas::onKeyRelease),
  };


// Object implementation
FXIMPLEMENT(FXGLCanvas,FXWindow,FXGLCanvasMap,ARRAYNUMBER(FXGLCanvasMap))


// For serialization
FXGLCanvas::FXGLCanvas(){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  sgnext=this;
  sgprev=this;
  ctx=0;
  }


// Make a canvas
FXGLCanvas::FXGLCanvas(FXComposite* p,FXGLVisual *vis,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXWindow(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  visual=vis;
  target=tgt;
  message=sel;
  sgnext=this;
  sgprev=this;
  ctx=0;
  }


// Make a canvas sharing display lists
FXGLCanvas::FXGLCanvas(FXComposite* p,FXGLVisual *vis,FXGLCanvas* sharegroup,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h):
  FXWindow(p,opts,x,y,w,h){
  flags|=FLAG_ENABLED|FLAG_SHOWN;
  visual=vis;
  target=tgt;
  message=sel;
  if(sharegroup){
    sgnext=sharegroup;
    sgprev=sharegroup->sgprev;
    sharegroup->sgprev=this;
    sgprev->sgnext=this;
    }
  else{
    sgnext=this;
    sgprev=this;
    }
  ctx=0;
  }


#ifdef WIN32
const char* FXGLCanvas::GetClass() const { return "FXGLCanvas"; }
#endif


// Return TRUE if it is sharing display lists
FXbool FXGLCanvas::isShared() const { return sgnext!=this; }


// Canvas is an object drawn by another
long FXGLCanvas::onPaint(FXObject*,FXSelector,void* ptr){
  return target && target->handle(this,MKUINT(message,SEL_PAINT),ptr);
  }


// Handle buttons if not handled in derived class
long FXGLCanvas::onLeftBtnPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target) target->handle(this,MKUINT(message,SEL_LEFTBUTTONPRESS),ptr);
    return 1;
    }
  return 0;
  }

long FXGLCanvas::onLeftBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    if(target) target->handle(this,MKUINT(message,SEL_LEFTBUTTONRELEASE),ptr);
    return 1;
    }
  return 0;
  }

long FXGLCanvas::onMiddleBtnPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target) target->handle(this,MKUINT(message,SEL_MIDDLEBUTTONPRESS),ptr);
    return 1;
    }
  return 0;
  }

long FXGLCanvas::onMiddleBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    if(target) target->handle(this,MKUINT(message,SEL_MIDDLEBUTTONRELEASE),ptr);
    return 1;
    }
  return 0;
  }

long FXGLCanvas::onRightBtnPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  if(isEnabled()){
    handle(this,MKUINT(0,SEL_FOCUS_SELF),ptr);
    grab();
    if(target) target->handle(this,MKUINT(message,SEL_RIGHTBUTTONPRESS),ptr);
    return 1;
    }
  return 0;
  }

long FXGLCanvas::onRightBtnRelease(FXObject*,FXSelector,void* ptr){
  if(isEnabled()){
    ungrab();
    if(target) target->handle(this,MKUINT(message,SEL_RIGHTBUTTONRELEASE),ptr);
    return 1;
    }
  return 0;
  }


// Mouse moved
long FXGLCanvas::onMotion(FXObject*,FXSelector,void* ptr){
  return isEnabled() && target && target->handle(this,MKUINT(message,SEL_MOTION),ptr);
  }


// Handle keyboard press/release 
long FXGLCanvas::onKeyPress(FXObject*,FXSelector,void* ptr){
  flags&=~FLAG_TIP;
  return isEnabled() && target && target->handle(this,MKUINT(message,SEL_KEYPRESS),ptr);
  }


long FXGLCanvas::onKeyRelease(FXObject*,FXSelector,void* ptr){
  return isEnabled() && target && target->handle(this,MKUINT(message,SEL_KEYRELEASE),ptr);
  }


/*******************************************************************************/


// Create X window (GL CANVAS)
void FXGLCanvas::create(){
  FXGLCanvas *canvas;
  void *sharedctx=NULL;
  FXWindow::create();
#ifdef HAVE_OPENGL
  if(!ctx){
    if(!visual->info){ fxerror("%s::create(): visual unsuitable for OpenGL.\n",getClassName()); }
    if(sgnext!=this){
      
      // Find another member of the group which is already created, and get its context
      canvas=sgnext;
      while(canvas!=this){
        sharedctx=canvas->ctx;
        if(sharedctx) break;
        canvas=canvas->sgnext;
        }
      
      // The visuals have to match, the book says...
      if(sgnext->visual!=canvas->visual){ 
        fxerror("%s::create(): trying to share display lists with incompatible visuals\n",getClassName()); 
        }
      }
#ifndef WIN32
    // Make context
    ctx=glXCreateContext(DISPLAY(getApp()),(XVisualInfo*)visual->info,(GLXContext)sharedctx,TRUE);
    if(!ctx){ fxerror("%s::create(): glXCreateContext() failed.\n",getClassName()); }
#else
    // Make that the pixel format of the device context 
    HDC hdc=::GetDC((HWND)xid);
    if(!SetPixelFormat(hdc,visual->pixelformat,(PIXELFORMATDESCRIPTOR*)visual->info)){ 
      fxerror("%s::create(): SetPixelFormat() failed.\n",getClassName()); 
      }
    
    // Make context
    ctx=(void*)wglCreateContext(hdc);
    if(!ctx){ fxerror("%s::create(): wglCreateContext() failed.\n",getClassName()); }

    // I hope I didn't get this backward; the new context obviously has no
    // display lists yet, but the old one may have, as it has already been around
    // for a while.  If you see this fail and can't explain why, then that might
    // be what's going on.  Report this to fox-users@cfdrc.com
    if(sharedctx && !wglShareLists((HGLRC)sharedctx,(HGLRC)ctx)){ fxerror("%s::create(): wglShareLists() failed.\n",getClassName()); }
    ::ReleaseDC((HWND)xid,hdc);
#endif
    }
#endif
  }


// Detach the GL Canvas
void FXGLCanvas::detach(){
#ifdef HAVE_OPENGL
  if(ctx){
    // Will this leak memory?
    ctx=0;
    }
#endif
  FXWindow::detach();
  }


// Destroy the GL Canvas
void FXGLCanvas::destroy(){
#ifdef HAVE_OPENGL
  if(ctx){
#ifndef WIN32
    glXDestroyContext(DISPLAY(getApp()),(GLXContext)ctx);
#else
    wglDeleteContext((HGLRC)ctx);
#endif
    ctx=0;
    }
#endif
  FXWindow::destroy();
  }


//  Make the rendering context of GL Canvas current
FXbool FXGLCanvas::makeCurrent(){
#ifdef HAVE_OPENGL
  if(ctx){
#ifndef WIN32
    return glXMakeCurrent(DISPLAY(getApp()),xid,(GLXContext)ctx);
#else
    HDC hdc=::GetDC((HWND)xid);
    if(visual->hPalette){
      SelectPalette(hdc,(HPALETTE)visual->hPalette,FALSE);
      RealizePalette(hdc);
      }
    BOOL bStatus=wglMakeCurrent(hdc,(HGLRC)ctx);
    return bStatus;
#endif
    }
#endif
  return FALSE;
  }
  
  
//  Make the rendering context of GL Canvas current
FXbool FXGLCanvas::makeNonCurrent(){
#ifdef HAVE_OPENGL
  if(ctx){
#ifndef WIN32
    return glXMakeCurrent(DISPLAY(getApp()),None,(GLXContext)NULL);
#else
    HDC hdc=wglGetCurrentDC();
    BOOL bStatus=wglMakeCurrent(NULL,NULL);
//    ::ReleaseDC((HWND)xid,hdc);
    return bStatus;
#endif
    }
#endif
  return FALSE;
  }


// Used by GL to swap the buffers in double buffer mode, or flush a single buffer
void FXGLCanvas::swapBuffers(){
#ifdef HAVE_OPENGL
#ifndef WIN32
  glXSwapBuffers(DISPLAY(getApp()),xid);
#else
  SwapBuffers(wglGetCurrentDC());
#endif
#endif
  }


// Save object to stream
void FXGLCanvas::save(FXStream& store) const {
  FXWindow::save(store);
  store << sgnext;
  store << sgprev;
  }

      
// Load object from stream
void FXGLCanvas::load(FXStream& store){
  FXWindow::load(store);
  store >> sgnext;
  store >> sgprev;
  }


// Close and release any resources
FXGLCanvas::~FXGLCanvas(){
  sgnext->sgprev=sgprev;
  sgprev->sgnext=sgnext;
  sgnext=(FXGLCanvas*)-1;
  sgprev=(FXGLCanvas*)-1;
#ifdef HAVE_OPENGL
  if(ctx){
#ifndef WIN32
    glXDestroyContext(DISPLAY(getApp()),(GLXContext)ctx);
#else
    wglDeleteContext((HGLRC)ctx);
#endif
    }
#endif
  }
