/********************************************************************************
*                                                                               *
*      F i l e   I n f o r m a t i o n   a n d   M a n i p u l a t i o n        *
*                                                                               *
*********************************************************************************
* Copyright (C) 2000 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* Contributed by: Sean Hubbell                                                  *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXFile.cpp,v 1.20 2000/05/01 15:15:22 gui Exp $                       *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXFile.h"

#ifdef __CYGWIN__
#define _MAX_PATH MAX_PATH
#endif


/*
  Notes:
  - Windows flavors of some of these functions are not perfect yet.
*/

/*******************************************************************************/

// Get current working directory
FXString FXFile::getCurrentDirectory(){
  FXchar buffer[MAXPATHLEN];
#ifndef WIN32
  if(getcwd(buffer,MAXPATHLEN)) return FXString(buffer);
#else
  if(GetCurrentDirectory(MAXPATHLEN,buffer)) return FXString(buffer);
#endif
  return "";
  }


// Change current directory
FXbool FXFile::setCurrentDirectory(const FXString& path){
#ifdef WIN32
  return !path.empty() && SetCurrentDirectory(path.text());
#else
  return !path.empty() && chdir(path.text())==0;
#endif
  }


// Get current drive prefix "a:", if any
// This is the same method as used in VC++ CRT.
FXString FXFile::getCurrentDrive(){
#ifdef WIN32
  FXchar buffer[MAXPATHLEN];
  if(GetCurrentDirectory(MAXPATHLEN,buffer) && isalpha(buffer[0]) && buffer[1]==':'){ 
    buffer[0]=tolower(buffer[0]);
    return FXString(buffer,2);
    }
#endif
  return "";
  }


// Change current drive prefix "a:"
// This is the same method as used in VC++ CRT.
FXbool FXFile::setCurrentDrive(const FXString& prefix){
#ifdef WIN32
  FXchar buffer[3];
  if(!prefix.empty() && isalpha(prefix[0]) && prefix[1]==':'){
    buffer[0]=prefix[0];
    buffer[1]=':';
    buffer[2]='\0';
    return SetCurrentDirectory(buffer);
    }
  return FALSE;
#else
  return TRUE;
#endif
  }
  

// Get user's home directory
FXString FXFile::getHomeDirectory(){
  static FXchar* home=NULL;
  if(!home){
#ifdef WIN32
    HKEY hKey;
    char    szEnv[_MAX_PATH+5];     // "HOME="
    DWORD   dwSize=_MAX_PATH;
    if(RegOpenKeyEx(HKEY_CURRENT_USER,"Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders",0,KEY_READ,&hKey)==ERROR_SUCCESS){
      strcpy(szEnv,"HOME=");
      
      // Change "Personal" to "Desktop" if you want...
      if(RegQueryValueEx(hKey,"Personal",NULL,NULL,(LPBYTE)szEnv+5,&dwSize)==ERROR_SUCCESS){
#ifdef _MSC_VER
        _putenv(szEnv);
#else
        putenv(szEnv);
#endif
        }
      RegCloseKey(hKey);
      }
#endif
    home=getenv("HOME");
    if(!home) home="";
    }
  return FXString(home);
  }


// Get executable path 
FXString FXFile::getExecPath(){
  return FXString(getenv("PATH"));
  }
  

// Return directory part of pathname, assuming full pathname
FXString FXFile::directory(const FXString& file){
  register FXint n,i;
  if(!file.empty()){
    i=0;
#ifdef WIN32
    if(isalpha(file[0]) && file[1]==':') i=2;
#endif
    if(file[i]==PATHSEP) i++;
    n=i;
    while(file[i]){
      if(file[i]==PATHSEP) n=i; 
      i++;
      }
    return FXString(file.text(),n);
    }
  return "";
  }


// Return name and extension part of pathname
FXString FXFile::name(const FXString& file){
  register FXint f,n;
  if(!file.empty()){
    n=0;
#ifdef WIN32
    if(isalpha(file[0]) && file[1]==':') n=2;
#endif
    f=n;
    while(file[n]){
      if(file[n++]==PATHSEP) f=n; 
      }
    return FXString(file.text()+f,n-f);
    }
  return "";
  }


// Return file title, i.e. document name only
FXString FXFile::title(const FXString& file){
  register FXint f,e;
  if(!file.empty()){
    e=0;
#ifdef WIN32
    if(isalpha(file[0]) && file[1]==':') e=2;
#endif
    f=e;
    while(file[e]){
      if(file[e++]==PATHSEP) f=e; 
      }
    e=f;
    while(file[e]){
      if(file[e]=='.') break; 
      e++;
      }
    return FXString(file.text()+f,e-f);
    }
  return "";
  }


// Return extension
FXString FXFile::extension(const FXString& file){
  register FXint n,e;
  if(!file.empty()){
    n=e=file.length();
    while(0<e && file[e-1]!=PATHSEP){
      if(file[e-1]=='.'){
        return FXString(file.text()+e,n-e);
        }
      e--;
      }
    }
  return "";
  }


// Return drive letter prefix "c:"
FXString FXFile::drive(const FXString& file){
#ifdef WIN32
  FXchar buffer[3];
  if(!file.empty() && isalpha(file[0]) && file[1]==':'){
    buffer[0]=tolower(file[0]);
    buffer[1]=':';
    buffer[2]='\0';
    return FXString(buffer);
    }
#endif
  return "";
  }


// Perform tilde expansion
FXString FXFile::expand(const FXString& file){
#ifndef WIN32
  FXint len=file.length();
  if(0<len && file[0]=='~'){
    struct passwd *pwd;
    FXint i=1;
    while(i<len && file[i]!=PATHSEP) i++;
    if(i==1){
      pwd=getpwuid(getuid());
      }
    else{
      pwd=getpwnam(file.mid(1,i-1).text());
      }
    if(pwd && pwd->pw_dir){
      return pwd->pw_dir+file.mid(i,len-i);
      }
    }
#else
  if(!file.empty() && file[0]=='~' && (file[1]==PATHSEP || file[1]=='\0')){
    return getHomeDirectory()+file.mid(1,MAXPATHLEN);
    }
#endif
  return file;
  }


// Simplify a file path; the path will remain relative if it was relative
FXString FXFile::simplify(const FXString& file){
  if(!file.empty()){
    FXString result=file;
    FXint beg,end,head,cur;
    head=0;
#ifndef WIN32
    if(result[0]==PATHSEP){ 
      head++; 
      }
    else{ 
      while(result[head]=='.' && result[head+1]=='.' && result[head+2]==PATHSEP) head+=3; 
      if(result[head]=='.' && result[head+1]=='.' && result[head+2]==0) head+=2;
      }
#else
    if(result[0]==PATHSEP && result[1]==PATHSEP){    // UNC
      head+=2; 
      }
    else if(isalpha(result[0]) && result[1]==':' && result[2]==PATHSEP){ 
      head+=3; 
      }
    else if(result[0]==PATHSEP){ 
      head++; 
      }
    else{ 
      while(result[head]=='.' && result[head+1]=='.' && result[head+2]==PATHSEP) head+=3; 
      if(result[head]=='.' && result[head+1]=='.' && result[head+2]==0) head+=2;
      }
#endif
    for(beg=cur=head; file[beg]; beg=end){
      while(file[beg]==PATHSEP) beg++;
      for(end=beg; file[end] && file[end]!=PATHSEP; end++);
      if(beg==end) break;
      if(beg+1==end && file[beg]=='.') continue;
      if(beg+2==end && file[beg]=='.' && file[beg+1]=='.'){
        while(head<cur && result[cur-1]!=PATHSEP) cur--;
        continue;
        }
      if(head<cur && result[cur-1]!=PATHSEP) result[cur++]=PATHSEP;
      while(beg<end){ result[cur++]=file[beg++]; }
      }
    while(head<cur && result[cur-1]==PATHSEP) cur--;
    result[cur]=0;
    return result;
    }
  return "";
  }


// Build absolute pathname
FXString FXFile::absolute(const FXString& file){
  FXString pathfile=FXFile::simplify(FXFile::expand(file));
#ifndef WIN32
  if(pathfile[0]==PATHSEP) return pathfile;
#else
  if(pathfile[0]==PATHSEP && pathfile[1]==PATHSEP) return pathfile;   // UNC
  if(isalpha(pathfile[0]) && pathfile[1]==':' && pathfile[2]==PATHSEP) return pathfile;
  if(pathfile[0]==PATHSEP) return getCurrentDrive()+pathfile;
#endif
  return FXFile::simplify(getCurrentDirectory()+PATHSEPSTRING+pathfile);
  }


// Build absolute pathname from parts
FXString FXFile::absolute(const FXString& base,const FXString& file){
  FXString pathfile=FXFile::simplify(FXFile::expand(file));
#ifndef WIN32
  if(pathfile[0]==PATHSEP) return pathfile;
#else
  if(pathfile[0]==PATHSEP && pathfile[1]==PATHSEP) return pathfile;   // UNC
  if(isalpha(pathfile[0]) && pathfile[1]==':' && pathfile[2]==PATHSEP) return pathfile;
  if(pathfile[0]==PATHSEP) return getCurrentDrive()+pathfile;
#endif
  return FXFile::simplify(absolute(base)+PATHSEPSTRING+pathfile);
  }
  

// Search pathlist for file
FXString FXFile::search(const FXString& pathlist,const FXString& file){
  FXString pathfile=FXFile::simplify(FXFile::expand(file));
  FXString path;
  FXint beg,end;
#ifndef WIN32
  if(pathfile[0]==PATHSEP){
    if(exists(pathfile)) return pathfile;
    return "";
    }
#else
  if(pathfile[0]==PATHSEP && pathfile[1]==PATHSEP){
    if(exists(pathfile)) return pathfile;   // UNC
    return "";
    }
  if(isalpha(pathfile[0]) && pathfile[1]==':' && pathfile[2]==PATHSEP){
    if(exists(pathfile)) return pathfile;
    return "";
    }
  if(pathfile[0]==PATHSEP){
    pathfile=getCurrentDrive()+pathfile;
    if(exists(pathfile)) return pathfile;
    return "";
    }
#endif
  for(beg=0; pathlist[beg]; beg=end){
    while(pathlist[beg]==PATHLISTSEP) beg++;
    for(end=beg; pathlist[end] && pathlist[end]!=PATHLISTSEP; end++);
    if(beg==end) break;
    path=absolute(pathlist.mid(beg,end-beg),pathfile);
    if(exists(path)) return path;
    }
  return "";
  }


// Up one level, given absolute path
FXString FXFile::upLevel(const FXString& file){
  if(!file.empty()){
    FXint beg=0;
    FXint end=file.length();
#ifndef WIN32
    if(file[0]==PATHSEP) beg++;
#else
    if(file[0]==PATHSEP) beg++;
    else if(file[0]==PATHSEP && file[1]==PATHSEP) beg+=2;   // UNC
    else if(isalpha(file[0]) && file[1]==':' && file[2]==PATHSEP) beg+=3;
#endif
    if(beg<end && file[end-1]==PATHSEP) end--;
    while(beg<end && file[--end]!=PATHSEP);
    return file.mid(0,end);
    }
  return file;
  }


// Check if file represents absolute pathname
FXbool FXFile::isAbsolute(const FXString& file){
#ifndef WIN32
  return !file.empty() && file[0]==PATHSEP;
#else
  return !file.empty() && ((file[0]==PATHSEP && file[1]==PATHSEP) || (isalpha(file[0]) && file[1]==':' && file[2]==PATHSEP));
#endif
  }


// Does string represent topmost directory
FXbool FXFile::isTopDirectory(const FXString& file){
#ifndef WIN32
  return !file.empty() && (file[0]==PATHSEP && file[1]=='\0');
#else
  return !file.empty() && (file[0]==PATHSEP && file[1]=='\0') || (isalpha(file[0]) && file[1]==':' && file[2]==PATHSEP && file[3]=='\0');
#endif
  }


// Check if string represents a file
FXbool FXFile::isFile(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && S_ISREG(status.st_mode);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Check if string represents a link
FXbool FXFile::isLink(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && S_ISLNK(status.st_mode);
#else
  return FALSE;
#endif
  }

// Check if string represents a directory
FXbool FXFile::isDirectory(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && S_ISDIR(status.st_mode);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && (atts&FILE_ATTRIBUTE_DIRECTORY);
#endif
  }


// Check if owner has full permissions
FXbool FXFile::isOwnerReadWriteExecute(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IRUSR) && (status.st_mode&S_IWUSR) && (status.st_mode&S_IXUSR);
#else
  return TRUE;
#endif
  }


// Check if owner can read 
FXbool FXFile::isOwnerReadable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IRUSR);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF);
#endif
  }


// Check if owner can write
FXbool FXFile::isOwnerWritable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && stat(file.text(),&status)==0 && (status.st_mode&S_IWUSR);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_READONLY);
#endif
  }


// Check if owner can execute
FXbool FXFile::isOwnerExecutable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IXUSR);
#else
  return TRUE;
#endif
  }


// Check if group has full permissions
FXbool FXFile::isGroupReadWriteExecute(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IRGRP) && (status.st_mode&S_IWGRP) && (status.st_mode&S_IXGRP);
#else
  return TRUE;
#endif
  }


// Check if group can read
FXbool FXFile::isGroupReadable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IRGRP);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF);
#endif
  }


// Check if group can write
FXbool FXFile::isGroupWritable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IWGRP);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_READONLY);
#endif
  }


// Check if group can execute
FXbool FXFile::isGroupExecutable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IXGRP);
#else
  return TRUE;
#endif
  }


// Check if everybody has full permissions
FXbool FXFile::isOtherReadWriteExecute(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IROTH) && (status.st_mode&S_IWOTH) && (status.st_mode&S_IXOTH);
#else
  return TRUE;
#endif
  }


// Check if everybody can read
FXbool FXFile::isOtherReadable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IROTH);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF);
#endif
  }


// Check if everybody can write
FXbool FXFile::isOtherWritable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IWOTH);
#else
  DWORD atts;
  return file.text() && ((atts=GetFileAttributes(file.text()))!=0xFFFFFFFF) && !(atts&FILE_ATTRIBUTE_READONLY);
#endif
  }


// Check if everybody can execute
FXbool FXFile::isOtherExecutable(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) && (status.st_mode&S_IXOTH);
#else
  return TRUE;
#endif
  }


// Return time file was last modified
long FXFile::modified(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)status.st_mtime : 0L;  
#else
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)status.st_mtime : 0L;  
#endif
  }


// Return time file was last accessed
long FXFile::accessed(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)status.st_atime : 0L;  
#else
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)status.st_atime : 0L;  
#endif
  }


// Return time when created
long FXFile::created(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)status.st_ctime : 0L;  
#else
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)status.st_ctime : 0L;  
#endif
  }


// Return time when touched
long FXFile::touched(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)FXMAX3(status.st_ctime,status.st_atime,status.st_mtime) : 0L;  
#else
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (long)FXMAX3(status.st_ctime,status.st_atime,status.st_mtime) : 0L;  
#endif
  }


// Get file info
FXbool FXFile::info(const FXString& file,struct stat& info){
#ifndef WIN32
  return file.text() && (stat(file.text(),&info)==0);  
#else
  return file.text() && (stat(file.text(),&info)==0);  
#endif
  }


// Get file size
unsigned long FXFile::size(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (unsigned long)status.st_size : 0L;  
#else
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? (unsigned long)status.st_size : 0L;  
#endif
  }


FXbool FXFile::exists(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0);   
#else
  return file.text() && (GetFileAttributes(file.text())!=0xFFFFFFFF);
#endif
  }


FXuint FXFile::mode(const FXString& file){
#ifndef WIN32
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? status.st_mode : 0;  
#else
  struct stat status;
  return file.text() && (stat(file.text(),&status)==0) ? status.st_mode : 0;  
#endif
  }


