/********************************************************************************
*                                                                               *
*                          D i c t i o n a r y    C l a s s                     *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXDict.cpp,v 1.4 2000/02/20 19:17:50 jeroen Exp $                        *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXObject.h"
#include "FXDict.h"

  
/*
  Notes:
  - The hash algorithm should yield a number in the range [0...FXDict::EMPTY)
    We need FXDict::EMPTY and FXDict::UNUSED for flag purposes.
  - Since the algorithm doubles the table size when exceeding MAX_LOAD,
    it would be prudent to keep MIN_LOAD less than .5*MAX_LOAD;
    otherwise, the algorithm might hip-hop between halving and doubling,
    which would be quite expensive!!
  - Not many people seem to know that hash tables don't have to be prime
    numbers; in fact, a table size of 2**n and odd probe distance are very
    easy to arrange, and this works just as well!
  - We store the hash key, so that 99.999% of the time we can compare hash numbers;
    only when hash numbers match do we need to compare keys.
    Thus, with a good hash function, the number of calls to strcmp() should be 
    roughly the same as the number of successful lookups.
  - The hash table should NEVER get full, or stuff will loop forever!!
*/


#define DEF_HASH_SIZE      4                // Initial table size (MUST be power of 2)
#define MAX_LOAD           80               // Maximum hash table load factor (%)
#define MIN_LOAD           10               // Minimum hash table load factor (%)
#define HASH1(x,n) (((unsigned int)(x)*13)%(n))           // Probe Position [0..n-1]
#define HASH2(x,n) (1|(((unsigned int)(x)*17)%((n)-1)))   // Probe Distance [1..n-1]



/*******************************************************************************/


// Object implementation
FXIMPLEMENT(FXDict,FXObject,NULL,0)


// Construct empty dictionary
FXDict::FXDict(){
  FXMALLOC(&dict,FXDictEntry,DEF_HASH_SIZE);
  for(FXuint i=0; i<DEF_HASH_SIZE; i++) dict[i].hash=-1;
  total=DEF_HASH_SIZE;
  number=0;
  }


// Defaul implementation 
void *FXDict::createData(const void* ptr){ return (void*)ptr; }


// Defaul implementation 
void FXDict::deleteData(void*){ }


// Resize table
void FXDict::size(FXint m){
  register FXint i,n,p,x,h;
  FXDictEntry *k;
  FXASSERT(number<=total);
  if(m<DEF_HASH_SIZE) m=DEF_HASH_SIZE;
  n=total;
  while((n>>2)>m) n>>=1;            // Shrink until n/4 <= m
  while((n>>1)<m) n<<=1;            // Grow until m <= n/2
  FXASSERT(m<=(n>>1));
  FXASSERT(DEF_HASH_SIZE<=n);
  if(n!=total){
    FXTRACE((200,"FXDict::size: %08x: resizing from %d to %d\n",this,total,n));
    FXASSERT(m<=n);
    FXMALLOC(&k,FXDictEntry,n);
    for(i=0; i<n; i++) k[i].hash=-1;
    for(i=0; i<total; i++){
      h=dict[i].hash;
      if(0<=h){
        p=HASH1(h,n);
        FXASSERT(0<=p && p<n);
        x=HASH2(h,n);
        FXASSERT(1<=x && x<n);
        while(k[p].hash!=-1) p=(p+x)%n;
        FXASSERT(k[p].hash<0);
        k[p]=dict[i];
        }
      }
    FXFREE(&dict);
    dict=k;
    total=n;
    }
  }


// Insert a new entry, leave it alone if already existing
void* FXDict::insert(const FXchar* ky,const void* pdata,FXbool mrk){
  register FXint p,i,x,h,n;
  register void *ptr;
  if(!ky){ fxerror("FXDict::insert: NULL key argument.\n"); }
  FXASSERT(number<total);
  h=fxstrhash(ky);
  FXASSERT(0<=h);
  p=HASH1(h,total);
  FXASSERT(0<=p && p<total);
  x=HASH2(h,total);
  FXASSERT(1<=x && x<total);
  i=-1;
  n=total;
  while(n && dict[p].hash!=-1){
    if((i==-1)&&(dict[p].hash==-2)) i=p;
    if(dict[p].hash==h && strcmp(dict[p].key,ky)==0){
      return dict[p].data;
      }
    p=(p+x)%total;
    n--;
    }
  if(i==-1) i=p;
  FXTRACE((200,"FXDict::insert: %08x: inserting: \"%s\"\n",this,ky));
  FXASSERT(0<=i && i<total);
  FXASSERT(dict[i].hash<0);
  ptr=createData(pdata);
  dict[i].hash=h;
  dict[i].mark=mrk;
  dict[i].key=fxstrdup(ky);
  dict[i].data=ptr;
  number++;
  if((100*number)>=(MAX_LOAD*total)) size(number);
  FXASSERT(number<total);
  return ptr;
  }


// Add or replace entry
void* FXDict::replace(const FXchar* ky,const void* pdata,FXbool mrk){
  register FXint p,i,x,h,n;
  register void *ptr;
  if(!ky){ fxerror("FXDict::replace: NULL key argument.\n"); }
  FXASSERT(number<total);
  h=fxstrhash(ky);
  FXASSERT(0<=h);
  p=HASH1(h,total);
  FXASSERT(0<=p && p<total);
  x=HASH2(h,total);
  FXASSERT(1<=x && x<total);
  i=-1;
  n=total;
  while(n && dict[p].hash!=-1){
    if((i==-1)&&(dict[p].hash==-2)) i=p;
    if(dict[p].hash==h && strcmp(dict[p].key,ky)==0){
      if(dict[p].mark<=mrk){
        FXTRACE((200,"FXDict::replace: %08x: replacing: \"%s\"\n",this,ky));
        deleteData(dict[p].data);
        dict[p].mark=mrk;
        dict[p].data=createData(pdata);
        }
      return dict[p].data;
      }
    p=(p+x)%total;
    n--;
    }
  if(i==-1) i=p;
  FXTRACE((200,"FXDict::replace: %08x: inserting: \"%s\"\n",this,ky));
  FXASSERT(0<=i && i<total);
  FXASSERT(dict[i].hash<0);
  ptr=createData(pdata);
  dict[i].hash=h;
  dict[i].mark=mrk;
  dict[i].key=fxstrdup(ky);
  dict[i].data=ptr;
  number++;
  if((100*number)>=(MAX_LOAD*total)) size(number);
  FXASSERT(number<total);
  return ptr;
  }


// Remove entry
void* FXDict::remove(const FXchar* ky){
  register FXint p,x,h,n;
  if(!ky){ fxerror("FXDict::remove: NULL key argument.\n"); }
  if(0<number){
    h=fxstrhash(ky);
    FXASSERT(0<=h);
    p=HASH1(h,total);
    FXASSERT(0<=p && p<total);
    x=HASH2(h,total);
    FXASSERT(1<=x && x<total);
    FXASSERT(number<total);
    n=total;
    while(n && dict[p].hash!=-1){
      if(dict[p].hash==h && strcmp(dict[p].key,ky)==0){
        FXTRACE((120,"FXDict::remove: %08x removing: \"%s\"\n",this,ky));
        dict[p].hash=-2;
        dict[p].mark=FALSE;
        FXFREE(&dict[p].key);
        deleteData(dict[p].data);
        dict[p].key=NULL;
        dict[p].data=NULL;
        number--;
        if((100*number)<=(MIN_LOAD*total)) size(number);
        FXASSERT(number<total);
        return NULL;
        }
      p=(p+x)%total;
      n--;
      }
    }
  return NULL;
  }


// Find entry
void* FXDict::find(const FXchar* ky){
  register FXint p,x,h,n;
  if(!ky){ fxerror("FXDict::find: NULL key argument.\n"); }
  if(0<number){
    h=fxstrhash(ky);
    FXASSERT(0<=h);
    p=HASH1(h,total);
    FXASSERT(0<=p && p<total);
    x=HASH2(h,total);
    FXASSERT(1<=x && x<total);
    FXASSERT(number<total);
    n=total;
    while(n && dict[p].hash!=-1){
      if(dict[p].hash==h && strcmp(dict[p].key,ky)==0){
        return dict[p].data;
        }
      p=(p+x)%total;
      n--;
      }
    }
  return NULL;
  }


// Get first non-empty entry
FXint FXDict::first() const {
  register FXint pos=0;
  while(pos<total){ if(0<=dict[pos].hash) break; pos++; }
  FXASSERT(total<=pos || 0<=dict[pos].hash);
  return pos;
  }


// Get last non-empty entry
FXint FXDict::last() const {
  register FXint pos=total-1;
  while(0<=pos){ if(0<=dict[pos].hash) break; pos--; }
  FXASSERT(pos<0 || 0<=dict[pos].hash);
  return pos;
  }


// Find next entry
FXint FXDict::next(FXint pos) const {
  FXASSERT(0<=pos && pos<total);
  while(++pos <= total-1){ if(0<=dict[pos].hash) break; }
  FXASSERT(total<=pos || 0<=dict[pos].hash);
  return pos;
  } 


// Find previous entry
FXint FXDict::prev(FXint pos) const {
  FXASSERT(0<=pos && pos<total);
  while(--pos >= 0){ if(0<=dict[pos].hash) break; }
  FXASSERT(pos<0 || 0<=dict[pos].hash);
  return pos;
  } 


// Remove all
void FXDict::clear(){
  register FXint i;
  for(i=0; i<total; i++){
    if(dict[i].hash>=0){ 
      FXFREE(&dict[i].key); 
      deleteData(dict[i].data);
      }
    }
  FXFREE(&dict);
  total=0;
  number=0;
  }


// Save data
void FXDict::save(FXStream& store) const {
  FXObject::save(store);
  store << total;
  store << number;
  for(FXint i=0; i<total; i++){
    store << dict[i].hash;
    if(dict[i].hash>=0){
      FXuint len=strlen(dict[i].key);
      store << len;
      store << dict[i].mark;
      store.save(dict[i].key,len);
      }
    }
  }


// Load data
void FXDict::load(FXStream& store){
  FXObject::load(store);
  store >> total;
  store >> number;
  for(FXint i=0; i<total; i++){
    store >> dict[i].hash;
    if(dict[i].hash>=0){
      FXuint len;
      store >> len;
      store >> dict[i].mark;
      FXMALLOC(&dict[i].key,FXchar,len+1);
      store.load(dict[i].key,len);
      dict[i].key[len]='\0';
      }
    }
  }


// Destroy table
FXDict::~FXDict(){
  clear();
  dict=(FXDictEntry*)-1;
  }


/*******************************************************************************/

// Object implementation
FXIMPLEMENT(FXStringDict,FXDict,NULL,0)

  
// Construct string dict
FXStringDict::FXStringDict(){
  FXTRACE((100,"FXStringDict::FXStringDict %08x\n",this));
  }


// Create string 
void *FXStringDict::createData(const void* ptr){
  FXint len=strlen((char*)ptr)+1; void *p;
  FXMALLOC(&p,FXchar,len);
  memcpy(p,ptr,len);
  FXASSERT(p);
  return p;
  }


// Delete string 
void FXStringDict::deleteData(void* ptr){
  FXASSERT(ptr);
  FXFREE(&ptr);
  }


// Save data
void FXStringDict::save(FXStream& store) const {
  FXDict::save(store);
  FXASSERT(0);
  }


// Load data
void FXStringDict::load(FXStream& store){ 
  FXDict::load(store);
  FXASSERT(0);
  }


// Destructor
FXStringDict::~FXStringDict(){
  FXTRACE((100,"FXStringDict::~FXStringDict %08x\n",this));
  clear();
  }


/*******************************************************************************/

// Object implementation
FXIMPLEMENT(FXSectionDict,FXDict,NULL,0)

  
// Construct string dict
FXSectionDict::FXSectionDict(){
  FXTRACE((100,"FXSectionDict::FXSectionDict %08x\n",this));
  }


// Create data 
void *FXSectionDict::createData(const void*){
  FXStringDict *ptr=new FXStringDict;
  FXASSERT(ptr);
  return ptr;
  }


// Delete data 
void FXSectionDict::deleteData(void* ptr){
  FXASSERT(ptr);
  delete ((FXStringDict*)ptr);
  }


// Save data
void FXSectionDict::save(FXStream& store) const {
  FXDict::save(store);
  FXASSERT(0);
  }


// Load data
void FXSectionDict::load(FXStream& store){ 
  FXDict::load(store);
  FXASSERT(0);
  }


// Destructor
FXSectionDict::~FXSectionDict(){
  FXTRACE((100,"FXSectionDict::~FXSectionDict %08x\n",this));
  clear();
  }
