/********************************************************************************
*                                                                               *
*      D e v i c e   C o n t e x t   D r a w i n g   o n   a   W i n d o w      *
*                                                                               *
*********************************************************************************
* Copyright (C) 1999 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXDCWindow.cpp,v 1.49 2000/03/06 22:12:29 jeroen Exp $                   *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "fxkeys.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXVisual.h"
#include "FXFont.h"
#include "FXCursor.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXBitmap.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXComposite.h"
#include "FXRootWindow.h"
#include "FXShell.h"
#include "FXRegion.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXProfiler.h"
  
/*
  Notes:
  
  - Associate a DC with a surface before you begin using it:
  
     long SomeWidget::onPaint(FXObject*,FXSelector,void* ptr){
       FXDCWindow dc(this,ptr);
       dc.drawLine(...);
       ... jadajadajada ...
       return 1;
       }

    The association is automatically broken when you go out of scope; the
    destructor of the FXDCWindow does this for you.

  - Optimizations: only perform style/attribute changes just before an actual
    drawing command takes place:- X-Windows apparently already does this;
    MS-Windows also?
      
  - We assume the following initial state:
      
    BLIT Function:        BLT_SRC
    Foreground:           black (0)
    Background:           white (1)
    Line Width:           0 (meaning thinnest/fastest, no guaranteed pixelation)
    Cap Style:            CAP_BUTT
    Join Style:           JOIN_MITER
    Line Style:           LINE_SOLID
    Fill Style:           FILL_SOLID
    Fill Rule:            RULE_EVEN_ODD
    Font:                 None
    Other Paremeters:     To Be Determined
    
  - Under X-Windows, end() will restore the GC to the state above; flags 
    keeps track of which changes have been made to minimize the necessary 
    updating.
    
  - Under X, graphics_exposures should be OFF:- at least some SGI IRIX machines
    have broken implementations of graphics_exposures.
*/

#define DISPLAY(app) ((Display*)((app)->display))


/********************************************************************************
*                                    X-Windows                                  *
********************************************************************************/

#ifndef WIN32


// Construct for expose event painting
FXDCWindow::FXDCWindow(FXDrawable* drawable,FXEvent* event):FXDC(drawable->getApp()){
  begin(drawable);
  rect.x=clip.x=event->rect.x;
  rect.y=clip.y=event->rect.y;
  rect.w=clip.w=event->rect.w;
  rect.h=clip.h=event->rect.h;
  XSetClipRectangles(DISPLAY(getApp()),(GC)gc,0,0,(XRectangle*)&clip,1,Unsorted);
  flags|=GCClipMask;
  }


// Construct for normal painting
FXDCWindow::FXDCWindow(FXDrawable* drawable):FXDC(drawable->getApp()){
  begin(drawable);
  }

  
// Destruct
FXDCWindow::~FXDCWindow(){
  end();
  }


// Begin locks in a drawable surface
void FXDCWindow::begin(FXDrawable *drawable){
  if(!drawable){ fxerror("FXDCWindow::begin: NULL drawable.\n"); }
  if(!drawable->id()){ fxerror("FXDCWindow::begin: drawable not created yet.\n"); }
  surface=drawable;
  visual=drawable->getVisual();
  rect.x=clip.x=0;
  rect.y=clip.y=0;
  rect.w=clip.w=drawable->getWidth();
  rect.h=clip.h=drawable->getHeight();
  devfg=~0;
  devbg=0;
  gc=visual->gc;
  flags=0;
  }


// End unlock the drawable surface; restore it
void FXDCWindow::end(){
  XGCValues gcv;
  if(flags){
    if(flags&GCFunction) gcv.function=BLT_SRC;
    if(flags&GCForeground) gcv.foreground=BlackPixel(DISPLAY(getApp()),DefaultScreen(DISPLAY(getApp())));
    if(flags&GCBackground) gcv.background=WhitePixel(DISPLAY(getApp()),DefaultScreen(DISPLAY(getApp())));
    if(flags&GCLineWidth) gcv.line_width=0;
    if(flags&GCCapStyle) gcv.cap_style=CAP_BUTT;
    if(flags&GCJoinStyle) gcv.join_style=JOIN_MITER;
    if(flags&GCLineStyle) gcv.line_style=LINE_SOLID;
    if(flags&GCFillStyle) gcv.fill_style=FILL_SOLID;
    if(flags&GCStipple) gcv.stipple=getApp()->stipples[STIPPLE_WHITE];    // Needed for IRIX6.4 bug workaround!
    if(flags&GCFillRule) gcv.fill_rule=RULE_EVEN_ODD;
    if(flags&GCFont) gcv.font=getApp()->getNormalFont()->id();
    if(flags&GCClipMask) gcv.clip_mask=None;
    if(flags&GCClipXOrigin) gcv.clip_x_origin=0;
    if(flags&GCClipYOrigin) gcv.clip_y_origin=0;
    if(flags&GCDashOffset) gcv.dash_offset=0;
    if(flags&GCDashList) gcv.dashes=4;
    if(flags&GCTileStipXOrigin) gcv.ts_x_origin=0;
    if(flags&GCTileStipYOrigin) gcv.ts_y_origin=0;
    if(flags&GCGraphicsExposures) gcv.graphics_exposures=True;
    if(flags&GCSubwindowMode) gcv.subwindow_mode=ClipByChildren;
    XChangeGC(DISPLAY(getApp()),(GC)gc,flags,&gcv);
    flags=0;
    }
  surface=NULL;
  }


// Read back pixel
FXColor FXDCWindow::readPixel(FXint x,FXint y){
  FXColor color=FXRGBA(0,0,0,0); 
  if(!surface){ fxerror("FXDCWindow::readPixel: DC not connected to drawable.\n"); }
  if(0<=x && 0<=y && x<surface->getWidth() && y<surface->getHeight()){
    XImage* xim=XGetImage(DISPLAY(getApp()),surface->id(),x,y,1,1,AllPlanes,ZPixmap);
    if(xim && xim->data){
      color=visual->getColor(XGetPixel(xim,0,0));
      XDestroyImage(xim);
      }
    }
  return color; 
  }


void FXDCWindow::drawPoint(FXint x,FXint y){ 
  if(!surface){ fxerror("FXDCWindow::drawPoint: DC not connected to drawable.\n"); }
  XDrawPoint(DISPLAY(getApp()),surface->id(),(GC)gc,x,y); 
  }


void FXDCWindow::drawPoints(const FXPoint* points,FXuint npoints){ 
  if(!surface){ fxerror("FXDCWindow::drawPoints: DC not connected to drawable.\n"); }
  XDrawPoints(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,CoordModeOrigin); 
  }


void FXDCWindow::drawPointsRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::drawPointsRel: DC not connected to drawable.\n"); }
  XDrawPoints(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,CoordModePrevious); 
  }


void FXDCWindow::drawLine(FXint x1,FXint y1,FXint x2,FXint y2){
  if(!surface){ fxerror("FXDCWindow::drawLine: DC not connected to drawable.\n"); }
  XDrawLine(DISPLAY(getApp()),surface->id(),(GC)gc,x1,y1,x2,y2);
  }


void FXDCWindow::drawLines(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::drawLines: DC not connected to drawable.\n"); }
  XDrawLines(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,CoordModeOrigin); 
  }


void FXDCWindow::drawLinesRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::drawLinesRel: DC not connected to drawable.\n"); }
  XDrawLines(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,CoordModePrevious); 
  }


void FXDCWindow::drawLineSegments(const FXSegment* segments,FXuint nsegments){
  if(!surface){ fxerror("FXDCWindow::drawLineSegments: DC not connected to drawable.\n"); }
  XDrawSegments(DISPLAY(getApp()),surface->id(),(GC)gc,(XSegment*)segments,nsegments); 
  }
 

void FXDCWindow::drawRectangle(FXint x,FXint y,FXint w,FXint h){
  if(!surface){ fxerror("FXDCWindow::drawRectangle: DC not connected to drawable.\n"); }
  XDrawRectangle(DISPLAY(getApp()),surface->id(),(GC)gc,x,y,w,h);
  }


void FXDCWindow::drawRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  if(!surface){ fxerror("FXDCWindow::drawRectangles: DC not connected to drawable.\n"); }
  XDrawRectangles(DISPLAY(getApp()),surface->id(),(GC)gc,(XRectangle*)rectangles,nrectangles);
  }


void FXDCWindow::drawArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  if(!surface){ fxerror("FXDCWindow::drawArc: DC not connected to drawable.\n"); }
  XDrawArc(DISPLAY(getApp()),surface->id(),(GC)gc,x,y,w,h,ang1,ang2);
  }


void FXDCWindow::drawArcs(const FXArc* arcs,FXuint narcs){
  if(!surface){ fxerror("FXDCWindow::drawArcs: DC not connected to drawable.\n"); }
  XDrawArcs(DISPLAY(getApp()),surface->id(),(GC)gc,(XArc*)arcs,narcs);
  }


void FXDCWindow::fillRectangle(FXint x,FXint y,FXint w,FXint h){
  if(!surface){ fxerror("FXDCWindow::fillRectangle: DC not connected to drawable.\n"); }
  XFillRectangle(DISPLAY(getApp()),surface->id(),(GC)gc,x,y,w,h);
  }


void FXDCWindow::fillRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  if(!surface){ fxerror("FXDCWindow::fillRectangles: DC not connected to drawable.\n"); }
  XFillRectangles(DISPLAY(getApp()),surface->id(),(GC)gc,(XRectangle*)rectangles,nrectangles);
  }


void FXDCWindow::fillArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  if(!surface){ fxerror("FXDCWindow::fillArc: DC not connected to drawable.\n"); }
  XFillArc(DISPLAY(getApp()),surface->id(),(GC)gc,x,y,w,h,ang1,ang2);
  }


void FXDCWindow::fillArcs(const FXArc* arcs,FXuint narcs){
  if(!surface){ fxerror("FXDCWindow::fillArcs: DC not connected to drawable.\n"); }
  XFillArcs(DISPLAY(getApp()),surface->id(),(GC)gc,(XArc*)arcs,narcs);
  }


void FXDCWindow::fillPolygon(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillArcs: DC not connected to drawable.\n"); }
  XFillPolygon(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,Convex,CoordModeOrigin);
  }


void FXDCWindow::fillConcavePolygon(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygon: DC not connected to drawable.\n"); }
  XFillPolygon(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,Nonconvex,CoordModeOrigin);
  }


void FXDCWindow::fillComplexPolygon(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygon: DC not connected to drawable.\n"); }
  XFillPolygon(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,Complex,CoordModeOrigin);
  }


void FXDCWindow::fillPolygonRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillPolygonRel: DC not connected to drawable.\n"); }
  XFillPolygon(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,Convex,CoordModePrevious);
  }


void FXDCWindow::fillConcavePolygonRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygonRel: DC not connected to drawable.\n"); }
  XFillPolygon(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,Nonconvex,CoordModePrevious);
  }


void FXDCWindow::fillComplexPolygonRel(const FXPoint* points,FXuint npoints){
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygonRel: DC not connected to drawable.\n"); }
  XFillPolygon(DISPLAY(getApp()),surface->id(),(GC)gc,(XPoint*)points,npoints,Complex,CoordModePrevious);
  }


void FXDCWindow::drawText(FXint x,FXint y,const FXchar* string,FXuint length){
  if(!surface){ fxerror("FXDCWindow::drawText: DC not connected to drawable.\n"); }
  XDrawString(DISPLAY(getApp()),surface->id(),(GC)gc,x,y,(char*)string,length);
  }


void FXDCWindow::drawImageText(FXint x,FXint y,const FXchar* string,FXuint length){
  if(!surface){ fxerror("FXDCWindow::drawImageText: DC not connected to drawable.\n"); }
  XDrawImageString(DISPLAY(getApp()),surface->id(),(GC)gc,x,y,(char*)string,length);
  }


void FXDCWindow::drawArea(const FXDrawable* source,FXint sx,FXint sy,FXint sw,FXint sh,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawArea: DC not connected to drawable.\n"); }
  if(!source || !source->id()){ fxerror("FXDCWindow::drawArea: illegal source specified.\n"); }
  XCopyArea(DISPLAY(getApp()),source->id(),surface->id(),(GC)gc,sx,sy,sw,sh,dx,dy);
  }


void FXDCWindow::drawImage(const FXImage* image,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawImage: DC not connected to drawable.\n"); }
  if(!image || !image->id()){ fxerror("FXDCWindow::drawImage: illegal image specified.\n"); }
  XCopyArea(DISPLAY(getApp()),image->id(),surface->id(),(GC)gc,0,0,image->width,image->height,dx,dy);
  }


void FXDCWindow::drawBitmap(const FXBitmap* bitmap,FXint dx,FXint dy) {
  if(!surface) fxerror("FXDCWindow::drawBitmap: DC not connected to drawable.\n");
  if(!bitmap || !bitmap->id()) fxerror("FXDCWindow::drawBitmap: illegal bitmap specified.\n");
  XCopyPlane(DISPLAY(getApp()),bitmap->id(),surface->id(),(GC)gc,0,0,bitmap->width,bitmap->height,dx,dy,1);
  }


// Draw a vanilla icon
void FXDCWindow::drawIcon(const FXIcon* icon,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawIcon: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIcon: illegal icon specified.\n"); }
  XGCValues gcv;  
  if(icon->getOptions()&IMAGE_OPAQUE){
    XCopyArea(DISPLAY(getApp()),icon->id(),surface->id(),(GC)gc,0,0,icon->width,icon->height,dx,dy);
    }
  else{
    gcv.clip_mask=icon->shape;
    gcv.clip_x_origin=dx;
    gcv.clip_y_origin=dy;
    XChangeGC(DISPLAY(getApp()),(GC)gc,GCClipMask|GCClipXOrigin|GCClipYOrigin,&gcv);
    XCopyArea(DISPLAY(getApp()),icon->id(),surface->id(),(GC)gc,0,0,icon->width,icon->height,dx,dy);
    XSetClipRectangles(DISPLAY(getApp()),(GC)gc,0,0,(XRectangle*)&clip,1,Unsorted); // Restore old clip rectangle
    flags|=GCClipMask;
    }
  }


// Draw a shaded icon, like when it is selected
void FXDCWindow::drawIconShaded(const FXIcon* icon,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawIconShaded: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIconShaded: illegal icon specified.\n"); }
  XGCValues gcv;  
  gcv.clip_mask=icon->shape;
  gcv.clip_x_origin=dx;
  gcv.clip_y_origin=dy;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCClipMask|GCClipXOrigin|GCClipYOrigin,&gcv);
  XCopyArea(DISPLAY(getApp()),icon->id(),surface->id(),(GC)gc,0,0,icon->width,icon->height,dx,dy);
  gcv.function=BLT_SRC;
  gcv.stipple=getApp()->stipples[STIPPLE_GRAY];
  gcv.fill_style=FILL_STIPPLED;
  gcv.ts_x_origin=dx;
  gcv.ts_y_origin=dy;
  gcv.foreground=visual->getPixel(getApp()->selbackColor);
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCForeground|GCFunction|GCTileStipXOrigin|GCTileStipYOrigin|GCStipple|GCFillStyle,&gcv);
  XFillRectangle(DISPLAY(getApp()),surface->id(),(GC)gc,dx,dy,icon->width,icon->height);
  gcv.function=rop;
  gcv.fill_style=fill;
  gcv.ts_x_origin=tx;
  gcv.ts_y_origin=ty;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCTileStipXOrigin|GCTileStipYOrigin|GCFunction|GCFillStyle,&gcv);                // Restore old raster op function and fill style
  XSetClipRectangles(DISPLAY(getApp()),(GC)gc,0,0,(XRectangle*)&clip,1,Unsorted); // Restore old clip rectangle
  flags|=GCClipMask;
  }


// This draws a sunken icon
void FXDCWindow::drawIconSunken(const FXIcon* icon,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawIconSunken: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->etch){ fxerror("FXDCWindow::drawIconSunken: illegal icon specified.\n"); }
  FXColor base=getApp()->baseColor;
  FXuint r=FXREDVAL(base);
  FXuint g=FXGREENVAL(base);
  FXuint b=FXBLUEVAL(base);
  FXColor clr=FXRGB((85*r)/100,(85*g)/100,(85*b)/100);
  XGCValues gcv;  
  gcv.clip_mask=icon->etch;
  gcv.clip_x_origin=dx+1;
  gcv.clip_y_origin=dy+1;
  gcv.foreground=visual->getPixel(getApp()->hiliteColor);
  gcv.function=BLT_SRC;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCForeground|GCClipMask|GCClipXOrigin|GCClipYOrigin|GCFunction,&gcv);
  XCopyPlane(DISPLAY(getApp()),icon->etch,surface->id(),(GC)gc,0,0,icon->width,icon->height,dx+1,dy+1,1);
  gcv.clip_mask=icon->etch;
  gcv.clip_x_origin=dx;
  gcv.clip_y_origin=dy;
  gcv.foreground=visual->getPixel(clr);
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCForeground|GCClipMask|GCClipXOrigin|GCClipYOrigin,&gcv);
  XCopyPlane(DISPLAY(getApp()),icon->etch,surface->id(),(GC)gc,0,0,icon->width,icon->height,dx,dy,1);
  gcv.foreground=devfg;
  gcv.function=rop;          
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCForeground|GCFunction,&gcv);               // Restore old raster op function and color
  XSetClipRectangles(DISPLAY(getApp()),(GC)gc,0,0,(XRectangle*)&clip,1,Unsorted); // Restore old clip rectangle
  flags|=GCClipMask;
  }


void FXDCWindow::drawHashBox(FXint x,FXint y,FXint w,FXint h,FXint b){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::drawHashBox: DC not connected to drawable.\n"); }
  gcv.stipple=getApp()->stipples[STIPPLE_GRAY];
  gcv.fill_style=FILL_STIPPLED;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCStipple|GCFillStyle,&gcv);
  XFillRectangle(DISPLAY(getApp()),surface->id(),(GC)gc,x,y,w-b,b);
  XFillRectangle(DISPLAY(getApp()),surface->id(),(GC)gc,x+w-b,y,b,h-b);
  XFillRectangle(DISPLAY(getApp()),surface->id(),(GC)gc,x+b,y+h-b,w-b,b);
  XFillRectangle(DISPLAY(getApp()),surface->id(),(GC)gc,x,y+b,b,h-b);
  gcv.stipple=getApp()->stipples[STIPPLE_WHITE];    // Needed for IRIX6.4 bug workaround!
  gcv.fill_style=fill;    // Restore old fill style
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCStipple|GCFillStyle,&gcv);
  }


void FXDCWindow::setForeground(FXColor clr){
  if(!surface){ fxerror("FXDCWindow::setForeground: DC not connected to drawable.\n"); }
  devfg=visual->getPixel(clr);
  XSetForeground(DISPLAY(getApp()),(GC)gc,devfg);
  flags|=GCForeground;
  fg=clr;
  }


void FXDCWindow::setBackground(FXColor clr){
  if(!surface){ fxerror("FXDCWindow::setBackground: DC not connected to drawable.\n"); }
  devbg=visual->getPixel(clr);
  XSetBackground(DISPLAY(getApp()),(GC)gc,devbg);
  flags|=GCBackground;
  bg=clr;
  }


void FXDCWindow::setDashes(FXuint dashoffset,const FXchar *dashlist,FXuint n){
  if(!surface){ fxerror("FXDCWindow::setDashes: DC not connected to drawable.\n"); }
  XSetDashes(DISPLAY(getApp()),(GC)gc,dashoffset,(char*)dashlist,n);
  flags|=(GCDashList|GCDashOffset);
  }


void FXDCWindow::setLineWidth(FXuint linewidth){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineWidth: DC not connected to drawable.\n"); }
  gcv.line_width=linewidth;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCLineWidth,&gcv);
  flags|=GCLineWidth;
  width=linewidth;
  }


void FXDCWindow::setLineCap(FXCapStyle capstyle){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineCap: DC not connected to drawable.\n"); }
  gcv.cap_style=capstyle;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCCapStyle,&gcv);
  flags|=GCCapStyle;
  cap=capstyle;
  }


void FXDCWindow::setLineJoin(FXJoinStyle joinstyle){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineJoin: DC not connected to drawable.\n"); }
  gcv.join_style=joinstyle;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCJoinStyle,&gcv);
  flags|=GCJoinStyle;
  join=joinstyle;
  }


void FXDCWindow::setLineStyle(FXLineStyle linestyle){ 
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setLineStyle: DC not connected to drawable.\n"); }
  gcv.line_style=linestyle;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCLineStyle,&gcv);
  flags|=GCLineStyle;
  style=linestyle;
  }


void FXDCWindow::setFillStyle(FXFillStyle fillstyle){
  if(!surface){ fxerror("FXDCWindow::setFillStyle: DC not connected to drawable.\n"); }
  XSetFillStyle(DISPLAY(getApp()),(GC)gc,fillstyle);
  flags|=GCFillStyle;
  fill=fillstyle;
  }
  
  
void FXDCWindow::setFillRule(FXFillRule fillrule){
  if(!surface){ fxerror("FXDCWindow::setFillRule: DC not connected to drawable.\n"); }
  XSetFillRule(DISPLAY(getApp()),(GC)gc,fillrule);
  flags|=GCFillRule;
  rule=fillrule;
  }


void FXDCWindow::setFunction(FXFunction func){
  if(!surface){ fxerror("FXDCWindow::setFunction: DC not connected to drawable.\n"); }
  XSetFunction(DISPLAY(getApp()),(GC)gc,func);
  flags|=GCFunction;
  rop=func;
  }


void FXDCWindow::setTile(FXImage* image,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setTile: DC not connected to drawable.\n"); }
  if(!image || !image->id()){ fxerror("FXDCWindow::setTile: illegal image specified.\n"); }
  gcv.tile=image->id();
  gcv.ts_x_origin=dx;
  gcv.ts_y_origin=dy;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCTileStipXOrigin|GCTileStipYOrigin|GCTile,&gcv);
  if(dx) flags|=GCTileStipXOrigin;
  if(dy) flags|=GCTileStipYOrigin;
  tile=image;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setStipple(FXBitmap* bitmap,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  if(!bitmap || !bitmap->id()){ fxerror("FXDCWindow::setStipple: illegal image specified.\n"); }
  gcv.stipple=bitmap->id();
  gcv.ts_x_origin=dx;
  gcv.ts_y_origin=dy;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCTileStipXOrigin|GCTileStipYOrigin|GCStipple,&gcv);
  if(dx) flags|=GCTileStipXOrigin;
  if(dy) flags|=GCTileStipYOrigin;
  flags|=GCStipple;
  stipple=bitmap;
  pattern=STIPPLE_NONE;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setStipple(FXStipplePattern pat,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  if(pat>STIPPLE_CROSSDIAG) pat=STIPPLE_CROSSDIAG;
  FXASSERT(getApp()->stipples[pat]);
  gcv.stipple=getApp()->stipples[pat];
  gcv.ts_x_origin=dx;
  gcv.ts_y_origin=dy;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCTileStipXOrigin|GCTileStipYOrigin|GCStipple,&gcv);
  if(dx) flags|=GCTileStipXOrigin;
  if(dy) flags|=GCTileStipYOrigin;
  stipple=NULL;
  pattern=pat;
  flags|=GCStipple;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setClipRegion(const FXRegion& region){
  if(!surface){ fxerror("FXDCWindow::setClipRegion: DC not connected to drawable.\n"); }
  XSetRegion(DISPLAY(getApp()),(GC)gc,(Region)region.region);///// Should intersect region and rect??
  flags|=GCClipMask;
  }


void FXDCWindow::setClipRectangle(FXint x,FXint y,FXint w,FXint h){
  if(!surface){ fxerror("FXDCWindow::setClipRectangle: DC not connected to drawable.\n"); }
  clip.x=FXMAX(x,rect.x);
  clip.y=FXMAX(y,rect.y);
  clip.w=FXMIN(x+w,rect.x+rect.w)-clip.x;
  clip.h=FXMIN(y+h,rect.y+rect.h)-clip.y;
  if(clip.w<=0) clip.w=0;
  if(clip.h<=0) clip.h=0;
  XSetClipRectangles(DISPLAY(getApp()),(GC)gc,0,0,(XRectangle*)&clip,1,Unsorted);
  flags|=GCClipMask;
  }
 

void FXDCWindow::clearClipRectangle(){
  if(!surface){ fxerror("FXDCWindow::clearClipRectangle: DC not connected to drawable.\n"); }
  clip=rect;
  XSetClipRectangles(DISPLAY(getApp()),(GC)gc,0,0,(XRectangle*)&clip,1,Unsorted);
  flags|=GCClipMask;
  }


void FXDCWindow::setClipMask(FXBitmap* bitmap,FXint dx,FXint dy){
  XGCValues gcv;  
  if(!surface){ fxerror("FXDCWindow::setClipMask: DC not connected to drawable.\n"); }
  if(!bitmap || !bitmap->id()){ fxerror("FXDCWindow::setClipMask: illegal mask specified.\n"); }
  gcv.clip_mask=bitmap->id();
  gcv.clip_x_origin=dx;
  gcv.clip_y_origin=dy;
  XChangeGC(DISPLAY(getApp()),(GC)gc,GCClipMask|GCClipXOrigin|GCClipYOrigin,&gcv);
  if(dx) flags|=GCClipXOrigin;
  if(dy) flags|=GCClipYOrigin;
  flags|=GCClipMask;
  mask=bitmap;
  cx=dx;
  cy=dy;
  }


void FXDCWindow::clearClipMask(){
  if(!surface){ fxerror("FXDCWindow::clearClipMask: DC not connected to drawable.\n"); }
  clip=rect;
  XSetClipRectangles(DISPLAY(getApp()),(GC)gc,0,0,(XRectangle*)&clip,1,Unsorted);
  flags|=GCClipMask;
  mask=NULL;
  cx=0;
  cy=0;
  }


void FXDCWindow::setTextFont(FXFont *fnt){
  if(!surface){ fxerror("FXDCWindow::setTextFont: DC not connected to drawable.\n"); }
  if(!fnt || !fnt->id()){ fxerror("FXDCWindow::setTextFont: illegal or NULL font specified.\n"); }
  XSetFont(DISPLAY(getApp()),(GC)gc,fnt->id());
  flags|=GCFont;
  font=fnt;
  }


void FXDCWindow::clipChildren(FXbool yes){ 
  if(!surface){ fxerror("FXDCWindow::clipChildren: window has not yet been created.\n"); }
  if(yes){
    XSetSubwindowMode(DISPLAY(getApp()),(GC)gc,ClipByChildren);
    flags&=~GCSubwindowMode;
    }
  else{
    XSetSubwindowMode(DISPLAY(getApp()),(GC)gc,IncludeInferiors);
    flags|=GCSubwindowMode;
    }
  }


/********************************************************************************
*                                   MS-Windows                                  *
********************************************************************************/

#else 

// This one is not defined in the Cygwin header files
#ifndef PS_JOIN_MASK
#define PS_JOIN_MASK 0x0000F000
#endif

// Construct for expose event painting
FXDCWindow::FXDCWindow(FXDrawable* drawable,FXEvent* event):FXDC(drawable->getApp()){
  FXPROFILE("FXDCWindow::FXDCWindow()");
  needsNewBrush=FALSE;
  FXMALLOC(&brush,LOGBRUSH,1);
  needsNewPen=FALSE;
  FXMALLOC(&pen,EXTLOGPEN,1);
  needsPath=FALSE;
  hOldPalette=NULL;
  begin(drawable);
  rect.x=clip.x=event->rect.x;
  rect.y=clip.y=event->rect.y;
  rect.w=clip.w=event->rect.w;
  rect.h=clip.h=event->rect.h;
  HRGN hrgn=CreateRectRgn(clip.x,clip.y,clip.x+clip.w,clip.y+clip.h);
  SelectClipRgn((HDC)dc,hrgn);
  DeleteObject(hrgn);
  }


// Construct for normal painting
FXDCWindow::FXDCWindow(FXDrawable* drawable):FXDC(drawable->getApp()){
  FXPROFILE("FXDCWindow::FXDCWindow()");
  needsNewBrush=FALSE;
  FXMALLOC(&brush,LOGBRUSH,1);
  needsNewPen=FALSE;
  FXMALLOC(&pen,EXTLOGPEN,1);
  needsPath=FALSE;
  hOldPalette=NULL;
  begin(drawable);
  }

  
// Destruct
FXDCWindow::~FXDCWindow(){
  FXPROFILE("FXDCWindow::~FXDCWindow()");
  FXFREE(&brush);
  FXFREE(&pen);
  end();
  }


// Begin locks in a drawable surface
void FXDCWindow::begin(FXDrawable *drawable){
  FXPROFILE("FXDCWindow::begin()");
  if(!drawable){ fxerror("FXDCWindow::begin: NULL drawable.\n"); }
  if(!drawable->id()){ fxerror("FXDCWindow::begin: drawable not created yet.\n"); }

  surface=drawable;// Careful:- surface->id() can be HWND or HBITMAP depending on drawable
  visual=drawable->getVisual();
  dc=drawable->GetDC();
  rect.x=clip.x=0;
  rect.y=clip.y=0;
  rect.w=clip.w=drawable->getWidth();
  rect.h=clip.h=drawable->getHeight();

  // Select and realize palette, if necessary
  if(visual->hPalette){
    hOldPalette=SelectPalette((HDC)dc,(HPALETTE)visual->hPalette,FALSE);
    RealizePalette((HDC)dc);
    }

  // Create our default pen (black, solid, one pixel wide)
  pen->elpPenStyle=PS_GEOMETRIC|PS_SOLID|PS_ENDCAP_ROUND|PS_JOIN_BEVEL;
  pen->elpWidth=1;
  pen->elpBrushStyle=BS_SOLID;
  pen->elpColor=PALETTERGB(0,0,0);
  pen->elpHatch=0;
  LOGBRUSH lb;
  lb.lbStyle=pen->elpBrushStyle;
  lb.lbColor=pen->elpColor;
  lb.lbHatch=pen->elpHatch;
  SelectObject((HDC)dc,ExtCreatePen(pen->elpPenStyle,pen->elpWidth,&lb,0,NULL));

  // Create our default brush (solid white, for fills)
  brush->lbStyle=BS_SOLID;
  brush->lbColor=PALETTERGB(255,255,255);
  brush->lbHatch=0;
  SelectObject((HDC)dc,CreateBrushIndirect(brush));

  // Text alignment
  SetTextAlign((HDC)dc,TA_BASELINE|TA_LEFT);

  // Polygon fill mode
  SetPolyFillMode((HDC)dc,ALTERNATE);

  // Default text font
  setTextFont(FXApp::instance()->getNormalFont());
  }


// End unlocks the drawable surface 
void FXDCWindow::end(){
  FXPROFILE("FXDCWindow::end()");
  if(!surface){ fxerror("FXDCWindow::end: DC not connected to drawable.\n"); }
  DeleteObject(SelectObject((HDC)dc,GetStockObject(BLACK_PEN)));
  DeleteObject(SelectObject((HDC)dc,GetStockObject(WHITE_BRUSH)));
  if(visual->hPalette){
    SelectPalette((HDC)dc,(HPALETTE)hOldPalette,FALSE);
    }
  surface->ReleaseDC((HDC)dc);
  DWORD dwFlags=GetWindowLong((HWND)surface->id(),GWL_STYLE);
  SetWindowLong((HWND)surface->id(),GWL_STYLE,dwFlags|WS_CLIPCHILDREN);
  surface=NULL;
  dc=0;
  }


// Read back pixel
FXColor FXDCWindow::readPixel(FXint x,FXint y){
  FXColor color=FXRGBA(0,0,0,0); 
  if(!surface){ fxerror("FXDCWindow::readPixel: DC not connected to drawable.\n"); }
  if(0<=x && 0<=y && x<surface->getWidth() && y<surface->getHeight()){
    COLORREF clr=GetPixel((HDC)dc,x,y);
    color=FXRGB(GetRValue(clr),GetGValue(clr),GetBValue(clr)); 
    }
  return color;
  }


// Draw pixel in current foreground color
void FXDCWindow::drawPoint(FXint x,FXint y){ 
  FXPROFILE("FXDCWindow::drawPoint()");
  if(!surface){ fxerror("FXDCWindow::drawPoint: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  SetPixel((HDC)dc,x,y,pen->elpColor);
  }


void FXDCWindow::drawPoints(const FXPoint* points,FXuint npoints){ 
  FXPROFILE("FXDCWindow::drawPoints()");
  if(!surface){ fxerror("FXDCWindow::drawPoints: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<npoints; i++) drawPoint(points[i].x,points[i].y);
  }


void FXDCWindow::drawPointsRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::drawPointsRel()");
  if(!surface){ fxerror("FXDCWindow::drawPointsRel: DC not connected to drawable.\n"); }
  int x=0,y=0;
  for(FXuint i=0; i < npoints; i++){
    x+=points[i].x;
    y+=points[i].y;
    drawPoint(x,y);
    }
  }


void FXDCWindow::drawLine(FXint x1,FXint y1,FXint x2,FXint y2){
  FXPROFILE("FXDCWindow::drawLine()");
  if(!surface){ fxerror("FXDCWindow::drawLine: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  MoveToEx((HDC)dc,x1,y1,NULL);
  if(needsPath) BeginPath((HDC)dc);
  LineTo((HDC)dc,x2,y2);
  if(needsPath){
    EndPath((HDC)dc);
    StrokePath((HDC)dc);
    }
  }


void FXDCWindow::drawLines(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::drawLines()");
  if(!surface){ fxerror("FXDCWindow::drawLines: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  MoveToEx((HDC)dc,points[0].x,points[0].y,NULL);
  if(needsPath) BeginPath((HDC)dc);
  if((npoints>8192) || (npoints<4)){
    MoveToEx((HDC)dc,points[0].x,points[0].y,NULL);
    for(FXuint i=1; i<npoints; i++) LineTo((HDC)dc,points[i].x,points[i].y);
    }
  else{
    POINT longPoints[8192];
    for(FXuint i=0; i<npoints; i++){
      longPoints[i].x = points[i].x;
      longPoints[i].y = points[i].y;
      }
    Polyline((HDC)dc,longPoints,npoints);
    //MoveToEx((HDC)dc,longPoints[npoints-1].x,longPoints[npoints-1].y,NULL);
    }
  //for(FXuint i=1; i<npoints; i++) LineTo((HDC)dc,points[i].x,points[i].y);
  if(needsPath){
    EndPath((HDC)dc);
    StrokePath((HDC)dc);
    }
  }


void FXDCWindow::drawLinesRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::drawLinesRel()");
  if(!surface){ fxerror("FXDCWindow::drawLinesRel: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  int x=points[0].x,y=points[0].y;
  MoveToEx((HDC)dc,x,y,NULL);
  if(needsPath) BeginPath((HDC)dc);
  for(FXuint i=1; i < npoints; i++){
    x+=points[i].x;
    y+=points[i].y;
    LineTo((HDC)dc,x,y);
    }
  if(needsPath){
    EndPath((HDC)dc);
    StrokePath((HDC)dc);
    }
  }


void FXDCWindow::drawLineSegments(const FXSegment* segments,FXuint nsegments){
  FXPROFILE("FXDCWindow::drawLineSegments()");
  if(!surface){ fxerror("FXDCWindow::drawLineSegments: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  if(needsPath) BeginPath((HDC)dc);
  for(FXuint i=0; i<nsegments; i++){
    MoveToEx((HDC)dc,segments[i].x1,segments[i].y1,NULL);
    LineTo((HDC)dc,segments[i].x2,segments[i].y2);
    }
  if(needsPath){
    EndPath((HDC)dc);
    StrokePath((HDC)dc);
    }
  }


// Unfilled rectangle
void FXDCWindow::drawRectangle(FXint x,FXint y,FXint w,FXint h){
  FXPROFILE("FXDCWindow::drawRectangle()");
  if(!surface){ fxerror("FXDCWindow::drawRectangle: DC not connected to drawable.\n"); }
  if(fill==FILL_STIPPLED || fill==FILL_OPAQUESTIPPLED){
    // Assume correct brush is already selected
    if(needsNewBrush) updateBrush();
    PatBlt((HDC)dc,x,y,w,1,PATCOPY);
    PatBlt((HDC)dc,x,y,1,h,PATCOPY);
    PatBlt((HDC)dc,x,y+h,w,1,PATCOPY);
    PatBlt((HDC)dc,x+w,y,1,h,PATCOPY);
    }
  else{
    if(needsNewPen) updatePen();
    SaveDC((HDC)dc);
    SelectObject((HDC)dc,GetStockObject(NULL_BRUSH));
    if(needsPath) BeginPath((HDC)dc);
    Rectangle((HDC)dc,x,y,x+w+1,y+h+1);
    if(needsPath){
      EndPath((HDC)dc);
      StrokePath((HDC)dc);
      }
    RestoreDC((HDC)dc,-1);
    }
  }


void FXDCWindow::drawRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  FXPROFILE("FXDCWindow::drawRectangles()");
  if(!surface){ fxerror("FXDCWindow::drawRectangles: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  SaveDC((HDC)dc);
  SelectObject((HDC)dc,GetStockObject(NULL_BRUSH));
  if(needsPath) BeginPath((HDC)dc);
  for(FXuint i=0; i < nrectangles; i++) Rectangle((HDC)dc,rectangles[i].x,rectangles[i].y,rectangles[i].x+rectangles[i].w+1,rectangles[i].y+rectangles[i].h+1);
  if(needsPath){
    EndPath((HDC)dc);
    StrokePath((HDC)dc);
    }
  RestoreDC((HDC)dc,-1);
  }


// Draw arc; angles in degrees*64, ang2 relative to ang1
void FXDCWindow::drawArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  FXPROFILE("FXDCWindow::drawArc()");
  if(!surface){ fxerror("FXDCWindow::drawArc: DC not connected to drawable.\n"); }
  if(needsNewPen) updatePen();
  ang2+=ang1;
  int xStart=int(x+0.5*w+w*cos(ang1*PI/(180.0*64.0)));
  int yStart=int(y+0.5*h-h*sin(ang1*PI/(180.0*64.0)));
  int xEnd=int(x+0.5*w+w*cos(ang2*PI/(180.0*64.0)));
  int yEnd=int(y+0.5*h-h*sin(ang2*PI/(180.0*64.0)));
  if(needsPath) BeginPath((HDC)dc);
  Arc((HDC)dc,x,y,x+w,y+h,xStart,yStart,xEnd,yEnd);
  if(needsPath){
    EndPath((HDC)dc);
    StrokePath((HDC)dc);
    }
  }


void FXDCWindow::drawArcs(const FXArc* arcs,FXuint narcs){
  FXPROFILE("FXDCWindow::drawArcs()");
  if(!surface){ fxerror("FXDCWindow::drawArcs: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<narcs; i++){
    drawArc(arcs[i].x,arcs[i].y,arcs[i].w,arcs[i].h,arcs[i].a,arcs[i].b);
    }
  }


// Fill using currently selected ROP code
void FXDCWindow::fillRectangle(FXint x,FXint y,FXint w,FXint h){
  FXPROFILE("FXDCWindow::fillRectangle()");
  if(!surface){ fxerror("FXDCWindow::fillRectangle: DC not connected to drawable.\n"); }
  if(needsNewBrush) updateBrush();
  HPEN hPen=(HPEN)SelectObject((HDC)dc,GetStockObject(NULL_PEN));
  Rectangle((HDC)dc,x,y,x+w+1,y+h+1);
//  RECT r;
//  r.left=x; r.top=y; r.right=x+w; r.bottom=y+h;
//  FillRect((HDC)dc,&r,brush);
  SelectObject((HDC)dc,hPen);
  }


// Fill using currently selected ROP code
void FXDCWindow::fillRectangles(const FXRectangle* rectangles,FXuint nrectangles){
  FXPROFILE("FXDCWindow::fillRectangles()");
  if(!surface){ fxerror("FXDCWindow::fillRectangles: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<nrectangles; i++){
    fillRectangle(rectangles[i].x,rectangles[i].y,rectangles[i].w,rectangles[i].h);
    }
  }


// Draw filled arc; angles are in degrees*64; ang2 is relative from ang1
void FXDCWindow::fillArc(FXint x,FXint y,FXint w,FXint h,FXint ang1,FXint ang2){
  FXPROFILE("FXDCWindow::fillArc()");
  if(!surface){ fxerror("FXDCWindow::fillArc: DC not connected to drawable.\n"); }
  if(needsNewBrush) updateBrush();
  ang2+=ang1;
  int xStart=int(x+0.5*w+w*cos(ang1*PI/(180.0*64.0)));
  int yStart=int(y+0.5*h-h*sin(ang1*PI/(180.0*64.0)));
  int xEnd=int(x+0.5*w+w*cos(ang2*PI/(180.0*64.0)));
  int yEnd=int(y+0.5*h-h*sin(ang2*PI/(180.0*64.0)));
  HPEN hPen=(HPEN)SelectObject((HDC)dc,GetStockObject(NULL_PEN));
  Pie((HDC)dc,x,y,x+w,y+h,xStart,yStart,xEnd,yEnd);
  SelectObject((HDC)dc,hPen);
  }


void FXDCWindow::fillArcs(const FXArc* arcs,FXuint narcs){
  FXPROFILE("FXDCWindow::fillArcs()");
  if(!surface){ fxerror("FXDCWindow::fillArcs: DC not connected to drawable.\n"); }
  for(FXuint i=0; i<narcs; i++){
    fillArc(arcs[i].x,arcs[i].y,arcs[i].w,arcs[i].h,arcs[i].a,arcs[i].b);
    }
  }


// Filled simple polygon
void FXDCWindow::fillPolygon(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillPolgon()");
  if(!surface){ fxerror("FXDCWindow::fillPolygon: DC not connected to drawable.\n"); }
  if(needsNewBrush) updateBrush();
  POINT *pts=new POINT[npoints];
  for(FXuint i=0; i<npoints; i++){
    pts[i].x=points[i].x;
    pts[i].y=points[i].y;
    }
  HPEN hPen=(HPEN)SelectObject((HDC)dc,GetStockObject(NULL_PEN));
  Polygon((HDC)dc,pts,npoints);
  SelectObject((HDC)dc,hPen);
  delete [] pts;
  }


void FXDCWindow::fillConcavePolygon(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillConcavePolygon()");
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygon: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


void FXDCWindow::fillComplexPolygon(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillComplexPolygon()");
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygon: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


// Filled simple polygon with relative points
void FXDCWindow::fillPolygonRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillPolygonRel()");
  if(!surface){ fxerror("FXDCWindow::fillPolygonRel: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


void FXDCWindow::fillConcavePolygonRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillConcavePolygonRel()");
  if(!surface){ fxerror("FXDCWindow::fillConcavePolygonRel: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


void FXDCWindow::fillComplexPolygonRel(const FXPoint* points,FXuint npoints){
  FXPROFILE("FXDCWindow::fillComplexPolygonRel()");
  if(!surface){ fxerror("FXDCWindow::fillComplexPolygonRel: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  }


// Draw string (only foreground bits)
void FXDCWindow::drawText(FXint x,FXint y,const FXchar* string,FXuint length){
  FXPROFILE("FXDCWindow::drawText()");
  if(!surface){ fxerror("FXDCWindow::drawText: DC not connected to drawable.\n"); }
  SaveDC((HDC)dc);
  SetBkMode((HDC)dc,TRANSPARENT);
  TextOut((HDC)dc,x,y,string,length);
  RestoreDC((HDC)dc,-1);
  }


// Draw string (both foreground and background bits)
void FXDCWindow::drawImageText(FXint x,FXint y,const FXchar* string,FXuint length){
  FXPROFILE("FXDCWindow::drawImageText()");
  if(!surface){ fxerror("FXDCWindow::drawImageText: DC not connected to drawable.\n"); }
  SaveDC((HDC)dc);
  SetBkMode((HDC)dc,OPAQUE);
  TextOut((HDC)dc,x,y,string,length);
  RestoreDC((HDC)dc,-1);
  }


// Draw area from source
// Some of these ROP codes do not have names; the full list can be found in the MSDN docs
// at Platform SDK/Reference/Appendixes/Win32 Appendixes/Raster Operation Codes/Ternary Raster Operations
void FXDCWindow::drawArea(const FXDrawable* source,FXint sx,FXint sy,FXint sw,FXint sh,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawArea()");
  if(!surface){ fxerror("FXDCWindow::drawArea: DC not connected to drawable.\n"); }
  if(!source || !source->id()){ fxerror("FXDCWindow::drawArea: illegal source specified.\n"); }
 // if(needsNewBrush) updateBrush();
  HDC shdc=(HDC)source->GetDC();
  switch(rop){
    case BLT_CLR:                     // D := 0
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,BLACKNESS); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,SRCAND); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,SRCERASE); 
      break;
    case BLT_SRC:                     // D := S
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,SRCCOPY); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,0x220326);
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,SRCINVERT); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,SRCPAINT); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,NOTSRCERASE); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,0x990066); // Not sure about this one
      break;
    case BLT_NOT_DST:                 // D := ~D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,DSTINVERT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,0xDD0228); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,NOTSRCCOPY); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,MERGEPAINT); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,0x7700E6); 
      break;
    case BLT_SET:                     // D := 1
      BitBlt((HDC)dc,dx,dy,sw,sh,shdc,sx,sy,WHITENESS); 
    default: 
      break;
    }
  source->ReleaseDC(shdc);
  }


// Draw image 
void FXDCWindow::drawImage(const FXImage* image,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawImage()");
  if(!surface){ fxerror("FXDCWindow::drawImage: DC not connected to drawable.\n"); }
  if(!image || !image->id()){ fxerror("FXDCWindow::drawImage: illegal image specified.\n"); }
 // if(needsNewBrush) updateBrush();
  HDC dcMem=(HDC)image->GetDC();
  switch(rop){
    case BLT_CLR:                     // D := 0
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,BLACKNESS); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,SRCAND); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,SRCERASE); 
      break;
    case BLT_SRC:                     // D := S
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,SRCCOPY); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,0x220326);
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,SRCINVERT); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,SRCPAINT); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,NOTSRCERASE); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,0x990066); // Not sure about this one
      break;
    case BLT_NOT_DST:                 // D := ~D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,DSTINVERT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,0xDD0228); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,NOTSRCCOPY); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,MERGEPAINT); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,0x7700E6); 
      break;
    case BLT_SET:                     // D := 1
      BitBlt((HDC)dc,dx,dy,image->width,image->height,dcMem,0,0,WHITENESS); 
    default: 
      break;
    }
  image->ReleaseDC(dcMem);
  }


// Draw bitmap (Contributed by Michal Furmanczyk <mf@cfdrc.com>)
void FXDCWindow::drawBitmap(const FXBitmap* bitmap,FXint dx,FXint dy) {
  FXPROFILE("FXDCWindow::drawBitmap()");
  if(!surface) fxerror("FXDCWindow::drawBitmap: DC not connected to drawable.\n");
  if(!bitmap || !bitmap->id()) fxerror("FXDCWindow::drawBitmap: illegal bitmap specified.\n");
  //if(needsNewBrush) updateBrush();
  HDC dcMem=(HDC)bitmap->GetDC();
  switch(rop){
    case BLT_CLR:                     // D := 0
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,BLACKNESS); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,SRCAND); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,SRCERASE); 
      break;
    case BLT_SRC:                     // D := S
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,SRCCOPY); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,0x220326);
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,SRCINVERT); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,SRCPAINT); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,NOTSRCERASE); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,0x990066); // Not sure about this one
      break;
    case BLT_NOT_DST:                 // D := ~D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,DSTINVERT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,0xDD0228); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,NOTSRCCOPY); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,MERGEPAINT); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,0x7700E6); 
      break;
    case BLT_SET:                     // D := 1
      BitBlt((HDC)dc,dx,dy,bitmap->width,bitmap->height,dcMem,0,0,WHITENESS); 
    default: 
      break;
    }
  bitmap->ReleaseDC(dcMem);
  }


// Draw icon
void FXDCWindow::drawIcon(const FXIcon* icon,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawIcon()");
  if(!surface){ fxerror("FXDCWindow::drawIcon: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIcon: illegal icon specified.\n"); }
  HDC hdcSrc=(HDC)icon->GetDC();
  if(icon->getOptions()&IMAGE_OPAQUE){
    BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcSrc,0,0,SRCCOPY); 
    }
  else{
    HDC hdcMsk=CreateCompatibleDC((HDC)dc);
    SelectObject(hdcMsk,icon->shape);
    COLORREF crOldBack=SetBkColor((HDC)dc,RGB(255,255,255));
    COLORREF crOldText=SetTextColor((HDC)dc,RGB(0,0,0));
    BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcMsk,0,0,SRCAND);
    BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcSrc,0,0,SRCPAINT);
    DeleteDC(hdcMsk);
    SetBkColor((HDC)dc,crOldBack);
    SetTextColor((HDC)dc,crOldText);
    }
  icon->ReleaseDC(hdcSrc);
  }


// This may be done faster, I suspect; but I'm tired of looking at this now;
// at least it's correct as it stands..
void FXDCWindow::drawIconShaded(const FXIcon* icon,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::drawIconShaded()");
  if(!surface){ fxerror("FXDCWindow::drawIconShaded: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIconShaded: illegal icon specified.\n"); }
  HDC hdcSrc=(HDC)icon->GetDC();
  HDC hdcMsk=CreateCompatibleDC((HDC)dc);
  FXColor selbackColor=getApp()->selbackColor;
  SelectObject(hdcMsk,icon->shape);
  COLORREF crOldBack=SetBkColor((HDC)dc,RGB(255,255,255));
  COLORREF crOldText=SetTextColor((HDC)dc,RGB(0,0,0));

  // Paint icon
  BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcMsk,0,0,SRCAND);
  BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcSrc,0,0,SRCPAINT);

  HBRUSH hBrush=CreatePatternBrush((HBITMAP)getApp()->stipples[STIPPLE_GRAY]);
  HBRUSH hOldBrush=(HBRUSH)SelectObject((HDC)dc,hBrush);

  // Make black where pattern is 0 and shape is 0 [DPSoa]
  BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcMsk,0,0,0x00A803A9);

  SetTextColor((HDC)dc,RGB(FXREDVAL(selbackColor),FXGREENVAL(selbackColor),FXBLUEVAL(selbackColor)));
  SetBkColor((HDC)dc,RGB(0,0,0));

  // Make selbackcolor where pattern is 0 and shape is 0 [DPSoo]
  BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcMsk,0,0,0x00FE02A9);

  SelectObject((HDC)dc,hOldBrush);
  DeleteObject(hBrush);
  DeleteDC(hdcMsk);
  SetBkColor((HDC)dc,crOldBack);
  SetTextColor((HDC)dc,crOldText);
  icon->ReleaseDC(hdcSrc);
  }



// Draw a sunken or etched-in icon.  
void FXDCWindow::drawIconSunken(const FXIcon* icon,FXint dx,FXint dy){
  if(!surface){ fxerror("FXDCWindow::drawIconSunken: DC not connected to drawable.\n"); }
  if(!icon || !icon->id() || !icon->shape){ fxerror("FXDCWindow::drawIconSunken: illegal icon specified.\n"); }
  HDC hdcSrc=(HDC)icon->GetDC();
  HDC hdcMono=CreateCompatibleDC((HDC)dc);
  SelectObject(hdcMono,icon->etch);

  // Apparently, BkColor and TextColor apply to the source
  COLORREF crOldBack=SetBkColor((HDC)dc,RGB(255,255,255)); 
  COLORREF crOldText=SetTextColor((HDC)dc,RGB(0,0,0));

  // While brush colors apply to the pattern
  FXColor hiliteColor=getApp()->hiliteColor;
  HBRUSH hbrHilite=CreateSolidBrush(RGB(FXREDVAL(hiliteColor),FXGREENVAL(hiliteColor),FXBLUEVAL(hiliteColor)));
  HBRUSH oldBrush=(HBRUSH)SelectObject((HDC)dc,hbrHilite);

  // BitBlt the black bits in the monochrome bitmap into highlight colors
  // in the destination DC (offset a bit). This BitBlt(), and the next one,
  // use an unnamed raster op (0xB8074a) whose effect is D := ((D ^ P) & S) ^ P.
  // Or at least I think it is ;) The code = PSDPxax, so that's correct JVZ
  BitBlt((HDC)dc,dx+1,dy+1,icon->getWidth(),icon->getHeight(),hdcMono,0,0,0xB8074A);
  FXColor shadowColor=getApp()->shadowColor;
  HBRUSH hbrShadow=CreateSolidBrush(RGB(FXREDVAL(shadowColor),FXGREENVAL(shadowColor),FXBLUEVAL(shadowColor)));
  SelectObject((HDC)dc,hbrShadow);

  // Now BitBlt the black bits in the monochrome bitmap into the
  // shadow color on the destination DC.
  BitBlt((HDC)dc,dx,dy,icon->getWidth(),icon->getHeight(),hdcMono,0,0,0xB8074A);
  DeleteDC(hdcMono);
  SelectObject((HDC)dc,oldBrush); 
  DeleteObject(hbrHilite);
  DeleteObject(hbrShadow);
  SetBkColor((HDC)dc,crOldBack);
  SetTextColor((HDC)dc,crOldText);
  icon->ReleaseDC(hdcSrc);
  }


void FXDCWindow::drawHashBox(FXint x,FXint y,FXint w,FXint h,FXint b){
  FXPROFILE("FXDCWindow::drawHashBox()");
  if(!surface){ fxerror("FXDCWindow::drawHashBox: DC not connected to drawable.\n"); }
  HBRUSH hBrush=(HBRUSH)SelectObject((HDC)dc,CreatePatternBrush((HBITMAP)getApp()->stipples[STIPPLE_GRAY]));
  HPEN hPen=(HPEN)SelectObject((HDC)dc,GetStockObject(NULL_PEN));
  PatBlt((HDC)dc,x,y,w-b,b,PATINVERT);
  PatBlt((HDC)dc,x+w-b,y,b,h-b,PATINVERT);
  PatBlt((HDC)dc,x+b,y+h-b,w-b,b,PATINVERT);
  PatBlt((HDC)dc,x,y+b,b,h-b,PATINVERT);
  SelectObject((HDC)dc,hPen);
  DeleteObject(SelectObject((HDC)dc,hBrush));
  }


void FXDCWindow::updatePen(){
  FXPROFILE("FXDCWindow::updatePen()");
  LOGBRUSH lb;
  lb.lbStyle=pen->elpBrushStyle;
  lb.lbColor=pen->elpColor;
  lb.lbHatch=pen->elpHatch;
  DeleteObject(SelectObject((HDC)dc,ExtCreatePen(pen->elpPenStyle,pen->elpWidth,&lb,0,NULL)));
  needsPath = (pen->elpWidth>1) ? TRUE : FALSE;
  needsNewPen=FALSE;
  }


void FXDCWindow::setForeground(FXColor clr){
  FXPROFILE("FXDCWindow::setForeground()");
  if(!surface){ fxerror("FXDCWindow::setForeground: DC not connected to drawable.\n"); }
  fg=clr;

  // Create a new pen with same settings and new color
  pen->elpColor=visual->getPixel(fg);
  needsNewPen=TRUE;

  // Reset the brush
  brush->lbColor=visual->getPixel(fg);
  needsNewBrush=TRUE;

  // And the text color
  SetTextColor((HDC)dc,visual->getPixel(fg));
  }


void FXDCWindow::setBackground(FXColor clr){
  FXPROFILE("FXDCWindow::setBackground()");
  if(!surface){ fxerror("FXDCWindow::setBackground: DC not connected to drawable.\n"); }
  bg=clr;
  SetBkColor((HDC)dc,visual->getPixel(bg));
  }


// Set dash pattern (for the LINE_ONOFF_DASH line style)
void FXDCWindow::setDashes(FXuint dashoffset,const FXchar *dashlist,FXuint n){
  FXPROFILE("FXDCWindow::setDashes()");
  if(!surface){ fxerror("FXDCWindow::setDashes: DC not connected to drawable.\n"); }
  }


void FXDCWindow::setLineWidth(FXuint linewidth){ 
  FXPROFILE("FXDCWindow::setLineWidth()");
  if(!surface){ fxerror("FXDCWindow::setLineWidth: DC not connected to drawable.\n"); }
  // Create a new pen with this line width
  pen->elpWidth=linewidth;
  width=linewidth;
  needsNewPen=TRUE;
  }


void FXDCWindow::setLineCap(FXCapStyle capstyle){
  FXPROFILE("FXDCWindow::setLineCap()");
  if(!surface){ fxerror("FXDCWindow::setLineCap: DC not connected to drawable.\n"); }

  // Modify its cap style
  DWORD dwPenType=pen->elpPenStyle&PS_TYPE_MASK;
  DWORD dwPenStyle=pen->elpPenStyle&PS_STYLE_MASK;
  DWORD dwEndCapAttributes=pen->elpPenStyle&PS_ENDCAP_MASK;
  DWORD dwJoinAttributes=pen->elpPenStyle&PS_JOIN_MASK;
  switch(capstyle){
    case CAP_NOT_LAST:
      dwPenType=PS_GEOMETRIC;
      dwEndCapAttributes=PS_ENDCAP_SQUARE;
      break;
    case CAP_BUTT:
      dwPenType=PS_GEOMETRIC;
      dwEndCapAttributes=PS_ENDCAP_FLAT;
      break;
    case CAP_ROUND:
      dwPenType=PS_GEOMETRIC;
      dwEndCapAttributes=PS_ENDCAP_ROUND;
      break;
    case CAP_PROJECTING:
      FXASSERT(FALSE);
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create a new pen with this style
  pen->elpPenStyle=dwPenType|dwPenStyle|dwEndCapAttributes|dwJoinAttributes;
  cap=capstyle;
  needsNewPen=TRUE;
  }


void FXDCWindow::setLineJoin(FXJoinStyle joinstyle){
  FXPROFILE("FXDCWindow::setLineJoin()");
  if(!surface){ fxerror("FXDCWindow::setLineJoin: DC not connected to drawable.\n"); }

  // Now modify the style flags
  DWORD dwPenType=pen->elpPenStyle&PS_TYPE_MASK;
  DWORD dwPenStyle=pen->elpPenStyle&PS_STYLE_MASK;
  DWORD dwEndCapAttributes=pen->elpPenStyle&PS_ENDCAP_MASK;
  DWORD dwJoinAttributes=pen->elpPenStyle&PS_JOIN_MASK;
  switch(joinstyle){
    case JOIN_MITER:
      dwPenType=PS_GEOMETRIC;
      dwJoinAttributes=PS_JOIN_MITER;
      break;
    case JOIN_ROUND:
      dwPenType=PS_GEOMETRIC;
      dwJoinAttributes=PS_JOIN_ROUND;
      break;
    case JOIN_BEVEL:
      dwPenType=PS_GEOMETRIC;
      dwJoinAttributes=PS_JOIN_BEVEL;
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create a new pen with this style
  pen->elpPenStyle=dwPenType|dwPenStyle|dwEndCapAttributes|dwJoinAttributes;
  join=joinstyle;
  needsNewPen=TRUE;
  }


void FXDCWindow::setLineStyle(FXLineStyle linestyle){
  FXPROFILE("FXDCWindow::setLineStyle()");
  if(!surface){ fxerror("FXDCWindow::setLineStyle: DC not connected to drawable.\n"); }

  // Modify style bits
  DWORD dwPenType=pen->elpPenStyle&PS_TYPE_MASK;
  DWORD dwPenStyle=pen->elpPenStyle&PS_STYLE_MASK;
  DWORD dwEndCapAttributes=pen->elpPenStyle&PS_ENDCAP_MASK;
  DWORD dwJoinAttributes=pen->elpPenStyle&PS_JOIN_MASK;
  switch(linestyle){
    case LINE_SOLID:
      dwPenType=PS_GEOMETRIC;
      dwPenStyle=PS_SOLID;
      break;
    case LINE_ONOFF_DASH:
      dwPenType=PS_COSMETIC;
      dwPenStyle=PS_DOT; // PS_ALTERNATE looks better, but is only supported for NT
      break;
    case LINE_DOUBLE_DASH:
      FXASSERT(FALSE);
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create a new pen with this style
  pen->elpPenStyle=dwPenType|dwPenStyle|dwEndCapAttributes|dwJoinAttributes;
  style=linestyle;
  needsNewPen=TRUE;
  }


static DWORD FXStipplePattern2Hatch(FXStipplePattern pat){
  FXPROFILE("FXStipplePattern2Hatch()");
  switch(pat){
    case STIPPLE_HORZ:
      return HS_HORIZONTAL;
    case STIPPLE_VERT:
      return HS_VERTICAL;
    case STIPPLE_CROSS:
      return HS_CROSS;
    case STIPPLE_DIAG:
      return HS_BDIAGONAL;
    case STIPPLE_REVDIAG:
      return HS_FDIAGONAL;
    case STIPPLE_CROSSDIAG:
      return HS_DIAGCROSS;
    default:
      FXASSERT(FALSE);
      return 0;
    }
  }


void FXDCWindow::updateBrush(){
  FXPROFILE("FXDCWindow::updateBrush()");
  // Get previous brush info
  DeleteObject(SelectObject((HDC)dc,GetStockObject(NULL_BRUSH)));
  switch(fill){
    case FILL_SOLID:
      brush->lbStyle=BS_SOLID;
      SetBkMode((HDC)dc,OPAQUE);
      break;
    case FILL_TILED:
      FXASSERT(FALSE);
      break;
    case FILL_STIPPLED:
      SetBrushOrgEx((HDC)dc,tx,ty,NULL);
      SetBkMode((HDC)dc,TRANSPARENT);
      if(stipple){
        brush->lbStyle=BS_PATTERN;
        brush->lbHatch=(LONG)stipple->id();    // This should be a HBITMAP
        }
      else{
        if(pattern>=STIPPLE_0 && pattern<=STIPPLE_16){
          brush->lbStyle=BS_PATTERN;
          brush->lbHatch=(LONG)getApp()->stipples[pattern];
          }
        else{
          brush->lbStyle=BS_HATCHED;
          brush->lbHatch=FXStipplePattern2Hatch(pattern);
          }
        }
      break;
    case FILL_OPAQUESTIPPLED:
      SetBrushOrgEx((HDC)dc,tx,ty,NULL);
      SetBkMode((HDC)dc,OPAQUE);
      if(stipple){
        brush->lbStyle=BS_PATTERN;
        brush->lbHatch=(LONG)stipple->id();    // This should be a HBITMAP
        }
      else{
        if(pattern>=STIPPLE_0 && pattern<=STIPPLE_16){
          brush->lbStyle=BS_PATTERN;
          brush->lbHatch=(LONG)getApp()->stipples[pattern];
          }
        else{
          brush->lbStyle=BS_HATCHED;
          brush->lbHatch=FXStipplePattern2Hatch(pattern);
          }
        }
      break;
    default:
      FXASSERT(FALSE);
    }

  // Create new brush and select it into the DC
  SelectObject((HDC)dc,CreateBrushIndirect(brush));
  needsNewBrush=FALSE;
  }


void FXDCWindow::setFillStyle(FXFillStyle fillstyle){
  FXPROFILE("FXDCWindow::setFillStyle()");
  if(!surface){ fxerror("FXDCWindow::setFillStyle: DC not connected to drawable.\n"); }
  fill=fillstyle;
  needsNewBrush=TRUE;
  }
  
  
// Set fill rule
void FXDCWindow::setFillRule(FXFillRule fillrule){
  FXPROFILE("FXDCWindow::setFillRule()");
  if(!surface){ fxerror("FXDCWindow::setFillRule: DC not connected to drawable.\n"); }
  if(fillrule==RULE_EVEN_ODD)
    SetPolyFillMode((HDC)dc,ALTERNATE);
  else
    SetPolyFillMode((HDC)dc,WINDING);
  rule=fillrule;
  }


// Set blit function
void FXDCWindow::setFunction(FXFunction func){
  FXPROFILE("FXDCWindow::setFunction()");
  if(!surface){ fxerror("FXDCWindow::setFunction: DC not connected to drawable.\n"); }
  rop=func;

  // Also set ROP2 code for lines
  switch(rop){
    case BLT_CLR:                     // D := 0
      SetROP2((HDC)dc,R2_BLACK); 
      break;
    case BLT_SRC_AND_DST:             // D := S & D
      SetROP2((HDC)dc,R2_MASKPEN); 
      break;
    case BLT_SRC_AND_NOT_DST:         // D := S & ~D
      SetROP2((HDC)dc,R2_MASKPENNOT); 
      break;
    case BLT_SRC:                     // D := S
      SetROP2((HDC)dc,R2_COPYPEN); 
      break;
    case BLT_NOT_SRC_AND_DST:         // D := ~S & D
      SetROP2((HDC)dc,R2_MASKNOTPEN); 
      break;
    case BLT_SRC_XOR_DST:             // D := S ^ D
      SetROP2((HDC)dc,R2_XORPEN); 
      break;
    case BLT_SRC_OR_DST:              // D := S | D
      SetROP2((HDC)dc,R2_MERGEPEN); 
      break;
    case BLT_NOT_SRC_AND_NOT_DST:     // D := ~S & ~D  ==  D := ~(S | D)
      SetROP2((HDC)dc,R2_NOTMERGEPEN); 
      break;
    case BLT_NOT_SRC_XOR_DST:         // D := ~S ^ D
      SetROP2((HDC)dc,R2_NOTXORPEN); // Is this the right one?
      break;
    case BLT_NOT_DST:                 // D := ~D
      SetROP2((HDC)dc,R2_NOT); 
      break;
    case BLT_SRC_OR_NOT_DST:          // D := S | ~D
      SetROP2((HDC)dc,R2_MERGEPENNOT); 
      break;
    case BLT_NOT_SRC:                 // D := ~S
      SetROP2((HDC)dc,R2_NOTCOPYPEN); 
      break;
    case BLT_NOT_SRC_OR_DST:          // D := ~S | D
      SetROP2((HDC)dc,R2_MERGENOTPEN); 
      break;
    case BLT_NOT_SRC_OR_NOT_DST:      // D := ~S | ~D  ==  ~(S & D)
      SetROP2((HDC)dc,R2_NOTMASKPEN); 
      break;
    case BLT_SET:                     // D := 1
      SetROP2((HDC)dc,R2_WHITE); 
      break;
    default:
      FXASSERT(FALSE);
    }
  }


// Set tile image
void FXDCWindow::setTile(FXImage* image,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setTile()");
  if(!surface){ fxerror("FXDCWindow::setTile: DC not connected to drawable.\n"); }
  tile=image;
  tx=dx;
  ty=dy;
  }

////////////// FIXME: likely not a good idea to hang on to the FXImage; the FOX object may go
//////////////        away even though the handle may hang around...

// Set stipple pattern
void FXDCWindow::setStipple(FXBitmap* bitmap,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setStipple()");
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  stipple=bitmap;
  pattern=STIPPLE_NONE;
  needsNewBrush=TRUE;
  tx=dx;
  ty=dy;
  }


void FXDCWindow::setStipple(FXStipplePattern pat,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setStipple()");
  if(!surface){ fxerror("FXDCWindow::setStipple: DC not connected to drawable.\n"); }
  stipple=NULL;
  pattern=pat;
  needsNewBrush=TRUE;
  tx=dx;
  ty=dy;
  }

////////////// FIXME: likely not a good idea to hang on to the FXBitmap; the FOX object may go
//////////////        away even though the handle may hang around...


void FXDCWindow::setClipRegion(const FXRegion& region){
  if(!surface){ fxerror("FXDCWindow::setClipRegion: DC not connected to drawable.\n"); }
  SelectClipRgn((HDC)dc,(HRGN)region.region);
  }


void FXDCWindow::setClipRectangle(FXint x,FXint y,FXint w,FXint h){
  FXPROFILE("FXDCWindow::setClipRectangle()");
  if(!surface){ fxerror("FXDCWindow::setClipRectangle: DC not connected to drawable.\n"); }
  clip.x=FXMAX(x,rect.x);
  clip.y=FXMAX(y,rect.y);
  clip.w=FXMIN(x+w,rect.x+rect.w)-clip.x;
  clip.h=FXMIN(y+h,rect.y+rect.h)-clip.y;
  if(clip.w<=0) clip.w=0;
  if(clip.h<=0) clip.h=0;
  HRGN hrgn=CreateRectRgn(clip.x,clip.y,clip.x+clip.w,clip.y+clip.h);
  SelectClipRgn((HDC)dc,hrgn);
  DeleteObject(hrgn);
  }
 

void FXDCWindow::clearClipRectangle(){
  FXPROFILE("FXDCWindow::clearClipRectangle()");
  if(!surface){ fxerror("FXDCWindow::clearClipRectangle: DC not connected to drawable.\n"); }
  clip=rect;
  HRGN hrgn=CreateRectRgn(clip.x,clip.y,clip.x+clip.w,clip.y+clip.h);
  SelectClipRgn((HDC)dc,hrgn);
  DeleteObject(hrgn);
  }


void FXDCWindow::setClipMask(FXBitmap* bitmap,FXint dx,FXint dy){
  FXPROFILE("FXDCWindow::setClipMask()");
  if(!surface){ fxerror("FXDCWindow::setClipMask: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  mask=bitmap;
  cx=dx;
  cy=dy;
  }


void FXDCWindow::clearClipMask(){
  FXPROFILE("FXDCWindow::clearClipMask()");
  if(!surface){ fxerror("FXDCWindow::clearClipMask: DC not connected to drawable.\n"); }
  FXASSERT(FALSE);
  mask=NULL;
  cx=0;
  cy=0;
  }

  
void FXDCWindow::setTextFont(FXFont *fnt){
  FXPROFILE("FXDCWindow::setTextFont()");
  if(!surface){ fxerror("FXDCWindow::setTextFont: DC not connected to drawable.\n"); }
  if(!fnt || !fnt->id()){ fxerror("FXDCWindow::setTextFont: illegal or NULL font specified.\n"); }
  SelectObject((HDC)dc,fnt->id());
  font=fnt;
  }



// Window will clip against child windows
void FXDCWindow::clipChildren(FXbool yes){ 
  FXPROFILE("FXDCWindow::clipChildren()");
  if(!surface){ fxerror("FXDCWindow::clipChildren: window has not yet been created.\n"); }
  DWORD    dwFlags=GetWindowLong((HWND)surface->id(),GWL_STYLE);
  HPEN     hPen;
  HBRUSH   hBrush;
  HFONT    hFont;
//  RECT     cliprect;
  COLORREF textcolor;
  COLORREF backcolor;
  FXint    fillmode;
  if(yes){
    if(!(dwFlags&WS_CLIPCHILDREN)){
      hPen=(HPEN)SelectObject((HDC)dc,GetStockObject(NULL_PEN));
      hBrush=(HBRUSH)SelectObject((HDC)dc,GetStockObject(NULL_BRUSH));
      hFont=(HFONT)SelectObject((HDC)dc,GetStockObject(SYSTEM_FONT));
      textcolor=GetTextColor((HDC)dc);
      backcolor=GetBkColor((HDC)dc);
      fillmode=GetPolyFillMode((HDC)dc);
      //GetClipBox(dc,&cr);

      ReleaseDC((HWND)surface->id(),(HDC)dc);
      SetWindowLong((HWND)surface->id(),GWL_STYLE,dwFlags|WS_CLIPCHILDREN);
      dc=GetDC((HWND)surface->id());

      SelectObject((HDC)dc,hFont);
      SelectObject((HDC)dc,hPen);
      SelectObject((HDC)dc,hBrush);
  /*
  if(cr.left!=0 || cr.top!=0 || cr.right!=0 || cr.bottom!=0){
    HRGN hClipRgn=CreateRectRgn(cr.left,cr.top,cr.right,cr.bottom);
    ExtSelectClipRgn((HDC)dc,hClipRgn,RGN_COPY);
    DeleteObject(hClipRgn);
    }
  */
      if(visual->hPalette){
        SelectPalette((HDC)dc,(HPALETTE)visual->hPalette,FALSE);
        RealizePalette((HDC)dc);
        }
      SetTextAlign((HDC)dc,TA_BASELINE|TA_LEFT);
      SetTextColor((HDC)dc,textcolor);
      SetBkColor((HDC)dc,backcolor);
      SetPolyFillMode((HDC)dc,fillmode);
      }
    }
  else{
    if(dwFlags&WS_CLIPCHILDREN){
      hPen=(HPEN)SelectObject((HDC)dc,GetStockObject(NULL_PEN));
      hBrush=(HBRUSH)SelectObject((HDC)dc,GetStockObject(NULL_BRUSH));
      hFont=(HFONT)SelectObject((HDC)dc,GetStockObject(SYSTEM_FONT));
      textcolor=GetTextColor((HDC)dc);
      backcolor=GetBkColor((HDC)dc);
      fillmode=GetPolyFillMode((HDC)dc);
      //GetClipBox(dc,&cr);

      ReleaseDC((HWND)surface->id(),(HDC)dc);
      SetWindowLong((HWND)surface->id(),GWL_STYLE,dwFlags & ~WS_CLIPCHILDREN);
      dc=GetDC((HWND)surface->id());

      SelectObject((HDC)dc,hFont);
      SelectObject((HDC)dc,hPen);
      SelectObject((HDC)dc,hBrush);
  /*
  if(cr.left!=0 || cr.top!=0 || cr.right!=0 || cr.bottom!=0){
    HRGN hClipRgn=CreateRectRgn(cr.left,cr.top,cr.right,cr.bottom);
    ExtSelectClipRgn((HDC)dc,hClipRgn,RGN_COPY);
    DeleteObject(hClipRgn);
    }
  */
      if(visual->hPalette){
        SelectPalette((HDC)dc,(HPALETTE)visual->hPalette,FALSE);
        RealizePalette((HDC)dc);
        }
      SetTextAlign((HDC)dc,TA_BASELINE|TA_LEFT);
      SetTextColor((HDC)dc,textcolor);
      SetBkColor((HDC)dc,backcolor);
      SetPolyFillMode((HDC)dc,fillmode);
      }
    }
  }

#endif
