/********************************************************************************
*                                                                               *
*                     A r r o w   B u t t o n    O b j e c t                    *
*                                                                               *
*********************************************************************************
* Copyright (C) 1998 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXArrowButton.cpp,v 1.8 2000/02/28 17:26:36 jeroen Exp $                 *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXString.h"
#include "FXObject.h"
#include "FXDict.h"
#include "FXRegistry.h"
#include "FXAccelTable.h"
#include "FXApp.h"
#include "FXId.h"
#include "FXDC.h"
#include "FXDCWindow.h"
#include "FXFont.h"
#include "FXDrawable.h"
#include "FXImage.h"
#include "FXIcon.h"
#include "FXWindow.h"
#include "FXFrame.h"
#include "FXLabel.h"
#include "FXArrowButton.h"


// Justification
#define JUSTIFY_MASK    (JUSTIFY_HZ_APART|JUSTIFY_VT_APART)

// Arrow styles
#define ARROW_MASK  (ARROW_UP|ARROW_DOWN|ARROW_LEFT|ARROW_RIGHT|ARROW_REPEAT|ARROW_AUTOGRAY|ARROW_AUTOHIDE|ARROW_TOOLBAR)


/*******************************************************************************/


// Map
FXDEFMAP(FXArrowButton) FXArrowButtonMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXArrowButton::onPaint),
  FXMAPFUNC(SEL_UPDATE,0,FXArrowButton::onUpdate),
  FXMAPFUNC(SEL_ACTIVATE,0,FXArrowButton::onActivate),
  FXMAPFUNC(SEL_DEACTIVATE,0,FXArrowButton::onDeactivate),
  FXMAPFUNC(SEL_TIMEOUT,FXArrowButton::ID_REPEAT,FXArrowButton::onRepeat),
  FXMAPFUNC(SEL_UNGRABBED,0,FXArrowButton::onUngrabbed),
  FXMAPFUNC(SEL_ENTER,0,FXArrowButton::onEnter),
  FXMAPFUNC(SEL_LEAVE,0,FXArrowButton::onLeave),
  FXMAPFUNC(SEL_CLICKED,0,FXArrowButton::onClicked),
  };


// Object implementation
FXIMPLEMENT(FXArrowButton,FXFrame,FXArrowButtonMap,ARRAYNUMBER(FXArrowButtonMap))


// For deserialization
FXArrowButton::FXArrowButton(){
  flags|=FLAG_ENABLED;
  arrowColor=0;
  repeater=NULL;
  state=FALSE;
  fired=FALSE;
  }


// Make a text button
FXArrowButton::FXArrowButton(FXComposite* p,FXObject* tgt,FXSelector sel,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb):
  FXFrame(p,opts,x,y,w,h,pl,pr,pt,pb){
  flags|=FLAG_ENABLED;
  target=tgt;
  message=sel;
  arrowColor=getApp()->foreColor;
  repeater=NULL;
  state=FALSE;
  fired=FALSE;
  }


// Get default size
FXint FXArrowButton::getDefaultWidth(){
  return padleft+padright+9+(border<<1);
  }


FXint FXArrowButton::getDefaultHeight(){
  return padtop+padbottom+9+(border<<1);
  }


// Enable the window
void FXArrowButton::enable(){
  if(!(flags&FLAG_ENABLED)){
    FXFrame::enable();
    update();
    }
  }


// Disable the window
void FXArrowButton::disable(){
  if(flags&FLAG_ENABLED){
    FXFrame::disable();
    update();
    }
  }


// Set button state
void FXArrowButton::setState(FXbool s){
  if(state!=s){
    state=s;
    update();
    }
  }


// If window can have focus
FXbool FXArrowButton::canFocus() const { return 1; }


// Implement auto-hide or auto-gray modes
long FXArrowButton::onUpdate(FXObject* sender,FXSelector sel,void* ptr){
  if(!FXFrame::onUpdate(sender,sel,ptr)){
    if(options&ARROW_AUTOHIDE){if(shown()){hide();recalc();}}
    if(options&ARROW_AUTOGRAY){disable();}
    }
  return 1;
  }


// Handle repaint 
long FXArrowButton::onPaint(FXObject*,FXSelector,void* ptr){
  FXEvent   *ev=(FXEvent*)ptr;
  FXDCWindow dc(this,ev);
  FXPoint    points[3];
  FXint      xx,yy,ww,hh,q;
  

  // With borders
  if(options&(FRAME_RAISED|FRAME_SUNKEN)){

    // Toolbar style
    if(options&ARROW_TOOLBAR){
      
      // Enabled and cursor inside, and up
      if(isEnabled() && underCursor() && !state){
        dc.setForeground(backColor);
        dc.fillRectangle(border,border,width-border*2,height-border*2);
        if(options&FRAME_THICK) drawDoubleRaisedRectangle(dc,0,0,width,height);
        else drawRaisedRectangle(dc,0,0,width,height);
        }
      
      // Enabled and cursor inside and down
      else if(isEnabled() && state){
        dc.setForeground(hiliteColor);
        dc.fillRectangle(border,border,width-border*2,height-border*2);
        if(options&FRAME_THICK) drawDoubleSunkenRectangle(dc,0,0,width,height);
        else drawSunkenRectangle(dc,0,0,width,height);
        }
              
      // Disabled or unchecked or not under cursor
      else{
        dc.setForeground(backColor);
        dc.fillRectangle(0,0,width,height);
        }
      }
    
    // Normal style
    else{
      
      // Draw sunken if enabled and pressed
      if(isEnabled() && state){
        dc.setForeground(hiliteColor);
        dc.fillRectangle(border,border,width-border*2,height-border*2);
        if(options&FRAME_THICK) drawDoubleSunkenRectangle(dc,0,0,width,height);
        else drawSunkenRectangle(dc,0,0,width,height);
        }
      
      // Draw in up state if disabled or up
      else{
        dc.setForeground(backColor);
        dc.fillRectangle(border,border,width-border*2,height-border*2);
        if(options&FRAME_THICK) drawDoubleRaisedRectangle(dc,0,0,width,height);
        else drawRaisedRectangle(dc,0,0,width,height);
        }
      }
    }
  
  // No borders
  else{
    if(isEnabled() && state){
      dc.setForeground(hiliteColor);
      dc.fillRectangle(0,0,width,height);
      }
    else{
      dc.setForeground(backColor);
      dc.fillRectangle(0,0,width,height);
      }
    }

  // Compute size of the arrows....
  ww=width-padleft-padright-(border<<1);
  hh=height-padtop-padbottom-(border<<1);
  if(options&(ARROW_UP|ARROW_DOWN)){
    q=ww|1; if(q>(hh<<1)) q=(hh<<1)-1;
    ww=q; hh=q>>1;
    } 
  else{
    q=hh|1; if(q>(ww<<1)) q=(ww<<1)-1;
    ww=q>>1; hh=q;
    }

  if(options&JUSTIFY_LEFT) xx=padleft+border;
  else if(options&JUSTIFY_RIGHT) xx=width-ww-padright-border;
  else xx=(width-ww)/2;

  if(options&JUSTIFY_TOP) yy=padtop+border;
  else if(options&JUSTIFY_BOTTOM) yy=height-hh-padbottom-border;
  else yy=(height-hh)/2;

  if(state){ ++xx; ++yy; }

  if(isEnabled())
    dc.setForeground(arrowColor);
  else
    dc.setForeground(shadowColor);

  // NB Size of arrow should stretch
  if(options&ARROW_UP){
    points[0].x=xx+(ww>>1);
    points[0].y=yy-1;
    points[1].x=xx;
    points[1].y=yy+hh;
    points[2].x=xx+ww;
    points[2].y=yy+hh;
    dc.fillPolygon(points,3);
    }
  else if(options&ARROW_DOWN){
    points[0].x=xx+1;
    points[0].y=yy;
    points[1].x=xx+ww-1;
    points[1].y=yy;
    points[2].x=xx+(ww>>1);
    points[2].y=yy+hh;
    dc.fillPolygon(points,3);
    }
  else if(options&ARROW_LEFT){
    points[0].x=xx+ww;
    points[0].y=yy;
    points[1].x=xx+ww;
    points[1].y=yy+hh-1;
    points[2].x=xx;
    points[2].y=yy+(hh>>1);
    dc.fillPolygon(points,3);
    }
  else if(options&ARROW_RIGHT){
    points[0].x=xx;
    points[0].y=yy;
    points[1].x=xx;
    points[1].y=yy+hh-1;
    points[2].x=xx+ww;
    points[2].y=yy+(hh>>1);
    dc.fillPolygon(points,3);
    }
  return 1;
  }


// The FXArrowButton lost the grab for some reason
long FXArrowButton::onUngrabbed(FXObject* sender,FXSelector sel,void* ptr){
  FXFrame::onUngrabbed(sender,sel,ptr);
  flags&=~FLAG_PRESSED;
  flags|=FLAG_UPDATE;
  setState(FALSE);
  return 1;
  }


// Button being activated
long FXArrowButton::onActivate(FXObject*,FXSelector,void*){
  flags|=FLAG_PRESSED;
  flags&=~FLAG_UPDATE;
  setState(TRUE);
  if(options&ARROW_REPEAT){ repeater=getApp()->addTimeout(getApp()->scrollSpeed,this,ID_REPEAT); }
  fired=FALSE;
  return 1;
  }
  

// Button being deactivated
long FXArrowButton::onDeactivate(FXObject*,FXSelector,void*){
  FXuint click=state;
  flags&=~FLAG_PRESSED;
  flags|=FLAG_UPDATE;
  setState(FALSE);
  if(repeater) getApp()->removeTimeout(repeater);
  if(!fired){ handle(this,MKUINT(0,SEL_CLICKED),(void*)click); }
  fired=FALSE;
  return 1;
  }


// Repeat a click automatically
long FXArrowButton::onRepeat(FXObject*,FXSelector,void*){
  FXuint click=state;
  handle(this,MKUINT(0,SEL_CLICKED),(void*)click);
  repeater=getApp()->addTimeout(getApp()->scrollSpeed,this,ID_REPEAT);
  fired=TRUE;
  return 1;
  }


// Clicked on button
long FXArrowButton::onClicked(FXObject*,FXSelector,void* ptr){
  if(target){
    if(target->handle(this,MKUINT(message,SEL_CLICKED),ptr)) return 1;
    if(ptr && target->handle(this,MKUINT(message,SEL_COMMAND),ptr)) return 1;
    }
  return 0;
  }
 

// Entered button
long FXArrowButton::onEnter(FXObject* sender,FXSelector sel,void* ptr){
  FXFrame::onEnter(sender,sel,ptr);
  if(isEnabled()){
    if(flags&FLAG_PRESSED) setState(TRUE);
    if(options&ARROW_TOOLBAR) update();
    }
  return 1;
  }


// Left button
long FXArrowButton::onLeave(FXObject* sender,FXSelector sel,void* ptr){
  FXFrame::onLeave(sender,sel,ptr);
  if(isEnabled()){
    if(flags&FLAG_PRESSED) setState(FALSE);
    if(options&ARROW_TOOLBAR) update();
    }
  return 1;
  }


// Set arrow style
void FXArrowButton::setArrowStyle(FXuint style){
  FXuint opts=(options&~ARROW_MASK) | (style&ARROW_MASK);
  if(options!=opts){
    options=opts;
    update();
    }
  }


// Get arrow style
FXuint FXArrowButton::getArrowStyle() const { 
  return (options&ARROW_MASK); 
  }


// Set text color
void FXArrowButton::setArrowColor(FXColor clr){
  if(clr!=arrowColor){
    arrowColor=clr;
    update();
    }
  }


// Set text justify style
void FXArrowButton::setJustify(FXuint style){
  FXuint opts=(options&~JUSTIFY_MASK) | (style&JUSTIFY_MASK);
  if(options!=opts){
    options=opts;
    update();
    }
  }


// Get text justify style
FXuint FXArrowButton::getJustify() const { 
  return (options&JUSTIFY_MASK); 
  }


// Kill the timer
FXArrowButton::~FXArrowButton(){
  if(repeater) getApp()->removeTimeout(repeater);
  repeater=(FXTimer*)-1;
  }
