/********************************************************************************
*                                                                               *
*                           S t r i n g   O b j e c t                           *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXString.h,v 1.17 2000/03/16 18:03:16 jeroen Exp $                       *
********************************************************************************/
#ifndef FXSTRING_H
#define FXSTRING_H



/**
  FXString provides essential string manipulation capabilities to FOX.
*/
class FXAPI FXString {
private:
  FXchar* str;
public:
  
  /// Create empty string
  FXString();
  
  /// Copy construct
  FXString(const FXString& s);

  /// Construct and init
  FXString(const FXchar* s);

  /// Construct and init with substring
  FXString(const FXchar* s,FXint n);

  /// Construct and fill with constant
  FXString(FXchar c,FXint n);
  
  /// Construct string from two parts
  FXString(const FXchar *s1,const FXchar* s2);
  
  /// Size to some desired capacity
  void size(FXint sz);

  /// Size of text data, 0 for empty string
  FXint size() const;

  /// Length of text
  FXint length() const { return strlen(str); }

  /// Get text contents
  const FXchar* text() const { return (const FXchar*)str; }

  /// See if string is empty
  FXbool empty() const { return str[0]==0; }

  /// Return a non-const reference to the ith character
  FXchar& operator[](FXint i){ return str[i]; }

  /// Return a const reference to the ith character
  const FXchar& operator[](FXint i) const { return str[i]; }

  /// Assign another string to this
  FXString& operator=(const FXString& s);

  /// Assign a C-style string to this
  FXString& operator=(const FXchar* s);

  /// Fill with a constant
  FXString& fill(FXchar c,FXint n);
  
  /// Fill up to current length
  FXString& fill(FXchar c);
  
  /// Convert to lower case
  FXString& lower();
  
  /// Convert to upper case
  FXString& upper();
  
  /// Extract partition of delimiter separated string
  FXString extract(FXint part,FXchar delim) const;

  /// Extract partition of delimiter separated string
  FXString extract(FXint part,FXchar delim,FXchar esc) const;

  /// Insert character at specified position
  FXString& insert(FXint pos,FXchar c);

  /// Insert string at specified position
  FXString& insert(FXint pos,const FXString& s);

  /// Insert string at specified position
  FXString& insert(FXint pos,const FXchar* s);

  /// Insert first n characters of string at specified position
  FXString& insert(FXint pos,const FXchar* s,FXint n);
  
  /// Prepend string with input character
  FXString& prepend(FXchar c);

  /// Prepend string with input string
  FXString& prepend(const FXString& s);

  /// Prepend string with input string
  FXString& prepend(const FXchar *s);

  /// Prepend string with first n characters of input string
  FXString& prepend(const FXchar *s,FXint n);
  
  /// Append input character to this string
  FXString& append(FXchar c);

  /// Append input string to this string
  FXString& append(const FXString& s);

  /// Append input string to this string
  FXString& append(const FXchar *s);

  /// Append first n characters of input string to this string
  FXString& append(const FXchar *s,FXint n);
  
  /// Replace a single character
  FXString& replace(FXint pos,FXchar c);

  /// Replace the m characters at pos with input string
  FXString& replace(FXint pos,FXint m,const FXString& s);

  /// Replace the m characters at pos with input string
  FXString& replace(FXint pos,FXint m,const FXchar *s);

  /// Replaces the m characters at pos with first n characters of input string
  FXString& replace(FXint pos,FXint m,const FXchar *s,FXint n);
  
  /// Remove substring
  FXString& remove(FXint pos,FXint n=1);
  
  /// Substitute one character by another
  FXString& substitute(FXchar orig,FXchar sub);

  /// Remove leading and trailing whitespace
  FXString& trim();
  
  /// Remove leading whitespace
  FXString& trimBegin();
  
  /// Remove trailing whitespace
  FXString& trimEnd();
  
  /// Truncate string at pos
  FXString& trunc(FXint pos);

  /// Clear
  FXString& clear();

  /// Get leftmost part
  FXString left(FXint n) const;
  
  /// Get rightmost part
  FXString right(FXint n) const;
  
  /// Get some part in the middle
  FXString mid(FXint pos,FXint n) const;
  
  /**
  * Return all characters before the n-th occurrence of ch,
  * counting from the beginning of the string if n>0, or
  * from the end if n<0.
  * If the character ch is not found, the entire string,
  * or the empty string, is returned, respectively.
  * A NULL string is returned if n==0.
  */
  FXString before(FXchar ch,FXint n=1) const;
  
  /**
  * Return all characters after the nth occurrence of ch,
  * counting from the beginning of the string if n>0, or from
  * the end if n<0.
  * If the character ch is not found, the empty string, or
  * the entire string, is returned, respectively.
  * A NULL string is returned if n==0.
  */
  FXString after(FXchar ch,FXint n=1) const;
  
  /// Find a character, searching forward; return position or -1
  FXint findf(FXchar c,FXint pos=0) const;
  
  /// Find a character, searching backward; return position or -1
  FXint findb(FXchar c,FXint pos=2147483647) const;
  
  /// Find a substring, searching forward; return position or -1
  FXint findstrf(const FXString &substr,FXint pos=0) const;
  
  /// Find a substring, searching backward; return position or -1
  FXint findstrb(const FXString &substr,FXint pos=2147483647) const;
  
  /// Find number of occurances of character in string
  FXint count(FXchar c) const;
  
  /// Format a string a-la printf
  FXString& format(const char *fmt,...);
  FXString& vformat(const char *fmt,va_list args);
  
  /// Get hash value
  FXint hash() const;
  
  /// Compare
  friend FXAPI FXint compare(const FXString &s1,const FXString &s2){ return strcmp(s1.str,s2.str); }
  friend FXAPI FXint compare(const FXString &s1,const FXchar *s2){ return strcmp(s1.str,s2); }
  friend FXAPI FXint compare(const FXchar *s1,const FXString &s2){ return strcmp(s1,s2.str); }

  /// Compare up to n
  friend FXAPI FXint compare(const FXString &s1,const FXString &s2,FXint n){ return strncmp(s1.str,s2.str,n); }
  friend FXAPI FXint compare(const FXString &s1,const FXchar *s2,FXint n){ return strncmp(s1.str,s2,n); }
  friend FXAPI FXint compare(const FXchar *s1,const FXString &s2,FXint n){ return strncmp(s1,s2.str,n); }

  /// Compare case insensitive
  friend FXAPI FXint comparecase(const FXString &s1,const FXString &s2);
  friend FXAPI FXint comparecase(const FXString &s1,const FXchar *s2);
  friend FXAPI FXint comparecase(const FXchar *s1,const FXString &s2);

  /// Compare case insensitive up to n
  friend FXAPI FXint comparecase(const FXString &s1,const FXString &s2,FXint n);
  friend FXAPI FXint comparecase(const FXString &s1,const FXchar *s2,FXint n);
  friend FXAPI FXint comparecase(const FXchar *s1,const FXString &s2,FXint n);

  /// Comparison operators
  friend FXAPI FXbool operator==(const FXString &s1,const FXString &s2){ return strcmp(s1.str,s2.str)==0; }
  friend FXAPI FXbool operator==(const FXString &s1,const FXchar *s2){ return strcmp(s1.str,s2)==0; }
  friend FXAPI FXbool operator==(const FXchar *s1,const FXString &s2){ return strcmp(s1,s2.str)==0; }
  
  friend FXAPI FXbool operator!=(const FXString &s1,const FXString &s2){ return strcmp(s1.str,s2.str)!=0; }
  friend FXAPI FXbool operator!=(const FXString &s1,const FXchar *s2){ return strcmp(s1.str,s2)!=0; }
  friend FXAPI FXbool operator!=(const FXchar *s1,const FXString &s2){ return strcmp(s1,s2.str)!=0; }
  
  friend FXAPI FXbool operator<(const FXString &s1,const FXString &s2){ return strcmp(s1.str,s2.str)<0; }
  friend FXAPI FXbool operator<(const FXString &s1,const FXchar *s2){ return strcmp(s1.str,s2)<0; }
  friend FXAPI FXbool operator<(const FXchar *s1,const FXString &s2){ return strcmp(s1,s2.str)<0; }
  
  friend FXAPI FXbool operator<=(const FXString &s1,const FXString &s2){ return strcmp(s1.str,s2.str)<=0; }
  friend FXAPI FXbool operator<=(const FXString &s1,const FXchar *s2){ return strcmp(s1.str,s2)<=0; }
  friend FXAPI FXbool operator<=(const FXchar *s1,const FXString &s2){ return strcmp(s1,s2.str)<=0; } 
  
  friend FXAPI FXbool operator>(const FXString &s1,const FXString &s2){ return strcmp(s1.str,s2.str)>0; }
  friend FXAPI FXbool operator>(const FXString &s1,const FXchar *s2){ return strcmp(s1.str,s2)>0; }
  friend FXAPI FXbool operator>(const FXchar *s1,const FXString &s2){ return strcmp(s1,s2.str)>0; }
  
  friend FXAPI FXbool operator>=(const FXString &s1,const FXString &s2){ return strcmp(s1.str,s2.str)>=0; }
  friend FXAPI FXbool operator>=(const FXString &s1,const FXchar *s2){ return strcmp(s1.str,s2)>=0; }
  friend FXAPI FXbool operator>=(const FXchar *s1,const FXString &s2){ return strcmp(s1,s2.str)>=0; }

  /// Append operators
  FXString& operator+=(const FXString& s);
  FXString& operator+=(const FXchar* s);
  FXString& operator+=(FXchar c);

  /// Concatenate two strings
  friend FXAPI FXString operator+(const FXString& s1,const FXString& s2);
  friend FXAPI FXString operator+(const FXString& s1,const FXchar* s2);
  friend FXAPI FXString operator+(const FXchar* s1,const FXString& s2);

  /// Saving to a stream
  friend FXAPI FXStream& operator<<(FXStream& store,const FXString& s);

  /// Load from a stream
  friend FXAPI FXStream& operator>>(FXStream& store,FXString& s);

  /// Format a string a-la printf
  friend FXAPI FXString FXStringFormat(const FXchar *fmt,...);
  friend FXAPI FXString FXStringVFormat(const FXchar *fmt,va_list args);
  
  /// Convert number to a string
  friend FXAPI FXString FXStringVal(FXint num,FXint base=10);
  friend FXAPI FXString FXStringVal(FXuint num,FXint base=10);
  friend FXAPI FXString FXStringVal(FXfloat num,FXint prec=6,FXExponent exp=EXP_AUTO);
  friend FXAPI FXString FXStringVal(FXdouble num,FXint prec=6,FXExponent exp=EXP_AUTO);
  
  /// Convert string to a number
  friend FXAPI FXint FXIntVal(const FXString& s,FXint base=10);
  friend FXAPI FXuint FXUIntVal(const FXString& s,FXint base=10);
  friend FXAPI FXfloat FXFloatVal(const FXString& s);
  friend FXAPI FXdouble FXDoubleVal(const FXString& s);

  /// Delete
 ~FXString();
  };


#endif
